/*
 * phonet.h
 * --------
 *
 * Phonetic rules for "phonet.c".
 *
 * Copyright (c):
 * 1999-2007:  Joerg MICHAEL, Adalbert-Stifter-Str. 11, 30655 Hannover, Germany
 *     and
 * (version 1.0) 1999:  Heise Verlag, Helstorfer Str. 7, 30625 Hannover, Germany
 *
 * SCCS: @(#) phonet.h  1.4.2  2007-08-27
 *
 * This program is subject to the GNU Lesser General Public License (LGPL)
 * (formerly known as GNU Library General Public Licence)
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Actually, the LGPL is __less__ restrictive than the better known GNU General
 * Public License (GPL). See the GNU Library General Public License or the file
 * LIB_GPLP.TXT for more details and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * There is one important restriction: If you modify this program in any way
 * (e.g. add or change phonetic rules or modify the underlying logic or
 * translate this program into another programming language), you must also
 * release the changes under the terms of the LGPL.
 * That means you have to give out the source code to your changes,
 * and a very good way to do so is mailing them to the address given below.
 * I think this is the best way to promote further development and use
 * of this software.
 *
 * If you have any remarks, feel free to e-mail to:
 *     ct@ct.heise.de
 *
 * The author's email address is:
 *    astro.joerg@googlemail.com
 */


#ifndef _PHONET_H_
#define _PHONET_H_

#include <stdio.h>
#include <stdlib.h>

#include "ph_ext.h"


#define  PHONET_VERSION  "version 1.4.2, 2007-08-27"

/****  we assume 1 byte chars  ****/
#define  HASH_COUNT    256

/****  macro to make the rules easier readable  ****/
#define  null  NULL


/****  end marker for string table  ****/
static char PHONET_END[] = "";

static char *phonet_language = "unknown";
static int  *phonet_init = NULL;
static int  *phonet_hash = NULL;
static int  (*phonet_hash_1)[26][28] = NULL;   /***  these are pointers  ***/
static int  (*phonet_hash_2)[26][28] = NULL;   /***  to  array[26][28]   ***/
static char **phonet_rules = NULL;


/****  Syntax for phonetic rules:               ****/
/****    <search_string>  <1st_rule> <2nd_rule> ****/
/****  Syntax for search strings:               ****/
/****    <word> [<->..] [<] [<0-9>] [^[^]] [$]  ****/

/****  Remarks:                                 ****/
/****  a) All phonetic rules must be written    ****/
/****     in upper case.                        ****/
/****  b) The end of <word> may contain as an   ****/
/****     optional simple regular expression    ****/
/****     one array of letters (or umlaut's)    ****/
/****     enclosed in '(' and ')'.              ****/
/****  c) Rules with a '<' demand that the      ****/
/****     replacement string may not be longer  ****/
/****     than the search string.               ****/
/****  d) The placement of rules determines     ****/
/****     their priority. Therefore, the rules  ****/
/****     for "SH" must be placed before the    ****/
/****     rules for "S"                         ****/
/****     (otherwise, a conversion error will   ****/
/****      occur for "SH").                     ****/



#ifdef PHONET_ENGLISH

/*
 *
 * UNDER CONSTRUCTION
 * (no rules available at this time)
 *
 */

#endif    /****  PHONET_ENGLISH  ****/




#ifdef PHONET_NO_LANGUAGE

static char *phonet_language_no_language = "no language";
static int  phonet_init_no_language = 0;
static int  phonet_hash_no_language [HASH_COUNT];
static int  phonet_hash_no_language_1 [26][28];
static int  phonet_hash_no_language_2 [26][28];

/****  Phonetic rules for "no language"   ****/
/****  (i.e. umlauts will be "expanded")  ****/

static char *phonet_rules_no_language[] =
  {
/****  separator chars  ****/
    "",               " ",       " ",
    "'",               " ",       " ",
    "`$",              "",        "",
    "\"",              " ",       " ",
    ",",               ",",       ",",
    ";",               ",",       ",",
    "-",               " ",       " ",
    " ",               " ",       " ",
    ".",               ".",       ".",
    ":",               ".",       ".",
/****  German umlauts  ****/
    "",               "AE",      "AE",
    "",               "OE",      "OE",
    "",               "UE",      "UE",
    "",               "S",       "S",
/****  international umlauts  ****/
    "",               "A",       "A",
    "",               "A",       "A",
    "",               "A",       "A",
    "",               "A",       "A",
    "",               "A",       "A",
    "",               "AE",      "AE",
    "",               "C",       "C",
    "",               "DJ",      "DJ",
    "",               "E",       "E",
    "",               "E",       "E",
    "",               "E",       "E",
    "",               "E",       "E",
    "",               "I",       "I",
    "",               "I",       "I",
    "",               "I",       "I",
    "",               "I",       "I",
    "",               "NH",      "NH",
    "",               "O",       "O",
    "",               "O",       "O",
    "",               "O",       "O",
    "",               "O",       "O",
    "",               "OE",      "OE",
    "",               "OE",      "OE",
    "",               "SH",      "SH",
    "",               "TH",      "TH",
    "",               "U",       "U",
    "",               "U",       "U",
    "",               "U",       "U",
    "",               "Y",       "Y",
    "",               "Y",       "Y",
/****  "normal" letters (A-Z)  ****/
    "MC^",             "MAC",     "MAC",
    "MC^",             "MAC",     "MAC",
    "M^",             "MAC",     "MAC",
    "M'^",             "MAC",     "MAC",
    "O^",             "O",       "O",
    "O'^",             "O",       "O",
    "VAN DEN ^",      "VANDEN",   "VANDEN",
    PHONET_END,      PHONET_END,  PHONET_END
  };

#endif    /****  PHONET_NO_LANGUAGE  ****/




#ifdef PHONET_GERMAN

static char *phonet_language_german = "German";
static int  phonet_init_german = 0;
static int  phonet_hash_german [HASH_COUNT];
static int  phonet_hash_german_1 [26][28];
static int  phonet_hash_german_2 [26][28];

/****  Phonetic rules for German.     ****/
/****  First Rules:                   ****/
/****    The letters ,,C,J,Q,V,Y    ****/
/****    will "disappear".            ****/
/****  Second Rules (less accurate):  ****/
/****    The letters ,,,C,D,       ****/
/****    G,H,J,O,P,Q,S,V,W,X,Y        ****/
/****    will "disappear".            ****/

static char *phonet_rules_german[] =
  {
/****  separator chars  ****/
    "",               " ",       " ",
    "'",               " ",       " ",
    "`$",              "",        "",
    "\"",              " ",       " ",
    ",",               " ",       " ",
    ";",               " ",       " ",
    "-",               " ",       " ",
    " ",               " ",       " ",
    ".",               ".",       ".",
    ":",               ".",       ".",
/****  German umlauts  ****/
    "E",              "E",       "E",
    "U<",             "EU",      "EU",
    "V(AEOU)-<",      "EW",      null,
    "$",              "",       null,
    "<",              null,      "E",
    "",               "E",       null,
    "E",              "",       "",
    "U",              "",       "",
    "VER--<",         "W",      null,
    "V(AOU)-",        "W",      null,
    "BEL(GNRW)-^^",   "BL ",    "IBL ",
    "BER^^",          "BA",     "IBA",
    "E",              "",       "I",
    "VER--<",         "W",      null,
    "V(AOU)-",        "W",      null,
    "",               null,      "I",
    "CH<",            null,      "Z",
    "<",              "S",       "Z",
/****  international umlauts  ****/
    "<",              "A",       "A",
    "<",              "A",       "A",
    "<",              "A",       "A",
    "<",              "A",       "A",
    "<",              "A",       "A",
    "ER-",            "E",       "E",
    "U<",             "EU",      "EU",
    "V(AEOU)-<",      "EW",      null,
    "$",              "",       null,
    "<",              null,      "E",
    "",               "E",       null,
    "",               "Z",       "Z",
    "-",             "",        "",
    "",               "DI",      "TI",
    "<",              "E",       "E",
    "<",              "E",       "E",
    "<",              "E",       "E",
    "",               "E",       "E",
    "<",              "I",       "I",
    "<",              "I",       "I",
    "<",              "I",       "I",
    "",               "I",       "I",
    "-",             "",        "",
    "",               "NI",      "NI",
    "<",              "O",       "U",
    "<",              "O",       "U",
    "<",              "O",       "U",
    "<",              "O",       "U",
    "<",              "",       "",
    "(IJY)-<",        "E",       "E",
    "<",              "",       "",
    "",               "SH",      "Z",
    "",               "T",       "T",
    "<",              "U",       "U",
    "<",              "U",       "U",
    "<",              "U",       "U",
    "<",              "I",       "I",
    "<",              "I",       "I",
/****  "normal" letters (A-Z)  ****/
    "ABELLE$",         "ABL",     "ABL",
    "ABELL$",          "ABL",     "ABL",
    "ABIENNE$",        "ABIN",    "ABIN",
    "ACHME---^",       "ACH",     "AK",
    "ACEY$",           "AZI",     "AZI",
    "ADV",             "ATW",     null,
    "AEGL-",           "EK",      null,
    "AEU<",            "EU",      "EU",
    "AE2",             "E",       "E",
    "AFTRAUBEN------", "AFT ",    "AFT ",
    "AGL-1",           "AK",      null,
    "AGNI-^",          "AKN",     "AKN",
    "AGNIE-",          "ANI",     "ANI",
    "AGN(AEOU)-$",     "ANI",     "ANI",
    "AH(AIOUY)-",    "AH",      null,
    "AIA2",            "AIA",     "AIA",
    "AIE$",            "E",       "E",
    "AILL(EOU)-",      "ALI",     "ALI",
    "AINE$",           "EN",      "EN",
    "AIRE$",           "ER",      "ER",
    "AIR-",            "E",       "E",
    "AISE$",           "ES",      "EZ",
    "AISSANCE$",       "ESANS",   "EZANZ",
    "AISSE$",          "ES",      "EZ",
    "AIX$",            "EX",      "EX",
    "AJ(AEIOU)--", "A",     "A",
    "AKTIE",           "AXIE",    "AXIE",
    "AKTUEL",          "AKTUEL",  null,
    "ALOI^",           "ALOI",    "ALUI",  /** Do NOT merge these rules  **/
    "ALOY^",           "ALOI",    "ALUI",  /** (needed by "check_rules") **/
    "AMATEU(RS)-",     "AMAT",   "ANAT",
    "ANCH(OEI)-",      "ANSH",    "ANZ",
    "ANDERGEGANG----", "ANDA GE", "ANTA KE",
    "ANDERGEHE----",   "ANDA ",   "ANTA ",
    "ANDERGESETZ----", "ANDA GE", "ANTA KE",
    "ANDERGING----",   "ANDA ",   "ANTA ",
    "ANDERSETZ(ET)-----", "ANDA ", "ANTA ",
    "ANDERZUGEHE----", "ANDA ZU ", "ANTA ZU ",
    "ANDERZUSETZE-----", "ANDA ZU ", "ANTA ZU ",
    "ANER(BKO)---^^",  "AN",      null,
    "ANHAND---^$",     "AN H",    "AN ",
    "ANH(AEIOUY)--^^", "AN",   null,
    "ANIELLE$",        "ANIEL",   "ANIL",
    "ANIEL",           "ANIEL",   null,
    "ANSTELLE----^$",  "AN ST",   "AN ZT",
    "ANTI^^",          "ANTI",    "ANTI",
    "ANVER^^",         "ANFA",    "ANFA",
    "ATIA$",           "ATIA",    "ATIA",
    "ATIA(NS)--",      "ATI",     "ATI",
    "ATI(AOU)-",    "AZI",     "AZI",
    "AUAU--",          "",        "",
    "AUERE$",          "AUERE",   null,
    "AUERE(NS)-$",     "AUERE",   null,
    "AUERE(AIOUY)--",  "AUER",    null,
    "AUER(AIOUY)-", "AUER",    null,
    "AUER<",           "AUA",     "AUA",
    "AUF^^",           "AUF",     "AUF",
    "AULT$",           "O",       "U",
    "AUR(BCDFGKLMNQSTVWZ)-", "AUA", "AUA",
    "AUR$",            "AUA",     "AUA",
    "AUSSE$",          "OS",      "UZ",
    "AUS(ST)-^",       "AUS",     "AUS",
    "AUS^^",           "AUS",     "AUS",
    "AUTOFAHR----",    "AUTO ",   "AUTU ",
    "AUTO^^",          "AUTO",    "AUTU",
    "AUX(IY)-",        "AUX",     "AUX",
    "AUX",             "O",       "U",
    "AU",              "AU",      "AU",
    "AVER--<",         "AW",      null,
    "AVIER$",          "AWIE",    "AFIE",
    "AV(EI)-^",     "AW",      null,
    "AV(AOU)-",        "AW",      null,
    "AYRE$",           "EIRE",    "EIRE",
    "AYRE(NS)-$",      "EIRE",    "EIRE",
    "AYRE(AIOUY)--",   "EIR",     "EIR",
    "AYR(AIOUY)-",  "EIR",     "EIR",
    "AYR<",            "EIA",     "EIA",
    "AYER--<",         "EI",      "EI",
    "AY(AEIOUY)--", "A",       "A",
    "A",              "E",       "E",
    "A(IJY)<",         "EI",      "EI",
    "BABY^$",          "BEBI",    "BEBI",
    "BAB(IY)^",        "BEBI",    "BEBI",
    "BEAU^$",          "BO",      null,
    "BEA(BCMNRU)-^",   "BEA",     "BEA",
    "BEAT(AEIMORU)-^", "BEAT",    "BEAT",
    "BEE$",            "BI",      "BI",
    "BEIGE^$",         "BESH",    "BEZ",
    "BENOIT--",        "BENO",    "BENU",
    "BER(DT)-",        "BER",     null,
    "BERN(DT)-",       "BERN",    null,
    "BE(LMNRST)-^",    "BE",      "BE",
    "BETTE$",          "BET",     "BET",
    "BEVOR^$",         "BEFOR",   null,
    "BIC$",            "BIZ",     "BIZ",
    "BOWL(EI)-",       "BOL",     "BUL",
    "BP(AEIORUY)-", "B", "B",
    "BRINGEND-----^",  "BRI",     "BRI",
    "BRINGEND-----",   " BRI",    " BRI",
    "BROW(NS)-",       "BRAU",    "BRAU",
    "BUDGET7",         "BGE",    "BIKE",
    "BUFFET7",         "BFE",    "BIFE",
    "BYLLE$",          "BILE",    "BILE",
    "BYLL$",           "BIL",     "BIL",
    "BYPA--^",         "BEI",     "BEI",
    "BYTE<",           "BEIT",    "BEIT",
    "BY9^",            "B",      null,
    "B(SZ)$",         "BS",      null,
    "CACH(EI)-^",      "KESH",    "KEZ",
    "CAE--",           "Z",       "Z",
    "CA(IY)$",         "ZEI",     "ZEI",
    "CE(EIJUY)--",     "Z",       "Z",
    "CENT<",           "ZENT",    "ZENT",
    "CERST(EI)----^",  "KE",      "KE",
    "CER$",            "ZA",      "ZA",
    "CE3",             "ZE",      "ZE",
    "CH'S$",           "X",       "X",
    "CHS$",           "X",       "X",
    "CHAO(ST)-",       "KAO",     "KAU",
    "CHAMPIO-^",       "SHEMPI",  "ZENBI",
    "CHAR(AI)-^",      "KAR",     "KAR",
    "CHAU(CDFSVWXZ)-", "SHO",     "ZU",
    "CH(CF)-",        "SHE",     "ZE",
    "CHE(CF)-",        "SHE",     "ZE",
    "CHEM-^",          "KE",      "KE",    /**  or: "CHE", "KE" **/
    "CHEQUE<",         "SHEK",    "ZEK",
    "CHI(CFGPVW)-",    "SHI",     "ZI",
    "CH(AEUY)-<^",     "SH",      "Z",
    "CHK-",            "",        "",
    "CHO(CKPS)-^",     "SHO",     "ZU",
    "CHRIS-",          "KRI",     null,
    "CHRO-",           "KR",      null,
    "CH(LOR)-<^",      "K",       "K",
    "CHST-",           "X",       "X",
    "CH(SXZ)3",       "X",       "X",
    "CHTNI-3",         "CHN",     "KN",
    "CH^",             "K",       "K",     /**  or: "CH", "K"  **/
    "CH",              "CH",      "K",
    "CIC$",            "ZIZ",     "ZIZ",
    "CIENCEFICT----",  "EIENS ",  "EIENZ ",
    "CIENCE$",         "EIENS",   "EIENZ",
    "CIER$",           "ZIE",     "ZIE",
    "CYB-^",           "ZEI",     "ZEI",
    "CY9^",            "Z",      "ZI",
    "C(IJY)-<3",       "Z",       "Z",
    "CLOWN-",          "KLAU",    "KLAU",
    "CCH",             "Z",       "Z",
    "CCE-",            "X",       "X",
    "C(CK)-",          "",        "",
    "CLAUDET---",      "KLO",     "KLU",
    "CLAUDINE^$",      "KLODIN",  "KLUTIN",
    "COACH",           "KOSH",    "KUZ",
    "COLE$",           "KOL",     "KUL",
    "COUCH",           "KAUSH",   "KAUZ",
    "COW",             "KAU",     "KAU",
    "CQUES$",          "K",       "K",
    "CQUE",            "K",       "K",
    "CRASH--9",        "KRE",     "KRE",
    "CREAT-^",         "KREA",    "KREA",
    "CST",             "XT",      "XT",
    "CS<^",            "Z",       "Z",
    "C(SX)",          "X",       "X",
    "CT'S$",           "X",       "X",
    "CT(SXZ)",        "X",       "X",
    "CZ<",             "Z",       "Z",
    "C()3",     "Z",       "Z",
    "C.^",             "C.",      "C.",
    "C-",             "Z",       "Z",
    "C$",             "Z",      "ZI",
    "C'S$",            "X",       "X",
    "C<",              "K",       "K",
    "DAHER^$",         "DAHER",   null,
    "DARAUFFOLGE-----", "DARAUF ", "TARAUF ",
    "DAVO(NR)-^$",     "DAFO",    "TAFU",
    "DD(SZ)--<",       "",        "",
    "DD9",             "D",       null,
    "DEPOT7",          "DEPO",    "TEBU",
    "DESIGN",          "DISEIN",  "TIZEIN",
    "DE(LMNRST)-3^",   "DE",      "TE",
    "DETTE$",          "DET",     "TET",
    "DH$",             "T",       null,
    "DIC$",            "DIZ",     "TIZ",
    "DIDR-^",          "DIT",     null,
    "DIEDR-^",         "DIT",     null,
    "DJ(AEIOU)-^",     "I",       "I",
    "DMITR-^",         "DIMIT",   "TINIT",
    "DRY9^",           "DR",     null,
    "DT-",             "",        "",
    "DUIS-^",          "D",      "TI",
    "DURCH^^",         "DURCH",   "TURK",
    "DVA$",            "TWA",     null,
    "DY9^",            "D",      null,
    "DYS$",            "DIS",     null,
    "DS(CH)--<",       "T",       "T",
    "DST",             "ZT",      "ZT",
    "DZS(CH)--",       "T",       "T",
    "D(SZ)",          "Z",       "Z",
    "D(AEIORUY)-",  "D",       null,
    "D()-", "D",    null,
    "D'H^",            "D",       "T",
    "DH^",            "D",       "T",
    "D`H^",            "D",       "T",
    "D'S3$",           "Z",       "Z",
    "DS3$",           "Z",       "Z",
    "D^",              "D",       null,
    "D",               "T",       "T",
    "EAULT$",          "O",       "U",
    "EAUX$",           "O",       "U",
    "EAU",             "O",       "U",
    "EAV",             "IW",      "IF",
    "EAS3$",           "EAS",     null,
    "EA(AEIOY)-3",  "EA",      "EA",
    "EA3$",            "EA",      "EA",
    "EA3",             "I",       "I",
    "EBENSO^$",        "EBNSO",   "EBNZU",
    "EBENSO^^",        "EBNSO ",  "EBNZU ",
    "EBEN^^",          "EBN",     "EBN",
    "EE9",             "E",       "E",
    "EGL-1",           "EK",      null,
    "EHE(IUY)--1",     "EH",      null,
    "EHUNG---1",       "E",       null,
    "EH(AIOUY)-1",  "EH",      null,
    "EIEI--",          "",        "",
    "EIERE^$",         "EIERE",   null,
    "EIERE$",          "EIERE",   null,
    "EIERE(NS)-$",     "EIERE",   null,
    "EIERE(AIOUY)--",  "EIER",    null,
    "EIER(AIOUY)-", "EIER",    null,
    "EIER<",           "EIA",     null,
    "EIGL-1",          "EIK",     null,
    "EIGH$",           "EI",      "EI",
    "EIH--",           "E",       "E",
    "EILLE$",          "EI",      "EI",
    "EIR(BCDFGKLMNQSTVWZ)-", "EIA", "EIA",
    "EIR$",            "EIA",     "EIA",
    "EITRAUBEN------", "EIT ",    "EIT ",
    "EI",              "EI",      "EI",
    "EJ$",             "EI",      "EI",
    "ELIZ^",           "ELIS",    null,
    "ELZ^",            "ELS",     null,
    "EL-^",            "E",       "E",
    "ELANG----1",      "E",       "E",
    "EL(DKL)--1",      "E",       "E",
    "EL(MNT)--1$",     "E",       "E",
    "ELYNE$",          "ELINE",   "ELINE",
    "ELYN$",           "ELIN",    "ELIN",
    "EL(AEIOUY)-1", "EL", "EL",
    "EL-1",            "L",       "L",
    "EM-^",            null,      "E",
    "EM(DFKMPQT)--1",  null,      "E",
    "EM(AEIOUY)--1", null, "E",
    "EM-1",            null,      "N",
    "ENGAG-^",         "ANGA",    "ANKA",
    "EN-^",            "E",       "E",
    "ENTUEL",          "ENTUEL",  null,
    "EN(CDGKQSTZ)--1", "E",       "E",
    "EN(AEINOUY)-1", "EN", "EN",
    "EN-1",            "",        "",
    "ERH(AEIOU)-^", "ERH",     "ER",
    "ER-^",            "E",       "E",
    "ERREGEND-----",   " ER",     " ER",
    "ERT1$",           "AT",      null,
    "ER(DGLKMNRQTZ)-1", "ER",    null,
    "ER(AEIOUY)-1", "ER", "A",
    "ER1$",            "A",       "A",
    "ER<1",            "A",       "A",
    "ETAT7",           "ETA",     "ETA",
    "ETI(AOU)-",    "EZI",     "EZI",
    "EUERE$",          "EUERE",   null,
    "EUERE(NS)-$",     "EUERE",   null,
    "EUERE(AIOUY)--",  "EUER",    null,
    "EUER(AIOUY)-", "EUER",    null,
    "EUER<",           "EUA",     null,
    "EUEU--",          "",        "",
    "EUILLE$",         "",       "",
    "EUR$",            "R",      "R",
    "EUX",             "",       "",
    "EUSZ$",           "EUS",     null,
    "EUTZ$",           "EUS",     null,
    "EUYS$",           "EUS",     "EUZ",
    "EUZ$",            "EUS",     null,
    "EU",              "EU",      "EU",
    "EVER--<1",        "EW",      null,
    "EV(OU)-1",     "EW",      null,
    "EYER<",           "EIA",     "EIA",
    "EY<",             "EI",      "EI",
    "FACETTE",         "FASET",   "FAZET",
    "FANS--^$",        "FE",      "FE",
    "FAN-^$",          "FE",      "FE",
    "FAULT-",          "FOL",     "FUL",
    "FEE(DL)-",        "FI",      "FI",
    "FEHLER",          "FELA",    "FELA",
    "FE(LMNRST)-3^",   "FE",      "FE",
    "FOERDERN---^",    "FRD",    "FRT",
    "FOERDERN---",     " FRD",   " FRT",
    "FOND7",           "FON",     "FUN",
    "FRAIN$",          "FRA",     "FRA",
    "FRISEU(RS)-",     "FRIS",   "FRIZ",
    "FY9^",            "F",      null,
    "FRDERN---^",     "FRD",    "FRT",
    "FRDERN---",      " FRD",   " FRT",
    "GAGS^$",          "GEX",     "KEX",
    "GAG^$",           "GEK",     "KEK",
    "GD",              "KT",      "KT",
    "GEGEN^^",         "GEGN",    "KEKN",
    "GEGENGEKOM-----", "GEGN ",   "KEKN ",
    "GEGENGESET-----", "GEGN ",   "KEKN ",
    "GEGENKOMME-----", "GEGN ",   "KEKN ",
    "GEGENZUKOM---",  "GEGN ZU ", "KEKN ZU ",
    "GENDETWAS-----$", "GENT ",   "KENT ",
    "GENRE",           "IORE",    "IURE",
    "GE(LMNRST)-3^",   "GE",      "KE",
    "GER(DKT)-",       "GER",     null,
    "GETTE$",          "GET",     "KET",
    "GGF.",            "GF.",     null,
    "GG-",             "",        "",
    "GH",              "G",       null,
    "GI(AOU)-^",       "I",       "I",
    "GION-3",          "KIO",     "KIU",
    "G(CK)-",          "",        "",
    "GJ(AEIOU)-^",     "I",       "I",
    "GMBH^$",          "GMBH",    "GMBH",
    "GNAC$",           "NIAK",    "NIAK",
    "GNON$",           "NION",    "NIUN",
    "GN$",             "N",       "N",
    "GONCAL-^",        "GONZA",   "KUNZA",
    "GRY9^",           "GR",     null,
    "G(SXZ)-<",       "K",       "K",
    "GUCK-",           "KU",      "KU",
    "GUISEP-^",        "IUSE",    "IUZE",
    "GUI-^",           "G",       "K",
    "GUTAUSSEH------^", "GUT ",   "KUT ",
    "GUTGEHEND------^", "GUT ",   "KUT ",
    "GY9^",            "G",      null,
    "G(AEILORUY)-", "G",       null,
    "G()-", "G",    null,
    "G'S$",            "X",       "X",
    "GS$",            "X",       "X",
    "G^",              "G",       null,
    "G",               "K",       "K",
    "HA(HIUY)--1",     "H",       null,
    "HANDVOL---^",     "HANT ",   "ANT ",
    "HANNOVE-^",       "HANOF",   null,
    "HAVEN7$",         "HAFN",    null,
    "HEAD-",           "HE",      "E",
    "HELIEGEN------",  "E ",      "E ",
    "HESTEHEN------",  "E ",      "E ",
    "HE(LMNRST)-3^",   "HE",      "E",
    "HE(LMN)-1",       "E",       "E",
    "HEUR1$",          "R",      "R",
    "HE(HIUY)--1",     "H",       null,
    "HIH(AEIOUY)-1", "IH",     null,
    "HLH(AEIOUY)-1", "LH",     null,
    "HMH(AEIOUY)-1", "MH",     null,
    "HNH(AEIOUY)-1", "NH",     null,
    "HOBBY9^",          "HOBI",   null,
    "HOCHBEGAB-----^",  "HOCH ",  "UK ",
    "HOCHTALEN-----^",  "HOCH ",  "UK ",
    "HOCHZUFRI-----^",  "HOCH ",  "UK ",
    "HO(HIY)--1",       "H",      null,
    "HRH(AEIOUY)-1", "RH",     null,
    "HUH(AEIOUY)-1", "UH",     null,
    "HUIS^^",          "HS",     "IZ",
    "HUIS$",           "S",      "IZ",
    "HUI--1",          "H",       null,
    "HYGIEN^",         "HKIEN",  null,
    "HY9^",            "H",      null,
    "HY(BDGMNPST)-",   "",       null,
    "H.^",             null,      "H.",
    "HU--1",          "H",       null,
    "H^",              "H",       "",
    "H",               "",        "",
    "ICHELL---",       "ISH",     "IZ",
    "ICHI$",           "ISHI",    "IZI",
    "IEC$",            "IZ",      "IZ",
    "IEDENSTELLE------", "IDN ",  "ITN ",
    "IEI-3",           "",        "",
    "IELL3",           "IEL",     "IEL",
    "IENNE$",          "IN",      "IN",
    "IERRE$",          "IER",     "IER",
    "IERZULAN---",     "IR ZU ",  "IR ZU ",
    "IETTE$",          "IT",      "IT",
    "IEU",             "I",      "I",
    "IE<4",            "I",       "I",
    "IGL-1",           "IK",      null,
    "IGHT3$",          "EIT",     "EIT",
    "IGNI(EO)-",       "INI",     "INI",
    "IGN(AEOU)-$",     "INI",     "INI",
    "IHER(DGLKRT)--1", "IHE",     null,
    "IHE(IUY)--",      "IH",      null,
    "IH(AIOUY)-",    "IH",      null,
    "IJ(AOU)-",        "I",       "I",
    "IJ$",             "I",       "I",
    "IJ<",             "EI",      "EI",
    "IKOLE$",          "IKOL",    "IKUL",
    "ILLAN(STZ)--4",   "ILIA",    "ILIA",
    "ILLAR(DT)--4",    "ILIA",    "ILIA",
    "IMSTAN----^",     "IM ",     "IN ",
    "INDELERREGE------", "INDL ", "INTL ",
    "INFRAGE-----^$",  "IN ",     "IN ",
    "INTERN(AOU)-^",   "INTAN",   "INTAN",
    "INVER-",          "INWE",    "INFE",
    "ITI(AIOU)-",   "IZI",     "IZI",
    "IUSZ$",           "IUS",     null,
    "IUTZ$",           "IUS",     null,
    "IUZ$",            "IUS",     null,
    "IVER--<",         "IW",      null,
    "IVIER$",          "IWIE",    "IFIE",
    "IV(OU)-",      "IW",      null,
    "IV<3",            "IW",      null,
    "IY2",             "I",       null,
    "I()<4",        "I",       "I",
    "JAVIE---<^",      "ZA",      "ZA",
    "JEANS^$",         "JINS",    "INZ",
    "JEANNE^$",        "IAN",     "IAN",
    "JEAN-^",          "IA",      "IA",
    "JER-^",           "IE",      "IE",
    "JE(LMNST)-",      "IE",      "IE",
    "JI^",             "JI",      null,
    "JOR(GK)^$",       "IRK",    "IRK",
    "J",               "I",       "I",
    "KC(EIJ)-",       "X",       "X",
    "KD",              "KT",      null,
    "KE(LMNRST)-3^",   "KE",      "KE",
    "KG(AEILORUY)-", "K",      null,
    "KH<^",            "K",       "K",
    "KIC$",            "KIZ",     "KIZ",
    "KLE(LMNRST)-3^",  "KLE",     "KLE",
    "KOTELE-^",        "KOTL",    "KUTL",
    "KREAT-^",         "KREA",    "KREA",
    "KRS(TZ)--^",     "KRI",     null,
    "KRYS(TZ)--^",     "KRI",     null,
    "KRY9^",           "KR",     null,
    "KSCH---",         "K",       "K",
    "KSH--",           "K",       "K",
    "K(SXZ)7",        "X",       "X",    /**** implies "KST" -> "XT" ****/
    "KT'S$",           "X",       "X",
    "KTI(AIOU)-3",     "XI",      "XI",
    "KT(SXZ)",        "X",       "X",
    "KY9^",            "K",      null,
    "K'S$",            "X",       "X",
    "KS$",            "X",       "X",
    "LANGES$",         " LANGES", " LANKEZ",
    "LANGE$",          " LANGE",  " LANKE",
    "LANG$",           " LANK",   " LANK",
    "LARVE-",          "LARF",    "LARF",
    "LD(SZ)$",        "LS",      "LZ",
    "LD'S$",           "LS",      "LZ",
    "LDS$",           "LS",      "LZ",
    "LEAND-^",         "LEAN",    "LEAN",
    "LEERSTEHE-----^", "LER ",    "LER ",
    "LEICHBLEIB-----", "LEICH ",  "LEIK ",
    "LEICHLAUTE-----", "LEICH ",  "LEIK ",
    "LEIDERREGE------", "LEIT ",  "LEIT ",
    "LEIDGEPR----^",   "LEIT ",   "LEIT ",
    "LEINSTEHE-----",  "LEIN ",   "LEIN ",
    "LEL-",            "LE",      "LE",
    "LE(MNRST)-3^",    "LE",      "LE",
    "LETTE$",          "LET",     "LET",
    "LFGNAG-",         "LFGAN",   "LFKAN",
    "LICHERWEIS----",  "LICHA ", "LIKA ",
    "LIC$",            "LIZ",     "LIZ",
    "LIVE^$",          "LEIF",    "LEIF",
    "LT(SZ)$",        "LS",      "LZ",
    "LT'S$",           "LS",      "LZ",
    "LTS$",           "LS",      "LZ",
    "LUI(GS)--",       "LU",      "LU",
    "LV(AIO)-",        "LW",      null,
    "LY9^",            "L",      null,
    "LSTS$",           "LS",      "LZ",
    "LZ(BDFGKLMNPQRSTVWX)-", "LS", null,
    "L(SZ)$",         "LS",      null,
    "MAIR-<",          "MEI",     "NEI",
    "MANAG-",          "MENE",    "NENE",
    "MANUEL",          "MANUEL",  null,
    "MASSEU(RS)-",     "MAS",    "NAZ",
    "MATCH",           "MESH",    "NEZ",
    "MAURICE",         "MORIS",   "NURIZ",
    "MBH^$",           "MBH",     "MBH",
    "MB(Z)$",         "MS",      null,
    "MB(STZ)-",       "M",       "N",
    "MCG9^",           "MAK",     "NAK",
    "MC9^",            "MAK",     "NAK",
    "MEMOIR-^",        "MEMOA",   "NENUA",
    "MERHAVEN$",       "MAHAFN",  null,
    "ME(LMNRST)-3^",   "ME",      "NE",
    "MEN(STZ)--3",     "ME",      null,
    "MEN$",            "MEN",     null,
    "MIGUEL-",         "MIGE",    "NIKE",
    "MIKE^$",          "MEIK",    "NEIK",
    "MITHILFE----^$",  "MIT H",   "NIT ",
    "MN$",             "M",       null,
    "MN",              "N",       "N",
    "MPJUTE-",         "MPUT",    "NBUT",
    "MP(Z)$",         "MS",      null,
    "MP(STZ)-",       "M",       "N",
    "MP(BDJLMNPQVW)-", "MB",      "NB",
    "MY9^",            "M",      null,
    "M(Z)$",          "MS",      null,
    "MG7^",           "MAK",     "NAK",
    "M'G7^",           "MAK",     "NAK",
    "M^",             "MAK",     "NAK",
    "M'^",             "MAK",     "NAK",
    "M",               null,      "N",
    "NACH^^",          "NACH",    "NAK",
    "NADINE",          "NADIN",   "NATIN",
    "NAIV--",          "NA",      "NA",
    "NAISE$",          "NESE",    "NEZE",
    "NAUGENOMM------", "NAU ",    "NAU ",
    "NAUSOGUT$",     "NAUSO GUT", "NAUZU KUT",
    "NCH$",            "NSH",     "NZ",
    "NCOISE$",         "SOA",     "ZUA",
    "NCOIS$",          "SOA",     "ZUA",
    "NDAR$",           "NDA",     "NTA",
    "NDERINGEN------", "NDE ",    "NTE ",
    "NDRO(CDKTZ)-",    "NTRO",    null,
    "ND(BFGJLMNPQVW)-", "NT",     null,
    "ND(SZ)$",        "NS",      "NZ",
    "ND'S$",           "NS",      "NZ",
    "NDS$",           "NS",      "NZ",
    "NEBEN^^",         "NEBN",    "NEBN",
    "NENGELERN------", "NEN ",    "NEN ",
    "NENLERN(ET)---",  "NEN LE",  "NEN LE",
    "NENZULERNE---", "NEN ZU LE", "NEN ZU LE",
    "NE(LMNRST)-3^",   "NE",      "NE",
    "NEN-3",           "NE",      "NE",
    "NETTE$",          "NET",     "NET",
    "NGU^^",           "NU",      "NU",
    "NG(BDFJLMNPQRTVW)-", "NK",   "NK",
    "NH(AUO)-$",       "NI",      "NI",
    "NICHTSAHNEN-----", "NIX ",   "NIX ",
    "NICHTSSAGE----",  "NIX ",    "NIX ",
    "NICHTS^^",        "NIX",     "NIX",
    "NICHT^^",         "NICHT",   "NIKT",
    "NINE$",           "NIN",     "NIN",
    "NON^^",           "NON",     "NUN",
    "NOTLEIDE-----^",  "NOT ",   "NUT ",
    "NOT^^",           "NOT",     "NUT",
    "NTI(AIOU)-3",     "NZI",     "NZI",
    "NTIEL--3",        "NZI",     "NZI",
    "NT(SZ)$",        "NS",      "NZ",
    "NT'S$",           "NS",      "NZ",
    "NTS$",           "NS",      "NZ",
    "NYLON",           "NEILON",  "NEILUN",
    "NY9^",            "N",      null,
    "NSTZUNEH---",     "NST ZU ", "NZT ZU ",
    "NSZ-",            "NS",      null,
    "NSTS$",           "NS",      "NZ",
    "NZ(BDFGKLMNPQRSTVWX)-", "NS", null,
    "N(SZ)$",         "NS",      null,
    "OBERE-",          "OBER",    null,
    "OBER^^",          "OBA",     "UBA",
    "OEU2",            "",       "",
    "OE<2",            "",       "",
    "OGL-",            "OK",      null,
    "OGNIE-",          "ONI",     "UNI",
    "OGN(AEOU)-$",     "ONI",     "UNI",
    "OH(AIOUY)-",    "OH",      null,
    "OIE$",            "",       "",
    "OIRE$",           "OA",      "UA",
    "OIR$",            "OA",      "UA",
    "OIX",             "OA",      "UA",
    "OI<3",            "EU",      "EU",
    "OKAY^$",          "OKE",     "UKE",
    "OLYN$",           "OLIN",    "ULIN",
    "OO(DLMZ)-",       "U",       null,
    "OO$",             "U",       null,
    "OO-",             "",        "",
    "ORGINAL-----",    "ORI",     "URI",
    "OTI(AOU)-",    "OZI",     "UZI",
    "OUI^",            "WI",      "FI",
    "OUILLE$",         "ULIE",    "ULIE",
    "OU(DT)-^",        "AU",      "AU",
    "OUSE$",           "AUS",     "AUZ",
    "OUT-",            "AU",      "AU",
    "OU",              "U",       "U",
    "O(FV)$",          "AU",      "AU",  /****  due to "OW$" -> "AU"  ****/
    "OVER--<",         "OW",      null,
    "OV(AOU)-",        "OW",      null,
    "OW$",             "AU",      "AU",
    "OWS$",            "OS",      "UZ",
    "OJ(AEIOU)--",  "O",       "U",
    "OYER",            "OIA",     null,
    "OY(AEIOU)--",  "O",       "U",
    "O(JY)<",          "EU",      "EU",
    "OZ$",             "OS",      null,
    "O^",             "O",       "U",
    "O'^",             "O",       "U",
    "O",               null,      "U",
    "PATIEN--^",       "PAZI",    "PAZI",
    "PENSIO-^",        "PANSI",   "PANZI",
    "PE(LMNRST)-3^",   "PE",      "PE",
    "PFER-^",          "FE",      "FE",
    "P(FH)<",          "F",       "F",
    "PIC^$",           "PIK",     "PIK",
    "PIC$",            "PIZ",     "PIZ",
    "PIPELINE",       "PEIBLEIN", "PEIBLEIN",
    "POLYP-",          "POL",    null,
    "POLY^^",          "POLI",    "PULI",
    "PORTRAIT7",       "PORTRE",  "PURTRE",
    "POWER7",          "PAUA",    "PAUA",
    "PP(FH)--<",       "B",       "B",
    "PP-",             "",        "",
    "PRODUZ-^",        "PRODU",   "BRUTU",
    "PRODUZI--",       " PRODU",  " BRUTU",
    "PRIX^$",          "PRI",     "PRI",
    "PS-^^",           "P",       null,
    "P(SZ)^",         null,      "Z",
    "P(SZ)$",         "BS",      null,
    "PT-^",            "",        "",
    "PTI(AOU)-3",   "BZI",     "BZI",
    "PY9^",            "P",      null,
    "P(AEIORUY)-",  "P",       "P",
    "P()-", "P",    null,
    "P.^",             null,      "P.",
    "P^",              "P",       null,
    "P",               "B",       "B",
    "QI-",             "Z",       "Z",
    "QUARANT--",       "KARA",    "KARA",
    "QUE(LMNRST)-3",   "KWE",     "KFE",
    "QUE$",            "K",       "K",
    "QUI(NS)$",        "KI",      "KI",
    "QUIZ7",           "KWIS",    null,
    "Q(UV)7",          "KW",      "KF",
    "Q<",              "K",       "K",
    "RADFAHR----",     "RAT ",    "RAT ",
    "RAEFTEZEHRE-----", "REFTE ", "REFTE ",
    "RCH",             "RCH",     "RK",
    "REA(DU)---3^",    "R",       null,
    "REBSERZEUG------", "REBS ",  "REBZ ",
    "RECHERCH^",       "RESHASH", "REZAZ",
    "RECYCL--",        "RIZEI",  "RIZEI",
    "RE(ALST)-3^",     "RE",      null,
    "REE$",            "RI",      "RI",
    "RER$",            "RA",      "RA",
    "RE(MNR)-4",       "RE",      "RE",
    "RETTE$",          "RET",     "RET",
    "REUZ$",           "REUZ",    null,
    "REW$",            "RU",      "RU",
    "RH<^",            "R",       "R",
    "RJA(MN)--",       "RI",      "RI",
    "ROWD-^",          "RAU",     "RAU",
    "RTEMONNAIE-",     "RTMON",   "RTNUN",
    "RTI(AOU)-3",   "RZI",     "RZI",
    "RTIEL--3",        "RZI",     "RZI",
    "RV(AEOU)-3",      "RW",      null,
    "RY(KN)-$",        "RI",      "RI",
    "RY9^",            "R",      null,
    "RFTEZEHRE-----", "REFTE ",  "REFTE ",
    "SAISO-^",         "SES",     "ZEZ",
    "SAFE^$",          "SEIF",    "ZEIF",
    "SAUCE-^",         "SOS",     "ZUZ",
    "SCHLAGGEBEN-----<", "SHLAK ", "ZLAK ",
    "SCHSCH---7",      "",        "",
    "SCHTSCH",         "SH",      "Z",
    "SC(HZ)<",         "SH",      "Z",
    "SC",              "SK",      "ZK",
    "SELBSTST--7^^",   "SELB",    "ZELB",
    "SELBST7^^",       "SELBST",  "ZELBZT",
    "SERVICE7^",       "SRWIS",  "ZRFIZ",
    "SERVI-^",         "SERW",    null,
    "SE(LMNRST)-3^",   "SE",      "ZE",
    "SETTE$",          "SET",     "ZET",
    "SHP-^",           "S",       "Z",
    "SHST",            "SHT",     "ZT",
    "SHTSH",           "SH",      "Z",
    "SHT",             "ST",      "Z",
    "SHY9^",           "SH",     null,
    "SH^^",            "SH",      null,
    "SH3",             "SH",      "Z",
    "SICHERGEGAN-----^", "SICHA ", "ZIKA ",
    "SICHERGEHE----^", "SICHA ",  "ZIKA ",
    "SICHERGESTEL------^", "SICHA ", "ZIKA ",
    "SICHERSTELL-----^", "SICHA ", "ZIKA ",
    "SICHERZU(GS)--^", "SICHA ZU ", "ZIKA ZU ",
    "SIEGLI-^",        "SIKL",    "ZIKL",
    "SIGLI-^",         "SIKL",    "ZIKL",
    "SIGHT",           "SEIT",    "ZEIT",
    "SIGN",            "SEIN",    "ZEIN",
    "SKI(NPZ)-",       "SKI",     "ZKI",
    "SKI<^",           "SHI",     "ZI",
    "SODASS^$",        "SO DAS",  "ZU TAZ",
    "SODA^$",         "SO DAS",  "ZU TAZ",
    "SOGENAN--^",      "SO GEN",  "ZU KEN",
    "SOUND-",          "SAUN",    "ZAUN",
    "STAATS^^",        "STAZ",    "ZTAZ",
    "STADT^^",         "STAT",    "ZTAT",
    "STANDE$",         " STANDE", " ZTANTE",
    "START^^",         "START",   "ZTART",
    "STAURANT7",       "STORAN",  "ZTURAN",
    "STEAK-",          "STE",     "ZTE",
    "STEPHEN-^$",      "STEW",    null,
    "STERN",           "STERN",   null,
    "STRAF^^",         "STRAF",   "ZTRAF",
    "ST'S$",           "Z",       "Z",
    "STS$",           "Z",       "Z",
    "STST--",          "",        "",
    "STS(ACEHIOU)--", "ST", "ZT",
    "ST(SZ)",          "Z",       "Z",
    "SPAREN---^",      "SPA",     "ZPA",
    "SPAREND----",     " SPA",    " ZPA",
    "S(PTW)-^^",       "S",       null,
    "SP",              "SP",      null,
    "STYN(AE)-$",      "STIN",    "ZTIN",
    "ST",              "ST",      "ZT",
    "SUITE<",          "SIUT",    "ZIUT",
    "SUKE--$",         "S",       "Z",
    "SURF(EI)-",       "SRF",    "ZRF",
    "SV(AEIOU)-<^", "SW",   null,
    "SYB(IY)--^",      "SIB",     null,
    "SYL(KVW)--^",     "SI",      null,
    "SY9^",            "S",      null,
    "SZE(NPT)-^",      "ZE",      "ZE",
    "SZI(ELN)-^",      "ZI",      "ZI",
    "SZCZ<",           "SH",      "Z",
    "SZT<",            "ST",      "ZT",
    "SZ<3",            "SH",      "Z",
    "SL(KVW)--^",     "SI",      null,
    "S",               null,      "Z",
    "TCH",             "SH",      "Z",
    "TD(AEIORUY)-", "T",       null,
    "TD(ݟ)-", "T", null,
    "TEAT-^",          "TEA",     "TEA",
    "TERRAI7^",        "TERA",    "TERA",
    "TE(LMNRST)-3^",   "TE",      "TE",
    "TH<",             "T",       "T",
    "TICHT-",          "TIK",     "TIK",
    "TICH$",           "TIK",     "TIK",
    "TIC$",            "TIZ",     "TIZ",
    "TIGGESTELL-------", "TIK ",  "TIK ",
    "TIGSTELL-----",   "TIK ",    "TIK ",
    "TOAS-^",          "TO",      "TU",
    "TOILET-",         "TOLE",    "TULE",
    "TOIN-",           "TOA",     "TUA",
    "TRAECHTI-^",      "TRECHT",  "TREKT",
    "TRAECHTIG--",     " TRECHT", " TREKT",
    "TRAINI-",         "TREN",    "TREN",
    "TRCHTI-^",       "TRECHT",  "TREKT",
    "TRCHTIG--",      " TRECHT", " TREKT",
    "TSCH",            "SH",      "Z",
    "TSH",             "SH",      "Z",
    "TST",             "ZT",      "ZT",
    "T(S)",           "Z",       "Z",
    "TT(SZ)--<",       "",        "",
    "TT9",             "T",       "T",
    "TV^$",            "TV",      "TV",
    "TX(AEIOU)-3",     "SH",      "Z",
    "TY9^",            "T",      null,
    "TZ-",             "",        "",
    "T'S3$",           "Z",       "Z",
    "TS3$",           "Z",       "Z",
    "UEBEL(GNRW)-^^",  "BL ",    "IBL ",
    "UEBER^^",         "BA",     "IBA",
    "UE2",             "",       "I",
    "UGL-",            "UK",      null,
    "UH(AOUY)-",     "UH",      null,
    "UIE$",            "",       "I",
    "UM^^",            "UM",      "UN",
    "UNTERE--3",       "UNTE",    "UNTE",
    "UNTER^^",         "UNTA",    "UNTA",
    "UNVER^^",         "UNFA",    "UNFA",
    "UN^^",            "UN",      "UN",
    "UTI(AOU)-",    "UZI",     "UZI",
    "UVE-4",           "UW",      null,
    "UY2",             "UI",      null,
    "UZZ",             "AS",      "AZ",
    "VACL-^",          "WAZ",     "FAZ",
    "VAC$",            "WAZ",     "FAZ",
    "VAN DEN ^",       "FANDN",   "FANTN",
    "VANES-^",         "WANE",    null,
    "VATRO-",          "WATR",    null,
    "VA(DHJNT)--^",    "F",       null,
    "VEDD-^",          "FE",      "FE",
    "VE(BEHIU)--^",    "F",       null,
    "VEL(BDLMNT)-^",   "FEL",     null,
    "VENTZ-^",         "FEN",     null,
    "VEN(NRSZ)-^",     "FEN",     null,
    "VER(AB)-^$",      "WER",     null,
    "VERBAL^$",        "WERBAL",  null,
    "VERBAL(EINS)-^",  "WERBAL",  null,
    "VERTEBR--",       "WERTE",   null,
    "VEREIN-----",     "F",       null,
    "VEREN(AEIOU)-^",  "WEREN",   null,
    "VERIFI",          "WERIFI",  null,
    "VERON(AEIOU)-^",  "WERON",   null,
    "VERSEN^",         "FERSN",   "FAZN",
    "VERSIERT--^",     "WERSI",   null,
    "VERSIO--^",       "WERS",    null,
    "VERSUS",          "WERSUS",  null,
    "VERTI(GK)-",      "WERTI",   null,
    "VER^^",           "FER",     "FA",
    "VERSPRECHE-------", " FER",  " FA",
    "VER$",            "WA",      null,
    "VER",             "FA",      "FA",
    "VET(HT)-^",       "FET",     "FET",
    "VETTE$",          "WET",     "FET",
    "VE^",             "WE",      null,
    "VIC$",            "WIZ",     "FIZ",
    "VIELSAGE----",    "FIL ",    "FIL ",
    "VIEL",            "FIL",     "FIL",
    "VIEW",            "WIU",     "FIU",
    "VILL(AE)-",       "WIL",     null,
    "VIS(ACEIKUVWZ)-<^", "WIS",   null,
    "VI(ELS)--^",      "F",       null,
    "VILLON--",        "WILI",    "FILI",
    "VIZE^^",          "FIZE",    "FIZE",
    "VLIE--^",         "FL",      null,
    "VL(AEIOU)--",     "W",       null,
    "VOKA-^",          "WOK",     null,
    "VOL(ATUVW)--^",   "WO",      null,
    "VOR^^",           "FOR",     "FUR",
    "VR(AEIOU)--",     "W",       null,
    "VV9",             "W",       null,
    "VY9^",            "W",      "FI",
    "V(Y)-",          "W",       null,
    "V()-",  "W",   null,
    "V(AEIJLRU)-<",    "W",       null,
    "V.^",             "V.",      null,
    "V<",              "F",       "F",
    "WEITERENTWI-----^", "WEITA ", "FEITA ",
    "WEITREICH-----^", "WEIT ",   "FEIT ",
    "WEITVER^",       "WEIT FER", "FEIT FA",
    "WE(LMNRST)-3^",   "WE",      "FE",
    "WER(DST)-",       "WER",     null,
    "WIC$",            "WIZ",     "FIZ",
    "WIEDERU--",       "WIDE",    "FITE",
    "WIEDER^$",        "WIDA",    "FITA",
    "WIEDER^^",        "WIDA ",   "FITA ",
    "WIEVIEL",         "WI FIL",  "FI FIL",
    "WISUEL",          "WISUEL",  null,
    "WR-^",            "W",       null,
    "WY9^",            "W",      "FI",
    "W(BDFGJKLMNPQRSTZ)-", "F",   null,
    "W$",              "F",       null,
    "W",               null,      "F",
    "X<^",             "Z",       "Z",
    "XHAVEN$",         "XAFN",    null,
    "X(CSZ)",          "X",       "X",
    "XTS(CH)--",       "XT",      "XT",
    "XT(SZ)",          "Z",       "Z",
    "YE(LMNRST)-3^",   "IE",      "IE",
    "YE-3",            "I",       "I",
    "YOR(GK)^$",       "IRK",    "IRK",
    "Y(AOU)-<7",       "I",       "I",
    "Y(BKLMNPRSTX)-1", "",       null,
    "YVES^$",          "IF",      "IF",
    "YVONNE^$",        "IWON",    "IFUN",
    "Y.^",             "Y.",      null,
    "Y",               "I",       "I",
    "ZC(AOU)-",        "SK",      "ZK",
    "ZE(LMNRST)-3^",   "ZE",      "ZE",
    "ZIEJ$",           "ZI",      "ZI",
    "ZIGERJA(HR)-3",   "ZIGA IA", "ZIKA IA",
    "ZL(AEIOU)-",      "SL",      null,
    "ZS(CHT)--",       "",        "",
    "ZS",              "SH",      "Z",
    "ZUERST",          "ZUERST",  "ZUERST",
    "ZUGRUNDE^$",    "ZU GRUNDE", "ZU KRUNTE",
    "ZUGRUNDE",     "ZU GRUNDE ", "ZU KRUNTE ",
    "ZUGUNSTEN",     "ZU GUNSTN", "ZU KUNZTN",
    "ZUHAUSE-",        "ZU HAUS", "ZU AUZ",
    "ZULASTEN^$",     "ZU LASTN", "ZU LAZTN",
    "ZURUECK^^",       "ZURK",   "ZURIK",
    "ZURZEIT",        "ZUR ZEIT", "ZUR ZEIT",
    "ZURCK^^",        "ZURK",   "ZURIK",
    "ZUSTANDE",      "ZU STANDE", "ZU ZTANTE",
    "ZUTAGE",          "ZU TAGE", "ZU TAKE",
    "ZUVER^^",         "ZUFA",    "ZUFA",
    "ZUVIEL",          "ZU FIL",  "ZU FIL",
    "ZUWENIG",        "ZU WENIK", "ZU FENIK",
    "ZY9^",            "Z",      null,
    "ZYK3$",           "ZIK",     null,
    "Z(VW)7^",         "SW",      null,
    PHONET_END,      PHONET_END,  PHONET_END
  };

#endif    /****  PHONET_GERMAN  ****/




static int set_phonet_language (int rule_set)
/****  result:  0 :  success             ****/
/****          -1 :  language not found  ****/
{
 switch (rule_set)
   {
    #ifdef PHONET_NO_LANGUAGE
         case PHONET_NO_LANGUAGE :
                   phonet_language = phonet_language_no_language;
                   phonet_init   = & phonet_init_no_language;
                   phonet_hash   = phonet_hash_no_language;
                   phonet_hash_1 = & phonet_hash_no_language_1;
                   phonet_hash_2 = & phonet_hash_no_language_2;
                   phonet_rules  = phonet_rules_no_language;
             return (0);
    #endif

    #ifdef PHONET_ENGLISH
         case PHONET_ENGLISH :
                   phonet_language = phonet_language_english;
                   phonet_init   = & phonet_init_english;
                   phonet_hash   = phonet_hash_english;
                   phonet_hash_1 = & phonet_hash_english_1;
                   phonet_hash_2 = & phonet_hash_english_2;
                   phonet_rules  = phonet_rules_english;
             return (0);
    #endif

    #ifdef PHONET_GERMAN
         case PHONET_GERMAN  :
                   phonet_language = phonet_language_german;
                   phonet_init   = & phonet_init_german;
                   phonet_hash   = phonet_hash_german;
                   phonet_hash_1 = & phonet_hash_german_1;
                   phonet_hash_2 = & phonet_hash_german_2;
                   phonet_rules  = phonet_rules_german;
             return (0);
    #endif
   }
 return (-1);
}

#endif     /****  _PHONET_H_  ****/


/************************************************************/
/****  end of file "phonet.h"  ******************************/
/************************************************************/
