#include "Frame.h"
#include "AsteroidSize.h"
#include "UfoSize.h"
#include "Tools.h"
#include "Log.h"

#include <fstream>
#include <iomanip>

#if defined _LOG_IO || defined LOG_IO_FILE
char Frame::lastFrameID = 0;
#endif

//dadurch muss das array nicht erst in WORDs konvertiert werden
//unter diesem #define nichts mehr #includen!
#ifndef LITTLE_ENDIAN
	#error "big endian wird momentan nicht unterstuetzt!"
#else
	#define data ((unsigned short*) buf)
#endif

Frame::Frame(const char* buf):
	buf(buf),
	index(0),
	currentFrameID(buf[1024])
{
	#if defined _LOG_IO || defined LOG_IO_FILE
	if(Tools::getFrameDiff(lastFrameID,currentFrameID)>1)
		IOLog("Frame loss: " << Tools::getFrameDiff(lastFrameID,currentFrameID) - 1 << " frame(s)");
		lastFrameID = currentFrameID;
	#endif
	
	#ifdef _LOG_FRAMES_
	dump();
	#endif
}

//folgende Annahmen werden gemacht:
//der Frame besteht IMMER aus einem Objektteil, der Copyrightmeldung
//und einem Textteil (in GENAU dieser Reihenfolge)
void Frame::operator>>(Snapshot& snapshot)
{
	parseObjects(snapshot);
	parseText(snapshot);
	snapshot.setOutputID(buf[1025]);
}

//Asteroiden, Ufos, Schüsse und das Schiff zum Snapshot hinzufügen
//folgende Annahmen werden gemacht:
//der erste Befehl ist IMMER ein gültiger JMPL
//der zweite Befehl ist IMMER LABS
//ein Schuss hat IMMER die Skalierung 7
//ALLE VCTR mit Skalierung 6 gehören zum Schiff
//SVEC haben NIE eine Bedeutung
//die Copyrightmeldung kommt IMMER vor Ende des Frames
void Frame::parseObjects(Snapshot& snapshot)
{
	//Sprungadresse in data[0] wird als korrekt angenommen
	index = 1;
	//leere Frames werden vernachlässigt, daher zweiter Befehl immer LABS
	int absoluteY = (data[index] & 0x3ff) << 3;
	index++;
	int absoluteX = (data[index] & 0x3ff) << 3;
	int scaleFactor = data[index] >> 12;
	index++;
	//die Schleife wird durch die Copyrightmeldung gebrochen
	//die sollte immer vorkommen, daher keine weitere Abbruchbedingung
	for(;;index++)
	{
		assert(index < 512);
		switch (data[index] >> 12)
		{
		// JSRL
		//Subroutine aufrufen (i.e. Grafiken aus dem ROM zeichnen)
		case 0xc:
			switch (data[index] & 0xfff)
			{
			//Asteroid (4 Typen)
			AsteroidPhaenoType asteroidType;
			case 0x8f3:
				switch(scaleFactor)
				{
				case 0:
					asteroidType = AsteroidType1Big;
					break;
				case 15:
					asteroidType = AsteroidType1Medium;
					break;
				default:
					asteroidType = AsteroidType1Small;
					break;
				}
				snapshot.addAsteroid(wxPoint(absoluteX, absoluteY),asteroidType);
				break;
			case 0x8ff:
				switch(scaleFactor)
				{
				case 0:
					asteroidType = AsteroidType2Big;
					break;
				case 15:
					asteroidType = AsteroidType2Medium;
					break;
				default:
					asteroidType = AsteroidType2Small;
					break;
				}
				snapshot.addAsteroid(wxPoint(absoluteX, absoluteY),asteroidType);
				break;
			case 0x90d:
				switch(scaleFactor)
				{
				case 0:
					asteroidType = AsteroidType3Big;
					break;
				case 15:
					asteroidType = AsteroidType3Medium;
					break;
				default:
					asteroidType = AsteroidType3Small;
					break;
				}
				snapshot.addAsteroid(wxPoint(absoluteX, absoluteY),asteroidType);
				break;
			case 0x91a:
				switch(scaleFactor)
				{
				case 0:
					asteroidType = AsteroidType4Big;
					break;
				case 15:
					asteroidType = AsteroidType4Medium;
					break;
				default:
					asteroidType = AsteroidType4Small;
					break;
				}
				snapshot.addAsteroid(wxPoint(absoluteX, absoluteY),asteroidType);
				break;
			//UFO
			case 0x929:
				if(scaleFactor == 15)
					snapshot.addUfo(wxPoint(absoluteX,absoluteY), UfoSize::big);
				else
					snapshot.addUfo(wxPoint(absoluteX,absoluteY), UfoSize::small);
				break;
		 	//Copyrightmeldung; Anfang der Textinhalte
		 	case 0x852:
		 		index++;
		 		return;
			}
			break;
		//zu Koordinaten springen und Skalierung setzen
		// LABS
		case 0xa:
			absoluteY = (data[index] & 0x3ff)<<3;
			index++;
			absoluteX = (data[index] & 0x3ff)<<3;
			scaleFactor = data[index] >> 12;
			break;
		//VCTR verschiedener Helligkeit zeichnen
		// 7 -- Schuss
		case 7:
			index++;
			if(data[index]>>12)
			{
				snapshot.addShot(wxPoint(absoluteX,absoluteY));
			}
			break;
		case 6:
			//die beiden Eingabevektoren
			int absX1, absX2, absY1, absY2;
			int signX1, signX2, signY1, signY2;
			//das Ergebnis
			int length;
			int signX,signY;
			
			signY1 = (data[index] & 0x400) >> 10;
			absY1  = (data[index] & 0x3ff);
			index++;
			signX1 = (data[index] & 0x400) >> 10;
			absX1  = (data[index] & 0x3ff);
			index++;
			signY2 = (data[index] & 0x400) >> 10;
			absY2  = (data[index] & 0x3ff);
			index++;
			signX2 = (data[index] & 0x400) >> 10;
			absX2  = (data[index] & 0x3ff);
		
			if(signX1 ^ signX2)
				//(-a)-(+b) = -(a+b); (+a)-(-b) = (a+b)
				signX = signX1;
			else if(signX1)
				signX = absX2-absX1 > 0 ? 0 : 1;
			else
				signX = absX1-absX2 > 0 ? 0 : 1;
			if(signY1 ^ signY2)
			{
				//(-a)-(+b) = -(a+b); (+a)-(-b) = (a+b)
				signY = signY1;
				length = absY1 + absY2;
			}
			else
			{
				if(signY1)
					signY = absY2-absY1 > 0 ? 0 : 1;
				else
					signY = absY1-absY2 > 0 ? 0 : 1;
				length = signY ? (signY1 ? absY1 - absY2 : absY2-absY1): (signY1? absY2-absY1 : absY1 - absY2);
			}
	
			snapshot.addShip(alive,wxPoint(absoluteX,absoluteY),angle[(signX<<1)+(signY<<8)+(length>>3)]);
			break;
		}
	}
}

//alles nach der Copyrightmeldung parsen, i.e. Punkte, Leben, Highscore
//folgende Annahmen werden gemacht:
//der erste Befehl nach der Copyrightmeldung ist IMMER GENAU
//0xA36C1064
//der zweite Befehl nach der Copyrightmeldung ist IMMER uninteressant
//danach folgt ein LABS (*1)
//dazwischen ist NICHTS interessantes außer die Ziffern des Punktestands
//das LABS (*1) ist GENAU 0xA354E0A0, wenn das Spiel
//gerade läuft, ansonsten gelten die folgenden Annahmen nicht
//danach folgt ein LABS (*2)
//dazwischen ist JEDER befehl GENAU 0xCA6D (ersatzschiff)
//das LABS (*2) ist GENAU 0xA36C01E0
//nachdem LABS (*2) folgt noch ein LABS
//dazwischen ist NICHTS interessantes außer die Ziffern des Highscores
void Frame::parseText(Snapshot& snapshot)
{
	//die Punkteanzeige sollte immer direkt nach dem Copyright kommen
	assert(data[index] == 0xA36C && data[index+1] == 0x1064);
	//ueberspringe LABS und das folgende NOOP
	index += 3;
	int points = 0;
	while(data[index] >> 12 != 0xA)
	{
		assert(index < 512);
		points *= 10;
		switch(data[index] & 0xfff)
		{
		case 0xB2E:
			points += 1;
			break;
		case 0xB32:
			points += 2;
			break;
		case 0xB3A:
			points += 3;
			break;
		case 0xB41:
			points += 4;
			break;
		case 0xB48:
			points += 5;
			break;
		case 0xB4F:
			points += 6;
			break;
		case 0xB56:
			points += 7;
			break;
		case 0xB5B:
			points += 8;
			break;
		case 0xB63:
			points += 9;
		default:
			break;
		}
		index++;
	}
	snapshot.setPoints(points);
	//wenn gerade kein Spiel läuft, werden keine Leben angezeigt
	//da der ganze Frame dann sowieso relativ uninteressant ist, erstmal abbrechen
	if(!(data[index] == 0xA354 && data[index+1] == 0xE0A0))
	{
		snapshot.setHighScore(0);
		snapshot.setLives(0);
		return;
	}
	index += 2;
	int lives = 0;
	while(data[index] >> 12 != 0xA)
	{
		assert(data[index] == 0xca6d);
		assert(index < 512);
		lives++;
		index++;
	}
	snapshot.setLives(lives);
	//nun sollte zum Anfangspunkt des Highscores gesprungen werden
	assert(data[index] == 0xA36C && data[index+1] == 0x01E0);
	index += 2;
	points = 0;
	while(data[index] >> 12 != 0xA)
	{
		assert(index < 512);
		points *= 10;
		switch(data[index] & 0xfff)
		{
		case 0xB2E:
			points += 1;
			break;
		case 0xB32:
			points += 2;
			break;
		case 0xB3A:
			points += 3;
			break;
		case 0xB41:
			points += 4;
			break;
		case 0xB48:
			points += 5;
			break;
		case 0xB4F:
			points += 6;
			break;
		case 0xB56:
			points += 7;
			break;
		case 0xB5B:
			points += 8;
			break;
		case 0xB63:
			points += 9;
		default:
			break;
		}
		index++;
	}
	snapshot.setHighScore(points);
}

void Frame::dump()
{
	ofstream log("frame.log",ios::app);
	log << "[FL] ############################################# Start of frame no. " << (int)currentFrameID << endl;
	for(int i = 0; i < 512; ++i)
	{
		int operation = data[i] >> 12;
		log << "[FL] " << setw(3) << setfill('0') << right << hex << i << " ";
		log << setw(4) << setfill('0') << right << hex << data[i] << " ";
		if(operation > 0xa)
			log << "     ";
		else
			log << setw(4) << setfill('0') << right << hex << data[i+1] << " ";
		if(operation < 0xa)
		{
			if((data[i+1] >> 12)==0)
			{
				log << "NOOP" << endl;
				i++;
				continue;
			}
			else if((data[i] & 0x3ff)==0 && (data[i+1] & 0x3ff)==0)
			{
				log << "Schuss" << endl;
				i++;
				continue;
			}
		}
		switch(operation)
		{
		case 0xF:
			log << "SVEC (";
			if(data[i] & 0x4)
				log << "-";
			log << dec << ((data[i] & 0x3) << 8) << ",";
			
			if(data[i] & 0x400)
				log << "-";
			log << dec << (data[i] & 0x300) << ") z"  << (data[i] & 0xf0);
			break;
		case 0xE:
			log << "JMPL $" << setw(3) << setfill('0') << right << hex << (data[i] & 0xfff);
			break;
		case 0xD:
			log << "RTSL ";
			break;
		case 0xC:
			log << "JSRL " << " ";
			do
			{
			switch(data[i] & 0xfff)
			{
			case 0x852:
				log << "Copyright-Meldung";
				break;
			case 0x880:
				log << "Explosion groß (3)";
				break;
			case 0x896:
				log << "Explosion (2)";
				break;
			case 0x8B5:
				log << "Explosion (1)";
				break;
			case 0x8D0:
				log << "Explosion klein (0)";
				break;
			case 0x8F3:
				log << "Asteroid Typ 1";
				break;
			case 0x8FF:
				log << "Asteroid Typ 2";
				break;
			case 0x90D:
				log << "Asteroid Typ 3";
				break;
			case 0x91A:
				log << "Asteroid Typ 4";
				break;
			case 0x929:
				log << "UFO";
				break;
			case 0xA6D:
				log << "∆";
				break;
			case 0xA78:
				log << "A";
				break;
			case 0xa8d:
				log << "C";
				break;
			case 0xa93:
				log << "D";
				break;
			case 0xa9b:
				log << "E";
				break;
			case 0xaa3:
				log << "F";
				break;
			case 0xaba:
				log << "I";
				break;
			case 0xac7:
				log << "K";
				break;
			case 0xacd:
				log << "L";
				break;
			case 0xad8:
				log << "N";
				break;
			case 0xADD:
				log << "0";
				break;
			case 0xae3:
				log << "P";
				break;
			case 0xaf3:
				log << "R";
				break;
			case 0xafb:
				log << "S";
				break;
			case 0xb02:
				log << "T";
				break;
			case 0xb08:
				log << "U";
				break;
			case 0xB26:
				log << "Z";
				break;
			case 0xB2C:
				log << "_";
				break;
			case 0xB2E:
				log << "1";
				break;
			case 0xB32:
				log << "2";
				break;
			case 0xB3A:
				log << "3";
				break;
			case 0xB41:
				log << "4";
				break;
			case 0xB48:
				log << "5";
				break;
			case 0xB4F:
				log << "6";
				break;
			case 0xB56:
				log << "7";
				break;
			case 0xB5B:
				log << "8";
				break;
			case 0xB63:
				log << "9";
				break;
			default:
				break;
			}
			i++;
			}
			while((data[i] >> 12) == 0xC && (data[i] & 0xfff) != 0x852);
			i--;
			break;
		case 0xB:
			log << "HALT" << endl;
			return;
		case 0xA:
			log << "LABS (" << dec << setw(3) << (data[i+1] & 0x3ff) << "," << (data[i] & 0x3ff) << "), s" << dec << setw(2) << (data[i+1] >> 12);
			i++;
			break;
		default:
			log << "VCTR (";
			if(data[i+1] & 0x400)
				log << "-";
			else
				log << " ";
			log << dec << (data[i+1] & 0x3ff) << ",";
			if(data[i] & 0x400)
				log << "-";
			else
				log << " ";
			log << dec << setw(3) << (data[i] & 0x3ff) << "), s" << dec << setw(2) << operation << ", z" << dec << setw(2) << (data[i+1] >> 12);
			i++;
			break;
		}
		log << endl;
	}
}

//klingt komisch, ist aber so (automatisch generierte hash table)
const char Frame::angle[] = {0,0,-128,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,124,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,120,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,116,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,0,112,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,0,108,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,0,104,0,0,0,0,0,0,0,0,0,0,0,0,28,0,100,0,0,0,0,0,0,0,0,0,0,0,32,0,96,0,0,0,0,0,0,0,0,0,0,36,0,92,0,0,0,0,0,0,0,0,40,0,88,0,0,0,0,0,0,0,44,0,84,0,0,0,0,48,0,80,0,0,0,0,52,0,76,0,56,0,72,60,64,68,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-128,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-4,0,-124,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-8,0,-120,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-12,0,-116,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-16,0,-112,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-20,0,-108,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-24,0,-104,0,0,0,0,0,0,0,0,0,0,0,0,-28,0,-100,0,0,0,0,0,0,0,0,0,0,0,-32,0,-96,0,0,0,0,0,0,0,0,0,0,-36,0,-92,0,0,0,0,0,0,0,0,-40,0,-88,0,0,0,0,0,0,0,-44,0,-84,0,0,0,0,-48,0,-80,0,0,0,0,-52,0,-76,0,-56,0,-72,-60,-64,-68,-64};
char Frame::frameBuffer[1026] = {0};
