#include "TrackingLayer.h"
#include "Tools.h"
#include "SortedList.h"
#include "Log.h"
#include "rdtsc.h"

void* TrackingLayer::Entry()
{
	while(!TestDestroy())
		trackData();
	return 0;
}

void TrackingLayer::trackData()
{
	#ifdef _MEASURE_TIME_
	time_tsc start, end;
	start = getMicroSeconds();
	wxLogDebug("TrackingLayer::Entry start %lld",start);
	#endif
	
	snapshot = repository->popSnapshot();
	
	if(snapshot && Tools::getFrameDiff(latestFrameID,snapshot->getFrameID()) >= 0)
	{
		unsigned char maxUDPID = repository->getUDPID();
		
		if(latestUdpID == snapshot->getOutputID())
		{
			nextUdpID++;
			if((char)((char)nextUdpID - (char)maxUDPID) > 0)
				nextUdpID = maxUDPID;
			receivedAction.keys &= ~(Key::SHOOT| Key::HYPERSPACE);
		}
		else
		{
			receivedAction = repository->getAction(snapshot->getOutputID());
			nextUdpID = snapshot->getOutputID()+1;
		}
		
		latency = Tools::getFrameDiff(receivedAction.frameID, snapshot->getFrameID())+1;
		latestFrameID = snapshot->getFrameID();
		
		//wxLogDebug("FrameID: %d",latestFrameID);
		#ifdef _MEASURE_TIME_
		time_tsc startMatching, endMatching;
		startMatching = getMicroSeconds();
		wxLogDebug("TrackingLayer::Entry Matching start %lld",startMatching);
		#endif
		
		matchShip(snapshot->getShip());
		//matchAsteroidsAndExplosions(snapshot->getAsteroidsAndExplosions(),snapshot->getNumberAsteroidsAndExplosions());
		matchUfo(snapshot->getUfo());
		
		list<AsteroidSnapshot>* asteroids = snapshot->getAsteroids();
		
		for(int i = 0; i< 12; i++)
		{	
			if(asteroids[i].size() != 0 || asteroidsTracking[i].size() != 0)
				matchFlyingObject<AsteroidSnapshot>(asteroids[i],asteroidsTracking[i]);
		}
		
		matchFlyingObject<ShotSnapshot>(snapshot->getShots(),shotsTracking);
		
		#ifdef _MEASURE_TIME_
		endMatching = getMicroSeconds();
		wxLogDebug("TrackingLayer::Entry Matching end %lld ==> time %lld",endMatching,endMatching-startMatching);
		#endif
		
		calibrateShipAngles();
		
		#ifdef _MEASURE_TIME_
		time_tsc startBuild, endBuild;
		startBuild = getMicroSeconds();
		wxLogDebug("TrackingLayer::Entry Build start %lld",startBuild);
		#endif
			
		updateShotMirages();
		
		Universe* universe = buildUniverse();
		
		checkVictims(universe);
		
		#ifdef _MEASURE_TIME_
		endBuild = getMicroSeconds();
		wxLogDebug("TrackingLayer::Entry Build end %lld ==> time %lld",endBuild,endBuild-startBuild);
		#endif
		
		if(universe->getShip().isUsed())
			updateShipAngles();
			
		repository->setUniverse(universe);
		
		latestUdpID = snapshot->getOutputID();
	}
	delete snapshot;
	
	snapshot = 0;
	
	#ifdef _MEASURE_TIME_
	end = getMicroSeconds();
	wxLogDebug("TrackingLayer::Entry end %lld ==> time %lld",end,end-start);
	#endif
}

void TrackingLayer::updateShotMirages()
{
	unsigned char index;
	unsigned char maxUdpID = repository->getUDPID();
	
	
	//update existing mirages
	for(index = startMirage;shotMirages[index].getFrameID() != -1;index++)
	{
		if((char)(((char)shotMirages[index].getID()) - ((char)nextUdpID)) >= 0)
		{ 
			shotMirages[index].simulateFrames(latestFrameID);
		}
	}
	
	//new mirage
	if(repository->getLastOutput()  & Key::SHOOT)
	{	
		Angle angle = getEstimatedAngle();
		
		if(repository->getLastOutput() & Key::LEFT)
			angle -= 3;
		else if(repository->getLastOutput() & Key::RIGHT)
			angle += 3;
			
		wxPoint flightVector = Tools::internToShot(angle);
		wxPoint start = Tools::calcPosition(shipTracking.getCurrentPosition(),shipTracking.getFlightVector(),latency)+Tools::getShotStartPoint(angle);
		
		shotMirages[index].setFrameID(latestFrameID);
		shotMirages[index].setID(maxUdpID);
		shotMirages[index].setStartPosition(start);
		shotMirages[index].setFlightVector(flightVector);
		shotMirages[index].setPosition(Tools::calcPosition(start,flightVector,1));
	}
	
	for(index = startMirage; shotMirages[index].getFrameID() != latestFrameID && shotMirages[index].getFrameID() != -1;index++)
	{
		if(latestUdpID != snapshot->getOutputID())
		{
			int dist = -1;
			list<TrackingObject<ShotSnapshot> >::iterator shotMatching;
			
			for(list<TrackingObject<ShotSnapshot> >::iterator iteratorShots = shotsTracking.begin();iteratorShots!= shotsTracking.end();iteratorShots++)
			{
				if(Tools::pythagoras(iteratorShots->getStart()-shotMirages[index].getStartPosition()) < dist || (dist == -1 && Tools::pythagoras(iteratorShots->getStart()-shotMirages[index].getStartPosition()) <= 10000))
				{
					dist = Tools::pythagoras(iteratorShots->getStart()-shotMirages[index].getStartPosition());
					shotMatching = iteratorShots;
				}
			}	
			
			if(dist != -1)
			{
				shotMatching->setEstimatedFlightVector(shotMirages[index].getFlightVector());
				
				for(int i = 0 ;i < 4; i++)
				{
					if(shotIDs[i].ID == -1)
					{
						shotIDs[i].ID = shotMatching->getObjectID();
						shotIDs[i].frameID = shotMatching->getFrameID();
						break;
					}
				}
			}
		}
		shotMirages[index].setFrameID(-1);	
	}
	
	startMirage = index;
}

void TrackingLayer::checkVictims(Universe* universe)
{
	int minFrames = -1;
	int tempFrames;
	list<Shot>::iterator shots;
	list<Asteroid >::iterator iteratorAsteroids;
	bool ufo;
	list<Asteroid >::iterator asteroid;
	int highestLifeTime = 0;
	int nextHit = 0x0fffffff;
	
	unsigned char index;
	
	numShots = 0;
	
	for(index = startMirage; shotMirages[index].getFrameID() != -1; index++)
	{
		ufo = false;
		minFrames = -1;
		
		numShots++;
			
		for(iteratorAsteroids = universe->getAsteroids()->begin();iteratorAsteroids != universe->getAsteroids()->end();iteratorAsteroids++)
		{
			tempFrames = Tools::calcCollision(shotMirages[index].getPosition(),shotMirages[index].getFlightVector(),1,iteratorAsteroids->getAbsolutePosition(),iteratorAsteroids->getFlightVector(),iteratorAsteroids->getAsteroidSize().getValue(),69,1);
			
			if(tempFrames != -1 && (minFrames==-1 || tempFrames < minFrames))
			{
				minFrames = tempFrames;
				asteroid = iteratorAsteroids;
			}
		}
		
		if(universe->getUfo().isUsed())
		{
			tempFrames = Tools::calcCollision(shotMirages[index].getPosition(),shotMirages[index].getFlightVector(),1,universe->getUfo().getAbsolutePosition(),universe->getUfo().getFlightVector(),universe->getUfo().getUfoSize().getValue(),69,1);
			
			if(tempFrames != -1 && (minFrames == -1 || tempFrames < minFrames))
			{
				minFrames = tempFrames;
				ufo = true;
			}
		}
		
		if(ufo)
		{
			universe->getUfo().addHit();
		}
		else if(minFrames != -1)
		{
			asteroid->addHit();
		}
	}
	
	for(shots = universe->getShots()->begin(); shots != universe->getShots()->end(); shots++)
	{
		ufo = false;
		minFrames = -1;
		
		for(int i=0; i< 4;i++)
		{
			if(shotIDs[i].ID == shots->getObjectID())
			{
				shotIDs[i].frameID = universe->getFrameID();
				if(highestLifeTime < shots->getLifeTime())
					highestLifeTime = shots->getLifeTime();
			}
		}
		
		for(iteratorAsteroids = universe->getAsteroids()->begin();iteratorAsteroids != universe->getAsteroids()->end();iteratorAsteroids++)
		{
			tempFrames = Tools::calcCollision(shots->getAbsolutePosition(),shots->getFlightVector(),1,iteratorAsteroids->getAbsolutePosition(),iteratorAsteroids->getFlightVector(),iteratorAsteroids->getAsteroidSize().getValue(),69,1);
			
			if(tempFrames != -1 && (minFrames==-1 || tempFrames < minFrames))
			{
				minFrames = tempFrames;
				asteroid = iteratorAsteroids;
				if(nextHit> minFrames)
					nextHit = minFrames;
			}
		}
		
		if(universe->getUfo().isUsed())
		{
			tempFrames = Tools::calcCollision(shots->getAbsolutePosition(),shots->getFlightVector(),1,universe->getUfo().getAbsolutePosition(),universe->getUfo().getFlightVector(),universe->getUfo().getUfoSize().getValue(),69,1);
			
			if(tempFrames != -1 && (minFrames == -1 || tempFrames < minFrames))
			{
				minFrames = tempFrames;
				ufo = true;
			}
		}
		
		if(ufo)
			universe->getUfo().addHit();
		else if(minFrames != -1)
			asteroid->addHit();
	}
	
	for(int i = 0; i< 4; i++)
	{
		if(shotIDs[i].frameID != universe->getFrameID())
			shotIDs[i].ID = -1;
		else
			numShots++;
	}
	
	universe->getShip().setNumShots(numShots);
	
	if(numShots >= 4)
	{
		universe->getShip().setNextShot(72-highestLifeTime< nextHit?72-highestLifeTime:nextHit);
	}
	
}

Angle TrackingLayer::getEstimatedAngle()
{
	Angle action = 0;
	if(receivedAction.keys & Key::LEFT)
		action = 3;
	else if(receivedAction.keys & Key::RIGHT)
		action = -3;
	
	if((char)((char)repository->getUDPID())-((char)nextUdpID) > 0)
		action += repository->estimatedAngleOffset(nextUdpID);
	
	return minAngle+ action;
}

Universe* TrackingLayer::buildUniverse()
{
	Ship ship;
	Ufo ufo;
	list<Asteroid> * asteroids= new list<Asteroid>();
	Asteroid tempAsteroid;
	
	list<Shot>* shots = new list<Shot>();
	Shot tempShot;
	
	wxPoint relativePosition;
	wxPoint relativeFlightVector;
	int distance;
	int framesToCollision;
	Angle relativeAngle;
	Matrix* rotationMatrix = 0;
	Matrix* universeMatrix = 0;
	
	if(shipTracking.isUsed())
	{
		ship = Ship(shipTracking.getObjectID(),shipTracking.getCurrentSnapshot().getShipState(),Tools::calcPosition(shipTracking.getCurrentPosition(),shipTracking.getFlightVector(),latency),shipTracking.getCurrentSnapshot().getAngle(),wxPoint(0,0),0,Tools::pythagoras(shipTracking.getFlightVector()),shipTracking.getFlightVector(),wxPoint(0,0),minAngle,maxAngle,getEstimatedAngle());
		
		#ifdef _USE_RELATIVE_VIEW_
		rotationMatrix = Tools::createRotationMatrix(shipTracking.getCurrentSnapshot().getAngle());
		relativePosition = shipTracking.getCurrentPosition();
		relativePosition.x *= -1;
		relativePosition.y *= -1;
		universeMatrix = Tools::createTranslationMatrix(relativePosition,*rotationMatrix);
		#endif
	}
	
	if(ufoTracking.isUsed())
	{
		relativePosition = ufoTracking.getCurrentPosition();
		relativeFlightVector = ufoTracking.getFlightVector();
		relativeAngle = 0;
		distance = -1;
		framesToCollision = -1;
		
		if(ship.getShipState() == alive)
		{
			#ifdef _USE_RELATIVE_VIEW_
			relativePosition = Tools::transformByMatrix(relativePosition, *universeMatrix);
			relativeFlightVector = Tools::transformByMatrix(relativeFlightVector,*rotationMatrix);
			#endif
			distance = Tools::pythagoras(Tools::calcPosition(ufoTracking.getCurrentPosition(),ufoTracking.getFlightVector(),latency)-ship.getAbsolutePosition());
			framesToCollision = Tools::calcCollision(ship.getAbsolutePosition(), ship.getFlightVector(),ship.getSize(),Tools::calcPosition(ufoTracking.getCurrentPosition(),ufoTracking.getFlightVector(),latency),ufoTracking.getFlightVector(), ufoTracking.getCurrentSnapshot().getUfoSize().getValue());
		}
		ufo = Ufo(ufoTracking.getObjectID(),Tools::calcPosition(ufoTracking.getCurrentPosition(),ufoTracking.getFlightVector(),latency),0,relativePosition,relativeAngle,Tools::pythagoras(ufoTracking.getFlightVector()),ufoTracking.getFlightVector(),relativeFlightVector,distance, framesToCollision,ufoTracking.getLifeTime(),ufoTracking.getCurrentSnapshot().getUfoSize());
	}
	
	for(int i = 0; i< 12;i++)
	{
		AsteroidSize size(AsteroidSize::small);
		switch(i %3)
		{
		case 1:
			size = AsteroidSize::medium;
			break;
		case 2:
			size = AsteroidSize::big;
			break;
		}
			
		if(asteroidsTracking[i].size() != 0)
		{
			for(list<TrackingObject<AsteroidSnapshot> >::iterator iteratorAsteroids = asteroidsTracking[i].begin();iteratorAsteroids != asteroidsTracking[i].end(); iteratorAsteroids++)
			{
				relativePosition = iteratorAsteroids->getCurrentPosition();
				relativeFlightVector = iteratorAsteroids->getFlightVector();
				relativeAngle = 0;
				distance = -1; 
				framesToCollision = -1;
				
				if(ship.getShipState() == alive)
				{
					#ifdef _USE_RELATIVE_VIEW_
					relativePosition = Tools::transformByMatrix(relativePosition, *universeMatrix);
					relativeFlightVector = Tools::transformByMatrix(relativeFlightVector, *rotationMatrix);
					#endif
					distance = Tools::pythagoras(Tools::calcPosition(iteratorAsteroids->getCurrentPosition(),iteratorAsteroids->getFlightVector(),latency)- ship.getAbsolutePosition());
					framesToCollision = Tools::calcCollision(ship.getAbsolutePosition(), ship.getFlightVector(), ship.getSize(),Tools::calcPosition(iteratorAsteroids->getCurrentPosition(),iteratorAsteroids->getFlightVector(),latency),iteratorAsteroids->getFlightVector(),size.getValue());
					
				}
				
				tempAsteroid = Asteroid(iteratorAsteroids->getObjectID(),Tools::calcPosition(iteratorAsteroids->getCurrentPosition(),iteratorAsteroids->getFlightVector(),latency),0,relativePosition,relativeAngle,Tools::pythagoras(iteratorAsteroids->getFlightVector()),iteratorAsteroids->getFlightVector(),relativeFlightVector,distance, framesToCollision,iteratorAsteroids->getLifeTime(),size);
				asteroids->push_back(tempAsteroid);
			}
		}
	}
	
	if(shotsTracking.size() != 0)
	{
		for(list<TrackingObject<ShotSnapshot> >::iterator iteratorShots = shotsTracking.begin(); iteratorShots != shotsTracking.end(); iteratorShots++)
		{
			relativePosition = iteratorShots->getCurrentPosition();
			relativeFlightVector = iteratorShots->getFlightVector();
			relativeAngle = 0;
			distance = -1;
			framesToCollision = -1;
			
			if(ship.getShipState() == alive)
			{
				#ifdef _USE_RELATIVE_VIEW_
				relativePosition = Tools::transform(relativePosition,ship->getAbsolutePosition(),ship->getAbsoluteAngle());
				relativeFlightVector = Tools::transformByMatrix(relativeFlightVector,*rotationMatrix);
				#endif
				distance = Tools::pythagoras(Tools::calcPosition(iteratorShots->getCurrentPosition(),iteratorShots->getFlightVector(),latency) - ship.getAbsolutePosition());
				framesToCollision = Tools::calcCollision(ship.getAbsolutePosition(),ship.getFlightVector(),ship.getSize(),Tools::calcPosition(iteratorShots->getCurrentPosition(),iteratorShots->getFlightVector(),latency), iteratorShots->getFlightVector(),1);
			}
			
			tempShot = Shot(iteratorShots->getObjectID(),Tools::calcPosition(iteratorShots->getCurrentPosition(),iteratorShots->getFlightVector(),latency),0,relativePosition,relativeAngle,Tools::pythagoras(iteratorShots->getFlightVector()),iteratorShots->getFlightVector(),relativeFlightVector,distance, framesToCollision,iteratorShots->getLifeTime(),iteratorShots->isExactFlightVector());
			shots->push_back(tempShot);
		}
	}
	
	delete rotationMatrix;
	delete universeMatrix;
	
	shoot++;
	if(repository->getLastOutput() & Key::SHOOT)
		shoot = 0;
	
	return new Universe(latestFrameID,ship,asteroids,shots,ufo,latency,!(repository->getLastOutput()&Key::HYPERSPACE), shoot >= 1,receivedAction.keys);
}

template<class T>
void TrackingLayer::matchFlyingObject(list<T> & snapshots, list< TrackingObject<T > >&tracking)
{
	list<SortedList<MatchingRelation< int > > > distances;
	list<SortedList<MatchingRelation< int > > >::iterator iteratorDistance;
	
	SortedList<MatchingRelation< int > > tempList;
	TrackingObject<T>  tempTrackingObject;
	
	vector< SnapshotMatchingRelation<typename list< T >::iterator,typename list< SortedList< MatchingRelation< int > > >::iterator> > bestMatching(tracking.size());
	typename vector< SnapshotMatchingRelation< typename list< T >::iterator, typename list< SortedList< MatchingRelation< int > > >::iterator> >::iterator iteratorMatching; 
	
	typename list<TrackingObject<T> >::iterator iteratorTracking;
	typename list<T>::iterator iteratorSnapshot;
	
	int tempDist;
	int indexTracking;
	
	for(iteratorSnapshot = snapshots.begin(); iteratorSnapshot != snapshots.end();iteratorSnapshot++)
	{
		tempList = SortedList<MatchingRelation< int > >();
		distances.push_back(tempList);
		iteratorDistance = distances.end();
		iteratorDistance--;
		
		for(iteratorTracking = tracking.begin(),indexTracking = 0;iteratorTracking != tracking.end();iteratorTracking++, indexTracking++)
		{
			if(bestMatching[indexTracking].getDistance() != 0)
			{
				tempDist = iteratorTracking->matchTo(*iteratorSnapshot);
				
				if(tempDist!= -1)
				{
					if(tempDist < 100)
					{
						iteratorDistance->addElement(new MatchingRelation< int >(0,indexTracking));
						break;
					}
					iteratorDistance->addElement(new MatchingRelation< int >(tempDist,indexTracking));	
				}
			}
		}
		updateMatching<T>(iteratorSnapshot,iteratorDistance,bestMatching);		
	}
	
	//Elemente aktualisieren
	for(indexTracking = 0, iteratorTracking = tracking.begin(); iteratorTracking != tracking.end();iteratorTracking++,indexTracking++)
	{
		if(bestMatching[indexTracking].getDistance() != -1)
		{
			//wxLogDebug("Matching updated: Old:%d %d New: %d %d FrameID: %d",iteratorTracking->getCurrentPosition().x,iteratorTracking->getCurrentPosition().y,bestMatching[indexTracking].getSnapshotMatching()->getPos().x,bestMatching[indexTracking].getSnapshotMatching()->getPos().y,bestMatching[indexTracking].getSnapshotMatching()->getFrameID());
			iteratorTracking->insertSnapshot(*bestMatching[indexTracking].getSnapshotMatching());
			snapshots.erase(bestMatching[indexTracking].getSnapshotMatching());
		}
	}
	
	//Elemente loeschen, die nicht mehr vorhanden sind	
	iteratorTracking = tracking.begin();
	while(iteratorTracking != tracking.end())
	{
		if(iteratorTracking->getFrameID() != latestFrameID)
		{
			//wxLogDebug("Asteroid destroyed: %d %d FrameID: %d",iteratorTracking->getCurrentPosition().x,iteratorTracking->getCurrentPosition().y,iteratorTracking->getFrameID());
			iteratorTracking = tracking.erase(iteratorTracking);
		}
		else
			iteratorTracking++;
	}
	
	//Elemente hinzufuegen, die neu sind
	iteratorSnapshot = snapshots.begin();
	
	while(iteratorSnapshot != snapshots.end())
	{
		//wxLogDebug("Asteroid inserted: %d %d", iteratorSnapshot->getPos().x,iteratorSnapshot->getPos().y);
		tempTrackingObject = TrackingObject<T>(*iteratorSnapshot,currentObjectID++);
		tracking.push_back(tempTrackingObject);
		
		iteratorSnapshot++;
	}
}

template<class T>
void TrackingLayer::updateMatching(typename list< T>::iterator& snapshot,list < SortedList < MatchingRelation < int > > >::iterator& currentDistance,vector< SnapshotMatchingRelation<typename list< T >::iterator, list< SortedList< MatchingRelation< int > > >::iterator> >& matching)
{
	typename vector< SnapshotMatchingRelation<typename list< T >::iterator, list< SortedList< MatchingRelation< int > > >::iterator> >::iterator iteratorMatching;
	SortedList < MatchingRelation < int > >::Enumerator & enumerator = currentDistance->getEnumerator();
	
	SnapshotMatchingRelation<typename list< T >::iterator, list< SortedList< MatchingRelation< int > > >::iterator> *tempSnapshotMatchingRelation;
	
	typename list< T>::iterator iteratorSnapshot;
	list < SortedList < MatchingRelation < int > > >::iterator iteratorDistance;
	
	while(enumerator.hasNext())
	{
		MatchingRelation<int >& currentMatchingRelation = enumerator.getNext();
		
		if(matching[currentMatchingRelation.getMatchingRelation()].getDistance() == -1)
		{
			tempSnapshotMatchingRelation = new SnapshotMatchingRelation<typename list< T >::iterator, list< SortedList< MatchingRelation< int > > >::iterator>(snapshot,currentMatchingRelation.getDistance(),currentDistance);
			matching[currentMatchingRelation.getMatchingRelation()] = *tempSnapshotMatchingRelation;
			delete tempSnapshotMatchingRelation;
			break;
		}
		else
		{
			if(matching[currentMatchingRelation.getMatchingRelation()].getDistance() > currentMatchingRelation.getDistance())
			{
				iteratorSnapshot = matching[currentMatchingRelation.getMatchingRelation()].getSnapshotMatching();
				iteratorDistance = matching[currentMatchingRelation.getMatchingRelation()].getMatchingRelation();
				
				matching[currentMatchingRelation.getMatchingRelation()].setSnapshotMatching(snapshot);
				matching[currentMatchingRelation.getMatchingRelation()].setMatchingRelation(currentDistance);
				matching[currentMatchingRelation.getMatchingRelation()].setDistance(currentMatchingRelation.getDistance());
				updateMatching<T>(iteratorSnapshot,iteratorDistance,matching);
				break;
			}
		}
	}
}

void TrackingLayer::matchShip(ShipSnapshot & ship)
{
	if(ship.getFrameID() == -1)
	{
		shipTracking.clear();
	}
	else
	{
		if(shipTracking.isUnused())
			shipTracking.insertSnapshot(ship,currentObjectID++);
		else
			shipTracking.insertSnapshot(ship);
	}
}


void TrackingLayer::matchUfo(UfoSnapshot &ufo)
{
	
	if(ufo.getFrameID() == -1)
	{
		ufoTracking.clear();
	}
	else
	{
		if(ufoTracking.isUnused())
			ufoTracking.insertSnapshot(ufo,currentObjectID++);
		else
			ufoTracking.insertSnapshot(ufo);
	}
}

void TrackingLayer::updateShipAngles()
{
	if(receivedAction.keys & Key::HYPERSPACE)
	{
		return;
	}
	
	if(receivedAction.keys& Key::LEFT)
	{
		minAngle += 3;
		maxAngle += 3;
	}
	
	if(receivedAction.keys & Key::RIGHT)
	{
		minAngle -= 3;
		maxAngle -= 3;
	}
}

void TrackingLayer::calibrateShipAngles()
{
	if(snapshot->getShip().getFrameID() == -1)
		return;
	
	if((char)(minAngle - Tools::minInternAngle(snapshot->getShip().getAngle())) >= 0 &&  (char)(Tools::maxInternAngle(snapshot->getShip().getAngle()) - minAngle) >= 0)
	{
		//wxLogDebug("minAngle %d %d",minAngle - minInternAngle(snapshot->getShip().getAngle()),maxInternAngle(snapshot->getShip().getAngle()) - minAngle);
		if((char)(maxAngle - Tools::minInternAngle(snapshot->getShip().getAngle())) < 0 || (char)(Tools::maxInternAngle(snapshot->getShip().getAngle()) - maxAngle) < 0)
		{
			TrackingLog("Adjust maxAngle " << (int)(maxAngle - Tools::minInternAngle(snapshot->getShip().getAngle())) << " " <<(int) (Tools::maxInternAngle(snapshot->getShip().getAngle()) - maxAngle));
			maxAngle = Tools::maxInternAngle(snapshot->getShip().getAngle());
		}
	}
	else if((char)(maxAngle - Tools::minInternAngle(snapshot->getShip().getAngle())) >= 0 && (char)(Tools::maxInternAngle(snapshot->getShip().getAngle()) - maxAngle) >= 0)
	{
		//wxLogDebug("maxAngle %d %d",maxAngle - minInternAngle(snapshot->getShip().getAngle()),maxInternAngle(snapshot->getShip().getAngle()) - maxAngle);
		if((char)(minAngle - Tools::minInternAngle(snapshot->getShip().getAngle())) < 0 ||  (char)(Tools::maxInternAngle(snapshot->getShip().getAngle()) - minAngle) < 0)
		{ 
			TrackingLog("Adjust minAngle " <<  (int)(minAngle - Tools::minInternAngle(snapshot->getShip().getAngle())) << ", " << (int)(Tools::maxInternAngle(snapshot->getShip().getAngle()) - minAngle));
			minAngle = Tools::minInternAngle(snapshot->getShip().getAngle());
		}
	}
	else
	{
		TrackingLog("FrameID " << latestFrameID << " Adjust both minAngle " << (int)minAngle << " new minAngle " << (int)Tools::minInternAngle(snapshot->getShip().getAngle()) << " maxAngle " << (int)Tools::maxInternAngle(snapshot->getShip().getAngle()) << ")");
		minAngle = Tools::minInternAngle(snapshot->getShip().getAngle());
		maxAngle = Tools::maxInternAngle(snapshot->getShip().getAngle());
	}
}

