package extern;

import java.util.ArrayList;
import java.util.List;

/**
 * The <code>GameStatus</code> class represents game status which
 * can be decoded from a <code>FramePacket</code>.
 * <p>
 * It describes the currently visible ship, saucer, asteroids and shots.
 */
public class GameStatus implements Cloneable {
	protected int ticks;
	protected Ship ship = null;
	protected Saucer saucer = null;
	protected List<Asteroid> asteroids = new ArrayList<Asteroid>();
	protected List<Shot> shots = new ArrayList<Shot>();
	protected String displayText = "";
	protected int lastPoints = -1;

	private void clear() {
		ticks = 0;
		ship = null;
		saucer = null;
		asteroids.clear();
		shots.clear();
	}

	/**
	 * Fills this game status by interpreting the vector ram from <code>framePacket</code>.
	 *
	 * @param framePacket
	 * 			the <code>FramePacket</code> to interpret.
	 * @param ticks
	 * 			the current game-time for this game status
	 */
	public void interpretScreen(FramePacket framePacket, int ticks) {
		int dx = 0, dy = 0, vx = 0, vy = 0, vz = 0, vs = 0;
		int v1x = 0;
		int v1y = 0;
		int shipdetect = 0;

		this.clear();
		this.ticks = ticks;

		final int initialJump = framePacket.readVectorRAMWord(0);
		if (initialJump != 0xe001 && initialJump != 0xe201)
			return; // should not occur, first instruction is always a jump
		
		String displayText = "";
		for (int pc = 1; true; ) {
			final int currentWord = framePacket.readVectorRAMWord(pc);
			int op = currentWord >> 12;
			switch (op) {
			case 0xa: { // LABS
				final int nextWord = framePacket.readVectorRAMWord(pc + 1);
				vy = currentWord & 0x3ff;
				vx = nextWord & 0x3ff;
				vs = nextWord >> 12;
				break;
			}
			case 0xb: // HALT
				this.displayText = displayText;
				setLastPoints();
				return;
			case 0xc: // JSRL
				Asteroid.Size saucerSize = Asteroid.Size.BIG;
				if (vs == 15) saucerSize = Asteroid.Size.MIDDLE;
				else if (vs == 14) saucerSize = Asteroid.Size.SMALL;

				switch (currentWord & 0xfff) {
				case 0x8f3:
					asteroids.add(new Asteroid(vx, vy, Asteroid.Type.TYPE_1, saucerSize));
					break;
				case 0x8ff:
					asteroids.add(new Asteroid(vx, vy, Asteroid.Type.TYPE_2, saucerSize));
					break;
				case 0x90d:
					asteroids.add(new Asteroid(vx, vy, Asteroid.Type.TYPE_3, saucerSize));
					break;
				case 0x91a:
					asteroids.add(new Asteroid(vx, vy, Asteroid.Type.TYPE_4, saucerSize));
					break;
				case 0x929:
					saucer = new Saucer(vx, vy, vs == 15? Saucer.Size.BIG: Saucer.Size.SMALL);
					break;
				case 0xA78: displayText = displayText + "A"; break;
				case 0xA80: displayText = displayText + "B"; break;
				case 0xA8D: displayText = displayText + "C"; break;
				case 0xA93: displayText = displayText + "D"; break;
				case 0xA9B: displayText = displayText + "E"; break;
				case 0xAA3: displayText = displayText + "F"; break;
				case 0xAAA: displayText = displayText + "G"; break;
				case 0xAB3: displayText = displayText + "H"; break;
				case 0xABA: displayText = displayText + "I"; break;
				case 0xAC1: displayText = displayText + "J"; break;
				case 0xAC7: displayText = displayText + "K"; break;
				case 0xACD: displayText = displayText + "L"; break;
				case 0xAD2: displayText = displayText + "M"; break;
				case 0xAD8: displayText = displayText + "N"; break;
				case 0xADD: displayText = displayText + "O"; break;
				case 0xAE3: displayText = displayText + "P"; break;
				case 0xAEA: displayText = displayText + "Q"; break;
				case 0xAF3: displayText = displayText + "R"; break;
				case 0xAFB: displayText = displayText + "S"; break;
				case 0xB02: displayText = displayText + "T"; break;
				case 0xB08: displayText = displayText + "U"; break;
				case 0xB0E: displayText = displayText + "V"; break;
				case 0xB13: displayText = displayText + "W"; break;
				case 0xB1A: displayText = displayText + "X"; break;
				case 0xB1F: displayText = displayText + "Y"; break;
				case 0xB26: displayText = displayText + "Z"; break;
				case 0xB2C: displayText = displayText + " "; break;
				case 0xB2E: displayText = displayText + "1"; break;
				case 0xB32: displayText = displayText + "2"; break;
				case 0xB3A: displayText = displayText + "3"; break;
				case 0xB41: displayText = displayText + "4"; break;
				case 0xB48: displayText = displayText + "5"; break;
				case 0xB4F: displayText = displayText + "6"; break;
				case 0xB56: displayText = displayText + "7"; break;
				case 0xB5B: displayText = displayText + "8"; break;
				case 0xB63: displayText = displayText + "9"; break;
				}
				break;
			case 0xd: // RTSL
				this.displayText = displayText;
				setLastPoints();
				return;
			case 0xe: // JMPL
				// pc = currentWord & 0xfff;
				// break;
				this.displayText = displayText;
				setLastPoints();
				return;
			case 0xf: // SVEC
				if (false) {
					dy = currentWord & 0x300;
					if ((currentWord & 0x400) != 0)	dy = -dy;
					dx = (currentWord & 3) << 8;
					if ((currentWord & 4) != 0)	dx = -dx;
					//sf = (((currentWord & 8) >> 2) | ((currentWord & 0x800) >> 11)) + 2;
					vz = (currentWord & 0xf0) >> 4;
				}
				break;
			default:
				{
					final int nextWord = framePacket.readVectorRAMWord(pc + 1);
					dy = currentWord & 0x3ff;
					if ((currentWord & 0x400) != 0)	dy = -dy;
					dx = nextWord & 0x3ff;
					if ((nextWord & 0x400) != 0) dx = -dx;
					//sf = op;
					vz = nextWord >> 12;
					if (dx == 0 && dy == 0 && vz == 15)
						shots.add(new Shot(vx, vy));
					if (op == 6 && vz == 12 && dx != 0 && dy != 0) {
						switch (shipdetect) {
						case 0:
							v1x = dx;
							v1y = dy;
							++shipdetect;
							break;
						case 1:
							ship = new Ship(vx, vy, v1x - dx, v1y - dy);
							++shipdetect;
							break;
						}
					} else if (shipdetect == 1)
						shipdetect = 0;
					break;
				}
			}
			if (op <= 0xa) ++pc; // two word instruction
			if (op != 0xe) ++pc; // not JMPL
		}
	}

	@Override
	public String toString() {
		StringBuilder stringBuilder = new StringBuilder();
		if (ship != null) {
			stringBuilder.append("ship:")
				.append(ship.getX()).append(",")
				.append(ship.getY()).append(";")
				.append(ship.getOrientation().getX()).append(",")
				.append(ship.getOrientation().getY()).append("\n");
		}
		if (saucer != null) {
			stringBuilder.append("saucer:")
				.append(saucer.getX()).append(",")
				.append(saucer.getY()).append(";")
				.append(saucer.getSize()).append("\n");
		}
		for (Asteroid asteroid: asteroids) {
			stringBuilder.append("asteroid:")
				.append(asteroid.getX()).append(",")
				.append(asteroid.getY()).append(";")
				.append(asteroid.getType()).append(";")
				.append(asteroid.getSize()).append("\n");
		}
		for (Shot shot: shots) {
			stringBuilder.append("shot:")
				.append(shot.getX()).append(",")
				.append(shot.getY()).append("\n");
		}
		return stringBuilder.toString();
	}

	/**
	 * Returns the current game-time.
	 *
	 * @return the current game-time
	 */
	public int getTicks() {
		return ticks;
	}

	/**
	 * Returns the current ship.
	 *
	 * @return the current ship, or <code>null</code> if none is found.
	 */
	public Ship getShip() {
		return ship;
	}

	/**
	 * Returns the current saucer.
	 *
	 * @return the current saucer, or <code>null</code> if none is found.
	 */
	public Saucer getSaucer() {
		return saucer;
	}

	/**
	 * Returns the current list of asteroids.
	 *
	 * @return the current list of asteroids.
	 */
	public List<Asteroid> getAsteroids() {
		return asteroids;
	}

	/**
	 * Returns the current list of shots.
	 *
	 * @return the current list of shots.
	 */
	public List<Shot> getShots() {
		return shots;
	}
	
	public String getText() {
		return displayText;
	}
	
	public int getCurrentPoints() {
		String tmp = displayText;
		tmp = tmp.replace("O", "0");
		String[] fields = tmp.split(" ");
		int ret = -1;
		for (int i=0; i<fields.length; i++) {
			if (fields[i].length() > 0) {
				try {
					ret = Integer.valueOf(fields[i]);
					break;
				} catch (NumberFormatException e) {
					ret = -1;
					break;
				}
			}
		}
		return ret;
	}
	
	public void setLastPoints() {
		int tmp = getCurrentPoints();
		if (tmp >= 0) {
			lastPoints = tmp;
		}
	}
	
	public int getLastPoints() {
		return lastPoints;
	}
}
