package extern;

/**
 * The <code>Position</code> class represents simple 2D vector.
 * <p>
 * Some basic vector operations are provided.
 */
public class Position {
	private int x;
	private int y;

	/**
	 * Initializes the position (0, 0).
	 */
	public Position() {
		this(0, 0);
	}

	/**
	 * Initializes the position (x, y).
	 *
	 * @param x
	 * 			x-coordinate
	 * @param y
	 * 			y-coordinte
	 */
	public Position(int x, int y) {
		super();
		this.x = x;
		this.y = y;
	}

	/**
	 * Returns the x-coordinate.
	 *
	 * @return x-coordinate
	 */
	final public int getX() {
		return x;
	}

	/**
	 * Sets the x-coordinate.
	 *
	 * @param x
	 * 			the x-coordinate
	 */
	final public void setX(int x) {
		this.x = x;
	}

	/**
	 * Returns the y-coordinate.
	 *
	 * @return y-coordinate
	 */
	final public int getY() {
		return y;
	}

	/**
	 * Sets the y-coordinate
	 *
	 * @param y
	 * 			the y-coordinate
	 */
	final public void setY(int y) {
		this.y = y;
	}

	/** Substract vector <code>a</code> from <code>b</code>.
	 *
	 * @param a
	 * 			a vector
	 * @param b
	 * 			a vector
	 * @return the vector pointing from <code>b</code> to <code>a</code>
	 */
	public static Position substract(Position a, Position b) {
		return new Position(a.x - b.x, a.y - b.y);
	}

	/** Normalizes the vector.
	 *
	 *  The resulting vector holds <code>(minX <= x <= maxX, minY <= y <= maxY)</code>.
	 *
	 * @param minX
	 * 			minimum x value
	 * @param maxX
	 * 			maximum x value
	 * @param minY
	 * 			minimum y value
	 * @param maxY
	 * 			maximum y value
	 */
	public void normalize(int minX, int maxX, int minY, int maxY) {
		// normalize x: minX ... maxX
		while (x < minX) x += maxX - minX;
		while (x > maxX) x -= maxX - minX;
		// normalize y: minY ... maxY
		while (y < minY) y += maxY - minY;
		while (y > maxY) y -= maxY - minY;
	}

	/**
	 * Calculates the dot product for <code>a<code> and <code>b<code>.
	 *
	 * Results in <code>a.x * b.x + a.y * b.y</code>.
	 *
	 * @param a
	 * 			a vector
	 * @param b
	 * 			a vector
	 * @return the dot product for <code>a<code> and <code>b<code>
	 */
	public static int dotProduct(Position a, Position b) {
		return a.x * b.x + a.y * b.y;
	}

	/**
	 * Calculates the 2D cross product for <code>a<code> and <code>b<code>.
	 *
	 * Results in <code>a.x * b.y - a.y * b.x</code>.
	 *
	 * @param a
	 * 			a vector
	 * @param b
	 * 			a vector
	 * @return the 2D cross product for <code>a<code> and <code>b<code>
	 */
	public static int crossProduct2D(Position a, Position b) {
		return a.x * b.y - a.y * b.x;
	}
}
