package extern;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;

/**
 * The <code>ServerConnection</code> class implements the network communication with the server.
 * <p>
 * It uses blocking UDP sockets.
 */
public class ServerConnection {
	public final int SERVER_PORT = 1979;
	protected DatagramSocket socket;
	private boolean stopped = false;
	private String servername = "";

	/**
	 * Initializes this connection with server <code>serverName</code>.
	 *
	 * @param serverName
	 * 			the name of the server
	 * @throws SocketException
	 * @throws UnknownHostException
	 */
	public ServerConnection(String serverName) {
		this.open(serverName);
	}
	
	public void open(String serverName) {
		this.servername = serverName;
		try {
			this.socket = new DatagramSocket();
		} catch (SocketException e) {
		}
		try {
			this.socket.connect(InetAddress.getByName(this.servername), SERVER_PORT);
		} catch (UnknownHostException e) {
		}
	}

	/**
	 * Closes the network connection. Must be called to free resources.
	 */
	public void close() {
		socket.close();
	}

	/**
	 * Receive a <code>FramePacket</code> from the server.
	 *
	 * This call blocks and waits until a packet with the right size arrives.
	 *
	 * @param framePacket
	 * 			the <code>FramePacket</code> to be filled with the received data
	 * @throws IOException
	 * @see FramePacket
	 */
	public void receive(FramePacket framePacket) throws IOException {
		if (!this.socket.isConnected()) {
			try {
				this.socket.connect(InetAddress.getByName(this.servername), SERVER_PORT);
			} catch (UnknownHostException e) {
				System.out.println("ServerConnection.receive(): ERROR: can't connect with game");
				return;
			}
		}
		DatagramPacket datagramPacket = framePacket.getDatagramPacket();
		stopped = false;
		while (!stopped) {
			socket.receive(datagramPacket);
			if (datagramPacket.getLength() != FramePacket.SIZE) {
				System.out.println("ServerConnection.receive(): ERROR: received packet with wrong size");
				continue;
			}
			break;
		}
	}

	/**
	 * Send the <code>keysPacket</code> data to the server.
	 *
	 * @param keysPacket
	 * 			the data to be sent to the server
	 * @throws IOException
	 * @see KeysPacket
	 */
	public void send(KeysPacket keysPacket) throws IOException {
		if (!this.socket.isConnected()) {
			try {
				this.socket.connect(InetAddress.getByName(this.servername), SERVER_PORT);
			} catch (UnknownHostException e) {
				System.out.println("ServerConnection.receive(): ERROR: can't connect with game");
				return;
			}
		}
		socket.send(keysPacket.getDatagramPacket());
	}
	
	public void stop() {
		stopped = true;
		this.socket.close();
	}

	public void setServername(String servername) {
		this.close();
		this.open(servername);
	}
}
