package mainClasses;

import java.io.File;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.concurrent.Semaphore;

public class ExecutionManager extends Thread {

	public enum StartingDirection {
		FORWARD, BACKWARD
	}

	private List<ServerController> serverList = null;
	
	private GAPopulation population = null;
	
	private GANextGenerationGenerator nextGenerator = null;
	
	private Semaphore nextProgramSemaphore = null;
	
	private Random random = null;
	
	private String filename = "";
	
	private GASave saver = null;
	
	private boolean justStartBest = false;
	
	private int justStartNumber = -1;
	
	private StartingDirection startingDirection = StartingDirection.FORWARD;
	
	private String identifier = "";
	
	public ExecutionManager() {
		serverList = new ArrayList<ServerController>();
		nextProgramSemaphore = new Semaphore(1);
		random = new Random();
		identifier = createIdentifier();
	}
	
	public void addServer(ServerController addServer) {
		serverList.add(addServer);
	}
	
	public GAPopulation setPopulation(GAPopulation population) {
		this.population = population;
		return this.population;
	}
	
	public GAPopulation getPopulation() {
		return this.population;
	}
	
	public GAProgram getNextProgram(GAProgram prev) {
		if (justStartBest) {
			return getBestProgram();
		}
		if (justStartNumber > -1) {
			return getProgramNumber();
		}
		if (prev != null) {
			if (prev.isStarted() == true) {
				// Is old Program not rated
				return prev;
			}
		}
		GAProgram ret = null;
		try {
			nextProgramSemaphore.acquire();
		} catch (InterruptedException e) {
		}
		ret = getNextUnratedUnstartedProgram(prev, true);
		if (ret != null) {
			createIdentifier(ret);
			ret.setStarted();
		} else {
			if (generationFinished()) {
				createNextGeneration();
				ret = getNextUnratedUnstartedProgram(prev, true);
				if (ret != null) {
					createIdentifier(ret);
					ret.setStarted();
				}
			} else {
				ret = getNextUnratedUnstartedProgram(prev, false);
				if (ret != null) {
					createIdentifier(ret);
					ret.setStarted();
				}
			}
		}
		nextProgramSemaphore.release();
		return ret;
	}
	
	private GAProgram getBestProgram() {
		if (population == null) {
			return null;
		}

		GAProgram best = null;
		for (int i=0; i<population.getSize(); i++) {
			GAProgram program = population.get(i);
			if (best == null) {
				best = program;
			} else if (best.getRating() < program.getRating()) {
				best = program;
			}
		}
		return best;
	}
	
	private GAProgram getProgramNumber() {
		if (this.population == null) {
			return null;
		}
		
		if (this.justStartNumber < this.population.getSize()) {
			return this.population.get(this.justStartNumber);
		} else {
			return null;
		}
	}

	private boolean generationFinished() {
		if (population == null) {
			return true;
		}
	
		for (int i=0; i<population.getSize(); i++) {
			GAProgram program = population.get(i);
			// if no Rating or program is started (= running) then not finished
			if ((program.getRating() < 0) || (program.isStarted() == true)) {
				return false;
			}
		}
		return true;
	}

	private void createNextGeneration() {
		if (this.population == null) {
			return;
		}
		
		if (!(this.population.getSize() > 0)) {
			return;
		}
		
		if ((this.population.getSize() > 0) && (this.saver != null) && (!this.filename.equals(""))) {
			this.saver.save(this.filename + ".g" + this.population.getGeneration() + ".ga", this);
		}
		
		Date d = new Date();
		DateFormat df = DateFormat.getTimeInstance( DateFormat.MEDIUM, Locale.GERMANY );

		if (this.nextGenerator != null) {
			System.out.println(df.format(d) + " ExecutionManager.createNextGeneration(): create next generation");
			this.population.sortPopulation();
			this.nextGenerator.createNextGeneration(this.population);
		} else {
			System.out.println(df.format(d) + " ExecutionManager.createNextGeneration(): ERROR: no generator");
		}
		
		if ((this.population.getSize() > 0) && (this.saver != null) && (!this.filename.equals(""))) {
			while (true) {
				String tmpFilename = this.filename + ".g" + this.population.getGeneration() + ".ga";
				File file = new File(tmpFilename);
				if (file.exists()) {
					this.filename = this.filename + ".g" + this.population.getGeneration();
				} else {
					this.saver.save(tmpFilename, this);
					break;
				}
			}
		}
	}

	private boolean isFinished(StartingDirection startingDirection, int end, int i) {
		if (startingDirection == StartingDirection.FORWARD) {
			return (i < end);
		} else {
			return (i > end);
		}
	}
	
	private GAProgram getNextUnratedUnstartedProgram(GAProgram prev, boolean checkPrev) {
		if (population == null) {
			return null;
		}

		Date d = new Date();
		DateFormat df = DateFormat.getTimeInstance( DateFormat.MEDIUM, Locale.GERMANY ); 
	
		int start = 0; 
		int end = 0;
		int increment = 1;
		
		switch (startingDirection) {
		case FORWARD: start = 0; end = population.getSize(); increment = 1; break;
		case BACKWARD: start = population.getSize()-1; end = -1; increment = -1; break;
		}
		
		for (int i=start; isFinished(startingDirection, end, i); i = i + increment) {
			GAProgram program = population.get(i);
			if (program != null) {
				if ((program.getRating() < 0) && (program.isStarted() == false)) {
					if (checkPrev) {
						if (program != prev) {
							System.out.println(df.format(d) + " ExecutionManager.getNextUnratedUnstartedProgram: (1) " + i + ".");
							return program;
						}
					} else {
						System.out.println(df.format(d) + " ExecutionManager.getNextUnratedUnstartedProgram: (2) " + i + ".");
						return program;
					}
				}
				// timeout
				if ((program.getTimeout() < d.getTime()) && (program.isStarted() == true)) {
					program.setNotStarted();
				}
			}
		}
		return null;
	}

	public ServerController getServer(int i) {
		if (serverList != null) {
			return serverList.get(i);
		}
		return null;
	}

	public int getServerCount() {
		if (serverList != null) {
			return serverList.size();
		}
		return 0;
	}

	private String createIdentifier() {
		String letters = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
		String newIdentifier = "";
		for (int i=0; i<32; i++) {
			int pos = random.nextInt(letters.length());
			newIdentifier = newIdentifier + letters.charAt(pos);
		}
		return newIdentifier;
	}
	
	private void createIdentifier(GAProgram program) {
		program.setIdentifier(createIdentifier());
	}

	public void setRatingFinished(String identifier, int rating) {
		if (population == null) {
			return;
		}
		
		if ((this.justStartBest) || (this.justStartNumber > -1)) {
			// only the best program is started again and again
			// so no rating is usefull
			return;
		}
	
		try {
			nextProgramSemaphore.acquire();
		} catch (InterruptedException e) {
		}

		population.setRating(identifier, rating);
		
		if ((population.getSize() > 0) && (this.saver != null) && (!this.filename.equals(""))) {
			this.saver.save(this.filename + ".g" + population.getGeneration() + ".ga", this);
		}

		nextProgramSemaphore.release();
	}
	
	public GANextGenerationGenerator setNextGenerationGenerator(GANextGenerationGenerator generator) {
		this.nextGenerator = generator;
		return this.nextGenerator;
	}
	
	public GANextGenerationGenerator getNextGenerationGenerator() {
		return this.nextGenerator;
	}

	public void doShutdown() {
		for (int i=0; i<serverList.size(); i++) {
			serverList.get(i).doShutdown();
		}
	}
	
	public String setFilename(String filename) {
		this.filename = filename;
		if (this.filename.endsWith(".ga")) {
			this.filename = this.filename.substring(0,this.filename.length()-3);
		}
		int index = this.filename.lastIndexOf(".g");
		try {
			Integer.valueOf(this.filename.substring(index+2, this.filename.length()));
			this.filename = this.filename.substring(0,index);
		} catch (NumberFormatException e) {
		}
		return this.filename;
	}
	
	public void setSaver(GASave saver) {
		this.saver = saver;
	}
	
	public void setJustStartBest(boolean doit) {
		this.justStartBest = doit;
	}
	
	public boolean getJustStartBest() {
		return this.justStartBest;
	}
	
	public int setJustStartNumber(int number) {
		this.justStartNumber = number;
		return this.justStartNumber;
	}
	
	public int getJustStartNumber() {
		return this.justStartNumber;
	}
	
	public StartingDirection setStartingDirection(StartingDirection startingDirection) {
		this.startingDirection = startingDirection;
		return this.startingDirection;
	}
	
	public StartingDirection getStartingDirection() {
		return this.startingDirection;
	}
	
	public boolean merge(ExecutionManager second) {
		GAPopulation popSecond = second.getPopulation();
		// same population size
		if (this.population.getSize() != popSecond.getSize()) {
			return false;
		}
		int size = this.population.getSize();
		// all programs are equals
		for (int i=0; i<size; i++) {
			String programFirst = this.population.get(i).getProgramString();
			String programSecond = popSecond.get(i).getProgramString();
			if (!programFirst.equals(programSecond)) {
				return false;
			}
		}
		// copy rating (when not yet rated)
		for (int i=0; i<size; i++) {
			GAProgram progFirst = this.population.get(i); 
			if (progFirst.getRating() < 0) {
				// not yet rated
				GAProgram progSecond = popSecond.get(i);
				for (int j=0; j<progSecond.getCurrentRatingCount(); j++) {
					progFirst.addRating(progSecond.getSubRating(j));
				}
			}
		}
		return true;
	}
	
	public String getIdentifier() {
		return this.identifier;
	}
}
