package simulation;

import global.CreateProgramTree;
import global.Logger;

import java.io.IOException;
import java.io.StringReader;
import java.text.DateFormat;
import java.util.Date;
import java.util.Locale;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import mainClasses.GALibrary;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import simulation.SimulationExecutor.ExecutorStatus;

import commands.AICommand;
import commands.IAICommand;

public class SimulationController extends Thread {

	public enum Status {
		UNKNOWN, JUSTBEGIN, CREATED, INITIALIZED, STARTED, FINISHED, STOPPED
	}
	
	// JUSTBEGIN -> CREATED -> INITIALIZED -> STARTED -> FINISHED -> CREATED -> ...
	
	private SimulationExecutor executor = null;
	
	private ICommunication communicator = null;
	
	private ICommunicationToSimulation communicator2Simulation = null;
	
	private Logger logger = null;
	
	private CreateProgramTree creater = null;
	
	private SimulationDisplay display = null;
	
	private Thread displayThread = null;
	
	private boolean shutdown = false;

	private Status status = Status.UNKNOWN;
	
	private String currentManagerIdentifier = "";
	
	private String currentProgramIdentifier = "";
	
	private GALibrary library = null;
	
	public SimulationController() {
		this.setName("SimulationController");
	}
	
	public void setCommunicator(ICommunication iCommunicator) {
		communicator = iCommunicator;
	}
	
	public void setCommunicationToSimulation(ICommunicationToSimulation iCommunicatorToSimulation) {
		communicator2Simulation = iCommunicatorToSimulation;
		if (executor != null) {
			executor.setCommunicator(communicator2Simulation);
		}
	}
	
	public void setLogger(Logger newLogger) {
		logger = newLogger;
	}
	
	public void setCreater(CreateProgramTree newCreater) {
		creater = newCreater;
	}

	public void setDisplay(SimulationDisplay newDisplay) {
		display = newDisplay;
		if (displayThread != null) {
			// Stop thread
		}
		displayThread = new Thread(display);
		displayThread.setName("SimulationDisplay");
		if (executor != null) {
			executor.setDisplay(display);
		}
	}
	
	public void setLibrary(GALibrary library) {
		this.library = library;
	}
	
	public void doShutdown() {
		doShutdown(null);
	}
	
	public void doShutdown(Node node) {
		shutdown = true;
		if (display != null) {
			display.doShutdown();
		}
		if (executor != null) {
			executor.shutdown = true;
			executor.interrupt();
		}
		if (communicator != null) {
			communicator.doShutdown();
		}
		interrupt();
	}
	
	public void doCreate(Node node) {
		if ((executor == null) || (status == Status.FINISHED)) {
			try {
				String m = ((Element)node).getAttribute("manager").trim();
				this.currentManagerIdentifier = m;
			} catch (Exception e) {
			}
			executor = new SimulationExecutor();
			executor.setCommunicator(communicator2Simulation);
			executor.setDisplay(display);
			executor.setLogger(logger);
			status = Status.CREATED;
		}
	}
	
	public void doInitialise(Node node) {
		String m = "";
		try {
			m = ((Element)node).getAttribute("manager").trim();
		} catch (Exception e) {
		}
		if ((status == Status.CREATED) && (this.currentManagerIdentifier.equals(m))) {
			boolean correct = false;
			NodeList nList = node.getChildNodes();
			for (int i=0; i<nList.getLength(); i++) {
				Node subNode = nList.item(i);
				if (subNode.getNodeName().equals("code")) {
					if (executor != null) {
						IAICommand cmd = creater.interpret(subNode.getTextContent());
						executor.setCommandTree(cmd);
						correct = true;
					}
				}
			}
			if (correct) {
				currentProgramIdentifier = ((Element)node).getAttribute("identifier");
				display.initialise();
				status = Status.INITIALIZED;
			}
		}
	}
	
	public void doStart(Node node) {
		String m = "";
		try {
			m = ((Element)node).getAttribute("manager").trim();
		} catch (Exception e) {
		}
		if ((status == Status.INITIALIZED) && (this.currentManagerIdentifier.equals(m))) {
			if (executor != null) {
				if (!displayThread.isAlive()) {
					displayThread.start();
				}
				if (!executor.isAlive()) {
					executor.start();
				}
				status = Status.STARTED;
			}
		}
	}
	
	public void doStop(Node node) {
		String m = "";
		try {
			m = ((Element)node).getAttribute("manager").trim();
		} catch (Exception e) {
		}
		if ((status == Status.STARTED) && (this.currentManagerIdentifier.equals(m))) {
			if (executor != null) {
				executor.shutdown = true;
				executor.interrupt();
				display.doShutdown();
				status = Status.STOPPED;
			}
		}
	}
	
	public void doStatusReport(Node node) {
		String ret = ""; String val = "";
		switch (status) {
		case UNKNOWN: ret = "UNKNOWN"; break;
		case JUSTBEGIN: ret = "JUSTBEGIN"; break;
		case CREATED: ret = "CREATED"; break;
		case INITIALIZED: ret = "INITIALIZED"; break;
		case STARTED: ret = "STARTED";
					val = "<rating identifier=\"" + currentProgramIdentifier + "\">" + 
						  display.getLastPoints() + "</rating>";
					break;
		case FINISHED: ret = "FINISHED"; 
					val = "<rating identifier=\"" + currentProgramIdentifier + "\">" + 
						  display.getLastPoints() + "</rating>"; break;
		case STOPPED: ret = "STOPPED"; break;
		}
		ret = "<status text=\"" + ret + "\">" + val + "</status><!--ende-->";
		communicator.sendCommand(ret);
	}
	
	public void doChangeSetting(Node node) {
		String m = "";
		try {
			m = ((Element)node).getAttribute("manager").trim();
		} catch (Exception e) {
		}
		if ((status == Status.FINISHED) || (this.currentManagerIdentifier.equals(m))) {
			String servername = ((Element)node).getAttribute("servername");
			communicator2Simulation.setServername(servername);
		}
	}

	public void doAddLibraryProgram(Node node) {
		String m = "";
		try {
			m = ((Element)node).getAttribute("manager").trim();
		} catch (Exception e) {
		}
		if ((status == Status.CREATED) && (this.currentManagerIdentifier.equals(m))) {
			boolean correct = false;
			IAICommand cmd = null;
			NodeList nList = node.getChildNodes();
			int id = 0;
			for (int i=0; i<nList.getLength(); i++) {
				Node subNode = nList.item(i);
				if (subNode.getNodeName().equals("code")) {
					try {
						String tmp = ((Element)subNode).getAttribute("id").trim();
						id = Integer.valueOf(tmp);
					} catch (Exception e) {
					}
					cmd = creater.interpret(subNode.getTextContent());
					correct = true;
				}
			}
			
			if (correct) {
				library.setManagerID(m);
				library.addProgram(id, (AICommand)cmd);
			}
		}
	}
	
	public void doClearLibrary(Node node) {
		String m = "";
		try {
			m = ((Element)node).getAttribute("manager").trim();
		} catch (Exception e) {
		}
		if ((status == Status.CREATED) && (this.currentManagerIdentifier.equals(m))) {
			boolean force = false;
			try {
				force = ((Element)node).getAttribute("force").trim().equalsIgnoreCase("TRUE");
			} catch (Exception e) {
			}
			if (force || (library.getManagerID().equals(m))) {
				library.clear();
			}
		}
	}
	
	public void interpret(Document document) {
		String root = document.getFirstChild().getNodeName();
		if (!root.equals("status")) {
			Date d = new Date();
			DateFormat df = DateFormat.getTimeInstance( DateFormat.MEDIUM, Locale.GERMANY ); 

			System.out.println(df.format(d) + " SimulationController.interpret(): " + root);
		}
		if (root.equals("shutdown")) {
			doShutdown(document.getFirstChild());
		} else if (root.equals("create")) {
			doCreate(document.getFirstChild());
		} else if (root.equals("initialise")) {
			doInitialise(document.getFirstChild());
		} else if (root.equals("start")) {
			doStart(document.getFirstChild());
		} else if (root.equals("stop")) {
			doStop(document.getFirstChild());
		} else if (root.equals("status")) {
			doStatusReport(document.getFirstChild());
		} else if (root.equals("setting")) {
			doChangeSetting(document.getFirstChild());
		} else if (root.equals("library")) {
			doAddLibraryProgram(document.getFirstChild());
		} else if (root.equals("clearlibrary")) {
			doClearLibrary(document.getFirstChild());
		}
	}
	
	@Override
	public void run() {
		status = Status.JUSTBEGIN;
		if (communicator == null) {
			return;
		}
		while ((!isInterrupted()) && (!shutdown)) { 
			String command = communicator.getNextCommand();
			
			checkCurrentExecution();
			
			if (command == null) {
				continue;
			}
			if (command.equals("")) {
				continue;
			}
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance(); 
			DocumentBuilder builder = null;
			try {
				builder = factory.newDocumentBuilder();
				Document document = builder.parse(new InputSource(new StringReader(command)));
				interpret(document);
			} catch (ParserConfigurationException e) {
				System.out.println("SimulationController.run(): " + e.getMessage());
			} catch (SAXException e) {
				System.out.println("SimulationController.run(): " + e.getMessage());
			} catch (IOException e) {
				System.out.println("SimulationController.run(): " + e.getMessage());
			} 
		}
		status = Status.STOPPED;
	}

	private void checkCurrentExecution() {
		if (status == Status.STARTED) {
			if (executor != null) {
				if (executor.getStatus() == ExecutorStatus.FINISHED) {
					status = Status.FINISHED;
				}
			} else {
				System.out.println("SimulationController.checkCurrentExecution(): ERROR: state is started but no executor");
				status = Status.UNKNOWN;
			}
		}
	}

}
