package simulation;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import extern.Asteroid;
import extern.FramePacket;
import extern.GameStatus;
import extern.Saucer;
import extern.Ship;
import extern.Shot;

public class SimulationDisplay extends Memory implements Runnable {
	
	private FramePacket framePacket = null;
	
	private GameStatus gameStatus = null; 
	
	private ICommunicationToSimulation communication = null;
	
	private boolean shutdown = false;
	
	public SimulationDisplay() {
		super();
		size = 500;
		framePacket = new FramePacket();
		gameStatus = new GameStatus();
	}

	public void setCommunication(ICommunicationToSimulation newCommunication) {
		communication = newCommunication;
	}

	@Override 
	public void initialise() {
		super.initialise();
	}
	
	@Override 
	public int setSize(int newSize) {
		// Size unchangeable
		return getSize();
	}
	
	@Override 
	public int getSize() {
		return super.getSize();
	}
	
	@Override 
	public int getValue(int pos) {
		pos = pos % 300;

		int ret = 0;
		if (pos < 5) {
			// get own Ship infos
			Ship ship = gameStatus.getShip();
			if (ship != null) {
				//System.out.println("SimulationDisplay:getValue(" + pos + "): ox=" + 
				//		ship.getOrientation().getX() + " oy=" + ship.getOrientation().getY());
				switch (pos) {
				case 0: ret = 1000; break;
				case 1: ret = ship.getX() * 1000; break;
				case 2: ret = ship.getY() * 1000; break;
				case 3: ret = ship.getOrientation().getX() * 1000; break;
				case 4: ret = ship.getOrientation().getY() * 1000; break;
				}
			}
		} else if (pos < 10) {
			// get count infos
			switch (pos) {
			case 5: ret = (gameStatus.getShip() == null ? 0 : 1000); break;
			case 6: ret = (gameStatus.getSaucer() == null ? 0 : 1000); break;
			case 7: ret = gameStatus.getShots().size() * 1000; break;
			case 8: ret = gameStatus.getAsteroids().size() * 1000; break;
			case 9: { // New code must be move in a new version
					Ship ship = gameStatus.getShip();
					if (ship != null) {
						ret = (int)(Math.atan2(ship.getX(), ship.getY()) * 1000);
					}
				}
			}
		} else if (pos < 20) {
			// get saucer infos
			Saucer saucer = gameStatus.getSaucer();
			if (saucer != null) {
				switch (pos) {
				case 10: ret = 1000; break;
				case 11: ret = saucer.getX() * 1000; break;
				case 12: ret = saucer.getY() * 1000; break;
				case 13: ret = (saucer.getSize() == Saucer.Size.SMALL ? 1000 : 2000); break;
				case 14: 
					ret = calcMinDistanceShip(saucer.getX(), saucer.getY(), true) * 1000;
					break;
				case 15: 
					ret = calcMinDistanceShip(saucer.getX(), saucer.getY(), false) * 1000;
					break;
				case 16:
					ret = (int) (calcMinDistanceShip(saucer.getX(), saucer.getY()) * 1000);
					break;
				case 17: // New code must be move in a new version
					ret = (int) (calcTanShip(saucer.getX(), saucer.getY()) * 1000);
					break;
				}
			}
		} else if (pos < 30) {
			// return infos for the nearest asteroid
			Asteroid nearest = null;
			double dist = 20000000;
			List<Asteroid> aList = gameStatus.getAsteroids();
			int index = -1;
			for (int i=0; i<aList.size(); i++) {
				try {
					Asteroid asteroid = aList.get(i);
					if (asteroid != null) {
						double d = calcMinDistanceShip(asteroid.getX(), asteroid.getY());
						if (d < dist) {
							dist = d;
							nearest = asteroid;
							index = i;
						}
					}
				} catch (IndexOutOfBoundsException e) {
					break;
				}
			}

			if (nearest != null) {
				switch (pos % 10) {
				case 0: ret = 1000; break;
				case 1: ret = nearest.getX() * 1000; break;
				case 2: ret = nearest.getY() * 1000; break;
				case 3: 
					ret = calcMinDistanceShip(nearest.getX(), nearest.getY(), true) * 1000;
					break;
				case 4: 
					ret = calcMinDistanceShip(nearest.getX(), nearest.getY(), false) * 1000;
					break;
				case 5:
					ret = (int) (calcMinDistanceShip(nearest.getX(), nearest.getY()) * 1000);
					break;
				case 6: 
					switch (nearest.getSize()) {
					case SMALL: ret = 1000; break;
					case MIDDLE: ret = 2000; break;
					case BIG: ret = 3000; break;
					}
					break;
				case 7: 
					switch (nearest.getType()) {
					case TYPE_1: ret = 1000; break;
					case TYPE_2: ret = 2000; break;
					case TYPE_3: ret = 3000; break;
					case TYPE_4: ret = 4000; break;
					}
					break;
				case 8: // New code must be move in a new version
					ret = (int) (calcTanShip(nearest.getX(), nearest.getY()) * 1000);
					break;
				case 9:
					ret = index * 1000;
					break;
				}
			}
		} else {
			// get shot and/or asteroids infos
			int itemNo = (pos - 30) / 20;
			int sumPos = pos % 10;
			int pos2 = (pos - 30) % 20;
			if (pos2 < 10) {
				// get shot infos
				Shot shot = null;
				if (gameStatus.getShots().size() > itemNo) {
					try {
						shot = gameStatus.getShots().get(itemNo);
					} catch (IndexOutOfBoundsException e) {
						shot = null;
					}
				}
				if (shot != null) {
					switch (sumPos) {
					case 0: ret = 1000; break;
					case 1: ret = shot.getX() * 1000; break;
					case 2: ret = shot.getY() * 1000; break;
					case 3: 
						ret = calcMinDistanceShip(shot.getX(), shot.getY(), true) * 1000;
						break;
					case 4: 
						ret = calcMinDistanceShip(shot.getX(), shot.getY(), false) * 1000;
						break;
					case 5:
						ret = (int) (calcMinDistanceShip(shot.getX(), shot.getY()) * 1000);
						break;
					case 6: // New code must be move in a new version
						ret = (int) (calcTanShip(shot.getX(), shot.getY()) * 1000);
						break;
					}
				}
			} else {
				// get asteroid infos
				Asteroid asteroid = null;
				if (gameStatus.getAsteroids().size() > itemNo) {
					try {
						asteroid = gameStatus.getAsteroids().get(itemNo);
					} catch (IndexOutOfBoundsException e) {
						asteroid = null;
					}
				}
				if (asteroid != null) {
					switch (sumPos) {
					case 0: ret = 1000; break;
					case 1: ret = asteroid.getX() * 1000; break;
					case 2: ret = asteroid.getY() * 1000; break;
					case 3: 
						ret = calcMinDistanceShip(asteroid.getX(), asteroid.getY(), true) * 1000;
						break;
					case 4: 
						ret = calcMinDistanceShip(asteroid.getX(), asteroid.getY(), false) * 1000;
						break;
					case 5:
						ret = (int) (calcMinDistanceShip(asteroid.getX(), asteroid.getY()) * 1000);
						break;
					case 6: 
						switch (asteroid.getSize()) {
						case SMALL: ret = 1000; break;
						case MIDDLE: ret = 2000; break;
						case BIG: ret = 3000; break;
						}
						break;
					case 7: 
						switch (asteroid.getType()) {
						case TYPE_1: ret = 1000; break;
						case TYPE_2: ret = 2000; break;
						case TYPE_3: ret = 3000; break;
						case TYPE_4: ret = 4000; break;
						}
						break;
					case 8: // New code must be move in a new version
						ret = (int) (calcTanShip(asteroid.getX(), asteroid.getY()) * 1000);
						break;
					}
				}
			}
		}
		return ret;
	}
	
	@Override 
	public int setValue(int pos, int value) {
		// Values unchangeable
		return 0;
	}

	@Override
	public void run() {
		shutdown = false;
		while (!shutdown) {
			if ((communication != null) && (framePacket != null)) {
				communication.getFrame(framePacket);
			}
			if ((gameStatus != null) && (framePacket != null)) {
				gameStatus.interpretScreen(framePacket, 0);
			}
		}
	}

	public void doShutdown() {
		shutdown = true;
		if (communication != null) {
			communication.stop();
		}
	}
	
	public boolean checkPlayerShipVisible() {
		if (gameStatus != null) {
			if (gameStatus.getShip() != null) {
				return true;
			}
		}
		return false;
	}
	
	public int getLastPoints() {
		return gameStatus.getLastPoints();
	}
	
	public String getDisplayText() {
		return gameStatus.getText();
	}
	
	public class DistValues {
		public int distSquare = 0;
		public int dX = 0;
		public int dY = 0;
	}

	public class DistValuesComparator implements Comparator<DistValues> {

		@Override
		public int compare(DistValues arg0, DistValues arg1) {
			if (arg0.distSquare < arg1.distSquare) {
				return -1;
			} else if (arg0.distSquare > arg1.distSquare) {
				return 1;
			} else {
				return 0;
			}
		}
		
	}

	public double calcTanShip(int X1, int Y1) {
		Ship ship = gameStatus.getShip();
		if (ship != null) {
			double x = (double)calcMinDistanceShip(X1, Y1, true);
			double y = (double)calcMinDistanceShip(X1, Y1, false);
			return Math.atan2(x, y);
		} else {
			return 0.0;
		}
	}
	
	public double calcMinDistanceShip(int X1, int Y1) {
		Ship ship = gameStatus.getShip();
		if (ship != null) {
			DistValues[] dists = calcMinDistance(X1, Y1, ship.getX(), ship.getY());
			return Math.sqrt(dists[0].distSquare);
		} else {
			return 200000000.0;
		}
	}
	
	public int calcMinDistanceShip(int X1, int Y1, boolean wantX) {
		Ship ship = gameStatus.getShip();
		if (ship != null) {
			return calcMinDistance(X1, Y1, ship.getX(), ship.getY(), wantX);
		} else {
			return 0;
		}
	}
	
	public int calcMinDistance(int X1, int Y1, int X2, int Y2, boolean wantX) {
		DistValues[] dists = calcMinDistance(X1, Y1, X2, Y2);
		if (wantX) {
			return dists[0].dX;
		} else {
			return dists[0].dY;
		}		
	}
	
	public DistValues[] calcMinDistance(int X1, int Y1, int X2, int Y2) {
		final int maxX = 1024;
		final int minY = 128;
		final int maxY = 896;
		int X1_ = X1 + maxX; 
		int Y1_ = Y1 - minY + maxY;
		int X2_ = X2 + maxX;
		int Y2_ = Y2 - minY + maxY;
		DistValues[] dists = new DistValues[9];
		for (int i=0; i<dists.length; i++) {
			dists[i] = new DistValues();
		}
		dists[0].dX = (X1 - X2);
		dists[0].dY = (Y1 - Y2);
		dists[1].dX = (X1_ - X2);
		dists[1].dY = (Y1 - Y2);
		dists[2].dX = (X1 - X2);
		dists[2].dY = (Y1_ - Y2);
		dists[3].dX = (X1_ - X2);
		dists[3].dY = (Y1_ - Y2);
		dists[4].dX = (X1 - X2_);
		dists[4].dY = (Y1 - Y2);
		dists[5].dX = (X1 - X2);
		dists[5].dY = (Y1 - Y2_);
		dists[6].dX = (X1 - X2_);
		dists[6].dY = (Y1 - Y2_);
		dists[7].dX = (X1 - X2_);
		dists[7].dY = (Y1_ - Y2);
		dists[8].dX = (X1_ - X2);
		dists[8].dY = (Y1 - Y2_);
		for (int i = 0; i<dists.length; i++) {
			dists[i].distSquare = dists[i].dX * dists[i].dX + dists[i].dY * dists[i].dY;
		}
		
		Arrays.sort(dists,new DistValuesComparator());
		
		return dists;
	}
}
