package de.heise.anniversary.contest;

import java.net.DatagramPacket;

/**
 * The <code>FramePacket</code> class represents the server reply packet.
 * <p>
 * It contains the vector ram, the frame number and the ping value.
 *
 * @see ServerConnection
 */
public class FramePacket {
	public static final int SIZE = 1024 + 2;
	private byte payload[] = new byte[SIZE];
	private DatagramPacket datagramPacket = new DatagramPacket(payload, payload.length);

	/**
	 * Returns the <code>DatagramPacket</code> for receiving the network data
	 *
	 * @return the <code>DatagramPacket</code>
	 *
	 * @see ServerConnection
	 */
	public DatagramPacket getDatagramPacket() {
		return datagramPacket;
	}

	/**
	 * Returns frame number.
	 *
	 * The frame number gets increased for every frame the server sends.
	 * It can be used to detect missing frames or wrong frame order.
	 *
	 * @return the frame number
	 */
	public byte getFrameNo() {
		return payload[SIZE - 1];
	}

	/**
	 * Returns ping value.
	 *
	 * The server returns the ping value from last processed <code>KeysPacket</code>.
	 * Use this to detect network latency.
	 *
	 * @return the ping value
	 *
	 * @see KeysPacket#setPing(byte)
	 * @see KeysPacket#getPing()
	 */
	public byte getPing() {
		return payload[SIZE - 1]; //
	}

	/**
	 * Allows access to the 16-bit vector ram words.
	 *
	 * @param index
	 * 			the index for the word
	 *
	 * @return the 16-bit word at index <code>index</code>
	 */
	public int readVectorRAMWord(int index) {
		return (payload[2 * index] & 0xff) | (payload[2 * index + 1] & 0xff) << 8;
	}
}
