// game_state.cpp
// --------------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#include "stdafx.hpp"

#include "game_state.hpp"

#include "../algo.hpp"
#include "ast_cossin.hpp"
#include "screen_state.hpp"

#define foreach BOOST_FOREACH

namespace intrepid
{
// struct asteroid

    void asteroid::adjust(screen_asteroid const &ast)
    {
        if (point::to_vram_pos(position) != point::to_vram_pos(ast.position)
            || lifetime != ast.lifetime || (!tracked && ast.tracked))
        {
            position = ast.position;
            velocity = ast.velocity;
            lifetime = ast.lifetime;
            precision_x = ast.precision_x;
            precision_y = ast.precision_y;
            shape = ast.shape;
            size = ast.size;
            tracked = ast.tracked;
        }
    }

// struct ship_t

    void ship_t::adjust(screen_ship const &shp)
    {
        present = shp.present;
        angle = shp.min_angle;
        position = shp.position;
        velocity = shp.velocity;
    }

// struct game_state

    void game_state::adjust(screen_state const &screen_st)
    {
        uint8_t ping = screen_st.get_ping() - 1;
        comm_model::ping_to_keys_map const &ping_to_keys
            = screen_st.get_ping_to_keys();
        for (int n = 0; n < screen_st.get_frame_diff(); ++n)
        {
            advance(ping_to_keys[ping]);
        }
        ship.adjust(screen_st.get_ship());
        for (int n = 0; n < 27; ++n)
        {
            asteroids[n].adjust(screen_st.get_asteroids()[n]);
        }
        find_shots(screen_st);
        saucer.adjust(screen_st.get_saucer());
        frame_count = screen_st.frame_->even() ? 1 : 0;
    }

    void game_state::find_shots(screen_state const &screen_st)
    {
        array<screen_shot, 6> const &screen_shots = screen_st.get_shots();
        num_shots = 0;
        foreach (screen_shot const &scr_shot, screen_shots)
        {
            if (scr_shot.present)
            {
                ++ num_shots;
            }
        }
        bool shots_found[6];
        fill(shots_found, false);
        for (int k = 2; k < 6; ++k)
        {
            shot &sht = shots[k];
            if (sht.lifetime)
            {
                bool found = false;
                for (size_t n = 0; n != screen_shots.size(); ++n)
                {
                    screen_shot const &scr_shot = screen_shots[n];
                    if (scr_shot.present)
                    {
                        if ((sht.position.x & -8) == (scr_shot.position.x & -8)
                             && (sht.position.y & -8)
                                 == (scr_shot.position.y & -8))
                        {
                            found = true;
                            shots_found[n] = true;
                            break;
                        }
                    }
                }
                if (!found)
                {
                    sht.lifetime = 0;
                }
            }
        }
        int k = 0;
        for (size_t n = 0; n != screen_shots.size() && k < 2; ++n)
        {
            screen_shot const &scr_shot = screen_shots[n];
            if (scr_shot.present && !shots_found[n])
            {
                shot &sht = shots[k++];
                if (sht.lifetime)
                {
                    ++ sht.frames_alive;
                    sht.velocity = torus_diff(scr_shot.position, sht.start_pos)
                                            / sht.frames_alive;
                    sht.position = scr_shot.position;
                }
                else
                {
                    sht.frames_alive = 0;
                    sht.velocity = point(0, 0);
                    sht.position = scr_shot.position;
                    sht.start_pos = scr_shot.position;
                    sht.lifetime = 73;
                }
                -- num_shots;
            }
        }
    }

    asteroid *game_state::get_empty_asteroid_slot()
    {
        foreach (asteroid &ast, asteroids)
        {
            if (!ast.lifetime)
            {
                return &ast;
            }
        }
        return 0;
    }

    shot *game_state::get_empty_ship_shot_slot()
    {
        for (size_t n = 2; n != shots.size(); ++n)
        {
            if (!shots[n].lifetime)
            {
                return &shots[n];
            }
        }
        return 0;
    }

    void game_state::advance(uint8_t keys)
    {
        advance_collisions();
        ast_rand();
        ++ frame_count;
        advance_fire(keys);
        advance_ship(keys);
        advance_move();
    }

    void game_state::advance_collisions()
    {
        foreach (shot &sh, shots)
        {
            if (sh.lifetime)
            {
                sh.asteroid_collision = 0;
                if (saucer.present && saucer.hits(sh.position))
                {
                    saucer.present = false;
                    sh.lifetime = 0;
                    -- num_shots;
                    continue;
                }
                foreach (asteroid &ast, asteroids)
                {
                    if (ast.lifetime == 100 && ast.hits(sh.position))
                    {
                        sh.asteroid_collision = &ast;
                        dissolve_asteroid(ast);
                        sh.lifetime = 0;
                        -- num_shots;
                        break;
                    }
                }
            }
        }
    }

    void game_state::advance_fire(uint8_t keys)
    {
        bool fire = (keys & comm_model::keys::fire) != 0;
        shot_fired = 0;
        if (fire && !last_frame_fire)
        {
            if (ship.present)
            {
                shot *sht = get_empty_ship_shot_slot();
                if (sht)
                {
                    ++ num_shots;
                    point offset = ast_cossin(ship.angle) >> 1;
                    sht->lifetime = 73;
                    sht->frames_alive = 0;
                    sht->asteroid_collision = false;
                    sht->velocity = (ship.velocity >> 8) + offset;
                    sht->velocity.x = min(sht->velocity.x, 111);
                    sht->velocity.x = max(sht->velocity.x, -111);
                    sht->velocity.y = min(sht->velocity.y, 111);
                    sht->velocity.y = max(sht->velocity.y, -111);
                    sht->position = ship.position + offset + (offset >> 1);
                    sht->position.torus_wrap();
                    shot_fired = sht;
                }
            }
        }
        last_frame_fire = fire;
    }

    void game_state::advance_move()
    {
        foreach (asteroid &ast, asteroids)
        {
            ast.move();
        }
        foreach (shot &sh, shots)
        {
            if (sh.move())
            {
                -- num_shots;
            }
        }
        saucer.move();
        ship.move();
    }

    inline void game_state::advance_ship(uint8_t keys)
    {
        if (keys & comm_model::keys::left)
        {
            ship.angle += 3;
        }
        else if (keys & comm_model::keys::right)
        {
            ship.angle -= 3;
        }
        if ((!frame_count & 1))
        {
            if (keys & comm_model::keys::thrust)
            {
                ship.velocity += 2 * ast_cossin(ship.angle);
            }
            else
            {
                if (ship.velocity.x > 0)
                {
                    ship.velocity.x -= ((ship.velocity.x >> 7) & -2) + 1;
                }
                else if (ship.velocity.x < 0)
                {
                    ship.velocity.x -= ((ship.velocity.x >> 7) & -2);
                }
                if (ship.velocity.y > 0)
                {
                    ship.velocity.y -= ((ship.velocity.y >> 7) & -2) + 1;
                }
                else if (ship.velocity.y < 0)
                {
                    ship.velocity.y -= ((ship.velocity.y >> 7) & -2);
                }
            }
            if (ship.velocity.x >= 0x4000)
            {
                ship.velocity.x = 0x3FFF;
            }
            else if (ship.velocity.x < -0x4000)
            {
                ship.velocity.x = -0x3FFF;
            }
            if (ship.velocity.y >= 0x4000)
            {
                ship.velocity.y = 0x3FFF;
            }
            else if (ship.velocity.y < -0x4000)
            {
                ship.velocity.y = -0x3FFF;
            }
        }
    }

    void game_state::dissolve_asteroid(asteroid &ast)
    {
        if (ast.size)
        {
            asteroid *ast1 = get_empty_asteroid_slot();
            if (ast1)
            {
                dissolve_asteroid_to(ast, *ast1);
                int offset = (ast1->velocity.x & 0x1F) << 1;
                ast1->position.x ^= offset;
                asteroid *ast2 = get_empty_asteroid_slot();
                if (ast2)
                {
                    dissolve_asteroid_to(ast, *ast2);
                    int offset = (ast2->velocity.y & 0x1F) << 1;
                    ast2->position.y ^= offset;
                }
            }
        }
        ast.lifetime = 38;
    }

    void game_state::rand_velocity(int &velocity)
    {
        uint8_t rnd = ast_rand() & 0x8F;
        if (rnd >= 128)
        {
            rnd |= 0xF0;
        }
        int v = velocity + static_cast<int8_t>(rnd);
        if (v >= 0)
        {
            if (v < 6)
            {
                v = 6;
            }
            else if (v > 31)
            {
                v = 31;
            }
        }
        else
        {
            if (v > -6)
            {
                v = -6;
            }
            else if (v < -31)
            {
                v = -31;
            }
        }
        velocity = v;
    }
} // end of namespace intrepid
