// game_state.hpp
// --------------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#ifndef INTREPID_GAME_STATE_HPP
#define INTREPID_GAME_STATE_HPP

#if defined(_MSC_VER) && (_MSC_VER >= 1200)
# pragma once
#endif // defined(_MSC_VER) && (_MSC_VER >= 1200)

#include "ast_random.hpp"
#include "geo.hpp"
#include "../comm_model.hpp"
#include "screen_state.hpp"

namespace intrepid
{
    struct asteroid
    {
        point         position;
        point         velocity;
        int8_t        lifetime; // 0 = Empty, 100 = Asteroid, < 99 Explosion
        int8_t        precision_x, precision_y;
        int8_t        shape;
        int8_t        size;
        bool          tracked;

        asteroid();

        bool hits(point p) const;
        bool is_near_ship(point p) const;

        void adjust(screen_asteroid const &ast);

        void move();
    };

    struct saucer_t : screen_saucer
    {
        bool hits(point p) const;

        void adjust(screen_saucer const &ast);
        void move();
    };

    struct shot
    {
        point   position;
        point   velocity;
        point   start_pos;
        int     lifetime;
        int     frames_alive;

        asteroid *asteroid_collision;

        shot();

        bool move();
    };

    struct ship_t
    {
        bool    present;
        uint8_t angle;
        point   position;
        point   velocity;

        ship_t();

        void adjust(screen_ship const &shp);

        void move();
    };

    struct game_state
    {
        typedef array<asteroid, 27> asteroids_array;
        typedef array<shot, 6> shots_array;

        asteroids_array    asteroids;
        shots_array        shots;
        ship_t             ship;
        saucer_t           saucer;
        int                num_shots;
        shot              *shot_fired;

        int     rand_seed;
        bool    last_frame_fire;
        uint8_t frame_count;

        game_state();

        void adjust(screen_state const &screen_st);

        void advance(uint8_t keys);
        void advance_collisions();
        void advance_fire(uint8_t keys);
        void advance_move();
        void advance_ship(uint8_t keys);
        uint8_t ast_rand();
        void dissolve_asteroid(asteroid &ast);
        void dissolve_asteroid_to(asteroid &ast, asteroid &child);
        void find_shots(screen_state const &screen_st);
        asteroid *get_empty_asteroid_slot();
        shot *get_empty_ship_shot_slot();
        void rand_velocity(int &velocity);
        void rand_velocity(point &velocity);
    };

// struct asteroid

    inline asteroid::asteroid()
        :   position(0, 0), velocity(0, 0), lifetime(0)
    {
    }

    inline bool asteroid::hits(point p) const
    {
        point dist = (position - p) >> 1;
        if (dist.x < 0)
        {
            dist.x ^= -1;
        }
        if (dist.y < 0)
        {
            dist.y ^= -1;
        }
        int radius;
        switch (size)
        {
        case 2: // Big Asteroid
            radius = 132;
            break;
        case 1: // Medium Asteroid
            radius = 72;
            break;
        default: // Small Asteroid
            radius = 42;
        }
        if (radius < dist.x || radius < dist.y)
        {
            // outside of dangerous square
            return false;
        }
        // inside of dangerous square, test if inside octagon
        return dist.x + dist.y <  radius + (radius >> 1);
    }

    inline bool asteroid::is_near_ship(point p) const
    {
        if (lifetime != 100)
        {
            return false;
        }
        point dist = (position - p) >> 1;
        if (dist.x < 0)
        {
            dist.x ^= -1;
        }
        if (dist.y < 0)
        {
            dist.y ^= -1;
        }
        int radius;
        switch (size)
        {
        case 2: // Big Asteroid
            radius = 132;
            break;
        case 1: // Medium Asteroid
            radius = 72;
            break;
        default: // Small Asteroid
            radius = 42;
        }
        radius += 120;
        if (radius < dist.x || radius < dist.y)
        {
            // outside of dangerous square
            return false;
        }
        // inside of dangerous square, test if inside octagon
        return dist.x + dist.y <  radius + (radius >> 1);
    }

    inline void asteroid::move()
    {
        if (lifetime)
        {
            if (lifetime == 100)
            {
                position += velocity;
                position.torus_wrap();
            }
            else
            {
                -- lifetime;
            }
        }
    }

// struct game_state

    inline game_state::game_state()
        :   rand_seed(-1), last_frame_fire(false)
    {
    }

    inline uint8_t game_state::ast_rand()
    {
        if (rand_seed >= 0)
        {
            uint8_t rnd = ast_random::get_rand(rand_seed);
            if (++ rand_seed >= 32767)
            {
                rand_seed -= 32767;
            }
            return rnd;
        }
        return 0;
    }

    inline void game_state::dissolve_asteroid_to(asteroid &ast,
                                                 asteroid &child)
    {
        uint8_t const rnd = ast_rand();
        child = ast;
        -- child.size;
        assert(child.size >= 0);
        child.shape = (rnd & 0x18) >> 3;
        rand_velocity(child.velocity);
    }

    inline void game_state::rand_velocity(point &velocity)
    {
        rand_velocity(velocity.x);
        ast_rand();
        ast_rand();
        ast_rand();
        rand_velocity(velocity.y);
    }

// struct saucer_t

    inline bool saucer_t::hits(point p) const
    {
        point dist = (position - p) >> 1;
        if (dist.x < 0)
        {
            dist.x ^= -1;
        }
        if (dist.y < 0)
        {
            dist.y ^= -1;
        }
        int radius;
        switch (size)
        {
        case 2:
            assert(false);
            return false;
        case 1: // Big Saucer
            radius = 72;
            break;
        default: // Small Saucer
            radius = 42;
        }
        if (radius < dist.x || radius < dist.y)
        {
            // outside of dangerous square
            return false;
        }
        // inside of dangerous square, test if inside octagon
        return dist.x + dist.y <  radius + (radius >> 1);
    }

    inline void saucer_t::adjust(screen_saucer const &ast)
    {
        screen_saucer::operator=(ast);
    }

    inline void saucer_t::move()
    {
        if (present)
        {
            position += velocity;
            position.torus_wrap();
        }
    }

// struct ship_t

    inline ship_t::ship_t()
        : present(false)
    {
    }

    inline void ship_t::move()
    {
        if (present)
        {
            position += velocity >> 8;
            position.torus_wrap();
        }
    }

// struct shot

    inline shot::shot()
        :   lifetime(0)
    {
    }

    inline bool shot::move()
    {
        if (lifetime)
        {
            position += velocity;
            position.torus_wrap();
            return --lifetime == 0; // died?
        }
        return false;
    }
} // end of namespace intrepid

#endif // include guard
