// geo.hpp
// -------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#ifndef INTREPID_GEO_HPP
#define INTREPID_GEO_HPP

#if defined(_MSC_VER) && (_MSC_VER >= 1200)
# pragma once
#endif // defined(_MSC_VER) && (_MSC_VER >= 1200)

namespace intrepid
{
    int const vram_screen_width = 1024;
    int const vram_screen_height = 768;
    int const screen_width = 8 * vram_screen_width;
    int const screen_height = 8 * vram_screen_height;

    int torus_diff_x(int x1, int x2);
    int torus_diff_y(int y1, int y2);
    int torus_max_x(int x1, int x2);
    int torus_max_y(int y1, int y2);
    int torus_min_x(int x1, int x2);
    int torus_min_y(int y1, int y2);
    int torus_wrap_x(int x);
    int torus_wrap_y(int y);
    int vram_torus_diff_x(int x1, int x2);
    int vram_torus_diff_y(int y1, int y2);

    struct point
    {
        int x, y;

        point();
        point(int x, int y);
        static point from_vram_pos(point vram_pos);
        static point to_vram_pos(point pos);

        void ceiling_division(int d);
        void floor_division(int d);
        void torus_diff(point other);
        void torus_max(point other);
        void torus_wrap();
        void vram_torus_diff(point other);

        void operator+=(point other);
        void operator-=(point other);
        void operator*=(int s);
        void operator/=(int s);
        void operator>>=(int n);
    };

    bool operator==(point p1, point p2);
    bool operator!=(point p1, point p2);
    bool operator<=(point p1, point p2); // partial order
    point operator+(point p1, point p2);
    point operator-(point p1, point p2);
    point operator*(int s, point p);
    point operator/(point p, int s);
    point operator>>(point p, int n);
    template<class Ch, class Tr>
    basic_ostream<Ch, Tr> &operator<<(basic_ostream<Ch, Tr> &os,
                                      point p);

    point ceiling_division(point p, int d);
    int   dot_product(point p1, point p2);
    point floor_division(point p, int d);
    int   norm2(point p);
    point torus_diff(point p1, point p2);
    point torus_max(point p1, point p2);
    point torus_wrap(point p);
    point vram_coord_transform(point p);
    point vram_torus_diff(point p1, point p2);

    struct rect
    {
        point bl, tr; // bottom-left, top-right inclusive
        rect();
        rect(point bl, point tr);
        static rect from_vram_pos(point vram_pos);

        bool contains(point p) const;
        bool empty() const;
        bool single_point() const;

        void intersect(rect const &other);
        void torus_intersect(rect const &other);
        void torus_wrap();

        void operator+=(rect const &other);
        void operator-=(rect const &other);
        void operator*=(int s);
    };

    bool operator==(rect const &r1, rect const &r2);
    rect operator+(rect r1, rect const &r2);
    rect operator-(rect r1, rect const &r2);
    rect operator*(int s, rect r);
    template<class Ch, class Tr>
    basic_ostream<Ch, Tr> &operator<<(basic_ostream<Ch, Tr> &os,
                                      rect const &r);
    rect operator*(int s, rect r);

    point get_center(rect const &r);
    point move_into(point p, rect const &r);
    point torus_get_center(rect const &r);
    point torus_move_into(point p, rect const &r);
    point torus_wrap(point p);

    inline int torus_diff_x(int x1, int x2)
    {
        int x = x1 - x2;
        while (x < - screen_width / 2)
        {
            x += screen_width;
        }
        while (x >= screen_width / 2)
        {
            x -= screen_width;
        }
        return x;
    }

    inline int torus_diff_y(int y1, int y2)
    {
        int y = y1 - y2;
        while (y < - screen_height / 2)
        {
            y += screen_height;
        }
        while (y >= screen_height / 2)
        {
            y -= screen_height;
        }
        return y;
    }

    inline int torus_max_x(int x1, int x2)
    {
        return torus_diff_x(x1, x2) >= 0 ? x1 : x2;
    }

    inline int torus_max_y(int y1, int y2)
    {
        return torus_diff_y(y1, y2) >= 0 ? y1 : y2;
    }

    inline int torus_min_x(int x1, int x2)
    {
        return torus_diff_x(x1, x2) <= 0 ? x1 : x2;
    }

    inline int torus_min_y(int y1, int y2)
    {
        return torus_diff_y(y1, y2) <= 0 ? y1 : y2;
    }

    inline int torus_wrap_x(int x)
    {
        while (x < 0)
        {
            x += screen_width;
        }
        while (x >= screen_width)
        {
            x -= screen_width;
        }
        return x;
    }

    inline int torus_wrap_y(int y)
    {
        while (y < 0)
        {
            y += screen_height;
        }
        while (y >= screen_height)
        {
            y -= screen_height;
        }
        return y;
    }

    inline int vram_torus_diff_x(int x1, int x2)
    {
        int x = x1 - x2;
        while (x < - vram_screen_width / 2)
        {
            x += vram_screen_width;
        }
        while (x >= vram_screen_width / 2)
        {
            x -= vram_screen_width;
        }
        return x;
    }

    inline int vram_torus_diff_y(int y1, int y2)
    {
        int y = y1 - y2;
        while (y < - vram_screen_height / 2)
        {
            y += vram_screen_height;
        }
        while (y >= vram_screen_height / 2)
        {
            y -= vram_screen_height;
        }
        return y;
    }

// struct point

    inline point::point()
    {
        // no default initialization
    }

    inline point::point(int xa, int ya)
        :   x(xa), y(ya)
    {
    }

    inline point point::from_vram_pos(point vram_pos)
    {
        return 8 * (vram_pos - point(0, 128));
    }

    inline point point::to_vram_pos(point pos)
    {
        return (pos >> 3) + point(0, 128);
    }

    inline void point::ceiling_division(int d)
    {
        x = intrepid::ceiling_division(x, d);
        y = intrepid::ceiling_division(y, d);
    }

    inline void point::floor_division(int d)
    {
        x = intrepid::floor_division(x, d);
        y = intrepid::floor_division(y, d);
    }

    inline void point::torus_diff(point other)
    {
        x = intrepid::torus_diff_x(x, other.x);
        y = intrepid::torus_diff_y(y, other.y);
    }

    inline void point::torus_max(point other)
    {
        x = torus_max_x(x, other.x);
        y = torus_max_y(y, other.y);
    }

    inline void point::torus_wrap()
    {
        x = torus_wrap_x(x);
        y = torus_wrap_y(y);
    }

    inline void point::vram_torus_diff(point other)
    {
        x = intrepid::vram_torus_diff_x(x, other.x);
        y = intrepid::vram_torus_diff_y(y, other.y);
    }

    inline void point::operator+=(point other)
    {
        x += other.x;
        y += other.y;
    }

    inline void point::operator-=(point other)
    {
        x -= other.x;
        y -= other.y;
    }

    inline void point::operator*=(int s)
    {
        x *= s;
        y *= s;
    }

    inline void point::operator/=(int s)
    {
        x /= s;
        y /= s;
    }

    inline void point::operator>>=(int n)
    {
        x >>= n;
        y >>= n;
    }

    inline bool operator==(point p1, point p2)
    {
        return p1.x == p2.x && p1.y == p2.y;
    }

    inline bool operator!=(point p1, point p2)
    {
        return !(p1 == p2);
    }

    inline bool operator<=(point p1, point p2)
    {
        return p1.x <= p2.x && p1.y <= p2.y;
    }

    inline point operator+(point p1, point p2)
    {
        p1 += p2;
        return p1;
    }

    inline point operator-(point p1, point p2)
    {
        p1 -= p2;
        return p1;
    }

    inline point operator*(int s, point p)
    {
        p *= s;
        return p;
    }

    inline point operator/(point p, int s)
    {
        p /= s;
        return p;
    }

    inline point operator>>(point p, int n)
    {
        p >>= n;
        return p;
    }

    template<class Ch, class Tr>
    inline basic_ostream<Ch, Tr> &operator<<(basic_ostream<Ch, Tr> &os,
                                             point p)
    {
        return os << '(' << p.x << ", " << p.y << ')';
    }

    inline point ceiling_division(point p, int d)
    {
        p.ceiling_division(d);
        return p;
    }

    inline int dot_product(point p1, point p2)
    {
        return p1.x * p2.x + p2.y * p2.y;
    }

    inline point floor_division(point p, int d)
    {
        p.floor_division(d);
        return p;
    }

    inline int norm2(point p)
    {
        return dot_product(p, p);
    }

    inline point torus_diff(point p1, point p2)
    {
        p1.torus_diff(p2);
        return p1;
    }

    inline point torus_max(point p1, point p2)
    {
        p1.torus_max(p2);
        return p1;
    }

    inline point torus_wrap(point p)
    {
        p.torus_wrap();
        return p;
    }

    inline point vram_torus_diff(point p1, point p2)
    {
        p1.vram_torus_diff(p2);
        return p1;
    }

// struct rect

    inline rect::rect()
    {
        // no default initialization
    }

    inline rect::rect(point bla, point tra)
        :   bl(bla), tr(tra)
    {
    }

    inline rect rect::from_vram_pos(point vram_pos)
    {
        rect r;
        r.bl = point::from_vram_pos(vram_pos);
        r.tr = r.bl + point(7, 7);
        return r;
    }

    inline bool rect::empty() const
    {
        return !(bl <= tr);
    }

    inline bool rect::contains(point p) const
    {
        return    bl.x <= p.x && p.x <= tr.x
               && bl.y <= p.y && p.y <= tr.y;
    }

    inline bool rect::single_point() const
    {
        return bl == tr;
    }

    inline void rect::intersect(rect const &other)
    {
        bl.x = max(bl.x, other.bl.x);
        bl.y = max(bl.y, other.bl.y);
        tr.x = min(tr.x, other.tr.x);
        tr.y = min(tr.y, other.tr.y);
    }

    inline void rect::torus_intersect(rect const &other)
    {
        bl.x = torus_max_x(bl.x, other.bl.x);
        bl.y = torus_max_y(bl.y, other.bl.y);
        tr.x = torus_min_x(tr.x, other.tr.x);
        tr.y = torus_min_y(tr.y, other.tr.y);
        torus_wrap();
    }

    inline void rect::torus_wrap()
    {
        bl.torus_wrap();
        tr.torus_wrap();
    }

    inline void rect::operator+=(rect const &other)
    {
        bl += other.bl;
        tr += other.tr;
    }

    inline void rect::operator-=(rect const &other)
    {
        bl -= other.bl;
        tr -= other.tr;
    }

    inline void rect::operator*=(int s)
    {
        bl *= s;
        tr *= s;
    }

    inline bool operator==(rect const &r1, rect const &r2)
    {
        return r1.bl == r2.bl && r1.tr == r2.tr;
    }

    inline rect operator+(rect r1, rect const &r2)
    {
        r1 += r2;
        return r1;
    }

    inline rect operator-(rect r1, rect const &r2)
    {
        r1 -= r2;
        return r1;
    }

    inline rect operator*(int s, rect r)
    {
        r *= s;
        return r;
    }

    template<class Ch, class Tr>
    inline basic_ostream<Ch, Tr> &operator<<(basic_ostream<Ch, Tr> &os,
                                             rect const &r)
    {
        return os << '(' << r.bl << ", " << r.tr << ')';
    }

    inline point get_center(rect const &r)
    {
        point p((r.bl.x + r.tr.x) >> 1,
                (r.bl.y + r.tr.y) >> 1);
        p.torus_wrap();
        return p;
    }

    inline point move_into(point p, rect const &r)
    {
        int x;
        if (p.x < r.bl.x)
        {
            x = r.bl.x;
        }
        else if (p.x > r.tr.x)
        {
            x = r.tr.x;
        }
        else
        {
            x = p.x;
        }
        int y;
        if (p.y < r.bl.y)
        {
            y = r.bl.y;
        }
        else if (p.y > r.tr.y)
        {
            y = r.tr.y;
        }
        else
        {
            y = p.y;
        }
        return point(x, y);
    }

    inline point torus_get_center(rect const &r)
    {
        point p(r.bl.x + (torus_diff_x(r.tr.x, r.bl.x) >> 1),
                r.bl.y + (torus_diff_y(r.tr.y, r.bl.y) >> 1));
        p.torus_wrap();
        return p;
    }

    inline point torus_move_into(point p, rect const &r)
    {
        int x;
        if (torus_diff_x(p.x, r.bl.x) < 0)
        {
            x = r.bl.x;
        }
        else if (torus_diff_x(p.x, r.tr.x) > 0)
        {
            x = r.tr.x;
        }
        else
        {
            x = p.x;
        }
        int y;
        if (torus_diff_y(p.y, r.bl.y) < 0)
        {
            y = r.bl.y;
        }
        else if (torus_diff_y(p.y, r.tr.y) > 0)
        {
            y = r.tr.y;
        }
        else
        {
            y = p.y;
        }
        return point(x, y);
    }
} // end of namespace intrepid

#endif // include guard
