// parse_frame.cpp
// ---------------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#include "stdafx.hpp"

#include "parse_frame.hpp"

namespace intrepid
{
    namespace
    {
        class vram_iterator
            : public boost::iterator_facade<vram_iterator,
                                            uint16_t const,
                                            boost::random_access_traversal_tag,
                                            uint16_t>
        {
        public:
            vram_iterator() { }
            explicit vram_iterator(uint8_t const *p)
                :   p_(p)
            { }

        private:
            friend class boost::iterator_core_access;

            void advance(ptrdiff_t n)
            {
                p_ += 2 * n;
            }

            void decrement()
            {
                p_ -= 2;
            }

            uint16_t dereference() const
            {
                // Vectorram is low endian.
                return p_[0] | (p_[1] << 8);
            }

            ptrdiff_t distance_to(vram_iterator const &other) const
            {
                return (other.p_ - p_) >> 1;
            }

            void increment()
            {
                p_ += 2;
            }

            bool equal(vram_iterator const &other) const
            {
                return p_ == other.p_;
            }

            uint8_t const *p_;
        };

        void parse_vectorram(uint8_t const vectorram[],
                             parsed_frame &frame)
        {
            point vpos;
            int vs = 0;
            int vz;
            int ship_detect = 0;
            int ship_index = 0;
            point v1;
            point d;
            int opcode;
            vram_iterator vram_end(vectorram + comm_model::vectorram_size);
            // Be careful with two-word-instructions: Never read beyond the end
            // of the vectorram:
            -- vram_end;
            frame.clear();
            // DBUG
            // DBUG
            for (vram_iterator pc(vectorram); pc != vram_end; )
            {
                int op = *pc++;
                int op2;
                opcode = op >> 12;
                switch (opcode)
                {
                case 0xA: // LABS
                    op2 = *pc++;
                    vpos.y = op & 0x3FF;
                    vpos.x = op2 & 0x3FF;
                    vs = op2 >> 12;
                    break;
                case 0xB: // HALT
                    return;
                case 0xC: // JSRL
                    switch (op & 0xFFF)
                    {
                    case 0x8F3:
                        frame.add(vpos, vram_object_type::asteroid1, vs);
                        break;
                    case 0x8FF:
                        frame.add(vpos, vram_object_type::asteroid2, vs);
                        break;
                    case 0x90D:
                        frame.add(vpos, vram_object_type::asteroid3, vs);
                        break;
                    case 0x91A:
                        frame.add(vpos, vram_object_type::asteroid4, vs);
                        break;
                    case 0x8D0:
                        frame.add(vpos, vram_object_type::explosion1, vs);
                        break;
                    case 0x8B5:
                        frame.add(vpos, vram_object_type::explosion2, vs);
                        break;
                    case 0x896:
                        frame.add(vpos, vram_object_type::explosion3, vs);
                        break;
                    case 0x880:
                        frame.add(vpos, vram_object_type::explosion4, vs);
                        break;
                    case 0x929:
                        frame.add(vpos, vram_object_type::saucer, vs);
                        break;
                    case 0x852: // Copyright message
                        return;
                    }
                    break;
                case 0xD: // RTSL
                    assert(false);
                    return;
                case 0xE: // JMPL
                    switch (op)
                    {
                    case 0xE001:
                        frame.set_even(true);
                        break;
                    case 0xE201:
                        frame.set_even(false);
                        break;
                    default:
                        assert(false);
                        return;
                    }
                case 0xF: // SVEC
                    vz = (op & 0xf0) >> 4;
                    if (vz && ship_detect == 0)
                    {
                        ship_detect = 1;
                        frame.add(vpos, vram_object_type::ship, vs);
                    }
                    break;
                default: // VCTR
                    op2 = *pc++;
                    d.x = op2 & 0x3ff;
                    if (op2 & 0x400)
                    {
                        d.x = - d.x;
                    }
                    d.y = op & 0x3ff;
                    if (op & 0x400)
                    {
                        d.y = - d.y;
                    }
                    vz = op2 >> 12;
                    if (d == point(0, 0) && vz == 15)
                    {
                        frame.add(vpos, vram_object_type::shot, vs);
                    }
                    if (opcode == 6 && vz == 12 && d.x != 0 && d.y != 0)
                    {
                        switch (ship_detect)
                        {
                        case 0:
                            frame.add(vpos, vram_object_type::ship, vs);
                            ++ ship_detect;
                            // fall into case 1
                        case 1:
                            v1 = d;
                            ++ ship_detect;
                            break;
                        case 2:
                            frame.set_ship_present(v1 - d);
                            ++ ship_detect;
                            break;
                        }
                    }
                    else if (ship_detect == 2)
                    {
                        ship_detect = 1;
                    }
                }
            }
            // No HALT found. Ignore.
            cout << "Vectorram: HALT instruction not found." << endl;
        }
    }

    void parse_frame(comm_model::frame_packet const &frame_packet,
                     comm_model::ping_to_keys_map const &ping_to_keys,
                     parsed_frame &frame)
    {
        parse_vectorram(frame_packet.vectorram, frame);
        frame.set_frameno(frame_packet.frameno);
        frame.set_ping(frame_packet.ping);
        frame.set_ping_to_keys(ping_to_keys);
    }

} // end of namespace intrepid
