/* Game physics in the asteroids universe 
	These routines know about speed, directions, "Winkelbytes" and distances
	
*/
/* $Id: dir.c,v 1.6 2008/05/17 21:12:01 raap Exp $ */

/* The known or measured game physics */
/*
	Size of the playing field: x = 0..1023, y = 0 .. 768

	Turning speed of our ship: 360 degrees in 42 frames

	Speed and range of our shots:
		Shots of our own ship have a lifetime of around 70 frames (+/- 1)
		Their speed is 8 per frame.
		If our ship is moving, the speed is increased (maybe added to the ship speed TODO (check that)
		We can have at most 4 shots at the same time on the screen.
		Our firing rate is 1 shot every two frames 
		The shots start at a distance of 20 from the ships coordinates with the ships direction
		or better they start at 2.5 times their speed of the middle of the ship
		
		if you hit fire and turn simultaneously, what will happen first ?
		Answer: first the shot is started, then the ship is turned
		
	Speed of Asteroids:
	
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "vector.h"
#include "dir.h"

/* Our current working hypothesis:
 Internally the asteroid game stores all direction vectors as a 1 byte value.
  This value is used as an index into a table of real (dx, dy) vectors.
  The table is ordered. Increasing the index results in increasing the angle of the resulting vector (0.. 2PI).
  So increasing the index means turning left.
  After 255 a wrap around to 0 occurs.

 If we know the internal dir index, we can get the true direction by a table lookup dir_list[dir_index].
 If we know the true direction, we can get the table index by get_dir_idx(direction).

 Sometimes we do not know the true direction or the index. 
 The ship has only 64 visible directions.
 
 We can record which direction index gives which visible direction.
 This is recorded in the table visible_dir[dir_idx].


*/

/* This is our measured ordered internal direction table: */
/* We can interpret this table either with or without binary point
	If you think about a binary point, the values in this table are of format 2.6
	two bits are before binary point, with the first one the sign
	and 6 bits are after the binary point.
	The vectors are all nearly 1 pixel long.
	 
	 Or we think about each entry as an 8 bit integer register (Vector Register). Then all vectors are 64 pixel long.
	 To compute the position offset of the moving object for 1 frame, we compute
	 offset = int( Spd * VR / 64 )
	 
	 The possible speed values are exact to 3 binary places after the dot, so we may think of the speed as
	 given by an internal speed register SR, which is 8 times the real speed
	 Then the offset formula is:
	 offset = int( SR * VR / 512)
	 
	 Each value x in the list is to be interpreted as x/64
	 
	 We use the words subpixel to describe 1/64th of a real pixel
	 and subframe to describe 1/8 of a real frame
	 
	 so our table is measured in subpixels per frame
	 
*/
delta dir_list [256] = { 
/*  dx,   dy,            angle in degrees */
 {  63,    0},           /* dir_idx =   0, dir = 0.00 deg. */
 {  63,    1},           /* dir_idx =   1, dir = 0.91 deg. */
 {  63,    3},           /* dir_idx =   2, dir = 2.73 deg. */
 {  63,    4},           /* dir_idx =   3, dir = 3.63 deg. */
 {  63,    6},           /* dir_idx =   4, dir = 5.44 deg. */
 {  63,    8},           /* dir_idx =   5, dir = 7.24 deg. */
 {  63,    9},           /* dir_idx =   6, dir = 8.13 deg. */
 {  62,   11},           /* dir_idx =   7, dir = 10.06 deg. */
 {  62,   12},           /* dir_idx =   8, dir = 10.95 deg. */
 {  62,   14},           /* dir_idx =   9, dir = 12.72 deg. */
 {  61,   15},           /* dir_idx =  10, dir = 13.82 deg. */
 {  61,   17},           /* dir_idx =  11, dir = 15.57 deg. */
 {  61,   18},           /* dir_idx =  12, dir = 16.44 deg. */
 {  60,   20},           /* dir_idx =  13, dir = 18.43 deg. */
 {  60,   21},           /* dir_idx =  14, dir = 19.29 deg. */
 {  59,   23},           /* dir_idx =  15, dir = 21.30 deg. */
 {  58,   24},           /* dir_idx =  16, dir = 22.48 deg. */
 {  58,   25},           /* dir_idx =  17, dir = 23.32 deg. */
 {  57,   27},           /* dir_idx =  18, dir = 25.35 deg. */
 {  56,   28},           /* dir_idx =  19, dir = 26.57 deg. */
 {  56,   30},           /* dir_idx =  20, dir = 28.18 deg. */
 {  55,   31},           /* dir_idx =  21, dir = 29.41 deg. */
 {  54,   32},           /* dir_idx =  22, dir = 30.65 deg. */
 {  53,   34},           /* dir_idx =  23, dir = 32.68 deg. */
 {  53,   35},           /* dir_idx =  24, dir = 33.44 deg. */
 {  52,   36},           /* dir_idx =  25, dir = 34.70 deg. */
 {  51,   38},           /* dir_idx =  26, dir = 36.69 deg. */
 {  50,   39},           /* dir_idx =  27, dir = 37.95 deg. */
 {  49,   40},           /* dir_idx =  28, dir = 39.23 deg. */
 {  48,   41},           /* dir_idx =  29, dir = 40.50 deg. */
 {  47,   42},           /* dir_idx =  30, dir = 41.78 deg. */
 {  46,   44},           /* dir_idx =  31, dir = 43.73 deg. */
 {  45,   45},           /* dir_idx =  32, dir = 45.00 deg. */
 {  44,   46},           /* dir_idx =  33, dir = 46.27 deg. */
 {  42,   47},           /* dir_idx =  34, dir = 48.22 deg. */
 {  41,   48},           /* dir_idx =  35, dir = 49.50 deg. */
 {  40,   49},           /* dir_idx =  36, dir = 50.77 deg. */
 {  39,   50},           /* dir_idx =  37, dir = 52.05 deg. */
 {  38,   51},           /* dir_idx =  38, dir = 53.31 deg. */
 {  36,   52},           /* dir_idx =  39, dir = 55.30 deg. */
 {  35,   53},           /* dir_idx =  40, dir = 56.56 deg. */
 {  34,   53},           /* dir_idx =  41, dir = 57.32 deg. */
 {  32,   54},           /* dir_idx =  42, dir = 59.35 deg. */
 {  31,   55},           /* dir_idx =  43, dir = 60.59 deg. */
 {  30,   56},           /* dir_idx =  44, dir = 61.82 deg. */
 {  28,   56},           /* dir_idx =  45, dir = 63.43 deg. */
 {  27,   57},           /* dir_idx =  46, dir = 64.65 deg. */
 {  25,   58},           /* dir_idx =  47, dir = 66.68 deg. */
 {  24,   58},           /* dir_idx =  48, dir = 67.52 deg. */
 {  23,   59},           /* dir_idx =  49, dir = 68.70 deg. */
 {  21,   60},           /* dir_idx =  50, dir = 70.71 deg. */
 {  20,   60},           /* dir_idx =  51, dir = 71.57 deg. */
 {  18,   61},           /* dir_idx =  52, dir = 73.56 deg. */
 {  17,   61},           /* dir_idx =  53, dir = 74.43 deg. */
 {  15,   61},           /* dir_idx =  54, dir = 76.18 deg. */
 {  14,   62},           /* dir_idx =  55, dir = 77.28 deg. */
 {  12,   62},           /* dir_idx =  56, dir = 79.05 deg. */
 {  11,   62},           /* dir_idx =  57, dir = 79.94 deg. */
 {   9,   63},           /* dir_idx =  58, dir = 81.87 deg. */
 {   8,   63},           /* dir_idx =  59, dir = 82.76 deg. */
 {   6,   63},           /* dir_idx =  60, dir = 84.56 deg. */
 {   4,   63},           /* dir_idx =  61, dir = 86.37 deg. */
 {   3,   63},           /* dir_idx =  62, dir = 87.27 deg. */
 {   1,   63},           /* dir_idx =  63, dir = 89.09 deg. */
 {   0,   63},           /* dir_idx =  64, dir = 90.00 deg. */
 {  -2,   63},           /* dir_idx =  65, dir = 91.82 deg. */
 {  -3,   63},           /* dir_idx =  66, dir = 92.73 deg. */
 {  -5,   63},           /* dir_idx =  67, dir = 94.54 deg. */
 {  -6,   63},           /* dir_idx =  68, dir = 95.44 deg. */
 {  -8,   63},           /* dir_idx =  69, dir = 97.24 deg. */
 { -10,   63},           /* dir_idx =  70, dir = 99.02 deg. */
 { -11,   62},           /* dir_idx =  71, dir = 100.06 deg. */
 { -13,   62},           /* dir_idx =  72, dir = 101.84 deg. */
 { -14,   62},           /* dir_idx =  73, dir = 102.72 deg. */
 { -16,   61},           /* dir_idx =  74, dir = 104.70 deg. */
 { -17,   61},           /* dir_idx =  75, dir = 105.57 deg. */
 { -19,   61},           /* dir_idx =  76, dir = 107.30 deg. */
 { -20,   60},           /* dir_idx =  77, dir = 108.43 deg. */
 { -22,   60},           /* dir_idx =  78, dir = 110.14 deg. */
 { -23,   59},           /* dir_idx =  79, dir = 111.30 deg. */
 { -25,   58},           /* dir_idx =  80, dir = 113.32 deg. */
 { -26,   58},           /* dir_idx =  81, dir = 114.15 deg. */
 { -27,   57},           /* dir_idx =  82, dir = 115.35 deg. */
 { -29,   56},           /* dir_idx =  83, dir = 117.38 deg. */
 { -30,   56},           /* dir_idx =  84, dir = 118.18 deg. */
 { -32,   55},           /* dir_idx =  85, dir = 120.19 deg. */
 { -33,   54},           /* dir_idx =  86, dir = 121.43 deg. */
 { -34,   53},           /* dir_idx =  87, dir = 122.68 deg. */
 { -36,   53},           /* dir_idx =  88, dir = 124.19 deg. */
 { -37,   52},           /* dir_idx =  89, dir = 125.43 deg. */
 { -38,   51},           /* dir_idx =  90, dir = 126.69 deg. */
 { -39,   50},           /* dir_idx =  91, dir = 127.95 deg. */
 { -41,   49},           /* dir_idx =  92, dir = 129.92 deg. */
 { -42,   48},           /* dir_idx =  93, dir = 131.19 deg. */
 { -43,   47},           /* dir_idx =  94, dir = 132.46 deg. */
 { -44,   46},           /* dir_idx =  95, dir = 133.73 deg. */
 { -45,   45},           /* dir_idx =  96, dir = 135.00 deg. */
 { -46,   44},           /* dir_idx =  97, dir = 136.27 deg. */
 { -47,   42},           /* dir_idx =  98, dir = 138.22 deg. */
 { -48,   41},           /* dir_idx =  99, dir = 139.50 deg. */
 { -49,   40},           /* dir_idx = 100, dir = 140.77 deg. */
 { -50,   39},           /* dir_idx = 101, dir = 142.05 deg. */
 { -51,   38},           /* dir_idx = 102, dir = 143.31 deg. */
 { -52,   36},           /* dir_idx = 103, dir = 145.30 deg. */
 { -53,   35},           /* dir_idx = 104, dir = 146.56 deg. */
 { -54,   34},           /* dir_idx = 105, dir = 147.80 deg. */
 { -55,   32},           /* dir_idx = 106, dir = 149.81 deg. */
 { -56,   31},           /* dir_idx = 107, dir = 151.03 deg. */
 { -56,   30},           /* dir_idx = 108, dir = 151.82 deg. */
 { -57,   28},           /* dir_idx = 109, dir = 153.84 deg. */
 { -58,   27},           /* dir_idx = 110, dir = 155.04 deg. */
 { -58,   25},           /* dir_idx = 111, dir = 156.68 deg. */
 { -59,   24},           /* dir_idx = 112, dir = 157.86 deg. */
 { -59,   23},           /* dir_idx = 113, dir = 158.70 deg. */
 { -60,   21},           /* dir_idx = 114, dir = 160.71 deg. */
 { -61,   20},           /* dir_idx = 115, dir = 161.85 deg. */
 { -61,   18},           /* dir_idx = 116, dir = 163.56 deg. */
 { -61,   17},           /* dir_idx = 117, dir = 164.43 deg. */
 { -62,   15},           /* dir_idx = 118, dir = 166.40 deg. */
 { -62,   14},           /* dir_idx = 119, dir = 167.28 deg. */
 { -63,   12},           /* dir_idx = 120, dir = 169.22 deg. */
 { -63,   11},           /* dir_idx = 121, dir = 170.10 deg. */
 { -63,    9},           /* dir_idx = 122, dir = 171.87 deg. */
 { -63,    8},           /* dir_idx = 123, dir = 172.76 deg. */
 { -63,    6},           /* dir_idx = 124, dir = 174.56 deg. */
 { -64,    4},           /* dir_idx = 125, dir = 176.42 deg. */
 { -64,    3},           /* dir_idx = 126, dir = 177.32 deg. */
 { -64,    1},           /* dir_idx = 127, dir = 179.10 deg. */
 { -64,    0},           /* dir_idx = 128, dir = 180.00 deg. */
 { -64,   -2},           /* dir_idx = 129, dir = 181.79 deg. */
 { -64,   -3},           /* dir_idx = 130, dir = 182.68 deg. */
 { -64,   -5},           /* dir_idx = 131, dir = 184.47 deg. */
 { -63,   -6},           /* dir_idx = 132, dir = 185.44 deg. */
 { -63,   -8},           /* dir_idx = 133, dir = 187.24 deg. */
 { -63,  -10},           /* dir_idx = 134, dir = 189.02 deg. */
 { -63,  -11},           /* dir_idx = 135, dir = 189.90 deg. */
 { -63,  -13},           /* dir_idx = 136, dir = 191.66 deg. */
 { -62,  -14},           /* dir_idx = 137, dir = 192.72 deg. */
 { -62,  -16},           /* dir_idx = 138, dir = 194.47 deg. */
 { -61,  -17},           /* dir_idx = 139, dir = 195.57 deg. */
 { -61,  -19},           /* dir_idx = 140, dir = 197.30 deg. */
 { -61,  -20},           /* dir_idx = 141, dir = 198.15 deg. */
 { -60,  -22},           /* dir_idx = 142, dir = 200.14 deg. */
 { -59,  -23},           /* dir_idx = 143, dir = 201.30 deg. */
 { -59,  -25},           /* dir_idx = 144, dir = 202.96 deg. */
 { -58,  -26},           /* dir_idx = 145, dir = 204.15 deg. */
 { -58,  -27},           /* dir_idx = 146, dir = 204.96 deg. */
 { -57,  -29},           /* dir_idx = 147, dir = 206.97 deg. */
 { -56,  -30},           /* dir_idx = 148, dir = 208.18 deg. */
 { -56,  -32},           /* dir_idx = 149, dir = 209.74 deg. */
 { -55,  -33},           /* dir_idx = 150, dir = 210.96 deg. */
 { -54,  -34},           /* dir_idx = 151, dir = 212.20 deg. */
 { -53,  -36},           /* dir_idx = 152, dir = 214.19 deg. */
 { -52,  -37},           /* dir_idx = 153, dir = 215.43 deg. */
 { -51,  -38},           /* dir_idx = 154, dir = 216.69 deg. */
 { -50,  -39},           /* dir_idx = 155, dir = 217.95 deg. */
 { -49,  -41},           /* dir_idx = 156, dir = 219.92 deg. */
 { -48,  -42},           /* dir_idx = 157, dir = 221.19 deg. */
 { -47,  -43},           /* dir_idx = 158, dir = 222.46 deg. */
 { -46,  -44},           /* dir_idx = 159, dir = 223.73 deg. */
 { -45,  -45},           /* dir_idx = 160, dir = 225.00 deg. */
 { -44,  -46},           /* dir_idx = 161, dir = 226.27 deg. */
 { -43,  -47},           /* dir_idx = 162, dir = 227.54 deg. */
 { -42,  -48},           /* dir_idx = 163, dir = 228.81 deg. */
 { -41,  -49},           /* dir_idx = 164, dir = 230.08 deg. */
 { -39,  -50},           /* dir_idx = 165, dir = 232.05 deg. */
 { -38,  -51},           /* dir_idx = 166, dir = 233.31 deg. */
 { -37,  -52},           /* dir_idx = 167, dir = 234.57 deg. */
 { -36,  -53},           /* dir_idx = 168, dir = 235.81 deg. */
 { -34,  -54},           /* dir_idx = 169, dir = 237.80 deg. */
 { -33,  -55},           /* dir_idx = 170, dir = 239.04 deg. */
 { -32,  -56},           /* dir_idx = 171, dir = 240.26 deg. */
 { -30,  -56},           /* dir_idx = 172, dir = 241.82 deg. */
 { -29,  -57},           /* dir_idx = 173, dir = 243.03 deg. */
 { -27,  -58},           /* dir_idx = 174, dir = 245.04 deg. */
 { -26,  -58},           /* dir_idx = 175, dir = 245.85 deg. */
 { -25,  -59},           /* dir_idx = 176, dir = 247.04 deg. */
 { -23,  -59},           /* dir_idx = 177, dir = 248.70 deg. */
 { -22,  -60},           /* dir_idx = 178, dir = 249.86 deg. */
 { -20,  -61},           /* dir_idx = 179, dir = 251.85 deg. */
 { -19,  -61},           /* dir_idx = 180, dir = 252.70 deg. */
 { -17,  -61},           /* dir_idx = 181, dir = 254.43 deg. */
 { -16,  -62},           /* dir_idx = 182, dir = 255.53 deg. */
 { -14,  -62},           /* dir_idx = 183, dir = 257.28 deg. */
 { -13,  -63},           /* dir_idx = 184, dir = 258.34 deg. */
 { -11,  -63},           /* dir_idx = 185, dir = 260.10 deg. */
 { -10,  -63},           /* dir_idx = 186, dir = 260.98 deg. */
 {  -8,  -63},           /* dir_idx = 187, dir = 262.76 deg. */
 {  -6,  -63},           /* dir_idx = 188, dir = 264.56 deg. */
 {  -5,  -64},           /* dir_idx = 189, dir = 265.53 deg. */
 {  -3,  -64},           /* dir_idx = 190, dir = 267.32 deg. */
 {  -2,  -64},           /* dir_idx = 191, dir = 268.21 deg. */
 {   0,  -64},           /* dir_idx = 192, dir = 270.00 deg. */
 {   1,  -64},           /* dir_idx = 193, dir = 270.90 deg. */
 {   3,  -64},           /* dir_idx = 194, dir = 272.68 deg. */
 {   4,  -64},           /* dir_idx = 195, dir = 273.58 deg. */
 {   6,  -63},           /* dir_idx = 196, dir = 275.44 deg. */
 {   8,  -63},           /* dir_idx = 197, dir = 277.24 deg. */
 {   9,  -63},           /* dir_idx = 198, dir = 278.13 deg. */
 {  11,  -63},           /* dir_idx = 199, dir = 279.90 deg. */
 {  12,  -63},           /* dir_idx = 200, dir = 280.78 deg. */
 {  14,  -62},           /* dir_idx = 201, dir = 282.72 deg. */
 {  15,  -62},           /* dir_idx = 202, dir = 283.60 deg. */
 {  17,  -61},           /* dir_idx = 203, dir = 285.57 deg. */
 {  18,  -61},           /* dir_idx = 204, dir = 286.44 deg. */
 {  20,  -61},           /* dir_idx = 205, dir = 288.15 deg. */
 {  21,  -60},           /* dir_idx = 206, dir = 289.29 deg. */
 {  23,  -59},           /* dir_idx = 207, dir = 291.30 deg. */
 {  24,  -59},           /* dir_idx = 208, dir = 292.14 deg. */
 {  25,  -58},           /* dir_idx = 209, dir = 293.32 deg. */
 {  27,  -58},           /* dir_idx = 210, dir = 294.96 deg. */
 {  28,  -57},           /* dir_idx = 211, dir = 296.16 deg. */
 {  30,  -56},           /* dir_idx = 212, dir = 298.18 deg. */
 {  31,  -56},           /* dir_idx = 213, dir = 298.97 deg. */
 {  32,  -55},           /* dir_idx = 214, dir = 300.19 deg. */
 {  34,  -54},           /* dir_idx = 215, dir = 302.20 deg. */
 {  35,  -53},           /* dir_idx = 216, dir = 303.44 deg. */
 {  36,  -52},           /* dir_idx = 217, dir = 304.70 deg. */
 {  38,  -51},           /* dir_idx = 218, dir = 306.69 deg. */
 {  39,  -50},           /* dir_idx = 219, dir = 307.95 deg. */
 {  40,  -49},           /* dir_idx = 220, dir = 309.23 deg. */
 {  41,  -48},           /* dir_idx = 221, dir = 310.50 deg. */
 {  42,  -47},           /* dir_idx = 222, dir = 311.78 deg. */
 {  44,  -46},           /* dir_idx = 223, dir = 313.73 deg. */
 {  45,  -45},           /* dir_idx = 224, dir = 315.00 deg. */
 {  46,  -44},           /* dir_idx = 225, dir = 316.27 deg. */
 {  47,  -43},           /* dir_idx = 226, dir = 317.54 deg. */
 {  48,  -42},           /* dir_idx = 227, dir = 318.81 deg. */
 {  49,  -41},           /* dir_idx = 228, dir = 320.08 deg. */
 {  50,  -39},           /* dir_idx = 229, dir = 322.05 deg. */
 {  51,  -38},           /* dir_idx = 230, dir = 323.31 deg. */
 {  52,  -37},           /* dir_idx = 231, dir = 324.57 deg. */
 {  53,  -36},           /* dir_idx = 232, dir = 325.81 deg. */
 {  53,  -34},           /* dir_idx = 233, dir = 327.32 deg. */
 {  54,  -33},           /* dir_idx = 234, dir = 328.57 deg. */
 {  55,  -32},           /* dir_idx = 235, dir = 329.81 deg. */
 {  56,  -30},           /* dir_idx = 236, dir = 331.82 deg. */
 {  56,  -29},           /* dir_idx = 237, dir = 332.62 deg. */
 {  57,  -27},           /* dir_idx = 238, dir = 334.65 deg. */
 {  58,  -26},           /* dir_idx = 239, dir = 335.85 deg. */
 {  58,  -25},           /* dir_idx = 240, dir = 336.68 deg. */
 {  59,  -23},           /* dir_idx = 241, dir = 338.70 deg. */
 {  60,  -22},           /* dir_idx = 242, dir = 339.86 deg. */
 {  60,  -20},           /* dir_idx = 243, dir = 341.57 deg. */
 {  61,  -19},           /* dir_idx = 244, dir = 342.70 deg. */
 {  61,  -17},           /* dir_idx = 245, dir = 344.43 deg. */
 {  61,  -16},           /* dir_idx = 246, dir = 345.30 deg. */
 {  62,  -14},           /* dir_idx = 247, dir = 347.28 deg. */
 {  62,  -13},           /* dir_idx = 248, dir = 348.16 deg. */
 {  62,  -11},           /* dir_idx = 249, dir = 349.94 deg. */
 {  63,  -10},           /* dir_idx = 250, dir = 350.98 deg. */
 {  63,   -8},           /* dir_idx = 251, dir = 352.76 deg. */
 {  63,   -6},           /* dir_idx = 252, dir = 354.56 deg. */
 {  63,   -5},           /* dir_idx = 253, dir = 355.46 deg. */
 {  63,   -3},           /* dir_idx = 254, dir = 357.27 deg. */
 {  63,   -2}            /* dir_idx = 255, dir = 358.18 deg. */
};

/* External accessor function */
delta * get_dir(int idx){
	return &(dir_list[idx]);
};


delta visible_dir_list[256] = {
 {1536,    0},           /* dir_idx =   0, visible = 0.00 deg., true dir = 0.00 deg. */
 {1536,    0},           /* dir_idx =   1, visible = 0.00 deg., true dir = 0.91 deg. */
 {1536,    0},           /* dir_idx =   2, visible = 0.00 deg., true dir = 2.73 deg. */
 {1536,    0},           /* dir_idx =   3, visible = 0.00 deg., true dir = 3.63 deg. */
 {1528,  152},           /* dir_idx =   4, visible = 5.68 deg., true dir = 5.44 deg. */
 {1528,  152},           /* dir_idx =   5, visible = 5.68 deg., true dir = 7.24 deg. */
 {1528,  152},           /* dir_idx =   6, visible = 5.68 deg., true dir = 8.13 deg. */
 {1528,  152},           /* dir_idx =   7, visible = 5.68 deg., true dir = 10.06 deg. */
 {1504,  296},           /* dir_idx =   8, visible = 11.13 deg., true dir = 10.95 deg. */
 {1504,  296},           /* dir_idx =   9, visible = 11.13 deg., true dir = 12.72 deg. */
 {1504,  296},           /* dir_idx =  10, visible = 11.13 deg., true dir = 13.82 deg. */
 {1504,  296},           /* dir_idx =  11, visible = 11.13 deg., true dir = 15.57 deg. */
 {1472,  440},           /* dir_idx =  12, visible = 16.64 deg., true dir = 16.44 deg. */
 {1472,  440},           /* dir_idx =  13, visible = 16.64 deg., true dir = 18.43 deg. */
 {1472,  440},           /* dir_idx =  14, visible = 16.64 deg., true dir = 19.29 deg. */
 {1472,  440},           /* dir_idx =  15, visible = 16.64 deg., true dir = 21.30 deg. */
 {1416,  584},           /* dir_idx =  16, visible = 22.41 deg., true dir = 22.48 deg. */
 {1416,  584},           /* dir_idx =  17, visible = 22.41 deg., true dir = 23.32 deg. */
 {1416,  584},           /* dir_idx =  18, visible = 22.41 deg., true dir = 25.35 deg. */
 {1416,  584},           /* dir_idx =  19, visible = 22.41 deg., true dir = 26.57 deg. */
 {1360,  720},           /* dir_idx =  20, visible = 27.90 deg., true dir = 28.18 deg. */
 {1360,  720},           /* dir_idx =  21, visible = 27.90 deg., true dir = 29.41 deg. */
 {1360,  720},           /* dir_idx =  22, visible = 27.90 deg., true dir = 30.65 deg. */
 {1360,  720},           /* dir_idx =  23, visible = 27.90 deg., true dir = 32.68 deg. */
 {1280,  856},           /* dir_idx =  24, visible = 33.77 deg., true dir = 33.44 deg. */
 {1280,  856},           /* dir_idx =  25, visible = 33.77 deg., true dir = 34.70 deg. */
 {1280,  856},           /* dir_idx =  26, visible = 33.77 deg., true dir = 36.69 deg. */
 {1280,  856},           /* dir_idx =  27, visible = 33.77 deg., true dir = 37.95 deg. */
 {1192,  976},           /* dir_idx =  28, visible = 39.31 deg., true dir = 39.23 deg. */
 {1192,  976},           /* dir_idx =  29, visible = 39.31 deg., true dir = 40.50 deg. */
 {1192,  976},           /* dir_idx =  30, visible = 39.31 deg., true dir = 41.78 deg. */
 {1192,  976},           /* dir_idx =  31, visible = 39.31 deg., true dir = 43.73 deg. */
 {1088, 1088},           /* dir_idx =  32, visible = 45.00 deg., true dir = 45.00 deg. */
 {1088, 1088},           /* dir_idx =  33, visible = 45.00 deg., true dir = 46.27 deg. */
 {1088, 1088},           /* dir_idx =  34, visible = 45.00 deg., true dir = 48.22 deg. */
 {1088, 1088},           /* dir_idx =  35, visible = 45.00 deg., true dir = 49.50 deg. */
 { 976, 1192},           /* dir_idx =  36, visible = 50.69 deg., true dir = 50.77 deg. */
 { 976, 1192},           /* dir_idx =  37, visible = 50.69 deg., true dir = 52.05 deg. */
 { 976, 1192},           /* dir_idx =  38, visible = 50.69 deg., true dir = 53.31 deg. */
 { 976, 1192},           /* dir_idx =  39, visible = 50.69 deg., true dir = 55.30 deg. */
 { 856, 1280},           /* dir_idx =  40, visible = 56.23 deg., true dir = 56.56 deg. */
 { 856, 1280},           /* dir_idx =  41, visible = 56.23 deg., true dir = 57.32 deg. */
 { 856, 1280},           /* dir_idx =  42, visible = 56.23 deg., true dir = 59.35 deg. */
 { 856, 1280},           /* dir_idx =  43, visible = 56.23 deg., true dir = 60.59 deg. */
 { 720, 1360},           /* dir_idx =  44, visible = 62.10 deg., true dir = 61.82 deg. */
 { 720, 1360},           /* dir_idx =  45, visible = 62.10 deg., true dir = 63.43 deg. */
 { 720, 1360},           /* dir_idx =  46, visible = 62.10 deg., true dir = 64.65 deg. */
 { 720, 1360},           /* dir_idx =  47, visible = 62.10 deg., true dir = 66.68 deg. */
 { 584, 1416},           /* dir_idx =  48, visible = 67.59 deg., true dir = 67.52 deg. */
 { 584, 1416},           /* dir_idx =  49, visible = 67.59 deg., true dir = 68.70 deg. */
 { 584, 1416},           /* dir_idx =  50, visible = 67.59 deg., true dir = 70.71 deg. */
 { 584, 1416},           /* dir_idx =  51, visible = 67.59 deg., true dir = 71.57 deg. */
 { 440, 1472},           /* dir_idx =  52, visible = 73.36 deg., true dir = 73.56 deg. */
 { 440, 1472},           /* dir_idx =  53, visible = 73.36 deg., true dir = 74.43 deg. */
 { 440, 1472},           /* dir_idx =  54, visible = 73.36 deg., true dir = 76.18 deg. */
 { 440, 1472},           /* dir_idx =  55, visible = 73.36 deg., true dir = 77.28 deg. */
 { 296, 1504},           /* dir_idx =  56, visible = 78.87 deg., true dir = 79.05 deg. */
 { 296, 1504},           /* dir_idx =  57, visible = 78.87 deg., true dir = 79.94 deg. */
 { 296, 1504},           /* dir_idx =  58, visible = 78.87 deg., true dir = 81.87 deg. */
 { 296, 1504},           /* dir_idx =  59, visible = 78.87 deg., true dir = 82.76 deg. */
 { 152, 1528},           /* dir_idx =  60, visible = 84.32 deg., true dir = 84.56 deg. */
 { 152, 1528},           /* dir_idx =  61, visible = 84.32 deg., true dir = 86.37 deg. */
 { 152, 1528},           /* dir_idx =  62, visible = 84.32 deg., true dir = 87.27 deg. */
 { 152, 1528},           /* dir_idx =  63, visible = 84.32 deg., true dir = 89.09 deg. */
 {   0, 1536},           /* dir_idx =  64, visible = 90.00 deg., true dir = 90.00 deg. */
 {-152, 1528},           /* dir_idx =  65, visible = 95.68 deg., true dir = 91.82 deg. */
 {-152, 1528},           /* dir_idx =  66, visible = 95.68 deg., true dir = 92.73 deg. */
 {-152, 1528},           /* dir_idx =  67, visible = 95.68 deg., true dir = 94.54 deg. */
 {-152, 1528},           /* dir_idx =  68, visible = 95.68 deg., true dir = 95.44 deg. */
 {-296, 1504},           /* dir_idx =  69, visible = 101.13 deg., true dir = 97.24 deg. */
 {-296, 1504},           /* dir_idx =  70, visible = 101.13 deg., true dir = 99.02 deg. */
 {-296, 1504},           /* dir_idx =  71, visible = 101.13 deg., true dir = 100.06 deg. */
 {-296, 1504},           /* dir_idx =  72, visible = 101.13 deg., true dir = 101.84 deg. */
 {-440, 1472},           /* dir_idx =  73, visible = 106.64 deg., true dir = 102.72 deg. */
 {-440, 1472},           /* dir_idx =  74, visible = 106.64 deg., true dir = 104.70 deg. */
 {-440, 1472},           /* dir_idx =  75, visible = 106.64 deg., true dir = 105.57 deg. */
 {-440, 1472},           /* dir_idx =  76, visible = 106.64 deg., true dir = 107.30 deg. */
 {-584, 1416},           /* dir_idx =  77, visible = 112.41 deg., true dir = 108.43 deg. */
 {-584, 1416},           /* dir_idx =  78, visible = 112.41 deg., true dir = 110.14 deg. */
 {-584, 1416},           /* dir_idx =  79, visible = 112.41 deg., true dir = 111.30 deg. */
 {-584, 1416},           /* dir_idx =  80, visible = 112.41 deg., true dir = 113.32 deg. */
 {-720, 1360},           /* dir_idx =  81, visible = 117.90 deg., true dir = 114.15 deg. */
 {-720, 1360},           /* dir_idx =  82, visible = 117.90 deg., true dir = 115.35 deg. */
 {-720, 1360},           /* dir_idx =  83, visible = 117.90 deg., true dir = 117.38 deg. */
 {-720, 1360},           /* dir_idx =  84, visible = 117.90 deg., true dir = 118.18 deg. */
 {-856, 1280},           /* dir_idx =  85, visible = 123.77 deg., true dir = 120.19 deg. */
 {-856, 1280},           /* dir_idx =  86, visible = 123.77 deg., true dir = 121.43 deg. */
 {-856, 1280},           /* dir_idx =  87, visible = 123.77 deg., true dir = 122.68 deg. */
 {-856, 1280},           /* dir_idx =  88, visible = 123.77 deg., true dir = 124.19 deg. */
 {-976, 1192},           /* dir_idx =  89, visible = 129.31 deg., true dir = 125.43 deg. */
 {-976, 1192},           /* dir_idx =  90, visible = 129.31 deg., true dir = 126.69 deg. */
 {-976, 1192},           /* dir_idx =  91, visible = 129.31 deg., true dir = 127.95 deg. */
 {-976, 1192},           /* dir_idx =  92, visible = 129.31 deg., true dir = 129.92 deg. */
 {-1088, 1088},          /* dir_idx =  93, visible = 135.00 deg., true dir = 131.19 deg. */
 {-1088, 1088},          /* dir_idx =  94, visible = 135.00 deg., true dir = 132.46 deg. */
 {-1088, 1088},          /* dir_idx =  95, visible = 135.00 deg., true dir = 133.73 deg. */
 {-1088, 1088},          /* dir_idx =  96, visible = 135.00 deg., true dir = 135.00 deg. */
 {-1192,  976},          /* dir_idx =  97, visible = 140.69 deg., true dir = 136.27 deg. */
 {-1192,  976},          /* dir_idx =  98, visible = 140.69 deg., true dir = 138.22 deg. */
 {-1192,  976},          /* dir_idx =  99, visible = 140.69 deg., true dir = 139.50 deg. */
 {-1192,  976},          /* dir_idx = 100, visible = 140.69 deg., true dir = 140.77 deg. */
 {-1280,  856},          /* dir_idx = 101, visible = 146.23 deg., true dir = 142.05 deg. */
 {-1280,  856},          /* dir_idx = 102, visible = 146.23 deg., true dir = 143.31 deg. */
 {-1280,  856},          /* dir_idx = 103, visible = 146.23 deg., true dir = 145.30 deg. */
 {-1280,  856},          /* dir_idx = 104, visible = 146.23 deg., true dir = 146.56 deg. */
 {-1360,  720},          /* dir_idx = 105, visible = 152.10 deg., true dir = 147.80 deg. */
 {-1360,  720},          /* dir_idx = 106, visible = 152.10 deg., true dir = 149.81 deg. */
 {-1360,  720},          /* dir_idx = 107, visible = 152.10 deg., true dir = 151.03 deg. */
 {-1360,  720},          /* dir_idx = 108, visible = 152.10 deg., true dir = 151.82 deg. */
 {-1416,  584},          /* dir_idx = 109, visible = 157.59 deg., true dir = 153.84 deg. */
 {-1416,  584},          /* dir_idx = 110, visible = 157.59 deg., true dir = 155.04 deg. */
 {-1416,  584},          /* dir_idx = 111, visible = 157.59 deg., true dir = 156.68 deg. */
 {-1416,  584},          /* dir_idx = 112, visible = 157.59 deg., true dir = 157.86 deg. */
 {-1472,  440},          /* dir_idx = 113, visible = 163.36 deg., true dir = 158.70 deg. */
 {-1472,  440},          /* dir_idx = 114, visible = 163.36 deg., true dir = 160.71 deg. */
 {-1472,  440},          /* dir_idx = 115, visible = 163.36 deg., true dir = 161.85 deg. */
 {-1472,  440},          /* dir_idx = 116, visible = 163.36 deg., true dir = 163.56 deg. */
 {-1504,  296},          /* dir_idx = 117, visible = 168.87 deg., true dir = 164.43 deg. */
 {-1504,  296},          /* dir_idx = 118, visible = 168.87 deg., true dir = 166.40 deg. */
 {-1504,  296},          /* dir_idx = 119, visible = 168.87 deg., true dir = 167.28 deg. */
 {-1504,  296},          /* dir_idx = 120, visible = 168.87 deg., true dir = 169.22 deg. */
 {-1528,  152},          /* dir_idx = 121, visible = 174.32 deg., true dir = 170.10 deg. */
 {-1528,  152},          /* dir_idx = 122, visible = 174.32 deg., true dir = 171.87 deg. */
 {-1528,  152},          /* dir_idx = 123, visible = 174.32 deg., true dir = 172.76 deg. */
 {-1528,  152},          /* dir_idx = 124, visible = 174.32 deg., true dir = 174.56 deg. */
 {-1536,    0},          /* dir_idx = 125, visible = 180.00 deg., true dir = 176.42 deg. */
 {-1536,    0},          /* dir_idx = 126, visible = 180.00 deg., true dir = 177.32 deg. */
 {-1536,    0},          /* dir_idx = 127, visible = 180.00 deg., true dir = 179.10 deg. */
 {-1536,    0},          /* dir_idx = 128, visible = 180.00 deg., true dir = 180.00 deg. */
 {-1536,    0},          /* dir_idx = 129, visible = 180.00 deg., true dir = 181.79 deg. */
 {-1536,    0},          /* dir_idx = 130, visible = 180.00 deg., true dir = 182.68 deg. */
 {-1536,    0},          /* dir_idx = 131, visible = 180.00 deg., true dir = 184.47 deg. */
 {-1528, -152},          /* dir_idx = 132, visible = 185.68 deg., true dir = 185.44 deg. */
 {-1528, -152},          /* dir_idx = 133, visible = 185.68 deg., true dir = 187.24 deg. */
 {-1528, -152},          /* dir_idx = 134, visible = 185.68 deg., true dir = 189.02 deg. */
 {-1528, -152},          /* dir_idx = 135, visible = 185.68 deg., true dir = 189.90 deg. */
 {-1504, -296},          /* dir_idx = 136, visible = 191.13 deg., true dir = 191.66 deg. */
 {-1504, -296},          /* dir_idx = 137, visible = 191.13 deg., true dir = 192.72 deg. */
 {-1504, -296},          /* dir_idx = 138, visible = 191.13 deg., true dir = 194.47 deg. */
 {-1504, -296},          /* dir_idx = 139, visible = 191.13 deg., true dir = 195.57 deg. */
 {-1472, -440},          /* dir_idx = 140, visible = 196.64 deg., true dir = 197.30 deg. */
 {-1472, -440},          /* dir_idx = 141, visible = 196.64 deg., true dir = 198.15 deg. */
 {-1472, -440},          /* dir_idx = 142, visible = 196.64 deg., true dir = 200.14 deg. */
 {-1472, -440},          /* dir_idx = 143, visible = 196.64 deg., true dir = 201.30 deg. */
 {-1416, -584},          /* dir_idx = 144, visible = 202.41 deg., true dir = 202.96 deg. */
 {-1416, -584},          /* dir_idx = 145, visible = 202.41 deg., true dir = 204.15 deg. */
 {-1416, -584},          /* dir_idx = 146, visible = 202.41 deg., true dir = 204.96 deg. */
 {-1416, -584},          /* dir_idx = 147, visible = 202.41 deg., true dir = 206.97 deg. */
 {-1360, -720},          /* dir_idx = 148, visible = 207.90 deg., true dir = 208.18 deg. */
 {-1360, -720},          /* dir_idx = 149, visible = 207.90 deg., true dir = 209.74 deg. */
 {-1360, -720},          /* dir_idx = 150, visible = 207.90 deg., true dir = 210.96 deg. */
 {-1360, -720},          /* dir_idx = 151, visible = 207.90 deg., true dir = 212.20 deg. */
 {-1280, -856},          /* dir_idx = 152, visible = 213.77 deg., true dir = 214.19 deg. */
 {-1280, -856},          /* dir_idx = 153, visible = 213.77 deg., true dir = 215.43 deg. */
 {-1280, -856},          /* dir_idx = 154, visible = 213.77 deg., true dir = 216.69 deg. */
 {-1280, -856},          /* dir_idx = 155, visible = 213.77 deg., true dir = 217.95 deg. */
 {-1192, -976},          /* dir_idx = 156, visible = 219.31 deg., true dir = 219.92 deg. */
 {-1192, -976},          /* dir_idx = 157, visible = 219.31 deg., true dir = 221.19 deg. */
 {-1192, -976},          /* dir_idx = 158, visible = 219.31 deg., true dir = 222.46 deg. */
 {-1192, -976},          /* dir_idx = 159, visible = 219.31 deg., true dir = 223.73 deg. */
 {-1088, -1088},         /* dir_idx = 160, visible = 225.00 deg., true dir = 225.00 deg. */
 {-1088, -1088},         /* dir_idx = 161, visible = 225.00 deg., true dir = 226.27 deg. */
 {-1088, -1088},         /* dir_idx = 162, visible = 225.00 deg., true dir = 227.54 deg. */
 {-1088, -1088},         /* dir_idx = 163, visible = 225.00 deg., true dir = 228.81 deg. */
 {-976, -1192},          /* dir_idx = 164, visible = 230.69 deg., true dir = 230.08 deg. */
 {-976, -1192},          /* dir_idx = 165, visible = 230.69 deg., true dir = 232.05 deg. */
 {-976, -1192},          /* dir_idx = 166, visible = 230.69 deg., true dir = 233.31 deg. */
 {-976, -1192},          /* dir_idx = 167, visible = 230.69 deg., true dir = 234.57 deg. */
 {-856, -1280},          /* dir_idx = 168, visible = 236.23 deg., true dir = 235.81 deg. */
 {-856, -1280},          /* dir_idx = 169, visible = 236.23 deg., true dir = 237.80 deg. */
 {-856, -1280},          /* dir_idx = 170, visible = 236.23 deg., true dir = 239.04 deg. */
 {-856, -1280},          /* dir_idx = 171, visible = 236.23 deg., true dir = 240.26 deg. */
 {-720, -1360},          /* dir_idx = 172, visible = 242.10 deg., true dir = 241.82 deg. */
 {-720, -1360},          /* dir_idx = 173, visible = 242.10 deg., true dir = 243.03 deg. */
 {-720, -1360},          /* dir_idx = 174, visible = 242.10 deg., true dir = 245.04 deg. */
 {-720, -1360},          /* dir_idx = 175, visible = 242.10 deg., true dir = 245.85 deg. */
 {-584, -1416},          /* dir_idx = 176, visible = 247.59 deg., true dir = 247.04 deg. */
 {-584, -1416},          /* dir_idx = 177, visible = 247.59 deg., true dir = 248.70 deg. */
 {-584, -1416},          /* dir_idx = 178, visible = 247.59 deg., true dir = 249.86 deg. */
 {-584, -1416},          /* dir_idx = 179, visible = 247.59 deg., true dir = 251.85 deg. */
 {-440, -1472},          /* dir_idx = 180, visible = 253.36 deg., true dir = 252.70 deg. */
 {-440, -1472},          /* dir_idx = 181, visible = 253.36 deg., true dir = 254.43 deg. */
 {-440, -1472},          /* dir_idx = 182, visible = 253.36 deg., true dir = 255.53 deg. */
 {-440, -1472},          /* dir_idx = 183, visible = 253.36 deg., true dir = 257.28 deg. */
 {-296, -1504},          /* dir_idx = 184, visible = 258.87 deg., true dir = 258.34 deg. */
 {-296, -1504},          /* dir_idx = 185, visible = 258.87 deg., true dir = 260.10 deg. */
 {-296, -1504},          /* dir_idx = 186, visible = 258.87 deg., true dir = 260.98 deg. */
 {-296, -1504},          /* dir_idx = 187, visible = 258.87 deg., true dir = 262.76 deg. */
 {-152, -1528},          /* dir_idx = 188, visible = 264.32 deg., true dir = 264.56 deg. */
 {-152, -1528},          /* dir_idx = 189, visible = 264.32 deg., true dir = 265.53 deg. */
 {-152, -1528},          /* dir_idx = 190, visible = 264.32 deg., true dir = 267.32 deg. */
 {-152, -1528},          /* dir_idx = 191, visible = 264.32 deg., true dir = 268.21 deg. */
 {   0, -1536},          /* dir_idx = 192, visible = 270.00 deg., true dir = 270.00 deg. */
 { 152, -1528},          /* dir_idx = 193, visible = 275.68 deg., true dir = 270.90 deg. */
 { 152, -1528},          /* dir_idx = 194, visible = 275.68 deg., true dir = 272.68 deg. */
 { 152, -1528},          /* dir_idx = 195, visible = 275.68 deg., true dir = 273.58 deg. */
 { 152, -1528},          /* dir_idx = 196, visible = 275.68 deg., true dir = 275.44 deg. */
 { 296, -1504},          /* dir_idx = 197, visible = 281.13 deg., true dir = 277.24 deg. */
 { 296, -1504},          /* dir_idx = 198, visible = 281.13 deg., true dir = 278.13 deg. */
 { 296, -1504},          /* dir_idx = 199, visible = 281.13 deg., true dir = 279.90 deg. */
 { 296, -1504},          /* dir_idx = 200, visible = 281.13 deg., true dir = 280.78 deg. */
 { 440, -1472},          /* dir_idx = 201, visible = 286.64 deg., true dir = 282.72 deg. */
 { 440, -1472},          /* dir_idx = 202, visible = 286.64 deg., true dir = 283.60 deg. */
 { 440, -1472},          /* dir_idx = 203, visible = 286.64 deg., true dir = 285.57 deg. */
 { 440, -1472},          /* dir_idx = 204, visible = 286.64 deg., true dir = 286.44 deg. */
 { 584, -1416},          /* dir_idx = 205, visible = 292.41 deg., true dir = 288.15 deg. */
 { 584, -1416},          /* dir_idx = 206, visible = 292.41 deg., true dir = 289.29 deg. */
 { 584, -1416},          /* dir_idx = 207, visible = 292.41 deg., true dir = 291.30 deg. */
 { 584, -1416},          /* dir_idx = 208, visible = 292.41 deg., true dir = 292.14 deg. */
 { 720, -1360},          /* dir_idx = 209, visible = 297.90 deg., true dir = 293.32 deg. */
 { 720, -1360},          /* dir_idx = 210, visible = 297.90 deg., true dir = 294.96 deg. */
 { 720, -1360},          /* dir_idx = 211, visible = 297.90 deg., true dir = 296.16 deg. */
 { 720, -1360},          /* dir_idx = 212, visible = 297.90 deg., true dir = 298.18 deg. */
 { 856, -1280},          /* dir_idx = 213, visible = 303.77 deg., true dir = 298.97 deg. */
 { 856, -1280},          /* dir_idx = 214, visible = 303.77 deg., true dir = 300.19 deg. */
 { 856, -1280},          /* dir_idx = 215, visible = 303.77 deg., true dir = 302.20 deg. */
 { 856, -1280},          /* dir_idx = 216, visible = 303.77 deg., true dir = 303.44 deg. */
 { 976, -1192},          /* dir_idx = 217, visible = 309.31 deg., true dir = 304.70 deg. */
 { 976, -1192},          /* dir_idx = 218, visible = 309.31 deg., true dir = 306.69 deg. */
 { 976, -1192},          /* dir_idx = 219, visible = 309.31 deg., true dir = 307.95 deg. */
 { 976, -1192},          /* dir_idx = 220, visible = 309.31 deg., true dir = 309.23 deg. */
 {1088, -1088},          /* dir_idx = 221, visible = 315.00 deg., true dir = 310.50 deg. */
 {1088, -1088},          /* dir_idx = 222, visible = 315.00 deg., true dir = 311.78 deg. */
 {1088, -1088},          /* dir_idx = 223, visible = 315.00 deg., true dir = 313.73 deg. */
 {1088, -1088},          /* dir_idx = 224, visible = 315.00 deg., true dir = 315.00 deg. */
 {1192, -976},           /* dir_idx = 225, visible = 320.69 deg., true dir = 316.27 deg. */
 {1192, -976},           /* dir_idx = 226, visible = 320.69 deg., true dir = 317.54 deg. */
 {1192, -976},           /* dir_idx = 227, visible = 320.69 deg., true dir = 318.81 deg. */
 {1192, -976},           /* dir_idx = 228, visible = 320.69 deg., true dir = 320.08 deg. */
 {1280, -856},           /* dir_idx = 229, visible = 326.23 deg., true dir = 322.05 deg. */
 {1280, -856},           /* dir_idx = 230, visible = 326.23 deg., true dir = 323.31 deg. */
 {1280, -856},           /* dir_idx = 231, visible = 326.23 deg., true dir = 324.57 deg. */
 {1280, -856},           /* dir_idx = 232, visible = 326.23 deg., true dir = 325.81 deg. */
 {1360, -720},           /* dir_idx = 233, visible = 332.10 deg., true dir = 327.32 deg. */
 {1360, -720},           /* dir_idx = 234, visible = 332.10 deg., true dir = 328.57 deg. */
 {1360, -720},           /* dir_idx = 235, visible = 332.10 deg., true dir = 329.81 deg. */
 {1360, -720},           /* dir_idx = 236, visible = 332.10 deg., true dir = 331.82 deg. */
 {1416, -584},           /* dir_idx = 237, visible = 337.59 deg., true dir = 332.62 deg. */
 {1416, -584},           /* dir_idx = 238, visible = 337.59 deg., true dir = 334.65 deg. */
 {1416, -584},           /* dir_idx = 239, visible = 337.59 deg., true dir = 335.85 deg. */
 {1416, -584},           /* dir_idx = 240, visible = 337.59 deg., true dir = 336.68 deg. */
 {1472, -440},           /* dir_idx = 241, visible = 343.36 deg., true dir = 338.70 deg. */
 {1472, -440},           /* dir_idx = 242, visible = 343.36 deg., true dir = 339.86 deg. */
 {1472, -440},           /* dir_idx = 243, visible = 343.36 deg., true dir = 341.57 deg. */
 {1472, -440},           /* dir_idx = 244, visible = 343.36 deg., true dir = 342.70 deg. */
 {1504, -296},           /* dir_idx = 245, visible = 348.87 deg., true dir = 344.43 deg. */
 {1504, -296},           /* dir_idx = 246, visible = 348.87 deg., true dir = 345.30 deg. */
 {1504, -296},           /* dir_idx = 247, visible = 348.87 deg., true dir = 347.28 deg. */
 {1504, -296},           /* dir_idx = 248, visible = 348.87 deg., true dir = 348.16 deg. */
 {1528, -152},           /* dir_idx = 249, visible = 354.32 deg., true dir = 349.94 deg. */
 {1528, -152},           /* dir_idx = 250, visible = 354.32 deg., true dir = 350.98 deg. */
 {1528, -152},           /* dir_idx = 251, visible = 354.32 deg., true dir = 352.76 deg. */
 {1528, -152},           /* dir_idx = 252, visible = 354.32 deg., true dir = 354.56 deg. */
 {1536,    0},           /* dir_idx = 253, visible = 0.00 deg., true dir = 355.46 deg. */
 {1536,    0},           /* dir_idx = 254, visible = 0.00 deg., true dir = 357.27 deg. */
 {1536,    0}            /* dir_idx = 255, visible = 0.00 deg., true dir = 358.18 deg. */

};


/* Possible visible directions: 64 */
delta visi_dir[] = {
 {1536,    0},           /* 0.00 deg. */
 {1528,  152},           /* 5.68 deg. */
 {1504,  296},           /* 11.13 deg. */
 {1472,  440},           /* 16.64 deg. */
 {1416,  584},           /* 22.41 deg. */
 {1360,  720},           /* 27.90 deg. */
 {1280,  856},           /* 33.77 deg. */
 {1192,  976},           /* 39.31 deg. */
 {1088, 1088},           /* 45.00 deg. */
 { 976, 1192},           /* 50.69 deg. */
 { 856, 1280},           /* 56.23 deg. */
 { 720, 1360},           /* 62.10 deg. */
 { 584, 1416},           /* 67.59 deg. */
 { 440, 1472},           /* 73.36 deg. */
 { 296, 1504},           /* 78.87 deg. */
 { 152, 1528},           /* 84.32 deg. */
 {   0, 1536},           /* 90.00 deg. */
 {-152, 1528},           /* 95.68 deg. */
 {-296, 1504},           /* 101.13 deg. */
 {-440, 1472},           /* 106.64 deg. */
 {-584, 1416},           /* 112.41 deg. */
 {-720, 1360},           /* 117.90 deg. */
 {-856, 1280},           /* 123.77 deg. */
 {-976, 1192},           /* 129.31 deg. */
 {-1088, 1088},          /* 135.00 deg. */
 {-1192,  976},          /* 140.69 deg. */
 {-1280,  856},          /* 146.23 deg. */
 {-1360,  720},          /* 152.10 deg. */
 {-1416,  584},          /* 157.59 deg. */
 {-1472,  440},          /* 163.36 deg. */
 {-1504,  296},          /* 168.87 deg. */
 {-1528,  152},          /* 174.32 deg. */
 {-1536,    0},          /* 180.00 deg. */
 {-1528, -152},          /* 185.68 deg. */
 {-1504, -296},          /* 191.13 deg. */
 {-1472, -440},          /* 196.64 deg. */
 {-1416, -584},          /* 202.41 deg. */
 {-1360, -720},          /* 207.90 deg. */
 {-1280, -856},          /* 213.77 deg. */
 {-1192, -976},          /* 219.31 deg. */
 {-1088, -1088},         /* 225.00 deg. */
 {-976, -1192},          /* 230.69 deg. */
 {-856, -1280},          /* 236.23 deg. */
 {-720, -1360},          /* 242.10 deg. */
 {-584, -1416},          /* 247.59 deg. */
 {-440, -1472},          /* 253.36 deg. */
 {-296, -1504},          /* 258.87 deg. */
 {-152, -1528},          /* 264.32 deg. */
 {   0, -1536},          /* 270.00 deg. */
 { 152, -1528},          /* 275.68 deg. */
 { 296, -1504},          /* 281.13 deg. */
 { 440, -1472},          /* 286.64 deg. */
 { 584, -1416},          /* 292.41 deg. */
 { 720, -1360},          /* 297.90 deg. */
 { 856, -1280},          /* 303.77 deg. */
 { 976, -1192},          /* 309.31 deg. */
 {1088, -1088},          /* 315.00 deg. */
 {1192, -976},           /* 320.69 deg. */
 {1280, -856},           /* 326.23 deg. */
 {1360, -720},           /* 332.10 deg. */
 {1416, -584},           /* 337.59 deg. */
 {1472, -440},           /* 343.36 deg. */
 {1504, -296},           /* 348.87 deg. */
 {1528, -152}            /* 354.32 deg. */

};

/* given a visi_dir_idx (0 .. 63)
	returns the lowest possible real dir_idx
*/
int min_dir_idx[64] = {
 253,		/* visi_dir_idx =   0, dir = 355.46 deg. */
   4,		/* visi_dir_idx =   1, dir = 5.44 deg. */
   8,		/* visi_dir_idx =   2, dir = 10.95 deg. */
  12,		/* visi_dir_idx =   3, dir = 16.44 deg. */
  16,		/* visi_dir_idx =   4, dir = 22.48 deg. */
  20,		/* visi_dir_idx =   5, dir = 28.18 deg. */
  24,		/* visi_dir_idx =   6, dir = 33.44 deg. */
  28,		/* visi_dir_idx =   7, dir = 39.23 deg. */
  32,		/* visi_dir_idx =   8, dir = 45.00 deg. */
  36,		/* visi_dir_idx =   9, dir = 50.77 deg. */
  40,		/* visi_dir_idx =  10, dir = 56.56 deg. */
  44,		/* visi_dir_idx =  11, dir = 61.82 deg. */
  48,		/* visi_dir_idx =  12, dir = 67.52 deg. */
  52,		/* visi_dir_idx =  13, dir = 73.56 deg. */
  56,		/* visi_dir_idx =  14, dir = 79.05 deg. */
  60,		/* visi_dir_idx =  15, dir = 84.56 deg. */
  64,		/* visi_dir_idx =  16, dir = 90.00 deg. */
  65,		/* visi_dir_idx =  17, dir = 91.82 deg. */
  69,		/* visi_dir_idx =  18, dir = 97.24 deg. */
  73,		/* visi_dir_idx =  19, dir = 102.72 deg. */
  77,		/* visi_dir_idx =  20, dir = 108.43 deg. */
  81,		/* visi_dir_idx =  21, dir = 114.15 deg. */
  85,		/* visi_dir_idx =  22, dir = 120.19 deg. */
  89,		/* visi_dir_idx =  23, dir = 125.43 deg. */
  93,		/* visi_dir_idx =  24, dir = 131.19 deg. */
  97,		/* visi_dir_idx =  25, dir = 136.27 deg. */
 101,		/* visi_dir_idx =  26, dir = 142.05 deg. */
 105,		/* visi_dir_idx =  27, dir = 147.80 deg. */
 109,		/* visi_dir_idx =  28, dir = 153.84 deg. */
 113,		/* visi_dir_idx =  29, dir = 158.70 deg. */
 117,		/* visi_dir_idx =  30, dir = 164.43 deg. */
 121,		/* visi_dir_idx =  31, dir = 170.10 deg. */
 125,		/* visi_dir_idx =  32, dir = 176.42 deg. */
 132,		/* visi_dir_idx =  33, dir = 185.44 deg. */
 136,		/* visi_dir_idx =  34, dir = 191.66 deg. */
 140,		/* visi_dir_idx =  35, dir = 197.30 deg. */
 144,		/* visi_dir_idx =  36, dir = 202.96 deg. */
 148,		/* visi_dir_idx =  37, dir = 208.18 deg. */
 152,		/* visi_dir_idx =  38, dir = 214.19 deg. */
 156,		/* visi_dir_idx =  39, dir = 219.92 deg. */
 160,		/* visi_dir_idx =  40, dir = 225.00 deg. */
 164,		/* visi_dir_idx =  41, dir = 230.08 deg. */
 168,		/* visi_dir_idx =  42, dir = 235.81 deg. */
 172,		/* visi_dir_idx =  43, dir = 241.82 deg. */
 176,		/* visi_dir_idx =  44, dir = 247.04 deg. */
 180,		/* visi_dir_idx =  45, dir = 252.70 deg. */
 184,		/* visi_dir_idx =  46, dir = 258.34 deg. */
 188,		/* visi_dir_idx =  47, dir = 264.56 deg. */
 192,		/* visi_dir_idx =  48, dir = 270.00 deg. */
 193,		/* visi_dir_idx =  49, dir = 270.90 deg. */
 197,		/* visi_dir_idx =  50, dir = 277.24 deg. */
 201,		/* visi_dir_idx =  51, dir = 282.72 deg. */
 205,		/* visi_dir_idx =  52, dir = 288.15 deg. */
 209,		/* visi_dir_idx =  53, dir = 293.32 deg. */
 213,		/* visi_dir_idx =  54, dir = 298.97 deg. */
 217,		/* visi_dir_idx =  55, dir = 304.70 deg. */
 221,		/* visi_dir_idx =  56, dir = 310.50 deg. */
 225,		/* visi_dir_idx =  57, dir = 316.27 deg. */
 229,		/* visi_dir_idx =  58, dir = 322.05 deg. */
 233,		/* visi_dir_idx =  59, dir = 327.32 deg. */
 237,		/* visi_dir_idx =  60, dir = 332.62 deg. */
 241,		/* visi_dir_idx =  61, dir = 338.70 deg. */
 245,		/* visi_dir_idx =  62, dir = 344.43 deg. */
 249		/* visi_dir_idx =  63, dir = 349.94 deg. */

};


/* given a visi_dir_idx (0 .. 63)
	returns the highest possible real dir_idx
*/
int max_dir_idx[64]={
   3,		/* visi_dir_idx =   0, dir = 3.63 deg. */
   7,		/* visi_dir_idx =   1, dir = 10.06 deg. */
  11,		/* visi_dir_idx =   2, dir = 15.57 deg. */
  15,		/* visi_dir_idx =   3, dir = 21.30 deg. */
  19,		/* visi_dir_idx =   4, dir = 26.57 deg. */
  23,           /* visi_dir_idx =   5, dir = 32.68 deg. */
  27,           /* visi_dir_idx =   6, dir = 37.95 deg. */
  31,           /* visi_dir_idx =   7, dir = 43.73 deg. */
  35,           /* visi_dir_idx =   8, dir = 49.50 deg. */
  39,           /* visi_dir_idx =   9, dir = 55.30 deg. */
  43,           /* visi_dir_idx =  10, dir = 60.59 deg. */
  47,           /* visi_dir_idx =  11, dir = 66.68 deg. */
  51,           /* visi_dir_idx =  12, dir = 71.57 deg. */
  55,           /* visi_dir_idx =  13, dir = 77.28 deg. */
  59,           /* visi_dir_idx =  14, dir = 82.76 deg. */
  63,           /* visi_dir_idx =  15, dir = 89.09 deg. */
  64,           /* visi_dir_idx =  16, dir = 90.00 deg. */
  68,           /* visi_dir_idx =  17, dir = 95.44 deg. */
  72,           /* visi_dir_idx =  18, dir = 101.84 deg. */
  76,           /* visi_dir_idx =  19, dir = 107.30 deg. */
  80,           /* visi_dir_idx =  20, dir = 113.32 deg. */
  84,           /* visi_dir_idx =  21, dir = 118.18 deg. */
  88,           /* visi_dir_idx =  22, dir = 124.19 deg. */
  92,           /* visi_dir_idx =  23, dir = 129.92 deg. */
  96,           /* visi_dir_idx =  24, dir = 135.00 deg. */
 100,           /* visi_dir_idx =  25, dir = 140.77 deg. */
 104,           /* visi_dir_idx =  26, dir = 146.56 deg. */
 108,           /* visi_dir_idx =  27, dir = 151.82 deg. */
 112,           /* visi_dir_idx =  28, dir = 157.86 deg. */
 116,           /* visi_dir_idx =  29, dir = 163.56 deg. */
 120,           /* visi_dir_idx =  30, dir = 169.22 deg. */
 124,           /* visi_dir_idx =  31, dir = 174.56 deg. */
 131,		/* visi_dir_idx =  32, dir = 184.47 deg. */
 135,           /* visi_dir_idx =  33, dir = 189.90 deg. */
 139,           /* visi_dir_idx =  34, dir = 195.57 deg. */
 143,           /* visi_dir_idx =  35, dir = 201.30 deg. */
 147,           /* visi_dir_idx =  36, dir = 206.97 deg. */
 151,           /* visi_dir_idx =  37, dir = 212.20 deg. */
 155,           /* visi_dir_idx =  38, dir = 217.95 deg. */
 159,           /* visi_dir_idx =  39, dir = 223.73 deg. */
 163,           /* visi_dir_idx =  40, dir = 228.81 deg. */
 167,           /* visi_dir_idx =  41, dir = 234.57 deg. */
 171,           /* visi_dir_idx =  42, dir = 240.26 deg. */
 175,           /* visi_dir_idx =  43, dir = 245.85 deg. */
 179,           /* visi_dir_idx =  44, dir = 251.85 deg. */
 183,           /* visi_dir_idx =  45, dir = 257.28 deg. */
 187,           /* visi_dir_idx =  46, dir = 262.76 deg. */
 191,           /* visi_dir_idx =  47, dir = 268.21 deg. */
 192,           /* visi_dir_idx =  48, dir = 270.00 deg. */
 196,           /* visi_dir_idx =  49, dir = 275.44 deg. */
 200,           /* visi_dir_idx =  50, dir = 280.78 deg. */
 204,           /* visi_dir_idx =  51, dir = 286.44 deg. */
 208,           /* visi_dir_idx =  52, dir = 292.14 deg. */
 212,           /* visi_dir_idx =  53, dir = 298.18 deg. */
 216,           /* visi_dir_idx =  54, dir = 303.44 deg. */
 220,           /* visi_dir_idx =  55, dir = 309.23 deg. */
 224,           /* visi_dir_idx =  56, dir = 315.00 deg. */
 228,           /* visi_dir_idx =  57, dir = 320.08 deg. */
 232,           /* visi_dir_idx =  58, dir = 325.81 deg. */
 236,           /* visi_dir_idx =  59, dir = 331.82 deg. */
 240,           /* visi_dir_idx =  60, dir = 336.68 deg. */
 244,           /* visi_dir_idx =  61, dir = 342.70 deg. */
 248,           /* visi_dir_idx =  62, dir = 348.16 deg. */
 252            /* visi_dir_idx =  63, dir = 354.56 deg. */

};


/* Compare two directions
	Returns 0 iff both are equal,
	< 0 if d1 is less than d2
	> 0 if d1 is greater than d2
*/
int dir_comp(delta *d1, delta *d2){
	if (fabs(angle(d1) - angle(d2)) < 0.000001) return 0;
	return ( sign(angle(d1) - angle(d2)) );
};


/* The ship has 64 visible (=external) directions 
	Internally it has a finer resolution (0.255).
	We have stored all visible directions in visi_dir 
	visi_dir is sorted so we do a binary search
*/
int get_visi_idx(delta *d) {
	int low = 0, hi = 63, mid;
	int res;
	
	while (low <= hi){
		mid = (low + hi)/2;
		if ( (res = dir_comp(&(visi_dir[mid]), d)) > 0 )
			hi = mid - 1;
		else if (res < 0)
			low = mid + 1;
		else
			return mid; 	/* found */
	};
#ifdef DEBUG	
	fprintf(stderr,"ERROR: Unknown visible direction (%d, %d) encountered !\n", d->dx, d->dy);
	exit (20);
#endif	
	return -1;
};


int get_idx(delta *d) {
	int low = 0, hi = 255, mid;
	int res;
	
	while (low <= hi){
		mid = (low + hi)/2;
		if ( (res = dir_comp(&(dir_list[mid]), d)) > 0 )
			hi = mid - 1;
		else if (res < 0)
			low = mid + 1;
		else
			return mid; 	/* found */
	};
#ifdef DEBUG	
	fprintf(stderr,"ERROR: Unknown direction (%d, %d) encountered !\n", d->dx, d->dy);
	exit (20);
#endif		
	return -1;
};

/* get any index into visible_dir which has *vis as direction 
	in other words, given a visible direction, return an index to one of the possible true directions
*/
static int get_any_visi_idx(delta *vis){
	int low = 0, hi = 255, mid;
	int res;
	
	while (low <= hi){
		mid = (low + hi)/2;
		if ( (res = dir_comp(&(visible_dir_list[mid]), vis)) > 0 )
			hi = mid - 1;
		else if (res < 0)
			low = mid + 1;
		else
			return mid; 	/* found */
	};
#ifdef DEBUG		
	fprintf(stderr,"ERROR: Unknown visi direction (%d, %d) encountered !\n", vis->dx, vis->dy);
	exit (20);
#endif	
	return -1;
};
	
	
/* Get dir min-max
	Given a visible direction, find out which possible true directions correspong to it
	returns the smallest and largest possible true direction in dmin, dmax
*/
void get_min_max_dir (delta *vis, delta *dmin, delta *dmax){
	int i, idx;
	idx = get_any_visi_idx(vis);
	
	i=idx;
	do{
		i = (i - 1) & 0xff;
	} while ( ( visible_dir_list[i].dx == vis->dx) && (visible_dir_list[i].dy == vis->dy) );
	*dmin = dir_list[ (i + 1) & 0xff];
	
	i=idx;
	do{
		i = (i + 1) & 0xff;
	} while ( ( visible_dir_list[i].dx == vis->dx) && (visible_dir_list[i].dy == vis->dy) );	
	*dmax = dir_list[ (i - 1) & 0xff];
	
};

/* Returns the largest integral value that is not greater than x. */
/* Truncates the quotient towards negative infinity */
int floor_div(int a, int b){
	return ( (int) floor( (double) a / (double) b));
};
	
	
/* Returns the smallest integral value that is not less than x. */
/* Truncates the quotient towards positive infinity */
int ceil_div(int a, int b){
	return ( (int) ceil( (double) a / (double) b));
};
	
/* Given a direction vector and a skalar divisor, 
	returns the largest integral vector that is not greater than <dir>/dt
*/ 
void floor_dir(delta *dir, int dt){
	/* Handle each quadrant seperately, just to make sure */
	if ((dir->dx >= 0) && (dir->dy >= 0) ) {
		dir->dx = ceil_div(dir->dx, dt);	/* If we increase dx, the angle gets smaller */
		dir->dy = floor_div(dir->dy, dt);  	/* If we decrease dy, the angle get smaller */
	};
	
	if ((dir->dx < 0) && (dir->dy >= 0) ) {
		dir->dx = ceil_div(dir->dx, dt);	/* If we move dx closer to 0, the angle gets smaller */
		dir->dy = ceil_div(dir->dy, dt);  	/* If we increase dy, the angle get smaller */
	};
		
	if ((dir->dx  < 0) && (dir->dy < 0) ) {
		dir->dx = floor_div(dir->dx, dt);	/* If we move dx further away from 0, the angle gets smaller */
		dir->dy = ceil_div(dir->dy, dt);  	/* If we move dy closer to 0, the angle gets smaller */
	};
		
	if ((dir->dx >= 0) && (dir->dy < 0) ) {
		dir->dx = floor_div(dir->dx, dt);	/* If we move dx closer to 0, the angle gets smaller */
		dir->dy = floor_div(dir->dy, dt);  	/* If we move dy further away from 0, the angle gets smaller */
	};
//	printf("Result of floor_dir = (%d, %d)\n",dir->dx, dir->dy);
};

/* Given a direction vector and a skalar divisor, 
	returns the smallest integral vector that is not smaller than <dir>/dt
*/ 
void ceil_dir(delta *dir, int dt){
		/* Handle each quadrant seperately, just to make sure */
	if ((dir->dx >= 0) && (dir->dy >= 0) ) {
		dir->dx = floor_div(dir->dx, dt);	/* If we increase dx, the angle gets smaller */
		dir->dy = ceil_div(dir->dy, dt);  	/* If we decrease dy, the angle get smaller */
	};

	if ((dir->dx < 0) && (dir->dy >= 0) ) {
		dir->dx = floor_div(dir->dx, dt);	/* If we move dx closer to 0, the angle gets smaller */
		dir->dy = floor_div(dir->dy, dt);  	/* If we increase dy, the angle get smaller */
	};
	
	if ((dir->dx  < 0) && (dir->dy < 0) ) {
		dir->dx = ceil_div(dir->dx, dt);	/* If we move dx further away from 0, the angle gets smaller */
		dir->dy = floor_div(dir->dy, dt);  	/* If we move dy closer to 0, the angle gets smaller */
	};
	
	if ((dir->dx >= 0) && (dir->dy < 0) ) {
		dir->dx = ceil_div(dir->dx, dt);	/* If we move dx closer to 0, the angle gets smaller */
		dir->dy = ceil_div(dir->dy, dt);  	/* If we move dy further away from 0, the angle gets smaller */
	};
//	printf("Result of ceil_dir = (%d, %d)\n",dir->dx, dir->dy);
};


		
/* This function tries to guess the true direction given an observed direction d and a time dt.
	The result is stored in *d_true
*/
void estimate_dir(delta *d, int dt, delta *d_true ){
	delta dist;
	delta d_min, d_max;
	int min_dx, max_dx, min_dy, max_dy;
	
	dist = *d;
//	printf("d.dx= %d, d.dy=%d, dt=%d\n",dist.dx, dist.dy, dt);
/*
	The true distance can be off by at most 7 in any direction, because we do not know the true subpixel position of the object before and after.
	We compute min and max distances, so that the max distance vector is always to the left (counter clockwise) of the min vector
*/
	
	/* Handle each quadrant seperately, just to make sure */
	if ((dist.dx >= 0) && (dist.dy >= 0) ) {
		min_dx = dist.dx + 7;	/* If we increase dx, the angle gets smaller */
		max_dx = dist.dx - 7;
		min_dy = dist.dy - 7;  	/* If we decrease dy, the angle get smaller */
		max_dy = dist.dy + 7;
	};
	
	if ((dist.dx < 0) && (dist.dy >= 0) ) {
		min_dx = dist.dx + 7;	/* If we move dx closer to 0, the angle gets smaller */
		max_dx = dist.dx - 7;
		min_dy = dist.dy + 7;  	/* If we increase dy, the angle get smaller */
		max_dy = dist.dy - 7;
	};
		
	if ((dist.dx  < 0) && (dist.dy < 0) ) {
		min_dx = dist.dx - 7;	/* If we move dx further away from 0, the angle gets smaller */
		max_dx = dist.dx + 7;
		min_dy = dist.dy + 7;  	/* If we move dy closer to 0, the angle gets smaller */
		max_dy = dist.dy - 7;
	};
		
	if ((dist.dx >= 0) && (dist.dy < 0) ) {
		min_dx = dist.dx - 7;	/* If we move dx closer to 0, the angle gets smaller */
		max_dx = dist.dx + 7;
		min_dy = dist.dy - 7;  	/* If we move dy further away from 0, the angle gets smaller */
		max_dy = dist.dy + 7;
	};
//	printf("min=(%d, %d), max=(%d, %d)\n",min_dx, min_dy, max_dx, max_dy);

	/* This is the minimum possible direction vector.
		We find an integral direction vector, which is not smaller 
	*/
	d_min.dx = min_dx;
	d_min.dy = min_dy;
	ceil_dir(&d_min, dt);
	 
	/* This is the maximum possible direction vector.
	We find an integral direction vector, which is not larger 
	*/	
	d_max.dx = max_dx;
	d_max.dy = max_dy;
	floor_dir(&d_max, dt);	
		
//	printf("min=(%d, %d), max=(%d, %d)\n",obj->dir_min.dx, obj->dir_min.dy, obj->dir_max.dx, obj->dir_max.dy);
	
	/* TODO this is not thought through, might be working 
		min(12, 8) and max(11,9) give (11,8) as middle (which seems to be correct)
	*/
	d_true->dx = (d_max.dx + d_min.dx) / 2;
	d_true->dy = (d_max.dy + d_min.dy) / 2;
	

//	PrtObj(obj, "Calculated dir: ");
};

/* Returns TRUE iff two direction are identical */
BOOL eq_dir(delta *dir1, delta *dir2){
	return ( (dir1->dx == dir2->dx) && (dir1->dy == dir2->dy) );
};


/* Verify that the given direction index is compatible with the given visible direction index 
	TODO not very efficient !
*/
BOOL check_dir(int dir_idx, int v_idx){
	int i;
	i=min_dir_idx[v_idx];
	
	while (i != dir_idx){
		if (i == max_dir_idx[v_idx]) return FALSE;
		i = (i + 1) & 0xff;
	};
	return TRUE;
};

/* We have an observed direction (maybe from ships visible orientation or from shots)
	and a guessed true direction index
	This routine makes sure that the guessed direction index matches the observed direction
	It returns a new guess for the direction index
*/
/* Synchronize the visible direction of the ship with the true direction */
int sync_dir(delta *observed, int guessed){
	delta v_dir;
	int v_idx, i, idx;
		
	v_idx = get_visi_idx(observed);	
	v_dir = visi_dir[v_idx];

#ifdef DEBUG
	if (! eq_dir(&v_dir, observed) ){
		
		fprintf(stderr, "Inconsistent visi dir!\n");
		printf("observed = <%d,%d>, v_dir = <%d,%d>\n", observed->dx, observed->dy, v_dir.dx, v_dir.dy);
		exit(20);
	};
#endif	
	idx = guessed;
	
	/* if our estimated direction is within bounds of visible direction, we are finished */
	if (check_dir(idx, v_idx)){
		return idx;
	};
	
#ifdef DEBUG
//	PrtObj(ship, "Out of sync ");
	printf("Out of sync\n");
#endif		
	/* We are off. The visible direction is not what we expected */
	/* Find the smallest possible correction */
	for (i=1; i<=128; i++){
		idx = (guessed + i) & 0xff;
		if (check_dir(idx, v_idx)){
//			printf("Synced \n");
			return idx;
		};
		idx = (guessed - i) & 0xff;
		if (check_dir(idx, v_idx)){
//			printf("Synced \n");
			return idx;
		};
	};
#ifdef DEBUG		
	fprintf(stderr,"Could not synchronize direction!\n");
	exit(20);
#endif	
	return FALSE;
};

