/* $Id: objects.h,v 1.9 2008/06/15 16:03:41 raap Exp $ */

#ifndef OBJECTS_H
#define OBJECTS_H

#include "util.h"
#include "vector.h"
#include "frame.h"

/* Previous values: 9, 18, 34 */
#define SMALL_AST_SIZE	(9*8)
#define MID_AST_SIZE	(18*8)
#define BIG_AST_SIZE	(34*8)


/* Our speed values are of different quality
	UNKNOW means we know nothing about the speed of the object
	ESTIMATED means we have an approximation of the speed, which can possibly be improved upon
	KNOWN means we know for sure the speed (direction and magnitude).
*/
enum speed_type {UNKNOWN, ESTIMATED, KNOWN};

/* We keep track of different type of objects. They are represented in this enum: 
 	SHIP: our own ship, UFO_S: small UFO, UFO_L: large UFO, SHOT: unknown SHOT
	OWN_SHOT: our own shot, UFO_SHOT: Shot by UFO, AST: Asteroid
*/
enum types {SHIP, UFO_S, UFO_L, SHOT, AST};
	
	
/* When we are shooting at an object, we generate a lot of information */
typedef struct {
	int	target_id;	/* We shoot at this target */
	int	hit_time;	/* time when we hit the target */
	int	turn_time;	/* time that we need to turn to target */
	int	shoot_time;	/* time that the shot needs to reach the target */
	int	shoot_dir;	/* direction at which we shoot */
	koord	hit_point;	/* where do we hit the target ? */	
} HitEvent;



typedef struct{
	koord	cur_xy;		/* Where is the object now */
	int	cur_time;	/* At which time was it seen */
	delta	distance;	/* distance traveled so far (dx and dy) */
	int	dt;		/* number of frames in which distance was traveled */
	delta	dir;		/* The object moves in this direction */
	int	dir_idx;	/* index into dir_list[], the so called "winkelbyte" */
	enum types type;	/* which kind of object is it */
	int	size;		/* approx. radius of the object */
	int	form;		/* for asteroids, form of the object (1 .. 4), 0 for all other objects */
	enum speed_type speed_good;	/* How reliable is the speed value */
	int	visi_idx;	/* only for own shots: visible direction of ship at time of shot */
	int	own;		/* only for shots: 1 if it is our own */
	int	last_update;	/* time of last update of distance and dt */
	koord	last_xy;	/* koord, when obj was last updated */
	int	birth_time;	/* When will this object first be visible ? */
	int	death_time;	/* Zombie objects. We expect the object to disappear at this time */
	int 	aura_time;	/* For normal asteroids and for zombies: -1
					For ghosts: Up to which time point are they visible */	
	int	id;		/* Each object gets a unique id. */
	HitEvent he;		/* Only used for own shots */
} object;


/* If we have a lot of objects of the same type, we keep them in a list (really an array) 
	Just to make sure we allocate space for 100 objects.
*/
#define OBJECTS_LIM 100
typedef object ObjList[OBJECTS_LIM]; 


/* For some types of object that we want to track, we keep a list of them and the number in the list */
typedef struct {
	ObjList	list;		/* a list with up to 100 objects */
	int	n;		/* number of objects in list */
} ObjTab;
	

void PrtObj(object *obj, char *s);
void init_shot(object *shot);
void init_ast(object *ast);
void init_ufo(object *ufo);
void init_ship(object *ship);
BOOL is_ufo(object *tar);
void tab_init (ObjTab *b);
void insert (ObjTab *b, object *obj);
int pass_out(ObjTab *b, int curtime);
void UpdatePos (object *obj, int curtime);
void UpdatePosTab (ObjTab *b, int curtime);		
void move_own_shot(object *obj, int speed, int dt, delta *offset);		
void game_state_update(FrameStatus *frame, int curtime);		
int match_deltas(delta *d1, delta *d2);
void game_state_init();
void update_ship_dir(object *ship, int t);	
void estimate_pos_list (ObjList list, int n, int newtime);
void update_pos (object *oldobj, object *newobj);		
void delete (ObjTab *tab, int idx) ;
void update_speed_list(ObjList list, int n);		
void PrtObjList(ObjTab *tab);		
int sure_shot(object *ship, object *tar, HitEvent *he);
void turn_ship (object *ship, int turndir);
void default_ast(object *ast);		
void update_speed(object *obj);	
BOOL calc_speed_ufo (object *ufo);
void predict_pos_list (ObjList list, int n, int newtime);
int find_obj(ObjList list, int n, object *ast);
int sure_shot_later (object *ship, object *tar);
void predict_pos_ship(object *ship, int t);
void update_speed_ship(object *ship, delta *visual_dir);		
void update_speed_list_ufos(ObjList list, int n);
int nearest_ast(ObjList list, int n, object *ship, delta *result);			
int turn_to_target (object *ship, object *tar, int *turn_dir, HitEvent *he);	
	
#endif
