package asteroid.model;

public class Collider extends ACompute {
	private static final int SHIP_SAVE = 0;
	private AsteroidVector mAsts;
	private int mForward;
	private boolean mSave;
	private Ship mShip;
	private int mShipFak;
	private boolean mShipFly;
	private ShotVector mShots;
	private Ufo mUfo;

	Collider( Ship ship, Ufo ufo, AsteroidVector asts, ShotVector shots) {
		mShip = ship;
		mUfo = ufo;
		mAsts = asts;
		mShots = shots;
	}

	void detect( int forward) {
		mForward = forward + 1;
		mSave = true;
		if (mShip.isVisible()) {
			mShipFly = (mShip.mMoveX8 != 0) || (mShip.mMoveY8 != 0);
			mShipFak = (mShip.mY8 * mShip.mMoveX8) - (mShip.mX8 * mShip.mMoveY8);
			mUfo.doIt( this);
			if (mSave) {
				mAsts.forAll( this);
			}
			if (mSave) {
				mShots.forAll( this);
			}
		}
	}

	public void forAsteroid( Asteroid ast) {
		forFlyable( ast);
	}

	void forFlyable( AFlyable fly) {
		if (mSave && (fly.isVisible())) {
			if (mShipFly) {
				mSave = !hitFirstFly( mShip, mShipFak, fly, mForward);
			}
			else {
				mSave = !hitFirstStand( mShip, fly, mForward);
			}
		}
	}

	public void forShot( Shot shot) {
		if (!shot.isShip()) {
			forFlyable( shot);
		}
	}

	public void forUfo( Ufo ufo) {
		forFlyable( ufo);
	}

	public static boolean hitAtPos( AFlyable fly1, AFlyable fly2, int x, int y, int forward) {
		int sizeX = fly1.sizeX() + fly2.sizeX() + SHIP_SAVE;
		if (isOutside( fly1.mX8, forward * fly1.mMoveX8, x, sizeX)) {
			return false;
		}
		if (isOutside( fly2.mX8, forward * fly2.mMoveX8, x, sizeX)) {
			return false;
		}
		int sizeY = fly2.sizeY() + fly1.sizeY() + SHIP_SAVE;
		if (isOutside( fly1.mY8, forward * fly1.mMoveY8, y, sizeY)) {
			return false;
		}
		if (isOutside( fly2.mY8, forward * fly2.mMoveY8, y, sizeY)) {
			return false;
		}
		return true;
	}

	public static boolean hitBy( AFlyable fly1, AFlyable fly2, int forward) {
		if ((fly1.mMoveX8 != 0) || (fly1.mMoveY8 != 0)) {
			return hitFirstFly( fly1, fly2, forward);
		}
		return hitFirstStand( fly1, fly2, forward);
	}

	public static boolean hitFirstFly( AFlyable fly1, AFlyable fly2, int forward) {
		int det = (fly1.mMoveX8 * fly2.mMoveY8) - (fly1.mMoveY8 * fly2.mMoveX8);
		if (det == 0) {
			return false; // beide fliegen parallel
		}
		int fak1 = (fly1.mY8 * fly1.mMoveX8) - (fly1.mX8 * fly1.mMoveY8);
		int fak2 = (fly2.mY8 * fly2.mMoveX8) - (fly2.mX8 * fly2.mMoveY8);
		int x = (fak1 * fly2.mMoveX8 / det) - (fak2 * fly1.mMoveX8 / det);
		int y = (fak1 * fly2.mMoveY8 / det) - (fak2 * fly1.mMoveY8 / det);
		return hitAtPos( fly1, fly2, x, y, forward);
	}

	private static boolean hitFirstFly( AFlyable fly1, int fak1, AFlyable fly2, int forward) {
		int det = (fly1.mMoveX8 * fly2.mMoveY8) - (fly1.mMoveY8 * fly2.mMoveX8);
		if (det == 0) {
			return false; // beide fliegen parallel
		}
		int fak2 = (fly2.mY8 * fly2.mMoveX8) - (fly2.mX8 * fly2.mMoveY8);
		int x = (fak1 * fly2.mMoveX8 / det) - (fak2 * fly1.mMoveX8 / det);
		int y = (fak1 * fly2.mMoveY8 / det) - (fak2 * fly1.mMoveY8 / det);
		return hitAtPos( fly1, fly2, x, y, forward);
	}

	public static boolean hitFirstStand( AFlyable fly1, AFlyable fly2, int forward) {
		int det = (fly2.mMoveY8 * fly2.mMoveY8) + (fly2.mMoveX8 * fly2.mMoveX8);
		if (det == 0) {
			return false; // beide stehen
		}
		int dx = fly1.mX8 - fly2.mX8;
		int dy = fly1.mY8 - fly2.mY8;
		int fak = (dy * fly2.mMoveX8) - (dx * fly2.mMoveY8);
		int x = fly1.mX8 + (fak * fly2.mMoveY8 / det);
		int y = fly1.mY8 - (fak * fly2.mMoveX8 / det);
		return hitAtPos( fly1, fly2, x, y, forward);
	}

	public static boolean isInside( int p0, int move, int p, int size) {
		int dp = p - p0;
		if (move < 0) {
			return (dp < size) && (dp > (move - size));
		}
		else {
			return (dp > -size) && (dp < (move + size));
		}
	}

	public int getKey() {
		if (mSave) {
			return 0;
		}
		return KEY_HYPERSPACE;
	}

	public static boolean isOutside( int p0, int move, int p, int size) {
		int dp = p - p0;
		if (move < 0) {
			return (dp > size) || (dp < (move - size));
		}
		else {
			return (dp < -size) || (dp > (move + size));
		}
	}
}
