package asteroid.udp;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.util.Iterator;

import asteroid.Operation;
import asteroid.SimpleLogging;
import asteroid.model.ICompute;

public class NetHandler extends AHandler {
	public static final int PORT = 1979;
	public static final int NAME_SIZE = 38;
	private DatagramChannel mChannel;
	private boolean mFirst;
	private int mFrame;
	private int mHeiseBusy;
	private ReceiveFiFo mReceives;
	private Selector mSelector;
	private ByteBuffer mSendKey = ByteBuffer.allocateDirect( KEY_SIZE);
	private ByteBuffer mSendName = ByteBuffer.allocateDirect( NAME_SIZE);
	private SendFiFo mSends = new SendFiFo();

	public NetHandler( InetAddress addr, int testLatenz) throws IOException {
		super( "NetHandler");
		mReceives = new ReceiveFiFo( testLatenz);
		mSendKey.put( AHandler.CT_MAME.getBytes());
		mSendName.put( AHandler.CT_NAME.getBytes());
		mSendName.put( AHandler.NAME.getBytes());
		mChannel = DatagramChannel.open();
		mChannel.connect( new InetSocketAddress( addr, PORT));
		mChannel.configureBlocking( false);
		mSelector = Selector.open();
		mChannel.register( mSelector, SelectionKey.OP_READ);
	}

	public void addKey( int key) {
		if (mHeiseBusy == 0) {
			mSends.nextWrite( key);
			try {
				send();
			}
			catch (IOException ex) {
				SimpleLogging.addException( ex);
				setOpen( false);
			}
		}
	}

	private void check( ByteBuffer bb) {
		if (mHeiseBusy > 0) {
			return;
		}
		if (mFirst) {
			mFrame = (mFrame + 1) & 0xFF;
			int frameNr = bb.get( FRAME_POS) & 0xFF;
			if (mFrame != frameNr) {
				SimpleLogging.addLog( "lost frame");
				mFrame = frameNr;
			}
		}
		else {
			mFrame = bb.get( FRAME_POS) & 0xFF;
			mFirst = true;
		}
	}

	public void close() {
		if (mSelector != null) {
			try {
				mSelector.close();
			}
			catch (IOException ex) {
				System.err.println( "exception: " + ex.getMessage());
			}
		}
		if (mChannel != null) {
			try {
				mChannel.close();
			}
			catch (IOException ex) {
				System.err.println( "exception: " + ex.getMessage());
			}
		}
	}

	public ByteBuffer getNext( boolean blocked) {
		return mReceives.nextRead( blocked);
	}

	private void receive( ByteBuffer bb) throws IOException {
		bb.clear();
		int readed = mChannel.read( bb);
		if (readed == FRAME_SIZE) {
			mReceives.received();
			mSaver.writeData( bb);
			mSaver.writeKey( 0);
		}
		else if (readed > 2) {
			String heiseCmd = Operation.getText( bb, 0, readed - 2);
			if (heiseCmd.startsWith( "busy")) {
				if (mHeiseBusy == 0) {
					mHeiseBusy = 50;
					SimpleLogging.addLog( "heise.de " + heiseCmd);
				}
			}
			else if (heiseCmd.startsWith( "game over")) {
				mHeiseBusy = 0;
				SimpleLogging.addLog( "heise.de " + heiseCmd);
			}
			else {
				mHeiseBusy = 0;
				SimpleLogging.addLog( "illegal heise command " + readed);
			}
			mReceives.eventRead();
		}
		else {
			SimpleLogging.addLog( "illegal packet size " + readed);
		}
	}

	public void run() {
		mTime = System.currentTimeMillis();
		try {
			sendKey( 0);
			while (!interrupted()) {
				if (mHeiseBusy > 0) {
					--mHeiseBusy;
					if (mHeiseBusy == 0) {
						sendKey( 0);
					}
					sleep(TIME_SLEEP);
				}
				if (!isOpen()) {
					mReceives.eventRead();
				}
				else if (mSelector.select( 100) > 0) {
					for (Iterator i = mSelector.selectedKeys().iterator(); i.hasNext();) {
						SelectionKey sel = (SelectionKey) i.next();
						if (sel.isReadable()) {
							ByteBuffer bb = mReceives.nextWrite();
							receive( bb);
							check( bb);
						}
						i.remove();
					}
				}
			}
		}
		catch (Exception ex) {
			SimpleLogging.addException( ex);
		}
		finally {
			close();
		}
	}

	private void send() throws IOException {
		int key = mSends.nextRead();
		if (key == -1) {
			return;
		}
		if ((key & ICompute.KEY_NAME) == ICompute.KEY_NAME) {
			mSendName.clear();
			mChannel.write( mSendName);
		}
		else {
			sendKey( key);
		}
	}

	private void sendKey( int key) throws IOException {
		mSendKey.clear();
		mSendKey.put( KEY_POS, (byte) key);
		mSendKey.put( KEY_PING, (byte) (key >> 16));
		mChannel.write( mSendKey);
	}
}
