package asteroid.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.util.StringTokenizer;

import javax.swing.JComponent;

import asteroid.Operation;
import asteroid.model.AEnemy;
import asteroid.model.AFlyable;
import asteroid.model.Angle;
import asteroid.model.Asteroid;
import asteroid.model.FlyDetect;
import asteroid.model.GameModel;
import asteroid.model.HighScores;
import asteroid.model.ICompute;
import asteroid.model.Ship;
import asteroid.model.Shot;
import asteroid.model.Ufo;

class GameView extends JComponent implements ICompute {
	private static final long serialVersionUID = -6626855104081248163L;
	private static final int WIDTH = 640;
	private static final int HEIGHT = 480;
	private static final double SHIP_SIZE = 5.0;
	private static final Color TEXT_COLOR = Color.WHITE;
	protected GameModel mModel;
	private boolean mDirection = false;
	private int mFade;
	private Font mFontHigh = new Font( "Monospace", Font.PLAIN, 18);
	private Font mFontObj = new Font( "Monospace", Font.PLAIN, 9);
	private Graphics mGraph;
	private boolean mIdentifier = true;
	private boolean mShowScore;
	private boolean mVelocity = true;

	public GameView( GameModel model) {
		mModel = model;
		Dimension size = new Dimension( WIDTH, HEIGHT);
		setMaximumSize( size);
		setMinimumSize( size);
		setPreferredSize( size);
		setOpaque( true);
		setDoubleBuffered( true);
	}

	private void drawArc( AFlyable fly, int size) {
		mGraph.drawArc( getScreenX8( fly.getX8()) - size, getScreenY8( fly.getY8()) - size, size << 1, size << 1, 0, 360);
	}

	private void drawDirection( int x, int y, int dy, int dx, double dir) {
		if (mDirection) {
			mGraph.setColor( Color.GRAY);
			int x0 = x + dx + dx;
			int y0 = y + dy + dy;
			int x1 = x + (int) Math.round( WIDTH * Math.cos( dir));
			int y1 = y + (int) Math.round( -WIDTH * Math.sin( dir));
			mGraph.drawLine( x0, y0, x1, y1);
		}
	}

	void drawIdentifier( AFlyable fly, int size) {
		if (mIdentifier) {
			mGraph.setColor( TEXT_COLOR);
			int x0 = getScreenX8( fly.getX8());
			int y0 = getScreenY8( fly.getY8());
			mGraph.drawString( fly.getIdent(), x0 + size, y0 - size);
		}
	}

	void drawStringLeft( String s, int x, int y, int width, FontMetrics fm) {
		mGraph.drawString( s, x + width - fm.stringWidth( s), y);
	}

	private void drawVelocity( AFlyable fly) {
		if (mVelocity && ((fly.getMoveX8() != 0) || (fly.getMoveY8() != 0))) {
			mGraph.setColor( Color.GRAY);
			int x0 = getScreenX8( fly.getX8());
			int y0 = getScreenY8( fly.getY8());
			int x1 = getScreenX8( fly.getX8() + (fly.getMoveX8() << 3));
			int y1 = getScreenY8( fly.getY8() + (fly.getMoveY8() << 3));
			mGraph.drawLine( x0, y0, x1, y1);
		}
	}

	public void forAsteroid( Asteroid ast) {
		if (ast.isVisible()) {
			mGraph.setColor( Color.MAGENTA);
			switch (ast.getZ()) {
				case 0: // grosser Asteroid
					drawArc( ast, 20);
					drawIdentifier( ast, 16);
					break;
				case 15: // mittlerer Asteroid
					drawArc( ast, 10);
					drawIdentifier( ast, 9);
					break;
				case 14: // kleiner Asteroid
				default:
					drawArc( ast, 5);
					drawIdentifier( ast, 6);
			}
			drawVelocity( ast);
		}
		else if (ast.isExplosion()) {
			forExplosion( ast);
		}
	}

	public void forAsteroid( FlyDetect detect) {
	}

	void forExplosion( AEnemy enemy) {
		mGraph.setColor( Color.ORANGE);
		switch (enemy.getType()) {
			case Operation.EXPLOSION_0:
				drawArc( enemy, 17);
				break;
			case Operation.EXPLOSION_1:
				drawArc( enemy, 13);
				break;
			case Operation.EXPLOSION_2:
				drawArc( enemy, 9);
				break;
			case Operation.EXPLOSION_3:
				drawArc( enemy, 5);
				break;
		}
	}

	public void forShip( Ship ship) {
		if ((mModel.getFrameType() == GameModel.FRAME_TYPE_GAME) || (mModel.getFrameType() == GameModel.FRAME_TYPE_GAME_OVER)) {
			double dir = Angle.toRadiant( mModel.getFireDirection());
			int dx = (int) Math.round( SHIP_SIZE * Math.cos( dir));
			int dy = (int) Math.round( -SHIP_SIZE * Math.sin( dir));
			int x = getScreenX8( ship.getX8());
			int y = getScreenY8( ship.getY8());
			switch (ship.getState()) {
				case AFlyable.STATE_HIDDEN:
				default:
					mGraph.setColor( Color.YELLOW);
					mGraph.drawLine( x - dx - dy, y - dy + dx, x - dx + dy, y - dy - dx);
					mGraph.drawLine( x - dx + dy, y - dy - dx, x + dx - (dy >> 1), y + dy + (dx >> 1));
					mGraph.drawLine( x - dx - dy, y - dy + dx, x + dx + (dy >> 1), y + dy - (dx >> 1));
					break;
				case AFlyable.STATE_VISIBLE:
					mFade = 255;
					mGraph.setColor( new Color( 255, 255, 255, mFade));
					mGraph.drawLine( x - dx - dy, y - dy + dx, x - dx + dy, y - dy - dx);
					mGraph.drawLine( x - dx + dy, y - dy - dx, x + dx + dx, y + dy + dy);
					mGraph.drawLine( x - dx - dy, y - dy + dx, x + dx + dx, y + dy + dy);
					drawVelocity( ship);
//					drawIdentifier( ship, 8);
					drawDirection( x, y, dy, dx, dir);
					break;
				case AFlyable.STATE_JUMP:
					if (mFade > 16) {
						mFade -= 16;
					}
					else {
						mFade = 0;
					}
					mGraph.setColor( new Color( 255, 255, 255, mFade));
					mGraph.drawLine( x - dx - dy, y - dy + dx, x - dx + dy, y - dy - dx);
					mGraph.drawLine( x - dx + dy, y - dy - dx, x + dx + dx, y + dy + dy);
					mGraph.drawLine( x - dx - dy, y - dy + dx, x + dx + dx, y + dy + dy);
					drawVelocity( ship);
//					drawIdentifier( ship, 8);
					break;
			}
		}
	}

	public void forShot( FlyDetect detect) {
	}

	public void forShot( Shot shot) {
		if (shot.isVisible()) {
			mGraph.setColor( shot.isShip() ? Color.GREEN : Color.PINK);
			drawArc( shot, 2);
			drawIdentifier( shot, 3);
			drawVelocity( shot);
		}
	}

	public void forUfo( Ufo ufo) {
		if (ufo.isVisible()) {
			int dx = 0;
			int dy = 0;
			switch (ufo.getZ()) {
				case 15: // grosses UFO
					dx = 12;
					dy = 5;
					break;
				case 14: // kleines UFO
				default:
					dx = 6;
					dy = 3;
					break;
			}
			mGraph.setColor( Color.RED);
			int x = getScreenX8( ufo.getX8());
			int y = getScreenY8( ufo.getY8());
			mGraph.drawLine( x - dx + dy, y - dy, x + dx - dy, y - dy);
			mGraph.drawLine( x - dx, y, x + dx, y);
			mGraph.drawLine( x - dx + dy, y + dy, x + dx - dy, y + dy);
			mGraph.drawLine( x - dx, y, x - dx + dy, y + dy);
			mGraph.drawLine( x + dx - dy, y + dy, x + dx, y);
			dy <<= 1;
			mGraph.drawLine( x - dx, y, x - dx + dy, y - dy);
			mGraph.drawLine( x - dx + dy, y - dy, x + dx - dy, y - dy);
			mGraph.drawLine( x + dx - dy, y - dy, x + dx, y);
			drawVelocity( ufo);
		}
		else if (ufo.isExplosion()) {
			forExplosion( ufo);
		}
	}

	public void setIdentifier( boolean value) {
		mIdentifier = value;
	}

	public boolean isIdentifier() {
		return mIdentifier;
	}

	public void paint( Graphics g) {
		g.setColor( Color.black);
		g.fillRect( 0, 0, getWidth(), getHeight());
		mGraph = g;
		if (mShowScore || (mModel.getFrameType() == GameModel.FRAME_TYPE_HIGH)) {
			mGraph.setFont( mFontHigh);
			paintHighScores();
		}
		else {
			mGraph.setFont( mFontObj);
			mModel.forShip( this);
			mModel.forUfo( this);
			mModel.forAllAsteroids( this);
			mModel.forAllShots( this);
		}
	}

	private void paintHighScores() {
		mGraph.setColor( TEXT_COLOR);
		FontMetrics fm = mGraph.getFontMetrics( mFontHigh);
		int h = fm.getHeight();
		int x = 230;
		int y = 100;
		mGraph.drawString( "HOECHSTERGEBNIS", x, y);
		y += h;
		for (int i = 0; i < HighScores.MAX_SCORES; ++i) {
			y += h;
			String s = mModel.getScore( i);
			if (s != null) {
				StringTokenizer st = new StringTokenizer( s);
				drawStringLeft( st.nextToken(), x, y, 20, fm);
				drawStringLeft( st.nextToken(), x + 10, y, 80, fm);
				mGraph.drawString( st.nextToken(), x + 110, y);
			}
		}
	}

//	private static int getScreenX( int x) {
//		return (WIDTH * x) >> 10; // 1024;
//	}
//
//	private static int getScreenY( int y) {
//		return (WIDTH * (895 - y)) >> 10;
//	}
//
	private static int getScreenX8( int x8) {
		return (WIDTH * x8) >> 13;
	}

	private static int getScreenY8( int y8) {
		return (WIDTH * (7160 - y8)) >> 13;
	}

	public void setShowScore( boolean value) {
		mShowScore = value;
	}

	public void setVelocity( boolean value) {
		mVelocity = value;
	}

	public boolean isVelocity() {
		return mVelocity;
	}
}
