package asteroid.ui;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.net.URL;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;

import asteroid.model.GameModel;
import asteroid.statistic.GameValues;

public class MainFrame extends JFrame {
	private static final long serialVersionUID = 3632598983042456571L;
	public static final int INDEX_GAME = 1;
	public static final int INDEX_LIST = 2;
	public static final int INDEX_HIGH = 3;
	public static final int INDEX_TEST = 4;
	public static final int INDEX_SAMPLE = 5;
	private static final String ICON_DIR = "/asteroid/pictures/";
	private JButton mBtnAuto = new JButton( new MainAction( "auto.gif", "auto", "Spiel starten."));
	private JToggleButton mBtnGame = new JToggleButton( new MainAction( "game.gif", "game", "Spiel Simulation anzeigen."));
	private JToggleButton mBtnHigh = new JToggleButton( new MainAction( "high.gif", "high", "Hoechstergebnisse anzeigen."));
	private JToggleButton mBtnIdent = new JToggleButton( new MainAction( "ident.gif", "ident", "Objektbezeichnung anzeigen."));
	private JToggleButton mBtnList = new JToggleButton( new MainAction( "list.gif", "list", "ROM Listing anzeigen."));
	private JToggleButton mBtnNone = new JToggleButton( new MainAction( "nothing.gif", "none", "keine Darstellung"));
	private JToggleButton mBtnRepeat = new JToggleButton( new MainAction( "repeat.gif", "repeat", "Mehrere Spiel automatisch starten."));
	private JToggleButton mBtnSample = new JToggleButton( new MainAction( "sample.gif", "sample", "Sample darstellen."));
	private JToggleButton mBtnTest = new JToggleButton( new MainAction( "test.gif", "test", "Kollision testen."));
	private JToggleButton mBtnTop = new JToggleButton( new MainAction( "top.gif", "top", "Spieler in Bestenliste eintragen."));
	private JToggleButton mBtnVelo = new JToggleButton( new MainAction( "velocity.gif", "velo", "Geschwindigkeitvektoren anzeigen."));
	private GameView mGameView;
	private int mLastType;
	private JComponent mListView;
	private GameModel mModel;
	private int mOffset;
	private JComponent mSampleView = new SampleView();
	private StatusView mStatusView;
	private JComponent mTestView = new TestView();

	public MainFrame( GameModel model, GameValues game) {
		super( "Asteroids");
		mModel = model;
		mLastType = model.getFrameType();
		setDefaultCloseOperation( JFrame.EXIT_ON_CLOSE);
		mListView = new ListView( model);
		mGameView = new GameView( model);
		mStatusView = new StatusView( model, game, 640, 130);
		getContentPane().add( createToolBar(), BorderLayout.NORTH);
		getContentPane().add( mStatusView, BorderLayout.CENTER);
		getContentPane().add( createViews(), BorderLayout.SOUTH);
		ButtonGroup grp = new ButtonGroup();
		grp.add( mBtnNone);
		grp.add( mBtnGame);
		grp.add( mBtnList);
		grp.add( mBtnTest);
		grp.add( mBtnSample);
		grp.add( mBtnHigh);
		updateButtons();
		mBtnVelo.setSelected( mGameView.isVelocity());
		mBtnIdent.setSelected( mGameView.isIdentifier());
		setView( INDEX_GAME);
	}

	private JToolBar createToolBar() {
		JToolBar tool = new JToolBar();
		tool.setFloatable( false);
		try {
			tool.setRollover( true);
		}
		catch (NoSuchMethodError event) {
		}
		tool.add( mBtnAuto);
		tool.add( mBtnRepeat);
		tool.add( mBtnTop);
		tool.addSeparator();
		tool.addSeparator();
		tool.add( mBtnNone);
		tool.add( mBtnGame);
		tool.add( mBtnList);
//		tool.add( mBtnTest);
		tool.add( mBtnSample);
		tool.add( mBtnHigh);
		tool.addSeparator();
		tool.addSeparator();
		tool.add( mBtnVelo);
		tool.add( mBtnIdent);
		return tool;
	}

	private Box createViews() {
		Box box = Box.createVerticalBox();
		box.add( mGameView);
		box.add( mListView);
		box.add( mTestView);
		box.add( mSampleView);
		return box;
	}

	public static Icon getIcon( String name) {
		try {
			URL url = MainFrame.class.getResource( ICON_DIR + name);
			if (url != null) {
				return new ImageIcon( url);
			}
		}
		catch (RuntimeException ex) {
		}
		return null;
	}

	public void refresh() {
		if (mGameView.isVisible()) {
			mGameView.repaint();
		}
		if (mTestView.isVisible()) {
			mTestView.repaint();
		}
		if (mSampleView.isVisible()) {
			mSampleView.repaint();
		}
		if (mListView.isVisible()) {
			mListView.repaint();
		}
		switch (++mOffset & 15) {
			case 4:
				mStatusView.updateView();
				break;
			case 8:
				updateTitle();
				updateButtons();
				break;
		}
	}

	private void updateButtons() {
		mBtnAuto.setEnabled( (mModel.getFrameType() == GameModel.FRAME_TYPE_READY) || (mModel.getFrameType() == GameModel.FRAME_TYPE_HIGH));
		mBtnRepeat.setSelected( mModel.isAutoStart());
		mBtnTop.setSelected( mModel.isTopList());
	}

	private void updateTitle() {
		if (mModel.getFrameType() != mLastType) {
			mLastType = mModel.getFrameType();
			switch (mLastType) {
				case GameModel.FRAME_TYPE_READY:
				case GameModel.FRAME_TYPE_HIGH:
					setTitle( "Asteroids");
					break;
				case GameModel.FRAME_TYPE_START:
					setTitle( "Asteroids: Spiel beginnt gleich!");
					break;
				case GameModel.FRAME_TYPE_GAME:
					setTitle( "Asteroids: Spiel aktiv...");
					break;
				case GameModel.FRAME_TYPE_GAME_OVER:
					setTitle( "Asteroids: Spiel zu Ende");
					break;
				case GameModel.FRAME_TYPE_END:
					setTitle( "Asteroids: Das Spiel ist zu Ende.");
					break;
				case GameModel.FRAME_TYPE_SCORE:
					setTitle( "Asteroids: Eintrag in die Bestenliste.");
					break;
			}
		}
	}

	public void setView( int index) {
		switch (index) {
			case INDEX_GAME:
				mBtnGame.setSelected( true);
				mGameView.setVisible( true);
				mListView.setVisible( false);
				mTestView.setVisible( false);
				mSampleView.setVisible( false);
				mGameView.setShowScore( false);
				break;
			case INDEX_LIST:
				mBtnList.setSelected( true);
				mGameView.setVisible( false);
				mListView.setVisible( true);
				mTestView.setVisible( false);
				mSampleView.setVisible( false);
				break;
			case INDEX_HIGH:
				mBtnHigh.setSelected( true);
				mGameView.setVisible( true);
				mListView.setVisible( false);
				mTestView.setVisible( false);
				mSampleView.setVisible( false);
				mGameView.setShowScore( true);
				break;
			case INDEX_TEST:
				mBtnTest.setSelected( true);
				mGameView.setVisible( false);
				mListView.setVisible( false);
				mTestView.setVisible( true);
				mSampleView.setVisible( false);
				break;
			case INDEX_SAMPLE:
				mBtnSample.setSelected( true);
				mGameView.setVisible( false);
				mListView.setVisible( false);
				mTestView.setVisible( false);
				mSampleView.setVisible( true);
				break;
			default:
				mBtnNone.setSelected( true);
				mGameView.setVisible( false);
				mListView.setVisible( false);
				mTestView.setVisible( false);
				mSampleView.setVisible( false);
		}
		pack();
	}

	private class MainAction extends AbstractAction {
		private static final long serialVersionUID = -7638695993616867697L;

		MainAction( String name, String cmd, String tip) {
			Icon icon = getIcon( name);
			if (icon != null) {
				putValue( Action.SMALL_ICON, icon);
			}
			else {
				putValue( Action.NAME, name);
			}
			putValue( Action.SHORT_DESCRIPTION, tip);
			putValue( Action.ACTION_COMMAND_KEY, cmd);
		}

		public void actionPerformed( ActionEvent e) {
			if ("auto".equals( e.getActionCommand())) {
				mBtnAuto.setEnabled( false);
				mModel.setAutoStart();
			}
			else if ("repeat".equals( e.getActionCommand())) {
				mModel.setRepeat( mBtnRepeat.isSelected());
			}
			else if ("top".equals( e.getActionCommand())) {
				mModel.setTopList( mBtnTop.isSelected());
			}
			else if ("velo".equals( e.getActionCommand())) {
				mGameView.setVelocity( mBtnVelo.isSelected());
			}
			else if ("ident".equals( e.getActionCommand())) {
				mGameView.setIdentifier( mBtnIdent.isSelected());
			}
			else if ("none".equals( e.getActionCommand())) {
				setView( 0);
			}
			else if ("game".equals( e.getActionCommand())) {
				setView( INDEX_GAME);
			}
			else if ("list".equals( e.getActionCommand())) {
				setView( INDEX_LIST);
			}
			else if ("high".equals( e.getActionCommand())) {
				setView( INDEX_HIGH);
			}
			else if ("test".equals( e.getActionCommand())) {
				setView( INDEX_TEST);
			}
			else if ("sample".equals( e.getActionCommand())) {
				setView( INDEX_SAMPLE);
			}
		}
	}
}
