package asteroid.model;

import asteroid.Operation;
import asteroid.PrintWriterFormat;

import junit.framework.TestCase;

public class OptimalTest1 extends TestCase {
	static final double PI_05 = Math.PI / 2;
	static final int[] HX = {
			Operation.WIDTH, Operation.WIDTH, Operation.WIDTH, 0, 0, 0, -Operation.WIDTH, -Operation.WIDTH, -Operation.WIDTH
		};
	static final int[] HY = {
			Operation.HEIGHT, 0, -Operation.HEIGHT, Operation.HEIGHT, 0, -Operation.HEIGHT, Operation.HEIGHT, 0, -Operation.HEIGHT
		};
	private GameModel mModel = new GameModel( null);
	private PrintWriterFormat mOut = new PrintWriterFormat( System.out);

	public OptimalTest1( String name) {
		super( name);
	}

	public static int getCollisionTimeEnemy( FlyInfo enemy, FlyInfo ship, int shipDir, int steps, int time, int hx, int hy) {
		int shotDir = (shipDir + (3 * steps)) & 0xFF;
		int ds = Math.abs( steps);
		int dx = Operation.signX8( enemy.getX8( time) - ship.mX8) + (ds * enemy.mMoveX8) + hx - Angle.posShotX( shotDir);
		int dy = Operation.signY8( enemy.getY8( time) - ship.mY8) + (ds * enemy.mMoveY8) + hy - Angle.posShotY( shotDir);
		int ox = Angle.vShotX( shotDir);
		int oy = Angle.vShotY( shotDir);
		int te = ((dx * oy) - (dy * ox)) / ((enemy.mMoveY8 * ox) - (enemy.mMoveX8 * oy));
		return (te >= 0) ? te : Short.MAX_VALUE;
	}

	public static int getCollisionTimeShot( FlyInfo enemy, FlyInfo ship, int shipDir, int steps, int time, int hx, int hy) {
		int shotDir = (shipDir + (3 * steps)) & 0xFF;
		int ds = Math.abs( steps);
		int dx = Operation.signX8( enemy.getX8( time) - ship.mX8) + (ds * enemy.mMoveX8) + hx - Angle.posShotX( shotDir);
		int dy = Operation.signY8( enemy.getY8( time) - ship.mY8) + (ds * enemy.mMoveY8) + hy - Angle.posShotY( shotDir);
		int ox = Angle.vShotX( shotDir);
		int oy = Angle.vShotY( shotDir);
		int ts = ((dx * enemy.mMoveY8) - (dy * enemy.mMoveX8)) / ((enemy.mMoveY8 * ox) - (enemy.mMoveX8 * oy));
		return (ts >= 0) ? ts : Short.MAX_VALUE;
	}

	static void findDirection( AEnemy enemy, Ship ship, int shipDir, int time) {
		enemy.mTimeEnemyOld = Integer.MAX_VALUE;
		testShadowDirection( enemy, ship, shipDir, time, 0, 0);
		double dx = Operation.signX8( enemy.getX8( time) - ship.mX8);
		double dy = Operation.signY8( enemy.getY8( time) - ship.mY8);
		if ((dy > Operation.HEIGHT_25) && (enemy.mMoveY8 > 0)) { // 1, 2, 3, 4
			testShadowDirection( enemy, ship, shipDir, time, 0, -Operation.HEIGHT);
			if ((dx > Operation.WIDTH_25) && (enemy.mMoveX8 > 0)) { // 4
				testShadowDirection( enemy, ship, shipDir, time, -Operation.WIDTH, -Operation.HEIGHT);
			}
			if ((dx < -Operation.WIDTH_25) && (enemy.mMoveX8 < 0)) { // 1
				testShadowDirection( enemy, ship, shipDir, time, Operation.WIDTH, -Operation.HEIGHT);
			}
		}
		else if ((dy < -Operation.HEIGHT_25) && (enemy.mMoveY8 < 0)) { // 13, 14, 15, 16
			testShadowDirection( enemy, ship, shipDir, time, 0, Operation.HEIGHT);
			if ((dx > Operation.WIDTH_25) && (enemy.mMoveX8 > 0)) { // 16
				testShadowDirection( enemy, ship, shipDir, time, -Operation.WIDTH, Operation.HEIGHT);
			}
			if ((dx < -Operation.WIDTH_25) && (enemy.mMoveX8 < 0)) { // 13
				testShadowDirection( enemy, ship, shipDir, time, Operation.WIDTH, Operation.HEIGHT);
			}
		}
		else {
			if ((dx > Operation.WIDTH_25) && (enemy.mMoveX8 > 0)) { // 4, 8, 12, 16
				testShadowDirection( enemy, ship, shipDir, time, -Operation.WIDTH, 0);
			}
			if ((dx < -Operation.WIDTH_25) && (enemy.mMoveX8 < 0)) { // 1, 5, 9, 13
				testShadowDirection( enemy, ship, shipDir, time, Operation.WIDTH, 0);
			}
		}
	}

	public static int findDirection( AEnemy enemy, Ship ship, int shipDir, int time, int hx, int hy) {
		double dx = Operation.signX8( enemy.getX8( time) - ship.mX8) + hx;
		double dy = Operation.signY8( enemy.getY8( time) - ship.mY8) + hy;
		int shotDir = Angle.getAngle( Math.atan2( dy, dx));
		int steps = Angle.signRot( shotDir - shipDir) / 3;
		for (int i = 0; i < 2; ++i) {
			steps = getNextDirection( enemy, ship, shipDir, steps, time, hx, hy);
		}
		return steps;
	}

	public static int getIntersectionX( FlyInfo enemy, FlyInfo ship, int steps, int time, int hx, int hy) {
		int dir = (Angle.sDirection + (3 * steps)) & 0xFF;
		int sx = Angle.posShotX( dir);
		int sy = Angle.posShotY( dir);
		int ox = Angle.vShotX( dir);
		int oy = Angle.vShotY( dir);
		int ex = Operation.signX8( enemy.getX8( time) - ship.mX8) + (Math.abs( steps) * enemy.mMoveX8);
		int ey = Operation.signY8( enemy.getY8( time) - ship.mY8) + (Math.abs( steps) * enemy.mMoveY8);
		int k1 = (ex * enemy.mMoveY8) - (ey * enemy.mMoveX8);
		int k2 = (sx * oy) - (sy * ox);
		return ship.mX8 + (((ox * k1) - (enemy.mMoveX8 * k2)) / ((enemy.mMoveY8 * ox) - (enemy.mMoveX8 * oy)));
	}

	public static int getIntersectionY( FlyInfo enemy, FlyInfo ship, int steps, int time, int hx, int hy) {
		int dir = (Angle.sDirection + (3 * steps)) & 0xFF;
		int sx = Angle.posShotX( dir);
		int sy = Angle.posShotY( dir);
		int ox = Angle.vShotX( dir);
		int oy = Angle.vShotY( dir);
		int ex = Operation.signX8( enemy.getX8( time) - ship.mX8) + (Math.abs( steps) * enemy.mMoveX8);
		int ey = Operation.signY8( enemy.getY8( time) - ship.mY8) + (Math.abs( steps) * enemy.mMoveY8);
		int k1 = (ex * enemy.mMoveY8) - (ey * enemy.mMoveX8);
		int k2 = (sx * oy) - (sy * ox);
		return ship.mY8 + (((oy * k1) - (enemy.mMoveY8 * k2)) / ((enemy.mMoveY8 * ox) - (enemy.mMoveX8 * oy)));
	}

	void iterationAll( int time) {
		for (int i = 0; i < AsteroidVector.MAX_ASTEROIDS; ++i) {
			AEnemy enemy = mModel.mAsts.get( i);
			if (enemy.isVisible()) {
//				enemy.setDirection( mModel.mShip, 0);
				if (i == 3) {
					i += 0;
				}
				findDirection( enemy, mModel.mShip, Angle.sDirection, time);
			}
		}
	}

	void iterationDir( int time, int hx, int hy) {
		for (int i = 0; i < AsteroidVector.MAX_ASTEROIDS; ++i) {
			AEnemy enemy = mModel.mAsts.get( i);
			if (enemy.isVisible()) {
//				enemy.setDirection( mModel.mShip, 0);
				if (i == 4) {
					i += 0;
				}
				Ship ship = mModel.mShip;
				int shipDir = Angle.sDirection;
				enemy.mStepsOld = getNextDirection( enemy, ship, shipDir, enemy.mStepsOld, time, hx, hy);
				iterationTime( enemy, ship, shipDir, time, hx, hy);
			}
		}
	}

	void iterationEnemy( AEnemy enemy, Ship ship, int shipDir, int time, int hx, int hy) {
		System.out.print( "[");
		System.out.print( enemy);
		System.out.print( "] [");
		System.out.print( ship);
		System.out.print( "] ");
		System.out.print( shipDir);
		System.out.print( " ");
		System.out.print( time);
		System.out.print( " ");
		System.out.print( hx);
		System.out.print( " ");
		System.out.print( hy);
		System.out.println();
		double sd = Angle.toRadiant( shipDir);
		double shotDir = iterationInit1( enemy, ship, time, hx, hy);
		for (int i = 0; i < 3; ++i) {
			shotDir = getNextDirection( enemy, ship, sd, shotDir, time, hx, hy);
		}
		enemy.mStepsOld = newSteps( enemy, ship, sd, time, hx, hy, shotDir);
		iterationTime( enemy, ship, shipDir, time, hx, hy);
		printEnemy( enemy);
		mOut.println();
	}

	void iterationInit( int time, int hx, int hy) {
		for (int i = 0; i < AsteroidVector.MAX_ASTEROIDS; ++i) {
			AEnemy enemy = mModel.mAsts.get( i);
			if (enemy.isVisible()) {
				if (i == 11) {
					i += 0;
				}
				enemy.mStepsOld = iterationInit( enemy, mModel.mShip, Angle.sDirection, time, hx, hy);
			}
		}
	}

	static int iterationInit( AEnemy enemy, Ship ship, int shipDir, int time, int hx, int hy) {
		double dx = Operation.signX8( enemy.getX8( time) - ship.mX8) + hx;
		double dy = Operation.signY8( enemy.getY8( time) - ship.mY8) + hy;
		int shotDir = Angle.getAngle( Math.atan2( dy, dx));
		int newSteps = Angle.signRot( shotDir - shipDir);
		return newSteps( enemy, ship, shipDir, time, hx, hy, newSteps);
	}

	static double iterationInit1( AEnemy enemy, Ship ship, int time, int hx, int hy) {
		double dx = Operation.signX8( enemy.getX8( time) - ship.mX8) + hx;
		double dy = Operation.signY8( enemy.getY8( time) - ship.mY8) + hy;
		return Math.atan2( dy, dx);
	}

	void iterationPrint() {
		mOut.savePos();
		mOut.print( GameModel.sFrame);
		mOut.fill( 8);
		for (int i = 0; i < AsteroidVector.MAX_ASTEROIDS; ++i) {
			AEnemy enemy = mModel.mAsts.get( i);
			if (enemy.isVisible()) {
				printEnemy( enemy);
			}
		}
		mOut.println();
	}

	void iterationPrint( int time, int hx, int hy) {
		iterationInit( time, hx, hy);
		iterationPrint();
		for (int i = 0; i < 2; ++i) {
			iterationDir( time, hx, hy);
			iterationPrint();
		}
		mOut.println();
	}

	private static void iterationTime( AEnemy enemy, Ship ship, int shipDir, int time, int hx, int hy) {
		enemy.mTimeShotOld = getCollisionTimeShot( enemy, ship, shipDir, enemy.mStepsOld, time, hx, hy);
		enemy.mTimeEnemyOld = getCollisionTimeEnemy( enemy, ship, shipDir, enemy.mStepsOld, time, hx, hy);
	}

	static int newSteps( AEnemy enemy, Ship ship, int shipDir, int time, int hx, int hy, int newSteps) {
		double dx = Operation.signX8( enemy.getX8( time) - ship.mX8) + hx;
		double dy = Operation.signY8( enemy.getY8( time) - ship.mY8) + hy;
		double a = enemy.mMoveY8 * dx;
		double b = enemy.mMoveX8 * dy;
		if (a < b) {
			if (newSteps < 0) {
				newSteps -= 2;
			}
		}
		else if (a > b) {
			if (newSteps > 0) {
				newSteps += 2;
			}
		}
		return newSteps / 3;
	}

	static int newSteps( AEnemy enemy, Ship ship, double shipDir, int time, int hx, int hy, double newSteps) {
		double dx = Operation.signX8( enemy.getX8( time) - ship.mX8) + hx;
		double dy = Operation.signY8( enemy.getY8( time) - ship.mY8) + hy;
		double a = enemy.mMoveY8 * dx;
		double b = enemy.mMoveX8 * dy;
		int steps = Angle.signRot( Angle.getAngle( newSteps - shipDir));
		if (a < b) {
			if (steps < 0) {
				steps -= 2.0;
			}
		}
		else if (a > b) {
			if (steps > 0) {
				steps += 2.0;
			}
		}
		return steps / 3;
	}

	private static boolean next( int[] perm, int first, int last) {
		if ((perm == null) || (last <= first)) {
			return false;
		}
		int i = last;
		for (;;) {
			int ii = i--;
			int akt = perm[i];
			if (akt < perm[ii]) {
				int j = last;
				while (akt >= perm[j]) {
					--j;
				}
				swap( perm, i, j);
				reverse( perm, ii, last);
				return true;
			}
			if (i == first) {
				reverse( perm, first, last);
				return false;
			}
		}
	}

	static int getNextDirection( AEnemy enemy, Ship ship, int shipDir, int steps, int time, int hx, int hy) {
		int shotDir = (shipDir + (3 * steps)) & 0xFF;
		int ds = Math.abs( steps);
		int dx0 = Operation.signX8( enemy.getX8( time) - ship.mX8);
		int dy0 = Operation.signY8( enemy.getY8( time) - ship.mY8);
		int dx = dx0 + (ds * enemy.mMoveX8) + hx - Angle.posShotX( shotDir);
		int dy = dy0 + (ds * enemy.mMoveY8) + hy - Angle.posShotY( shotDir);
		double gamma = Math.atan2( dy, dx);
		double shotSpeed8 = Angle.vShot( shotDir);
		double enemySpeed8 = Math.sqrt( (enemy.mMoveX8 * enemy.mMoveX8) + (enemy.mMoveY8 * enemy.mMoveY8));
		double betha = Math.atan2( enemy.mMoveY8, enemy.mMoveX8);
		double dBG = betha - gamma;
		double dAG = Math.asin( enemySpeed8 * Math.sin( dBG) / shotSpeed8);
		if (Math.abs( dAG) > PI_05) {
			dAG = Math.PI - dAG;
		}
		int newDir = Angle.getAngle( dAG + gamma);
		int newSteps = Angle.signRot( newDir - shipDir);
		if (Angle.signRot( dBG) >= 0) {
			switch (newSteps % 3) {
				case 2:
				case -1:
					newSteps += 1;
					newSteps /= 3;
					break;
				case 1:
				case -2:
					newSteps += 2;
					newSteps /= 3;
					break;
				case 0:
					newSteps /= 3;
					if (newSteps != steps) {
						int te = getCollisionTimeEnemy( enemy, ship, shipDir, newSteps, time, hx, hy);
						int ts = getCollisionTimeShot( enemy, ship, shipDir, newSteps, time, hx, hy);
						if ((te > 0) && (te < ts)) {
							newSteps += 1;
						}
					}
					break;
			}
		}
		else {
			switch (newSteps % 3) {
				case 2:
				case -1:
					newSteps -= 2;
					newSteps /= 3;
					break;
				case 1:
				case -2:
					newSteps -= 1;
					newSteps /= 3;
					break;
				case 0:
					newSteps /= 3;
					if (newSteps != steps) {
						int te = getCollisionTimeEnemy( enemy, ship, shipDir, newSteps, time, hx, hy);
						int ts = getCollisionTimeShot( enemy, ship, shipDir, newSteps, time, hx, hy);
						if ((te > 0) && (te < ts)) {
							newSteps -= 1;
						}
					}
					break;
			}
		}
		return newSteps;
	}

	static double getNextDirection( AEnemy enemy, Ship ship, double shipDir, double shotDir, int time, int hx, int hy) {
		double dt = Angle.toAngle( Math.abs( shotDir - shipDir)) / 3.0;
		int dx0 = Operation.signX8( enemy.getX8( time) - ship.mX8);
		int dy0 = Operation.signY8( enemy.getY8( time) - ship.mY8);
		int sod = Angle.getAngle( shotDir);
		double dx = dx0 + (dt * enemy.mMoveX8) + hx - Angle.posShotX( sod);
		double dy = dy0 + (dt * enemy.mMoveY8) + hy - Angle.posShotY( sod);
		double gamma = Math.atan2( dy, dx);
		double shotSpeed8 = Angle.vShot( sod);
		double enemySpeed8 = Math.sqrt( (enemy.mMoveX8 * enemy.mMoveX8) + (enemy.mMoveY8 * enemy.mMoveY8));
		double betha = Math.atan2( enemy.mMoveY8, enemy.mMoveX8);
		double dBG = betha - gamma;
		double dAG = Math.asin( enemySpeed8 * Math.sin( dBG) / shotSpeed8);
		if (Math.abs( dAG) > PI_05) {
			dAG = Math.PI - dAG;
		}
		return dAG + gamma;
	}

	void permStep( boolean[] used, int[] gen, int step) {
		if (step < gen.length) {
			for (int i = 0; i < gen.length; ++i) {
				if (!used[i]) {
					used[i] = true;
					gen[step] = i;
					permStep( used, gen, step + 1);
					used[i] = false;
				}
			}
		}
		else {
			for (int i = 0; i < gen.length; ++i) {
				mOut.print( gen[i]);
				mOut.print( ",");
			}
			mOut.println();
		}
	}

	void permutiere( int[] gen) {
		boolean[] used = new boolean[gen.length];
		permStep( used, gen, 0);
	}

	void printEnemy( AEnemy enemy) {
		mOut.savePos();
		mOut.print( enemy.getIdent());
		mOut.print( ":");
		mOut.print( enemy.mStepsOld);
		mOut.print( ",");
		mOut.print( enemy.mTimeEnemyOld);
		mOut.print( ",");
		mOut.print( enemy.mTimeShotOld);
		mOut.print( ",");
		mOut.print( enemy.mTimeEnemyOld - enemy.mTimeShotOld);
		mOut.fill( 22);
	}

	void printTime( int time, int[] gen) {
		mOut.print( "[");
		for (int i = 0; i < gen.length; ++i) {
			if (i > 0) {
				mOut.print( ",");
			}
			mOut.print( gen[i]);
		}
		mOut.print( "] time=");
		mOut.print( time);
		mOut.print( " (");
		mOut.print( Math.round( 10.0 * time / 6) / 100.0);
		mOut.println( "s)");
	}

	void printTimes( AEnemy enemy, int steps, int time, int hx, int hy) {
		mOut.savePos();
		mOut.print( enemy.getIdent());
		mOut.fill( 4);
		mOut.print( "x0=");
		mOut.print( getIntersectionX( enemy, mModel.mShip, steps, time, hx, hy));
		mOut.fill( 10);
		mOut.print( "y0=");
		mOut.print( getIntersectionY( enemy, mModel.mShip, steps, time, hx, hy));
		mOut.fill( 10);
		mOut.print( "t=[");
		int te = getCollisionTimeEnemy( enemy, mModel.mShip, Angle.sDirection, steps, time, hx, hy);
		mOut.print( te);
		mOut.print( ",");
		int ts = getCollisionTimeShot( enemy, mModel.mShip, Angle.sDirection, steps, time, hx, hy);
		mOut.print( ts);
		mOut.print( ",");
		mOut.print( te - ts);
		mOut.print( "]");
		mOut.fill( 20);
		mOut.print( "step=");
		mOut.print( steps);
		mOut.fill( 10);
		mOut.print( "dir=");
		mOut.print( (Angle.sDirection + (3 * steps)) & 0xFF);
		mOut.fill( 10);
		mOut.println();
	}

	private static void reverse( int[] perm, int first, int last) {
		while (first < last) {
			swap( perm, first++, last--);
		}
	}

	private static void swap( int[] perm, int i, int j) {
		int temp = perm[i];
		perm[i] = perm[j];
		perm[j] = temp;
	}

	public void testA1() {
		mModel.parse( "550,___R__,038,039,058,1,060,063    S:4192,4192,12,+00,+00,..   M0:2512,3720,15,-61,-17,..        _1:0000,0000,15,+00,+00,..        M2:3096,4632,15,-59,+24,..        M3:0968,2464,15,-56,-30,..        _4:0000,0000,00,+00,+00,..        _5:0000,0000,00,+00,+00,..        a0:0000,0000,00,+00,+00,..  A1:0904,3616,14,+06,-10,..  E2:1832,6392,00,+00,+00,..  A3:2056,6584,14,+08,+08,..  A4:0904,2544,14,+06,-20,..  A5:6008,4936,15,-14,-17,..  A6:7720,3904,15,-16,+08,..  E7:1808,6352,00,+00,+00,..  A8:0648,6568,14,+11,-27,..  A9:0200,7016,14,+06,-22,..  E10:0992,3760,15,+00,+00,.. A11:0328,1672,14,+12,-28,.. A12:7152,4680,15,+06,-21,.. a13:0000,0000,14,+00,+00,.. a14:0000,0000,14,+00,+00,.. a15:0000,0000,14,+00,+00,.. a16:0000,0000,15,+00,+00,.. a17:0000,0000,15,+00,+00,.. a18:0000,0000,14,+00,+00,.. a19:0000,0000,14,+00,+00,.. a20:0000,0000,14,+00,+00,.. a21:0000,0000,00,+00,+00,.. a22:0000,0000,00,+00,+00,.. a23:0000,0000,00,+00,+00,.. a24:0000,0000,00,+00,+00,.. a25:0000,0000,00,+00,+00,.. a26:0000,0000,00,+00,+00,.. u:0000,0000,00,+00,+00,..   ");
		Angle.sDirection = 61;
		iterationPrint( 0, 0, 0);
	}

	public void testA2() {
		mModel.parse( "550,___R__,038,039,058,1,060,063    S:4192,4192,12,+00,+00,..   M0:2512,3720,15,-61,-17,..        _1:0000,0000,15,+00,+00,..        M2:3096,4632,15,-59,+24,..        M3:0968,2464,15,-56,-30,..        _4:0000,0000,00,+00,+00,..        _5:0000,0000,00,+00,+00,..        a0:0000,0000,00,+00,+00,..  A1:0904,3616,14,+06,-10,..  E2:1832,6392,00,+00,+00,..  A3:2056,6584,14,+08,+08,..  A4:0904,2544,14,+06,-20,..  A5:6008,4936,15,-14,-17,..  A6:7720,3904,15,-16,+08,..  E7:1808,6352,00,+00,+00,..  A8:0648,6568,14,+11,-27,..  A9:0200,7016,14,+06,-22,..  E10:0992,3760,15,+00,+00,.. A11:0328,1672,14,+12,-28,.. A12:7152,4680,15,+06,-21,.. a13:0000,0000,14,+00,+00,.. a14:0000,0000,14,+00,+00,.. a15:0000,0000,14,+00,+00,.. a16:0000,0000,15,+00,+00,.. a17:0000,0000,15,+00,+00,.. a18:0000,0000,14,+00,+00,.. a19:0000,0000,14,+00,+00,.. a20:0000,0000,14,+00,+00,.. a21:0000,0000,00,+00,+00,.. a22:0000,0000,00,+00,+00,.. a23:0000,0000,00,+00,+00,.. a24:0000,0000,00,+00,+00,.. a25:0000,0000,00,+00,+00,.. a26:0000,0000,00,+00,+00,.. u:0000,0000,00,+00,+00,..   ");
		Angle.sDirection = 61;
		iterationPrint( 0, 0, -Operation.HEIGHT);
	}

	public void testA3() {
		mModel.parse( "550,___R__,038,039,058,1,060,063    S:4192,4192,12,+00,+00,..   M0:2512,3720,15,-61,-17,..        _1:0000,0000,15,+00,+00,..        M2:3096,4632,15,-59,+24,..        M3:0968,2464,15,-56,-30,..        _4:0000,0000,00,+00,+00,..        _5:0000,0000,00,+00,+00,..        a0:0000,0000,00,+00,+00,..  A1:0904,3616,14,+06,-10,..  E2:1832,6392,00,+00,+00,..  A3:2056,6584,14,+08,+08,..  A4:0904,2544,14,+06,-20,..  A5:6008,4936,15,-14,-17,..  A6:7720,3904,15,-16,+08,..  E7:1808,6352,00,+00,+00,..  A8:0648,6568,14,+11,-27,..  A9:0200,7016,14,+06,-22,..  E10:0992,3760,15,+00,+00,.. A11:0328,1672,14,+12,-28,.. A12:7152,4680,15,+06,-21,.. a13:0000,0000,14,+00,+00,.. a14:0000,0000,14,+00,+00,.. a15:0000,0000,14,+00,+00,.. a16:0000,0000,15,+00,+00,.. a17:0000,0000,15,+00,+00,.. a18:0000,0000,14,+00,+00,.. a19:0000,0000,14,+00,+00,.. a20:0000,0000,14,+00,+00,.. a21:0000,0000,00,+00,+00,.. a22:0000,0000,00,+00,+00,.. a23:0000,0000,00,+00,+00,.. a24:0000,0000,00,+00,+00,.. a25:0000,0000,00,+00,+00,.. a26:0000,0000,00,+00,+00,.. u:0000,0000,00,+00,+00,..   ");
		Angle.sDirection = 61;
		iterationAll( 0);
		iterationPrint();
		mOut.println();
	}

	public void testB1() {
		mModel.parse( "550,___R__,038,039,058,1,060,063    S:4192,4192,12,+00,+00,..   M0:2512,3720,15,-61,-17,..        _1:0000,0000,15,+00,+00,..        M2:3096,4632,15,-59,+24,..        M3:0968,2464,15,-56,-30,..        _4:0000,0000,00,+00,+00,..        _5:0000,0000,00,+00,+00,..        a0:0000,0000,00,+00,+00,..  A1:0904,3616,14,+06,-10,..  E2:1832,6392,00,+00,+00,..  A3:2056,6584,14,+08,+08,..  A4:0904,2544,14,+06,-20,..  A5:6008,4936,15,-14,-17,..  A6:7720,3904,15,-16,+08,..  E7:1808,6352,00,+00,+00,..  A8:0648,6568,14,+11,-27,..  A9:0200,7016,14,+06,-22,..  E10:0992,3760,15,+00,+00,.. A11:0328,1672,14,+12,-28,.. A12:7152,4680,15,+06,-21,.. a13:0000,0000,14,+00,+00,.. a14:0000,0000,14,+00,+00,.. a15:0000,0000,14,+00,+00,.. a16:0000,0000,15,+00,+00,.. a17:0000,0000,15,+00,+00,.. a18:0000,0000,14,+00,+00,.. a19:0000,0000,14,+00,+00,.. a20:0000,0000,14,+00,+00,.. a21:0000,0000,00,+00,+00,.. a22:0000,0000,00,+00,+00,.. a23:0000,0000,00,+00,+00,.. a24:0000,0000,00,+00,+00,.. a25:0000,0000,00,+00,+00,.. a26:0000,0000,00,+00,+00,.. u:0000,0000,00,+00,+00,..   ");
		iterationEnemy( mModel.mAsts.get( 5), mModel.mShip, 61, 0, 0, 0);
	}

	public void testB2() {
		mModel.parse( "550,___R__,038,039,058,1,060,063    S:4192,4192,12,+00,+00,..   M0:2512,3720,15,-61,-17,..        _1:0000,0000,15,+00,+00,..        M2:3096,4632,15,-59,+24,..        M3:0968,2464,15,-56,-30,..        _4:0000,0000,00,+00,+00,..        _5:0000,0000,00,+00,+00,..        a0:0000,0000,00,+00,+00,..  A1:0904,3616,14,+06,-10,..  E2:1832,6392,00,+00,+00,..  A3:2056,6584,14,+08,+08,..  A4:0904,2544,14,+06,-20,..  A5:6008,4936,15,-14,-17,..  A6:7720,3904,15,-16,+08,..  E7:1808,6352,00,+00,+00,..  A8:0648,6568,14,+11,-27,..  A9:0200,7016,14,+06,-22,..  E10:0992,3760,15,+00,+00,.. A11:0328,1672,14,+12,-28,.. A12:7152,4680,15,+06,-21,.. a13:0000,0000,14,+00,+00,.. a14:0000,0000,14,+00,+00,.. a15:0000,0000,14,+00,+00,.. a16:0000,0000,15,+00,+00,.. a17:0000,0000,15,+00,+00,.. a18:0000,0000,14,+00,+00,.. a19:0000,0000,14,+00,+00,.. a20:0000,0000,14,+00,+00,.. a21:0000,0000,00,+00,+00,.. a22:0000,0000,00,+00,+00,.. a23:0000,0000,00,+00,+00,.. a24:0000,0000,00,+00,+00,.. a25:0000,0000,00,+00,+00,.. a26:0000,0000,00,+00,+00,.. u:0000,0000,00,+00,+00,..   ");
		iterationEnemy( mModel.mAsts.get( 8), mModel.mShip, 61, 0, 0, -Operation.HEIGHT);
	}

	public void testB3() {
		int time = 409;
		int x = 648;
		int y = 6568;
		int mx = 11;
		int my = -27;
		int x1 = Operation.normX8( x + (time * mx));
		int y1 = Operation.normY8( y + (time * my));
		iterationEnemy( new Asteroid( 0, x, y, 14, mx, my), new Ship( 4192, 4192, 12, 0, 0), 153, time, 0, 0);
		iterationEnemy( new Asteroid( 0, x1, y1, 14, mx, my), new Ship( 4192, 4192, 12, 0, 0), 153, 0, 0, 0);
	}

	public void testB4() {
		int time = 220;
		int x = 904;
		int y = 3616;
		int mx = 6;
		int my = -10;
		int x1 = Operation.normX8( x + (time * mx));
		int y1 = Operation.normY8( y + (time * my));
		iterationEnemy( new Asteroid( 0, x, y, 14, mx, my), new Ship( 4192, 4192, 12, 0, 0), 112, time, 0, 6144);
		iterationEnemy( new Asteroid( 0, x1, y1, 14, mx, my), new Ship( 4192, 4192, 12, 0, 0), 112, 0, 0, 6144);
	}

	public void testB5() {
		int time = 226;
		int x = 904;
		int y = 2544;
		int mx = 6;
		int my = -20;
		int x1 = Operation.normX8( x + (time * mx));
		int y1 = Operation.normY8( y + (time * my));
		iterationEnemy( new Asteroid( 0, x, y, 14, mx, my), new Ship( 4192, 4192, 12, 0, 0), 244, time, 0, 0);
		iterationEnemy( new Asteroid( 0, x1, y1, 14, mx, my), new Ship( 4192, 4192, 12, 0, 0), 244, 0, 0, 0);
	}

//	public void testC() {
//		Parser.parse( mModel, "550,___R__,038,039,058,1,060,063	   S:4192,4192,12,+00,+00,..   M0:2512,3720,15,-61,-17,..	 _1:0000,0000,15,+00,+00,..	   M2:3096,4632,15,-59,+24,..	     M3:0968,2464,15,-56,-30,..	       _4:0000,0000,00,+00,+00,..	 _5:0000,0000,00,+00,+00,..	   a0:0000,0000,00,+00,+00,..  A1:0904,3616,14,+06,-10,..  E2:1832,6392,00,+00,+00,..  A3:2056,6584,14,+08,+08,..  A4:0904,2544,14,+06,-20,..  A5:6008,4936,15,-14,-17,..  A6:7720,3904,15,-16,+08,..  E7:1808,6352,00,+00,+00,..  A8:0648,6568,14,+11,-27,..  A9:0200,7016,14,+06,-22,..  E10:0992,3760,15,+00,+00,.. A11:0328,1672,14,+12,-28,.. A12:7152,4680,15,+06,-21,.. a13:0000,0000,14,+00,+00,.. a14:0000,0000,14,+00,+00,.. a15:0000,0000,14,+00,+00,.. a16:0000,0000,15,+00,+00,.. a17:0000,0000,15,+00,+00,.. a18:0000,0000,14,+00,+00,.. a19:0000,0000,14,+00,+00,.. a20:0000,0000,14,+00,+00,.. a21:0000,0000,00,+00,+00,.. a22:0000,0000,00,+00,+00,.. a23:0000,0000,00,+00,+00,.. a24:0000,0000,00,+00,+00,.. a25:0000,0000,00,+00,+00,.. a26:0000,0000,00,+00,+00,.. u:0000,0000,00,+00,+00,..   ");
//		Angle.sDirection = 61;
//		mOut.println();
//		printTimes( mModel.mAsts.get( 1), 29, 0, 0, 0);
//		printTimes( mModel.mAsts.get( 3), 38, 0, 0, 0);
//		printTimes( mModel.mAsts.get( 3), 43, 0, 0, -Operation.HEIGHT);
//		printTimes( mModel.mAsts.get( 4), 36, 0, 0, 0);
//		printTimes( mModel.mAsts.get( 4), 12, 0, 0, Operation.HEIGHT);
//		printTimes( mModel.mAsts.get( 5), -20, 0, 0, 0);
//		printTimes( mModel.mAsts.get( 6), -19, 0, 0, 0);
//		printTimes( mModel.mAsts.get( 8), 42, 0, 0, 0);
//		printTimes( mModel.mAsts.get( 8), 44, 0, 0, -Operation.HEIGHT);
//		printTimes( mModel.mAsts.get( 9), 38, 0, 0, 0);
//		printTimes( mModel.mAsts.get( 11), 40, 0, 0, 0);
//		printTimes( mModel.mAsts.get( 12), -25, 0, 0, 0);
//	}
//
	public void testD() {
		mModel.parse( "550,___R__,038,039,058,1,060,063    S:4192,4192,12,+00,+00,..   M0:2512,3720,15,-61,-17,..        _1:0000,0000,15,+00,+00,..        M2:3096,4632,15,-59,+24,..        M3:0968,2464,15,-56,-30,..        _4:0000,0000,00,+00,+00,..        _5:0000,0000,00,+00,+00,..        a0:0000,0000,00,+00,+00,..  A1:0904,3616,14,+06,-10,..  E2:1832,6392,00,+00,+00,..  A3:2056,6584,14,+08,+08,..  A4:0904,2544,14,+06,-20,..  A5:6008,4936,15,-14,-17,..  A6:7720,3904,15,-16,+08,..  E7:1808,6352,00,+00,+00,..  A8:0648,6568,14,+11,-27,..  A9:0200,7016,14,+06,-22,..  E10:0992,3760,15,+00,+00,.. A11:0328,1672,14,+12,-28,.. A12:7152,4680,15,+06,-21,.. a13:0000,0000,14,+00,+00,.. a14:0000,0000,14,+00,+00,.. a15:0000,0000,14,+00,+00,.. a16:0000,0000,15,+00,+00,.. a17:0000,0000,15,+00,+00,.. a18:0000,0000,14,+00,+00,.. a19:0000,0000,14,+00,+00,.. a20:0000,0000,14,+00,+00,.. a21:0000,0000,00,+00,+00,.. a22:0000,0000,00,+00,+00,.. a23:0000,0000,00,+00,+00,.. a24:0000,0000,00,+00,+00,.. a25:0000,0000,00,+00,+00,.. a26:0000,0000,00,+00,+00,.. u:0000,0000,00,+00,+00,..   ");
		Angle.sDirection = 61;
		int[] gen = { 1, 3, 4, 5, 6, 8, 9, 11, 12 };
		printTime( getTime( mModel.mAsts, mModel.mUfo, mModel.mShip, Angle.sDirection, gen), gen);
		System.out.println();
	}

	public void testE() {
		mModel.parse( "550,___R__,038,039,058,1,060,063    S:4192,4192,12,+00,+00,..   M0:2512,3720,15,-61,-17,..        _1:0000,0000,15,+00,+00,..        M2:3096,4632,15,-59,+24,..        M3:0968,2464,15,-56,-30,..        _4:0000,0000,00,+00,+00,..        _5:0000,0000,00,+00,+00,..        a0:0000,0000,00,+00,+00,..  A1:0904,3616,14,+06,-10,..  E2:1832,6392,00,+00,+00,..  A3:2056,6584,14,+08,+08,..  A4:0904,2544,14,+06,-20,..  A5:6008,4936,15,-14,-17,..  A6:7720,3904,15,-16,+08,..  E7:1808,6352,00,+00,+00,..  A8:0648,6568,14,+11,-27,..  A9:0200,7016,14,+06,-22,..  E10:0992,3760,15,+00,+00,.. A11:0328,1672,14,+12,-28,.. A12:7152,4680,15,+06,-21,.. a13:0000,0000,14,+00,+00,.. a14:0000,0000,14,+00,+00,.. a15:0000,0000,14,+00,+00,.. a16:0000,0000,15,+00,+00,.. a17:0000,0000,15,+00,+00,.. a18:0000,0000,14,+00,+00,.. a19:0000,0000,14,+00,+00,.. a20:0000,0000,14,+00,+00,.. a21:0000,0000,00,+00,+00,.. a22:0000,0000,00,+00,+00,.. a23:0000,0000,00,+00,+00,.. a24:0000,0000,00,+00,+00,.. a25:0000,0000,00,+00,+00,.. a26:0000,0000,00,+00,+00,.. u:0000,0000,00,+00,+00,..   ");
		Angle.sDirection = 61;
		int[] asts = { 1, 3, 4, 5, 6, 8, 9, 11, 12 };
		int[] perm = { 0, 1, 2, 3 };
		int[] gen = new int[perm.length];
		int max = Integer.MAX_VALUE;
		do {
			for (int i = 0; i < perm.length; ++i) {
				gen[i] = asts[perm[i]];
			}
			int time = getTime( mModel.mAsts, mModel.mUfo, mModel.mShip, Angle.sDirection, gen);
			if (time < max) {
				max = time;
				printTime( time, gen);
			}
		}
		while (next( perm, 0, perm.length - 1));
	}

	public void testF() {
		int max = 400000;
		long start = System.currentTimeMillis();
		for (int i = 0; i < max; ++i) {
			Math.atan2( 500.0, 100.0);
			Math.atan2( 300.0, 400.0);
		}
		long end = System.currentTimeMillis() - start;
		System.out.println( (1.0 * end / max) + "ms");
	}

	public void testG() {
		int max = 400000;
		long start = System.currentTimeMillis();
		for (int i = 0; i < max; ++i) {
//			double a = Math.sin( (500 + i) / 700);
			Math.sin( 0.6667);
			Math.asin( 0.689);
		}
		long end = System.currentTimeMillis() - start;
		System.out.println( (1.0 * end / max) + "ms");
	}

	static void testShadowDirection( AEnemy enemy, Ship ship, int shipDir, int time, int hx, int hy) {
		int steps = findDirection( enemy, ship, shipDir, time, hx, hy);
		int timeShot = getCollisionTimeShot( enemy, ship, shipDir, steps, time, hx, hy);
		int timeEnemy = getCollisionTimeEnemy( enemy, ship, shipDir, steps, time, hx, hy);
		if (timeEnemy < enemy.mTimeEnemyOld) {
			enemy.mStepsOld = steps;
			enemy.mTimeShotOld = timeShot;
			enemy.mTimeEnemyOld = timeEnemy;
		}
		if (timeEnemy < timeShot) {
			System.out.print( "ops: [");
			System.out.print( enemy);
			System.out.print( "] [");
			System.out.print( ship);
			System.out.print( "] ");
			System.out.print( shipDir);
			System.out.print( " ");
			System.out.print( time);
			System.out.print( " ");
			System.out.print( hx);
			System.out.print( " ");
			System.out.print( hy);
			System.out.println();
		}
	}

	static int getTime( AsteroidVector asts, Ufo ufo, Ship ship, int shipDir, int[] gen) {
		int time = 0;
		for (int i = 0; i < gen.length; ++i) {
			int index = gen[i];
			AEnemy enemy;
			if (index == Ufo.UFO_ID) {
				enemy = ufo;
			}
			else {
				enemy = asts.get( index);
			}
			findDirection( enemy, ship, shipDir, time);
			time += Math.abs( enemy.mStepsOld);
			time += enemy.mTimeEnemyOld;
			shipDir = (shipDir + enemy.mStepsOld) & 0xFF;
		}
		return time;
	}
}
