using System;
using System.Collections.Generic;
using System.Text;
using System.Collections;

namespace ct.Asteroid
{
  #region ENUMS
  public enum ShotType
  {
    UNKNOWN = 0,
    SHIP = 1,
    SAUCER = 2
  }
  public enum AsteroidType
  {
    UNKNOWN = 0,
    EINS = 1,
    ZWEI = 2,
    DREI = 3,
    VIER = 4
  }
  public enum AsteroidGroesse
  {
    UNKNOWN = 0,
    KLEIN = 8,
    MITTEL = 20,
    GROSS = 40
  }
  public enum SaucerGroesse
  {
    UNKNOWN = 0,
    MITTEL = 10,
    GROSS = 25
  }
  public class CAsteroidTarget
  {
    public CDrehung Drehung = null;
    public int abs_delta_drehung = 1000;
    public CPunkt P1, P4, P5;
    public int frame_P1, distance_P1;
    public int frame_P4, distance_P4;
    public int frame_P5, distance_P5;
  }
  #endregion

  #region CTargets, CDrehung
  public class CDrehung
  {
    private static Hashtable DrehungenHash = new Hashtable();
    private static double Bogenmass = (double)(180 / Math.PI);
    private static bool is_initialized = false;
    /// <summary>
    /// Wenn > -31 dreht er eher nach rechts - < 
    /// </summary>
    public static int Left_Margin = -31;
    public static int Right_Margin = 32;
    private static int[,] cls_DrehPaare = {  
                           {0,152,296,440,584,720,856,976,1088,1192,1280,1360,1416,1472,1504,1528,1536,1528,
                               1504,1472,1416,1360,1280,1192,1088,976,856,720,584,440,296,152,0,-152,-296,-440,
                               -584,-720,-856,-976,-1088,-1192,-1280,-1360,-1416,-1472,-1504,-1528,-1536,-1528,
                               -1504,-1472,-1416,-1360,-1280,-1192,-1088,-976,-856,-720,-584,-440,-296,-152},
                           {1536,1528,1504,1472,1416,1360,1280,1192,1088,976,856,720,584,440,296,152,0,-152,
                               -296,-440,-584,-720,-856,-976,-1088,-1192,-1280,-1360,-1416,-1472,-1504,-1528,
                               -1536,-1528,-1504,-1472,-1416,-1360,-1280,-1192,-1088,-976,-856,-720,-584,-440,
                               -296,-152,0,152,296,440,584,720,856,976,1088,1192,1280,1360,1416,1472,1504,1528}
        };
    public int dx, dy, Nummer;
    public double m;
    public double Winkel;
    public CGerade Drehlinie;
    public CPunkt Peilpunkt;

    #region PUBLIC FUNCTIONS
    public CDrehung ( int x, int y )
    {
      if (!is_initialized) init();
      dx = x; dy = y;
      if (dx == 0) m = (double) dy / 0.0001;
      else if (dy == 0) m = (double) 0.0001 / dx;
      else  m = (double) dy / dx;
      Winkel = berechneWinkel(dx, dy);
     }
    #endregion
    #region STATIC FUNCTIONS
     public static double berechneWinkel ( CPunkt start, CPunkt ziel )
     {
       int delta_x, delta_y;
       delta_x = ziel.x - start.x;
       delta_y = ziel.y - start.y;
       return berechneWinkel(delta_x, delta_y);
     }
     public static double berechneWinkel ( int delta_x, int delta_y )
    {
      double cls_winkel;
      cls_winkel = Math.Atan2(delta_y, delta_x) * Bogenmass;
      if (cls_winkel <= 90) cls_winkel = 90 - cls_winkel;
      else cls_winkel = 450 - cls_winkel;
      return cls_winkel;
    }
    public static void init ()
    {
      is_initialized = true;
      for (int i = 0; i < 64; i++)
      {
        CDrehung newDrehung = new CDrehung(cls_DrehPaare[0, i], cls_DrehPaare[1, i]);
        newDrehung.Nummer = i + 1;
        newDrehung.Peilpunkt = new CPunkt(newDrehung.dx, newDrehung.dy);
        newDrehung.Drehlinie = new CGerade(new CPunkt(0, 0), newDrehung.Peilpunkt);
        //TargetData.ConsoleWriteLine("{0}.Winkel:{1:#.00}", newDrehung.Nummer, newDrehung.Winkel);
        DrehungenHash.Add(newDrehung.Nummer, newDrehung);
        DrehungenHash.Add((int) -newDrehung.Winkel, newDrehung);
        DrehungenHash.Add(newDrehung.ToString(), newDrehung);
      }
    }
    public static CDrehung getDrehung ( double winkel )
    {
      return getDrehung((int)winkel);
    }
    public static CDrehung getDrehung ( int winkel )
    {
      CDrehung returnDrehung = null;
      int delta_winkel = 0, help_winkel;
      while (returnDrehung == null && delta_winkel < 10)
      {
        help_winkel = winkel + delta_winkel;
        if (help_winkel > 359) help_winkel -= 360;
        if (DrehungenHash.ContainsKey(-help_winkel))
        {
          returnDrehung = DrehungenHash[-help_winkel] as CDrehung;
          break;
        }
        help_winkel = winkel - delta_winkel;
        if (help_winkel < 0) help_winkel += 360;
        else if (DrehungenHash.ContainsKey(-help_winkel))
        {
          returnDrehung = DrehungenHash[-help_winkel] as CDrehung;
          break;
        }
        delta_winkel++;
      }
      return returnDrehung;
    }
    public static CDrehung getDrehung ( int x, int y )
    {
      if (!is_initialized) init();
      string hashKey = x.ToString() + " " + y.ToString();
      if (DrehungenHash.ContainsKey(hashKey))
      {
        return DrehungenHash[hashKey] as CDrehung; // Volltreffer 
      }
      else
      {
        int winkel_drehung = (int)berechneWinkel(x, y);
        return getDrehung(winkel_drehung);
      }
    }
    public static CDrehung getDrehung ( CPunkt punkt1, CPunkt punkt2 )
    {
      if (!is_initialized) init();
      int x, y;
      x = punkt2.x - punkt1.x;
      y = punkt2.y - punkt1.y;
        return getDrehung(x,y);
    }
    #endregion
    #region OVERRIDE and OPERATOR
    public static int operator - ( CDrehung drehung1, CDrehung drehung2 )
    {
      if (drehung1 == null || drehung2 == null) return 1000;
      int returnInt = drehung1.Nummer - drehung2.Nummer;
      if (returnInt < Left_Margin) returnInt += 64; 
      else if (returnInt > Right_Margin) returnInt -= 64;
      return returnInt;
    }
    public override string ToString ()
    {
      return string.Format("{0} {1}", dx, dy);
    }
    #endregion
  }
  #endregion

  #region CAsteroid, CShot, CSaucer, CShip <- CFlugobject
  public class CAsteroid : CFlugobject
  {
    public AsteroidType type = AsteroidType.UNKNOWN;
    public AsteroidGroesse groesse = AsteroidGroesse.UNKNOWN;
    private CAsteroidTarget asteroidTarget = new CAsteroidTarget();
    public Hashtable Shots = new Hashtable();
    public int Radius;
    public int Punktwert;
    private int cls_hitframe = 0;
    private CPunkt HitPunkt;
    private CKreis HitKreis;
    public int HitShip
    {
      get
      {
        CPunkt futurePosition;
        int tmp_int;
        if (Flugbahn == null) return -1;
        HitKreis = CTargets.Ship.Shipgebiet;
        HitKreis.Radius += (int)groesse;
        HitPunkt = TrifftGebiet(HitKreis);
        futurePosition = Berechneposition(2);
        if (futurePosition == null) return -1;
        if ( futurePosition.Entfernung_XY(CTargets.Ship.Berechneposition(2)) > 
          Position.Entfernung_XY(CTargets.Ship.Position)) return -1;
        if (HitPunkt == null || Fluggeschwindigkeit < 0.01) return -1;
        tmp_int = (int)HitPunkt.Entfernung_XY(Aktuelleposition) - (CShip.Shipradius + Radius);
        tmp_int = (int)(tmp_int / Mittlerefluggeschwindigkeit);
        return tmp_int;
      }
    }
    public int ShipDistance
    {
      get
      {
        int tmpInt = -(CShip.Shipradius + Radius) ;
        //if (Position == null || CTargets.Ship.Position == null) return 0x1FFFFF;
        tmpInt += (int)Position.Entfernung_XY(CTargets.Ship.Position);
        return tmpInt;
      }
    }
    public int HitFramesLeft { get { return cls_hitframe; } }
    public CDrehung HitDrehung
    {
      get
      {
        CDrehung returnDrehung = null;
        double shot_speed = 8, distance, shot_distance, min_dist = 0x1FFFFF;
        int frameCounter = 1; //, deltaframe = frameCounter / 2; Annherung mach ich spter
        CPunkt ShipPosition = CTargets.Ship.Berechneposition(1);
        CPunkt futureObjectPosition = this.Berechneposition(frameCounter);
        if (futureObjectPosition == null)
        {
          TargetData.ConsoleWriteLine("{0} Nix gefunden", Position);
          return null;
        }
        while (returnDrehung == null && frameCounter < FuturePositions.Count)
        {
          futureObjectPosition = FuturePositions[frameCounter] as CPunkt;
          distance = futureObjectPosition.Entfernung_XY(ShipPosition);
          shot_distance = Math.Abs(distance - (shot_speed * frameCounter));
          //if (shot_distance < Mittlerefluggeschwindigkeit + shot_speed)
          if (shot_distance < Radius)
          {
            returnDrehung = CDrehung.getDrehung(CDrehung.berechneWinkel(ShipPosition, futureObjectPosition));
            break;
          }
          frameCounter++;
          futureObjectPosition = Berechneposition(frameCounter);
        }
        return returnDrehung;
      }
    }
    public CDrehung HitDrehung_alt2
    {
      get
      {
        CDrehung returnDrehung = null;
        double shot_speed = 8, distance, shot_distance, min_dist = 0x1FFFFF;
        int frameCounter = 1; //, deltaframe = frameCounter / 2; Annherung mach ich spter
        CPunkt ShipPosition = CTargets.Ship.Berechneposition(1);
        CPunkt futureObjectPosition = this.Berechneposition(frameCounter);
        if (futureObjectPosition == null)
        {
          TargetData.ConsoleWriteLine("{0} Nix gefunden", Position);
          return null;
        }
        while (returnDrehung == null && frameCounter < 80)
        {
          distance = futureObjectPosition.Entfernung_XY(ShipPosition);
          shot_distance = Math.Abs(distance - (shot_speed * frameCounter));
          //if (shot_distance < Mittlerefluggeschwindigkeit + shot_speed)
          if (shot_distance < Radius)
          {
            returnDrehung = CDrehung.getDrehung(CDrehung.berechneWinkel(ShipPosition, futureObjectPosition));
            break;
          }
          frameCounter++;
          futureObjectPosition = Berechneposition(frameCounter);
        }
        return returnDrehung;
      }
    }
    public CDrehung HitDrehung_alt
    {
        get
        {
            CDrehung returnDrehung = null;
            double shot_speed = 8, distance, shot_distance, min_dist = 0x1FFFFF;
            int frameCounter = 1; //, deltaframe = frameCounter / 2; Annherung mach ich spter
            CPunkt ShipPosition = CTargets.Ship.Position;
            CPunkt futureObjectPosition = Berechneposition(frameCounter);
            if (futureObjectPosition == null) return null;
            while (returnDrehung == null && frameCounter < 80)
            {
                distance = futureObjectPosition.Entfernung_XY(ShipPosition);
                shot_distance = Math.Abs(distance - (shot_speed * frameCounter));
                if (shot_distance < Mittlerefluggeschwindigkeit + shot_speed)
                {
                    returnDrehung = CDrehung.getDrehung(CDrehung.berechneWinkel(ShipPosition, futureObjectPosition));
                }
                frameCounter++;
                futureObjectPosition = Berechneposition(frameCounter);
            }
            return returnDrehung;
        }
    }
    public CAsteroidTarget targetAsteroid
    {
      get
      {
        asteroidTarget.abs_delta_drehung = 1000;
        double shot_speed = 8, distance, shot_distance, min_dist = 0x1FFFFF;
        bool not_found = true;
        int frameCounter = 1; //, deltaframe = frameCounter / 2; Annherung mach ich spter
        CPunkt ShipPosition = CTargets.Ship.Position;
        CPunkt futureObjectPosition = Berechneposition(frameCounter);
        if (Flugbahn != null)
        {
          CDrehung shipDrehungTarget = null;
          //TargetData.ConsoleWriteLine("Starte Suche: Asteroid={0}" , Position);
          while (frameCounter < 80 && not_found)
          {
            shipDrehungTarget = CDrehung.getDrehung(CDrehung.berechneWinkel(ShipPosition, futureObjectPosition));
            asteroidTarget.P1 = shipDrehungTarget.Drehlinie.Schnittpunkt(Flugbahn);
            if (asteroidTarget.P1 == null ||
                futureObjectPosition.Entfernung_Quadrat(asteroidTarget.P1) >= Position.Entfernung_Quadrat(asteroidTarget.P1))
            {
              frameCounter++;
              futureObjectPosition = Berechneposition(frameCounter);
              continue;
            }
            else
            {
              asteroidTarget.Drehung = shipDrehungTarget;
              asteroidTarget.abs_delta_drehung = Math.Abs(shipDrehungTarget - CTargets.Ship.Drehung);
              not_found = false;
            } 
          }
        }
        return asteroidTarget;
      }
    }
  }
  public class CShot : CFlugobject
  {
    public ShotType type = ShotType.UNKNOWN;
    public int Serialnummer;
    public int Hit_Serialnummer = 0;
    public int Hit_Frame = 0;
    public int Target_Serialnummer = 0;
    private CPunkt HitPunkt;
    public int HitShip
    {
      get
      {
        if (type == ShotType.SHIP) return -999;
        int tmp_int;
        if (Flugbahn == null) return -1;
        HitPunkt = TrifftGebiet(CTargets.Ship.Shipgebiet);
        if (HitPunkt == null || Fluggeschwindigkeit < 0.01) return -1;
        tmp_int = (int)(HitPunkt.Entfernung_XY(Position) / Mittlerefluggeschwindigkeit);
        //tmp_int = (int)(HitPunkt.Entfernung_XY(Position) / Fluggeschwindigkeit);
        return tmp_int;
      }
    }
    public new CPunkt Berechneposition(int delta_frame)
    {
      int delta_x, delta_y;
      CPunkt returnPosition = Startposition;
      CPunkt Position1, Position2;
      //if (cls_counter < 4) return null;
      if (Letzteposition == null) return returnPosition;
      if (Aktuelleposition == null)
      {
        Position1 = Startposition;
        Position2 = Letzteposition;
      }
      else
      {
        Position1 = Letzteposition;
        Position2 = Aktuelleposition;
      }
      if (letzte_Flugzeit == 0) return Position2;
      if (Fluggeschwindigkeit == 0) return Position2;
      delta_x = (int)((Position2.x - Position1.x) / letzte_Flugzeit);
      delta_y = (int)((Position2.y - Position1.y) / letzte_Flugzeit);
      delta_x = (int)(delta_x * delta_frame);
      delta_y = (int)(delta_y * delta_frame);
      if (delta_x > 500) delta_x -= 1024;
      else if (delta_x < -500) delta_x += 1023;
      if (delta_y > 500)
      {
        delta_y -= 768;
      }
      else if (delta_y < -500)
      {
        delta_y += 767;
      }
      returnPosition.x = Position2.x + delta_x;
      returnPosition.y = Position2.y + delta_y;
      if (returnPosition.x > 1023) returnPosition.x -= 1024;
      else if (returnPosition.x < 0) returnPosition.x += 1023;
      if (returnPosition.y > 895)
      {
        returnPosition.y -= 768;
      }
      else if (returnPosition.y < 128) returnPosition.y += 767;
      return returnPosition;
    }

    public void setTarget ()
    {
      if (type == ShotType.SHIP)
      {
        if ( Flugbahn != null )
        {
          if (Hit_Serialnummer > 0)
          {
            //Guck mal ob er noch trifft
            //wenn ja RETURN
          }
          CPunkt futurePosition, targetFuturePosition;
          CAsteroid asteroid;
          int delta_frame = 1;
          int min_dist = 0x1FFFF, distance, tmp_Serialnummer=0;
          Hit_Serialnummer = 0;
          while (Hit_Serialnummer == 0 && delta_frame < 80)
          {
            futurePosition = Berechneposition(1);
            if (futurePosition == null) return;
            {
              for (int i = 0; i < CTargets.Asteroids.Count; i++)
              {
                asteroid = CTargets.Asteroids[i] as CAsteroid;
                if (asteroid.Flugbahn == null || asteroid.Mittlerefluggeschwindigkeit == 0) continue;
                targetFuturePosition = asteroid.Berechneposition(delta_frame);
                distance = (int) futurePosition.Entfernung_XY(targetFuturePosition);
                distance = (int) ( distance / asteroid.Mittlerefluggeschwindigkeit);
                distance -= (int) asteroid.groesse;
                if (distance < 0) //Treffer
                {
                  Hit_Serialnummer = asteroid.Serialnummer;
                  //TargetData.ConsoleWriteLine("Shot [{0}] trifft [{1}]", Serialnummer, Hit_Serialnummer);
                }
                //else if (distance < min_dist)
                //{
                //  min_dist = distance;
                //  tmp_Serialnummer = asteroid.Serialnummer;
                //}
              }
            }
            delta_frame++; //Achtung knnte knapp werden
          }
        }
      }
    }
  }
    public class CSaucer : CFlugobject
    {
        public SaucerGroesse groesse = SaucerGroesse.UNKNOWN;
        public bool Saucer_Present = false;
        public int HitFrame(CDrehung hit_drehung)
        {
            return 0;
        }
        public CDrehung HitDrehung
        {
            get
            {
                if (!Saucer_Present) return null;
                CDrehung returnDrehung = null;
                double shot_speed = 8, distance, shot_distance, min_dist = 0x1FFFFF;
                int frameCounter = 1; //, deltaframe = frameCounter / 2; Annherung mach ich spter
                CPunkt ShipPosition = CTargets.Ship.Position;
                CPunkt futureObjectPosition = Berechneposition(frameCounter);
                if (futureObjectPosition == null) return null;
                while (returnDrehung == null && frameCounter < 80)
                {
                    distance = futureObjectPosition.Entfernung_XY(ShipPosition);
                    shot_distance = Math.Abs(distance - (shot_speed * frameCounter));
                    if (shot_distance < Mittlerefluggeschwindigkeit + shot_speed)
                    {
                        returnDrehung = CDrehung.getDrehung(CDrehung.berechneWinkel(ShipPosition, futureObjectPosition));
                    }
                    frameCounter++;
                    futureObjectPosition = Berechneposition(frameCounter);
                }
                return returnDrehung;
            }
        }
        public CDrehung HitDrehung_old
        {
            get
            {
                if (!Saucer_Present) return null;
                CDrehung returnDrehung = null;
                double shot_speed = 8, distance, shot_distance, min_dist = 0x1FFFFF;
                int frameCounter = 1; //, deltaframe = frameCounter / 2; Annherung mach ich spter
                CPunkt ShipPosition = CTargets.Ship.Position;
                CPunkt futureObjectPosition = Berechneposition(frameCounter);
                if (futureObjectPosition == null) return null;
                while (returnDrehung == null && frameCounter < 80)
                {
                    distance = futureObjectPosition.Entfernung_XY(ShipPosition);
                    shot_distance = Math.Abs(distance - (shot_speed * frameCounter));
                    if (shot_distance < Mittlerefluggeschwindigkeit + shot_speed)
                    {
                        returnDrehung = CDrehung.getDrehung(CDrehung.berechneWinkel(ShipPosition, futureObjectPosition));
                    }
                    frameCounter++;
                    futureObjectPosition = Berechneposition(frameCounter);
                }
                return returnDrehung;
            }
        }
    }
  public class CShip : CFlugobject
  {
    public static int Shipradius = 20;
    public CKreis Shipgebiet
    {
      get 
      {
        return new CKreis(Position, Shipradius); 
      }
    }
    public CDrehung Drehung;
    public void Move ( ref KeysPacket keys, CPunkt Zielpunkt )
    {
      if (Aktuelleposition == null) return;
      double Beschleunigungsrate;
      int Zielgenauigkeit = 0;
      double distance = Aktuelleposition.Entfernung_XY(Zielpunkt);
      if (distance > 20)
      {
        CDrehung zielDrehung = CDrehung.getDrehung(Aktuelleposition, Zielpunkt);
        int delta_drehung = zielDrehung - Drehung;
        Zielgenauigkeit = 5 - (int)(20 / Math.Sqrt(distance) + 1);
        if ( Zielgenauigkeit < 0 ) Zielgenauigkeit = 0;
        if (Math.Abs(delta_drehung) <= Zielgenauigkeit)
        {
          if (distance > 196)
          {
            Beschleunigungsrate = 0.8;
          }
          else
          {
            Beschleunigungsrate = Math.Sqrt(distance) / 20 * 0.8;
          }
          if (Fluggeschwindigkeit < Beschleunigungsrate) keys.thrust(true);
        }
      }
    }
  }
  public class CFlugobject :IComparable
  {
    #region PRIVATE VARIABLES
    private int cls_counter = 0;
    private CPunkt cls_position = new CPunkt(0,0);
    private CPunkt NullPunkt = new CPunkt(0,0);
    private CKreis Kreisgebiet = new CKreis(0,0,0);
    private ArrayList Punkteliste = new ArrayList();
    #endregion
    #region PUBLIC VARIABLES
    public int Serialnummer;
    public int Parentnummer;
    public int ParentInsertFrame = -1;
    public int Nummer; //benutzt von ICompareable->CompareTo()
    public int AktuellerFrame;
    public CGerade Flugbahn;
    public double Flugdistanz_aktuell; // 
    public double Flugdistanz_gesamt; // 
    public double Fluggeschwindigkeit; // 
    public double Mittlerefluggeschwindigkeit; // 
    public double Flugwinkel; // 0 auf 12:00 Uhr
    public int Flugzeit; // in Frames
    public int letzte_Flugzeit;
    private int Flugzeitcounter = 0;
    public CPunkt Aktuelleposition = null;
    public CPunkt Letzteposition = null;
    public CPunkt Startposition = null;
    public bool Positiongesetzt = false;
    public ArrayList FuturePositions = new ArrayList(128);
    #endregion
    public CFlugobject()
    {

    }
    public void SetPosition ( int x_new, int y_new, int frameNumber )
    {
      CPunkt newPosition = new CPunkt(x_new, y_new);
      CPunkt Korrekturpunkt;
      #region Startposition
      if (Startposition == null)
      {
        Startposition = newPosition;
        AktuellerFrame = frameNumber;
        letzte_Flugzeit = 0;
        Flugzeit = 0;
      }
      #endregion
      #region Letzteposition
      else if (Letzteposition == null)
      {
        Letzteposition = newPosition;
        if (Letzteposition.Entfernung_XY(Startposition) != Letzteposition.Entfernung(Startposition))
        {
          Korrekturpunkt = Letzteposition.KorrektXY(ref Startposition);
          Startposition += Korrekturpunkt;
        }
        letzte_Flugzeit = frameNumber - AktuellerFrame;
        if (letzte_Flugzeit < 0) letzte_Flugzeit += 255;
        Flugzeit += letzte_Flugzeit;
        AktuellerFrame = frameNumber;
        //Flugdistanz_aktuell = Letzteposition.Entfernung(Startposition);
        Flugdistanz_aktuell = Letzteposition.Entfernung(Startposition);
        Flugdistanz_gesamt = Flugdistanz_aktuell;
        Fluggeschwindigkeit = Flugdistanz_aktuell / letzte_Flugzeit;
        Mittlerefluggeschwindigkeit = Fluggeschwindigkeit;
      }
      #endregion
      #region Aktuelleposition
      else if (Aktuelleposition == null)
      {
        Aktuelleposition = new CPunkt(x_new, y_new);
        if (Aktuelleposition.Entfernung_XY(Letzteposition) != Aktuelleposition.Entfernung(Letzteposition))
        {
          Korrekturpunkt = Aktuelleposition.KorrektXY(ref Letzteposition);
          Letzteposition += Korrekturpunkt;
          Startposition += Korrekturpunkt;
        }
        letzte_Flugzeit = frameNumber - AktuellerFrame;
        if (letzte_Flugzeit < 0) letzte_Flugzeit += 255;
        Flugzeit += letzte_Flugzeit;
        AktuellerFrame = frameNumber;
        //Flugdistanz_aktuell = Aktuelleposition.Entfernung(Letzteposition);
        Flugdistanz_aktuell = Aktuelleposition.Entfernung(Letzteposition);
        Flugdistanz_gesamt += Flugdistanz_aktuell;
        Fluggeschwindigkeit = Flugdistanz_aktuell / letzte_Flugzeit;
        //Mittlerefluggeschwindigkeit = Flugdistanz_gesamt / Flugzeit;
        Mittlerefluggeschwindigkeit = Fluggeschwindigkeit;
      }
      #endregion
      #region ELSE
      else
      {
        Aktuelleposition.CopyTo(ref Letzteposition);
        Aktuelleposition.x = x_new;
        Aktuelleposition.y = y_new;
        Korrekturpunkt = Aktuelleposition.KorrektXY(ref Letzteposition);
        if (Korrekturpunkt != NullPunkt)
        {
          Letzteposition += Korrekturpunkt;
          Startposition += Korrekturpunkt;
        }
        letzte_Flugzeit = frameNumber - AktuellerFrame;
        if (letzte_Flugzeit < 0) letzte_Flugzeit += 256;
        if (letzte_Flugzeit == 0)
        {
          Console.WriteLine("Flugzeit = 0");
        }
        Flugzeit += letzte_Flugzeit;
        AktuellerFrame = frameNumber;
        //Flugdistanz_aktuell = Aktuelleposition.Entfernung(Letzteposition);
        //Flugdistanz_gesamt = Aktuelleposition.Entfernung(Startposition); ;
        Flugdistanz_aktuell = Aktuelleposition.Entfernung(Letzteposition);
        Flugdistanz_gesamt = Aktuelleposition.Entfernung(Startposition); ;
        Fluggeschwindigkeit = Flugdistanz_aktuell / letzte_Flugzeit;
        if (Fluggeschwindigkeit > 30)
        {
          Fluggeschwindigkeit += 0;
        }
        //Mittlerefluggeschwindigkeit = Flugdistanz_gesamt / Flugzeit;
        if (Flugzeitcounter == 0)
        {
          Mittlerefluggeschwindigkeit = Fluggeschwindigkeit;
          Flugzeitcounter = 2;
        }
        else
        {
          Mittlerefluggeschwindigkeit += Fluggeschwindigkeit;
          Mittlerefluggeschwindigkeit /= Flugzeitcounter;
          Flugzeitcounter++;
        }
        Mittlerefluggeschwindigkeit = Flugdistanz_gesamt / Flugzeit;
        if (Fluggeschwindigkeit > 0)
        {
          if (Flugbahn == null) Flugbahn = new CGerade(Startposition, Aktuelleposition);
          else
          {
            if ( Korrekturpunkt != null )
              Flugbahn.Endpunkt(Startposition, Aktuelleposition);
            else
              Flugbahn.Endpunkt(Aktuelleposition);
          }
        }
      }
      #endregion
      if (this is CShot)
      {
        ((CShot)this).setTarget();
      }
      setFuturePositions();
      cls_counter++;
    }
    private void setFuturePositions()
    {
      CPunkt fPos;
      FuturePositions.Clear();
      if (Berechneposition(1) != null && Aktuelleposition != null)
      {
//        FuturePositions.Add(Position);
        for (int i = 1; i < 80; i++)
        {
          fPos = Berechneposition(i);
          FuturePositions.Add(new CPunkt(fPos.x,fPos.y));
//          TargetData.ConsoleWrite("{0} ", fPOs);
        }
      }
    }
    public CPunkt Berechneposition_new ( int delta_frame )
    {
      int delta_x, delta_y;
      CPunkt returnPosition = Startposition;
      CPunkt Position1, Position2;
      //if (cls_counter < 4) return null;
      if (Letzteposition == null) return returnPosition;
      if (Aktuelleposition == null)
      {
        Position1 = Startposition;
        Position2 = Letzteposition;
      }
      else
      {
        Position1 = Letzteposition;
        Position2 = Aktuelleposition;
      }
      if (letzte_Flugzeit == 0) return Position2;
      if (Fluggeschwindigkeit == 0) return Position2;
      delta_x = (int)((Position2.x - Position1.x) / letzte_Flugzeit);
      delta_y = (int)((Position2.y - Position1.y) / letzte_Flugzeit);
      delta_x = (int)(delta_x * delta_frame);
      delta_y = (int)(delta_y * delta_frame) ;
      if (delta_x > 500) delta_x -= 1024;
      else if (delta_x < -500) delta_x += 1023;
      if (delta_y > 500)
      {
        delta_y -= 768;
      }
      else if (delta_y < -500)
      {
        delta_y += 767;
      }
      cls_position.x = Position2.x + delta_x;
      cls_position.y = Position2.y + delta_y;
      if (cls_position.x > 1023) cls_position.x -= 1024;
      else if (cls_position.x < 0) cls_position.x += 1023;
      if (cls_position.y > 895)
      {
        cls_position.y -= 768;
      }
      else if (cls_position.y < 128) cls_position.y += 767;
      return cls_position;
    }
    public CPunkt Berechneposition ( int delta_frame )
    {
      int delta_x, delta_y;
      CPunkt returnPosition = Startposition;
      //if (cls_counter < 4) return null;
      if (Letzteposition == null) return returnPosition;
      if (Aktuelleposition == null)
      {
        returnPosition = Startposition * Letzteposition;
        return returnPosition;
      }
      if (Fluggeschwindigkeit == 0) return Aktuelleposition;
      if (letzte_Flugzeit == 0) return Aktuelleposition;
      delta_x = (int)(((Aktuelleposition.x - Letzteposition.x) * delta_frame) / letzte_Flugzeit);
      delta_y = (int)(((Aktuelleposition.y - Letzteposition.y) * delta_frame) / letzte_Flugzeit);
      if (delta_x > 500) delta_x -= 1024;
      else if (delta_x < -500) delta_x += 1023;
      if (delta_y > 500)
      {
        delta_y -= 768;
      }
      else if (delta_y < -500)
      {
        delta_y += 767;
      }
      cls_position.x = Aktuelleposition.x + delta_x;
      cls_position.y = Aktuelleposition.y + delta_y;
      if (cls_position.x > 1023) cls_position.x -= 1024;
      else if (cls_position.x < 0) cls_position.x += 1023;
      if (cls_position.y > 895)
      {
        cls_position.y -= 768;
      }
      else if (cls_position.y < 128) cls_position.y += 767;
      return cls_position;
    }
    public CPunkt TrifftGebiet(object gebietObject)
    {
      CPunkt returnPunkt = null;
      if (Flugbahn == null) return null;
      if (gebietObject is CKreis)
      {
        Kreisgebiet = gebietObject as CKreis;
        Punkteliste.Clear();
        Punkteliste = Flugbahn.Kreisschnittpunkt(Kreisgebiet);
        if ( Punkteliste.Count > 0 )
        {
          returnPunkt = Aktuelleposition.geringsteEntfernung(Punkteliste);
        }
      }
      return returnPunkt;
    }
    public CPunkt Position 
    {
      get
      {
        if (Aktuelleposition != null) return Aktuelleposition;
        if (Letzteposition != null) return Letzteposition;
        return Startposition;
      }
    }
    public int Counter { get { return cls_counter; } }
    #region INTERFACE ICOMPARABLE
    public int CompareTo ( object obj )
    {
      if (obj is CFlugobject)
      {
        CFlugobject flugobject = (CFlugobject)obj;

        return Nummer.CompareTo(flugobject.Nummer);
      }

      throw new ArgumentException("Objekt ist kein CFlugobject");
    }
    #endregion
  }
  #endregion

  #region GEOMETRISCHE OBJEKTE
  public class CGerade
  {
    private const double AUFLOESUNG_M = 0.0001;
    private double cls_m, cls_b;
    private CPunkt cls_start_punkt, cls_ende_punkt;
    #region CONSTRUCTORS (OK)
    public CGerade (CPunkt start_punkt, CPunkt ende_punkt)
    {
      cls_start_punkt = start_punkt;
      cls_ende_punkt = ende_punkt;
      setGerade();
    }
    public CGerade ( CPunkt start_punkt, double x, double y )
    {
      cls_start_punkt = start_punkt;
      cls_ende_punkt = new CPunkt(x, y);
      setGerade();
    }
    public CGerade ( double x, double y, CPunkt ende_punkt )
    {
      cls_start_punkt = new CPunkt(x, y);
      cls_ende_punkt = ende_punkt;
      setGerade();
    }
    public CGerade ( double x_start, double y_start, double x_ende, double y_ende )
    {
      cls_start_punkt = new CPunkt(x_start, y_start);
      cls_ende_punkt = new CPunkt(x_ende, y_ende);
      setGerade();
    }

    public CGerade ( double x_koordinate, double y_koordinate, double m_value )
    {
      cls_start_punkt = new CPunkt(x_koordinate, y_koordinate);
      setGerade(m_value);
    }
    public CGerade ( double m_value, double b_value )
    {
      double help_dbl = 500.0;
      cls_b = b_value;
      cls_m = Trim_M(m_value);
      cls_start_punkt = new CPunkt(0.0, cls_b);
      if (Math.Abs(cls_m) < 1)
      {
        cls_ende_punkt = new CPunkt(help_dbl, (help_dbl * cls_m + cls_b));
      }
      else
      {
        cls_ende_punkt = new CPunkt(((help_dbl - cls_b) / cls_m), help_dbl);
      }
    }
    #endregion
    #region PRIVATE FUNCTIONS
    private void setGerade (double m_value)
    {
      double help_dbl;
      cls_m = Trim_M(m_value);
      cls_b = cls_start_punkt.y_double - (cls_m * cls_start_punkt.x_double);
      if (Math.Abs(cls_m) < 1)
      {
        help_dbl = cls_start_punkt.x_double + 500;
        cls_ende_punkt = new CPunkt(help_dbl, (help_dbl * cls_m + cls_b));
      }
      else
      {
        help_dbl = cls_start_punkt.y_double + 500;
        cls_ende_punkt = new CPunkt(((help_dbl - cls_b) / cls_m), help_dbl);
      }
    }
    private void setGerade()
    {
      double delta_x, delta_y;
      delta_x = cls_ende_punkt.x_double - cls_start_punkt.x_double;
      delta_y = cls_ende_punkt.y_double - cls_start_punkt.y_double;
      if (delta_x == 0) delta_x = AUFLOESUNG_M;
      if (delta_y == 0) delta_y = AUFLOESUNG_M;
      cls_m = Trim_M(delta_y / delta_x);
      cls_b = cls_ende_punkt.y_double - (cls_m * cls_ende_punkt.x_double);
    }
    private double Trim_M ( double m_value ) //(OK)
    {
      double help_dbl = m_value;
      if (help_dbl == 0)
      {
        help_dbl = AUFLOESUNG_M; // m = 0.0001
      }
      else if (help_dbl < 0)
      {
        if (help_dbl < -(1 / AUFLOESUNG_M)) help_dbl = -(1 / AUFLOESUNG_M); // m = -10000
        else if (help_dbl > -AUFLOESUNG_M) help_dbl = -AUFLOESUNG_M; // m = -0.0001
      }
      else
      {
        if (help_dbl > (1 / AUFLOESUNG_M)) help_dbl = 1 / AUFLOESUNG_M; // m = 10000
        else if (help_dbl < AUFLOESUNG_M) help_dbl = AUFLOESUNG_M; // m = 0.0001
      }
      return help_dbl;
    }
    #endregion
    #region PUBLIC FUNCTIONS
    #region F(x) und F(y)
    public int Fx ( int x_value ){ return Fx((double) x_value); }
    public int Fx ( CPunkt referenzPunkt ) { return Fx(referenzPunkt.x_double); }
    public int Fx ( double x_value )
    {
      return (int) Fx_double(x_value);
    }
    public double Fx_double ( int x_value ) { return Fx_double((double)x_value); }
    public double Fx_double ( CPunkt referenzPunkt ) { return Fx_double(referenzPunkt.x_double); }
    public double Fx_double ( double x_value )
    {
      double result;
      result = cls_m * x_value + cls_b; 
      return result;
    }
    public int Fy ( int y_value ){ return Fy((double) y_value); }
    public int Fy ( CPunkt referenzPunkt ) { return Fy(referenzPunkt.y_double); }
    public int Fy ( double y_value )
    {
      return (int) Fy_double(y_value);
    }
    public double Fy_double ( int y_value ) { return Fy_double((double)y_value); }
    public double Fy_double ( CPunkt referenzPunkt ) { return Fy_double(referenzPunkt.y_double); }
    public double Fy_double ( double y_value )
    {
      double result;
      result = (y_value + cls_b) / m; 
      return result;
    }
    #endregion
    public CPunkt Schnittpunkt(CGerade referenzGerade)
    {
      double x_dbl, y_dbl, delta_b, delta_m;
      delta_b = this.cls_b - referenzGerade.b;
      delta_m = referenzGerade.m - this.cls_m;
      if (delta_m == 0) return null; //Geraden sind parallel -> kein Schnittpunkt
      if (delta_b == 0) //Schnittpunkt bei x=0
      {
        x_dbl = 0;
        y_dbl = this.cls_b;
      }
      else
      {
        x_dbl = delta_b / delta_m;
        y_dbl = Fx_double(x_dbl);
      }
      return new CPunkt(x_dbl, y_dbl);
    }
    public ArrayList Kreisschnittpunkt (CKreis referenzKreis)
    {
      ArrayList returnList = new ArrayList();
      CPunkt punkt1, punkt2;
      double help_x, help_y;
      double b0 = Fx_double(referenzKreis.Mittelpunkt) - referenzKreis.Mittelpunkt.y_double;
      double b0_quad = b0 * b0;
      double m1 = 1 + cls_m * cls_m;
      double r_quad = referenzKreis.Radius_double * referenzKreis.Radius_double;
      double diskriminante = r_quad * m1 - b0_quad;
      if (diskriminante == 0)
      {
        help_x = -(cls_m * b0) / m1;
        help_y = b0 / m1;
        punkt1 = new CPunkt(help_x + referenzKreis.Mittelpunkt.x_double, help_y + referenzKreis.Mittelpunkt.y_double);
        returnList.Add(punkt1);
      }
      else if (diskriminante > 0)
      {
        diskriminante = Math.Sqrt(diskriminante);
        help_x = -(cls_m * b0 + diskriminante) / m1;
        help_y = (b0 + cls_m * diskriminante) / m1;
        punkt1 = new CPunkt(help_x + referenzKreis.Mittelpunkt.x_double, help_y + referenzKreis.Mittelpunkt.y_double);
        help_x = -(cls_m * b0 - diskriminante) / m1;
        help_y = (b0 - cls_m * diskriminante) / m1;
        punkt2 = new CPunkt(help_x + referenzKreis.Mittelpunkt.x_double, help_y + referenzKreis.Mittelpunkt.y_double);
        returnList.Add(punkt1);
        returnList.Add(punkt2);
      }
      return returnList;
    }
    public void Endpunkt ( int x, int y )
    {
      cls_ende_punkt.x = x;
      cls_ende_punkt.y = y;
      setGerade();
    }
    public void Endpunkt ( double x, double y )
    {
      cls_ende_punkt.x_double = x;
      cls_ende_punkt.y_double = y;
      setGerade();
    }
    public void Endpunkt ( CPunkt endpunkt )
    {
      cls_ende_punkt = endpunkt;
      setGerade();
    }
    public void Endpunkt ( CPunkt startpunkt, CPunkt endpunkt )
    {
      cls_start_punkt = startpunkt;
      cls_ende_punkt = endpunkt;
      setGerade();
    }
    #endregion
    #region Properties
    public double m
    { //(OK)
      get { return Trim_M(cls_m); }
      set
      {
        setGerade(value);
      }
    }
    public double m_senkrechte
    { //(OK)
      get
      {
        double help_dbl = Trim_M(cls_m);
        help_dbl = -(1 / help_dbl);
        return help_dbl; 
      }
    }
    public double b
    { // (OK)
      get { return cls_b; }
      set //verschiebt start und ende Koordinaten
      {
        double delta_b = cls_b - value;
        cls_b = value;
        cls_start_punkt.y_double -= delta_b;
        cls_ende_punkt.y_double -= delta_b;
      }
    }
    #endregion
    #region OVERRIDE
    public override string ToString ()
    {
      return string.Format("m={0:#.000} b={1:#.0}",m,b);
    }
    #endregion
  }
  public class CKreis
  {
    CPunkt cls_mittelpunkt;
    int cls_radius;
    #region CONSTRUCTORS
    public CKreis ( CPunkt mittelpunkt, int radius )
    {
      cls_radius = radius;
      cls_mittelpunkt = mittelpunkt;
    }
    public CKreis ( int x_koordinate, int y_koordinate, int radius )
    {
      cls_radius = radius;
      cls_mittelpunkt = new CPunkt(x_koordinate, y_koordinate);
    }
    public CKreis ( double x_koordinate, double y_koordinate, int radius )
    {
      cls_radius = radius;
      cls_mittelpunkt = new CPunkt(x_koordinate, y_koordinate);
    }
    #endregion
    #region PUBLIC FUNCTIONS
    #endregion
    #region PROPERTIES
    public CPunkt Mittelpunkt { get { return cls_mittelpunkt; } }
    public int Radius { get { return cls_radius; } set { cls_radius = value; } }
    public double Radius_double { get { return (double)cls_radius; } set { cls_radius = (int)value; } }
    #endregion
  }
  public class CPunkt
  {
    private int x_int, y_int;
    private double x_dbl, y_dbl;
    public CPunkt ()
    {
      x_int = y_int = 0;
      x_dbl = y_dbl = 0.0;
    }
    public CPunkt ( int x, int y )
    {
      x_int = x;
      y_int = y;
      x_dbl = (double)x;
      y_dbl = (double)y;
    }
    public CPunkt ( double x, double y )
    {
      x_int = (int)x;
      y_int = (int)y;
      x_dbl = x;
      y_dbl = y;
    }
    public CPunkt KorrektXY ( ref CPunkt referenzPunkt )
    {
      int dx, dy;
      CPunkt returnPunkt = new CPunkt(0,0);
      dx = this.x - referenzPunkt.x;
      dy = this.y - referenzPunkt.y;
      if (dx < -512)
      {
        returnPunkt.x = -1024;
      }
      else if (dx > 511)
      {
        returnPunkt.x = 1024;
      }
      if (dy < -384)
      {
        returnPunkt.y = -768;  
      }
      else if (dy > 383)
      {
        returnPunkt.y = 768;
      }
      return returnPunkt;
    }
    public double Entfernung ( CPunkt referenzpunkt )
    {
      double delta_x, delta_y;
      delta_x = this.x_dbl - referenzpunkt.x_double;
      delta_y = this.y_dbl - referenzpunkt.y_double;
      return Math.Sqrt(delta_x * delta_x + delta_y * delta_y);
    }
    public double Entfernung_XY ( CPunkt referenzpunkt )
    {
      double dx, dy;
      dx = this.x_dbl - referenzpunkt.x_double;
      while (dx < -512) dx += 1024; // dx normalisieren auf -512 ... 511
      while (dx > 511) dx -= 1024;
      dy = this.y_dbl - referenzpunkt.y_double;
      while (dy < -384) dy += 768;  // dy normalisieren auf -384 ... 383
      while (dy > 383) dy -= 768;
      return Math.Sqrt(dx * dx + dy * dy);
    }
    public double Entfernung_Quadrat ( CPunkt referenzpunkt )
    {
      int delta_x, delta_y;
      delta_x = this.x - referenzpunkt.x;
      delta_y = this.y - referenzpunkt.y;
      return (delta_x * delta_x + delta_y * delta_y);
    }
    public CPunkt geringsteEntfernung ( CPunkt punkt1, CPunkt punkt2 )
    {
      double dist1 = this.Entfernung(punkt1);
      double dist2 = this.Entfernung(punkt2);
      if (dist1 < dist2) return punkt1;
      else return punkt2;
    }
    public CPunkt geringsteEntfernung (ArrayList punkteliste)
    {
      CPunkt returnPunkt = new CPunkt(0, 0);
      double dist, min_dist = 0x1FFFFF;
      bool punkt_gefunden = false;
      foreach(CPunkt cPunkt in punkteliste)
      {
        dist = this.Entfernung(cPunkt);
        if (dist < min_dist)
        {
          punkt_gefunden = true;
          cPunkt.CopyTo(ref returnPunkt);
          min_dist = dist;
        }
      }
      if (!punkt_gefunden) return null;
      return returnPunkt;
    }
    public CPunkt Distanzpunkt ( CGerade referenzgerade )
    {
      CGerade G1 = new CGerade(x_dbl, y_dbl, referenzgerade.m_senkrechte);
      return G1.Schnittpunkt(referenzgerade);
    }
    public void CopyTo ( ref CPunkt referenzPunkt )
    {
      referenzPunkt.x_double = this.x_dbl;
      referenzPunkt.y_double = this.y_dbl;
    }
    #region berladene Operatoren
    public static CPunkt operator + ( CPunkt punkt1, CPunkt punkt2 )
    {
      return new CPunkt(punkt1.x + punkt2.x, punkt1.y + punkt2.y);
    }
    public static CPunkt operator * ( CPunkt punkt1, CPunkt punkt2 )
    {
      int delta_x, delta_y;
      delta_x = punkt2.x - punkt1.x;
      delta_y = punkt2.y - punkt1.y;
      return new CPunkt(punkt2.x + delta_x, punkt2.y + delta_y);
    }
    public static bool operator ==( CPunkt punkt1, CPunkt punkt2 )
    {
      object obj1 = punkt1;
      object obj2 = punkt2;
      if (obj1 == null && obj2 == null) return true;
      else if (obj2 == null) return false;
      else return (punkt1.x == punkt2.x && punkt1.y == punkt2.y);

    }
    public static bool operator !=( CPunkt punkt1, CPunkt punkt2 )
    {
      return !(punkt1 == punkt2);
    }
     public override string ToString ()
    {
      return string.Format("x={0:0000} y={1:0000}", this.x, this.y);
    }
    public override int GetHashCode ()  { return 0; }
    public override bool Equals ( object o ) { return true; }
    #endregion
    #region PROPERTIES
    public int x { get { return x_int; } set { x_int = value; x_dbl = (double)value; } }
    public int y { get { return y_int; } set { y_int = value; y_dbl = (double)value; } }
    public double x_double { get { return x_dbl; } set { x_dbl = value; x_int = (int)value; } }
    public double y_double { get { return y_dbl; } set { y_dbl = value; y_int = (int)value; } }
    #endregion
  }
  #endregion
}
