;;; -*- Mode: LISP; Syntax: COMMON-LISP; Package: WELTRAUMPUTZE; Base: 10 -*-

(in-package :weltraumputze)

;;; Scene description structures

(defclass thing ()
  ((x :accessor x :initarg :x)
   (y :accessor y :initarg :y)
   (dx :accessor dx :initarg :dx :initform nil)
   (dy :accessor dy :initarg :dy :initform nil)
   (history :accessor history :initform nil)
   (history-complete :accessor history-complete :initform nil)))

(defclass asteroid (thing)
  ((size :accessor size :initarg :size)
   (atype :accessor atype :initarg :atype)))

(defclass flying-saucer (thing)
  ((size :accessor size :initarg :size)))

(defclass ship (thing)
  ((gun-dx :accessor gun-dx :initarg :gun-dx)
   (gun-dy :accessor gun-dy :initarg :gun-dy)))

(defclass shot (thing)
  ((first-frame :accessor first-frame :initarg :first-frame)))

(defclass scene ()
  ((targets :accessor targets :initform nil)
   (ship :accessor ship :initform nil)
   (angle-byte :accessor angle-byte :initform nil) ; list of possible angle bytes
   (shots :accessor shots :initform nil)
   (frame-no :accessor frame-no :initarg :frame-no)))

(defgeneric dim (thing))

(defmethod dim ((thing asteroid))
  (if (= (slot-value thing 'size) 3) 36 ; big asteroid
      (if (= (slot-value thing 'size) 2) 18 ; medium sized
          9))) ; small one (scale factor 14)

(defmethod dim ((thing flying-saucer))
  (if (= (slot-value thing 'size) 2) 16 ; big flying saucer
      8)) ; small one (scale factor 14)

(defmethod dim ((thing ship))
  12)

(defmethod dim ((thing shot))
  0.5)

(defun create-scene-from-vram (vram frame-number)
  (if (not vram)
      (error "No vector ram was specified for scene initialization."))
  (if (not (and (= (aref vram 0) #x01) (or (= (aref vram 1) #xe0) (= (aref vram 1) #xe2))))
      (error "Unexpected begin of vram sequence."))
  (let ((s (make-instance 'scene :frame-no frame-number)))
    (labels
        ((& (int1 int2)
           (boole boole-and int1 int2))
         (read-word (vram pos)
           (boole boole-ior (aref vram pos) (ash (aref vram (1+ pos)) 8)))
         (asize (n)
           (cond
             ((= n 0) ; big asteroid
              3)
             ((= n 15) ; medium
              2)
             (t ; small
              1)))
         (flsize (n)
           (cond
             ((= n 15) ; big flying saucer
              2)
             (t
              1))))
      (let (vy vx vs v1x v1y (ship-detect nil))
        (do ((pos 2 (+ pos 2))) ((> pos 1023))
          (let* ((word0 (read-word vram pos))
                 (word1 (read-word vram (+ pos 2)))
                 (op (ash word0 -12)))
            (cond
              ((= op #xa) ; LABS
               (setf vy (& word0 #x3ff)
                     vx (& word1 #x3ff)
                     vs (ash word1 -12)
                     pos (+ pos 2)))
              ((= op #xb) ; HALT
               (return))
              ((= op #xc) ; JSRL
               (let ((address (& word0 #xfff)))
                 (cond
                   ((member address '(#x8f3 #x8ff #x90d #x91a))
                    (push (make-instance 'asteroid :x vx :y vy :size (asize vs) :atype address) (targets s)))
                   ((= address #x929)
                    (push (make-instance 'flying-saucer :x vx :y vy :size (flsize vs)) (targets s))))))
              ((= op #xd) ; RTSL
               (return))
              ((= op #xe) ; JMPL
               (return))
              ((= op #xf) ; SVEC
               )
              (t ; VCTR
               (let ((dy (if (zerop (& word0 #x400)) (& word0 #x3ff) (- (& word0 #x3ff))))
                     (dx (if (zerop (& word1 #x400)) (& word1 #x3ff) (- (& word1 #x3ff))))
                     (vz (ash word1 -12)))
                 (if (and (zerop dx) (zerop dy) (= vz 15))
                     (push (make-instance 'shot :x vx :y vy) (shots s)))
                 (if (and (= op 6) (= vz 12) (/= dx 0) (/= dy 0))
                     (if ship-detect
                         (setf (ship s)
                               (make-instance 'ship :x vx :y vy :gun-dx (- v1x dx) :gun-dy (- v1y dy)))
                         (setf v1x dx
                               v1y dy
                               ship-detect t))
                     (setf ship-detect nil))
                 (setf pos (+ pos 2))))
              ))
          )))
    s))

(defgeneric match-things (a b frame-distance))

(defmethod match-things ((a thing) (b thing) frame-distance)
  ; In general, two arbitrary things don't match
  nil)

(defmethod match-things ((a asteroid) (b asteroid) frame-distance)
  (and (eq (size a) (size b))
       (eq (atype a) (atype b))
       (if (dx b)
           (< (flat-world-distance (x a) (y a)
                                   (+ (x b) (* (dx b) frame-distance))
                                   (+ (y b) (* (dy b) frame-distance))) 2)
           (< (flat-world-distance (x a) (y a) (x b) (y b)) (* frame-distance 8)))))

(defmethod match-things ((a flying-saucer) (b flying-saucer) frame-distance)
  (and (eq (size a) (size b))
       (< (flat-world-distance (x a) (y a) (x b) (y b)) (* frame-distance 8))))

(defmethod match-things ((a shot) (b shot) frame-distance)
  (if (dx b)
      (< (flat-world-distance (x a) (y a)
                              (+ (x b) (* (dx b) frame-distance))
                              (+ (y b) (* (dy b) frame-distance))) 2)
      (< (flat-world-distance (x a) (y a) (x b) (y b)) (* frame-distance 20))))

;; Synchronize targets and shots between two scenes, carrying over their history
(defun continue-scene-sequence (new-scene old-scene)
  (labels ((match-things-with-frame-distance (a b)
             (match-things a b (- (frame-no new-scene) (frame-no old-scene))))
           (pass-on-history-with-frame-no (a b)
             (pass-on-history a b (frame-no new-scene))))
    (synchronize-lists (targets new-scene) (targets old-scene)
                       #'match-things-with-frame-distance #'pass-on-history-with-frame-no)
    (synchronize-lists (shots new-scene) (shots old-scene)
                       #'match-things-with-frame-distance #'pass-on-history-with-frame-no)
    (if (and (ship new-scene) (ship old-scene))
        (pass-on-history-with-frame-no (ship new-scene) (ship old-scene)))))

(defun initialize-new-things (new-scene)
    ; Start a history for all new things
    (dolist (n (append (targets new-scene) (shots new-scene)))
      (if (and (not (history-complete n)) (null (history n)))
          (progn
            (setf (history n) (list (list (x n) (y n) (frame-no new-scene))))
            (when (eq (type-of n) 'shot)
              (setf (first-frame n) (frame-no new-scene))
              (let* ((ship (ship new-scene))
                     (distance (and ship (flat-world-distance (x ship) (y ship) (x n) (y n)))))
                     (when (and distance (> distance 18) (< distance 22))
                         (setf (dx n) (+ (* (/ (xwrap- (x n) (x ship)) distance) 8) (or (dx ship) 0)))
                         (setf (dy n) (+ (* (/ (ywrap- (y n) (y ship)) distance) 8) (or (dy ship) 0))))))
            ))))

(defun adjust-angle-byte (scene last-scene last-command)
  (let ((visible-angle (if (ship scene) (list (gun-dx (ship scene)) (gun-dy (ship scene))) nil)))
;    (if visible-angle
;        (format t "Visible angle: (~D, ~D) last angle byte: ~A last command: ~B, frame diff: ~D~%"
;                (first visible-angle) (second visible-angle)
;                (and last-scene (angle-byte last-scene))
;                last-command (- (frame-no scene) (frame-no last-scene))))
    (if (or (null last-scene) (null (angle-byte last-scene)))
        (setf (angle-byte scene)
              (if visible-angle
                  (visible-angle-to-angle-bytes visible-angle)
                  nil))
        (if visible-angle
            (setf (angle-byte scene)
                  (weed-out-impossible-angle-bytes
                   (if (= 1 (- (frame-no scene) (frame-no last-scene)))
                       (trace-angle-byte (angle-byte last-scene) last-command)
                       (do ((angle-bytes (angle-byte last-scene))
                            (n (- (frame-no scene) (frame-no last-scene)) (1- n)))
                           ((> n 0) angle-bytes)
                         (trace-angle-byte angle-bytes nil)))
                   visible-angle))
            (setf (angle-byte scene) (angle-byte last-scene))))

    ; Safety check
    (if (and visible-angle (null (angle-byte scene)))
        (format t "adjust-angle-byte: no possible angle left! visible angle: (~D, ~D), angle byte: ~D~%" (car visible-angle) (cadr visible-angle) (angle-byte scene)))
    ))

(defgeneric pass-on-history (new old frame-no))

(defmethod pass-on-history ((new thing) (old thing) frame-no)

  ; If the old history was already complete, just copy the calculated movement
  (if (setf (history-complete new) (history-complete old))
      (progn
        (setf (dx new) (dx old)
              (dy new) (dy old))
        (return-from pass-on-history)))

  ; History of new thing is the old history amended by the current position
  (setf (history new) (cons
                       (list (x new) (y new) frame-no)
                       (subseq (history old) 0 (min (length (history old)) 8))))

  ; As long as we have at least two history entries, we can calculate movement
  (if (> (length (history new)) 1)
      (progn
        ; Adjust history in case of a screen edge case
        (let ((xdiff (- (nth 0 (car (history new))) (nth 0 (cadr (history new)))))
              (ydiff (- (nth 1 (car (history new))) (nth 1 (cadr (history new))))))
          (if (or (> (abs xdiff) 1000) (> (abs ydiff) 750))
              (dolist (entry (cdr (history new)))
                (cond
                  ((> xdiff 1000) (incf (nth 0 entry) 1024))
                  ((< xdiff -1000) (decf (nth 0 entry) 1024)))
                (cond
                  ((> ydiff 750) (incf (nth 1 entry) 768))
                  ((< ydiff -750) (decf (nth 1 entry) 768))))))
        ; Calculate movement (dx, dy) based on history
        (let* ((first-entry (car (last (history new))))
               (frame-count (- frame-no (nth 2 first-entry))))
          (setf (dx new) (/ (- (x new) (nth 0 first-entry)) frame-count))
          (setf (dy new) (/ (- (y new) (nth 1 first-entry)) frame-count))
          ; For anything but a flying saucer, the movement calculated with a
          ; frame distance of 8 never changes
          (unless (or (/= frame-count 8) (member (type-of new) '(ship flying-saucer)))
            (setf (history-complete new) t))))

      ; Without two history entries to compare, set movement to nil for unknown
      (progn
        (setf (dx new) nil)
        (setf (dy new) nil)))
)

;; Traces the age of shots
(defmethod pass-on-history :after ((new shot) (old shot) frame-no)
  (setf (first-frame new) (first-frame old)))

(defun copy-scene (orig-scene)
  (let ((scene (make-instance 'scene)))
    (setf (targets scene) (mapcar #'copy-thing (targets orig-scene)))
    (setf (ship scene) (if (ship orig-scene) (copy-thing (ship orig-scene)) nil))
    (setf (angle-byte scene) (angle-byte orig-scene))
    (setf (shots scene) (mapcar #'copy-thing (shots orig-scene)))
    (setf (frame-no scene) (frame-no orig-scene))
    scene))

(defun copy-thing (orig-thing)
  (let ((thing (make-instance (type-of orig-thing))))
    (setf (x thing) (x orig-thing)
          (y thing) (y orig-thing)
          (dx thing) (or (dx orig-thing) 0)
          (dy thing) (or (dy orig-thing) 0))
    (cond
      ((eq (type-of thing) 'asteroid)
       (setf (size thing) (size orig-thing)
             (atype thing) (atype orig-thing)))
      ((eq (type-of thing) 'flying-saucer)
       (setf (size thing) (size orig-thing)))
      ((eq (type-of thing) 'ship)
       )
      ((eq (type-of thing) 'shot)
       (setf (first-frame thing) (first-frame orig-thing))))
    thing))

(defun update-scene (scene command previous-command)
  (labels ((xwrap+ (x dx)
             (let ((nx (+ x dx)))
               (cond
                 ((> nx 1023) (- nx 1024))
                 ((< nx 0) (+ nx 1024))
                 (t nx))))
           (ywrap+ (y dy)
             (let ((ny (+ y dy)))
               (cond
                 ((> ny 895) (- ny 768))
                 ((< ny 128) (+ ny 768))
                 (t ny))))
           (move-on (thing)
             (setf (x thing) (xwrap+ (x thing) (dx thing)))
             (setf (y thing) (ywrap+ (y thing) (dy thing))))
           (hit (thing shots)
             (dolist (shot shots)
               (if (< (flat-world-distance (x thing) (y thing) (x shot) (y shot)) (dim thing))
                   (return t))))
           (create-shot (scene)
             (multiple-value-bind (gx gy) (angle-byte-to-gun (angle-byte scene))
               (let ((shot-x (+ (* gx -5/8) (x (ship scene))))
                     (shot-y (+ (* gy -5/8) (y (ship scene)))))
                 (make-instance 'shot :x shot-x :y shot-y
                                :dx (+ gx (or (dx (ship scene)) 0)) :dy (+ gy (or (dy (ship scene)) 0))
                                :first-frame (+ (frame-no scene) 2))))))
    (mapcar #'move-on (targets scene))
    (mapcar #'move-on (shots scene))
    (if (ship scene)
        (move-on (ship scene)))
    (setf (frame-no scene) (1+ (frame-no scene)))
    (setf (targets scene) (remove-if #'(lambda (thing) (hit thing (shots scene))) (targets scene)))
    (setf (angle-byte scene)
          (trace-angle-byte (angle-byte scene) previous-command))
    (if (and (ship scene) (fire command))
        (setf (shots scene) (push (create-shot scene) (shots scene))))
    (setf (shots scene) (remove-if #'(lambda (shot) (> (- (frame-no scene) (first-frame shot)) 70))
                                   (shots scene)))
    (if (hyperspace command)
        (setf (ship scene) nil))))
    

(defun create-centered-scene (scene)
  ;; Without a ship, we don't know where to center
  (if (null (ship scene))
      scene)

  ;; Create a clone of the supplied scene, corrected by ship position
  (let ((cscene (make-instance 'scene))
        (x (x (ship scene)))
        (y (y (ship scene)))
        (dx (dx (ship scene)))
        (dy (dy (ship scene))))
    (flet ((create-thing (from-thing)
             (create-centered-thing from-thing x y (or dx 0) (or dy 0))))
      (setf (targets cscene) (mapcar #'create-thing (targets scene)))
      (setf (ship cscene) (create-thing (ship scene)))
      (setf (angle-byte cscene) (angle-byte scene))
      (setf (shots cscene) (mapcar #'create-thing (shots scene)))
      (setf (frame-no cscene) (frame-no scene)))
    cscene))

(defun create-centered-thing (thing x y dx dy)
  (let ((nt (make-instance (type-of thing))))
    (multiple-value-bind (nx ny) (cliprect-correction (x thing) (y thing) x y)
      (setf (x nt) nx (y nt) ny (dx nt) (- (or (dx thing) 0) dx) (dy nt) (- (or (dy thing) 0) dy)))
    (cond
      ((eq (type-of thing) 'asteroid)
       (setf (size nt) (size thing) (atype nt) (atype thing)))
      ((eq (type-of thing) 'flying-saucer)
       (setf (size nt) (size thing)))
      ((eq (type-of thing) 'ship)
       )
      ((eq (type-of thing) 'shot)
       (setf (first-frame nt) (first-frame thing))))
    nt))

(defun dump-scene (scene)
  (format t "Dumping scene ~A, frame no ~D, angle byte ~A:~%"
          scene (frame-no scene) (angle-byte scene))
  (if (ship scene)
      (format t "Ship at (~D, ~D) dir (~D, ~D)~%"
              (x (ship scene)) (y (ship scene)) (dx (ship scene)) (dy (ship scene)))
      (format t "Ship not visible~%"))
  (dolist (target (targets scene))
    (format t "~A at (~D, ~D) dir (~F, ~F) size ~D~%"
            target (x target) (y target) (dx target) (dy target) (size target)))
  (dolist (shot (shots scene))
    (format t "~A at (~D, ~D) dir (~F, ~F) age ~D~%"
            shot (x shot) (y shot) (dx shot) (dy shot) (- (frame-no scene) (first-frame shot))))
  (format t "~%"))