package de.curdreinert.asteroids.net;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;

import de.curdreinert.asteroids.base.Log;

public class UdpConnector {

	private DatagramSocket socket;

	private byte sendData[] = "ctmame@0".getBytes();

	private InetAddress address;

	private byte sentPing = 0;

	private byte expectedFrame = 0;

	private int port;

	private List<PacketListener> listeners = new ArrayList<PacketListener>();

	public UdpConnector(String host, int port) throws IOException {
		address = InetAddress.getByName(host);
		socket = new DatagramSocket();
		this.port = port;
	}

	public void run() throws IOException {
		send(new Command());

		byte receivedData[] = new byte[1026];
		DatagramPacket receivePacket = new DatagramPacket(receivedData, 1026);

		while (true) {
			receivedData[0] = 0;
			socket.receive(receivePacket);
			if (receivedData[0] != 0) {
				notify(receivedData);
			}
		}
	}

	private void notify(byte[] data) throws IOException {
		byte frame = data[1024];
		byte ping = data[1025];
		if (ping != sentPing) {
			Log.debug("Packets out of order, received " + ping + ", last sent "
					+ sentPing);
			notifyPacketDisorder(ping, sentPing);
		}
		if (frame != expectedFrame) {
			Log.warn("Missed " + (frame - expectedFrame) + " frame(s).");
			notifyFrameLoss(frame, expectedFrame);
			expectedFrame = ++frame;
			return;
		}
		expectedFrame = ++frame;
		Command command = notifyDataReceived(data);
		if (command != null) {
			send(command);
		}
	}

	public void send(Command command) throws IOException {
		sendData[6] = command.toNet();
		sendData[7] = (byte) ++sentPing;
		socket.send(new DatagramPacket(sendData, 8, address, port));
	}

	public void addListener(PacketListener listener) {
		listeners.add(listener);
	}

	public void removeListener(PacketListener listener) {
		listeners.remove(listener);
	}

	private Command notifyDataReceived(byte[] data) {
		Command command = null;
		for (PacketListener listener : listeners) {
			Command current = listener.packetReceived(data);
			if (command == null) {
				command = current;
			}
		}
		return command;
	}

	private void notifyPacketDisorder(byte received, byte sent) {
		for (PacketListener listener : listeners) {
			listener.packetDisorder(received, sent);
		}
	}

	private void notifyFrameLoss(byte received, byte expected) {
		for (PacketListener listener : listeners) {
			listener.frameLoss(received, expected);
		}
	}

}
