/******************************************************************************
* matrix33.inl
* The definitions of the matrix33-class. 
******************************************************************************/

inline Matrix22::Matrix22() : VecMatrix(2,2)
{
  //values are set to zero by MATRIX-constructor
}

/*****************************************************************************/
inline Matrix22::Matrix22(const Matrix22& a) : VecMatrix(2,2)
{
  m[0][0] = a.m[0][0]; m[0][1] = a.m[0][1];
  m[1][0] = a.m[1][0]; m[1][1] = a.m[1][1];
}

/*****************************************************************************/
inline Matrix22::Matrix22(const Vector2& v1, const Vector2& v2) : VecMatrix(2,2)
{
  m[0][0] = v1.x; m[0][1] = v2.x;
  m[1][0] = v1.y; m[1][1] = v2.y;
}

/*****************************************************************************/
inline Matrix22::Matrix22(const VecMatrix& M) : VecMatrix(2,2) 
{
  if (M.getrow() < 2 || M.getcolumn() < 2)
    return; 
  for (int i = 0; i < 2; i++)
    for (int j = 0; j < 2; j++)
      m[i][j] = M.m[i][j]; 
}

/*****************************************************************************/
inline Matrix22::~Matrix22()
{		
}

/*****************************************************************************/
inline VecFloat Matrix22::determinant() const
{
  return  m[0][0] * m[1][1] - m[0][1] * m[1][0];
}

/*****************************************************************************/
inline void Matrix22::identity()
{
  m[0][0] = (VecFloat)1.0; m[0][1] = 0;
  m[1][0] = 0; m[1][1] = (VecFloat)1.0;
}

/*****************************************************************************/
inline bool Matrix22::inverse()
{
  VecFloat temp00 =  m[1][1];
  VecFloat temp01 = -m[1][0];
  VecFloat temp10 = -m[0][1];
  VecFloat temp11 =  m[0][0];
 
  VecFloat d = m[0][0] * m[1][1] - m[0][1] * m[1][0];

  // check if singular
  if(d == 0)
  {
    identity();
    return false;
  }

  d = (VecFloat)1.0 / d;

  m[0][0] = temp00 * d; m[0][1] = temp01 * d; 
  m[1][0] = temp10 * d; m[1][1] = temp11 * d; 

  return true;
}

/*****************************************************************************/
inline void Matrix22::multiply(const Matrix22& a1, const Matrix22& a2)
{
  VecFloat temp00 = a1.m[0][0] * a2.m[0][0] + a1.m[0][1] * a2.m[1][0];
  VecFloat temp01 = a1.m[0][0] * a2.m[0][1] + a1.m[0][1] * a2.m[1][1];
  VecFloat temp10 = a1.m[1][0] * a2.m[0][0] + a1.m[1][1] * a2.m[1][0];
  VecFloat temp11 = a1.m[1][0] * a2.m[0][1] + a1.m[1][1] * a2.m[1][1];
 
  m[0][0] = temp00; m[0][1] = temp01;
  m[1][0] = temp10; m[1][1] = temp11; 
}


/*****************************************************************************/
inline void Matrix22::multiplyTranspose(const Matrix22& a1, const Matrix22& a2)
{
  VecFloat temp00 = a1.m[0][0] * a2.m[0][0] + a1.m[0][1] * a2.m[0][1];
  VecFloat temp01 = a1.m[0][0] * a2.m[1][0] + a1.m[0][1] * a2.m[1][1];
  VecFloat temp10 = a1.m[1][0] * a2.m[0][0] + a1.m[1][1] * a2.m[0][1];
  VecFloat temp11 = a1.m[1][0] * a2.m[1][0] + a1.m[1][1] * a2.m[1][1];
  
  m[0][0] = temp00; m[0][1] = temp01; 
  m[1][0] = temp10; m[1][1] = temp11; 
}

/*****************************************************************************/
inline void Matrix22::multiplyTranspose(const Matrix22& a1, const Matrix22& a2, const Vector2& v)
{
  VecFloat temp00 = v.x * a1.m[0][0] * a2.m[0][0] + v.y * a1.m[0][1] * a2.m[0][1];
  VecFloat temp01 = v.x * a1.m[0][0] * a2.m[1][0] + v.y * a1.m[0][1] * a2.m[1][1];
  VecFloat temp10 = v.x * a1.m[1][0] * a2.m[0][0] + v.y * a1.m[1][1] * a2.m[0][1];
  VecFloat temp11 = v.x * a1.m[1][0] * a2.m[1][0] + v.y * a1.m[1][1] * a2.m[1][1];
 
  m[0][0] = temp00; m[0][1] = temp01;
  m[1][0] = temp10; m[1][1] = temp11; 
  
}

/*****************************************************************************/
inline void Matrix22::multiplyTranspose(const Vector2& v1, const Vector2& v2)
{
  m[0][0] = v1.x * v2.x; m[0][1] = v1.x * v2.y;
  m[1][0] = v1.y * v2.x; m[1][1] = v1.y * v2.y; 
}

/*****************************************************************************/
inline VecFloat Matrix22::norm() const
{
  return   m[0][0] * m[0][0] + m[0][1] * m[0][1] + m[1][0] * m[1][0] + m[1][1] * m[1][1];
}

/*****************************************************************************/
inline void Matrix22::transposeMultiply(const Matrix22& a1, const Matrix22& a2)
{
  VecFloat temp00 = a1.m[0][0] * a2.m[0][0] + a1.m[1][0] * a2.m[1][0];
  VecFloat temp01 = a1.m[0][0] * a2.m[0][1] + a1.m[1][0] * a2.m[1][1];
  VecFloat temp10 = a1.m[0][1] * a2.m[0][0] + a1.m[1][1] * a2.m[1][0];
  VecFloat temp11 = a1.m[0][1] * a2.m[0][1] + a1.m[1][1] * a2.m[1][1];
  
  m[0][0] = temp00; m[0][1] = temp01; 
  m[1][0] = temp10; m[1][1] = temp11; 
}

/*****************************************************************************/
inline void Matrix22::operator-=(const Matrix22& a)
{
  m[0][0] -= a.m[0][0]; m[0][1] -= a.m[0][1];
  m[1][0] -= a.m[1][0]; m[1][1] -= a.m[1][1]; 
}

/*****************************************************************************/
inline void Matrix22::operator+=(const Matrix22& a)
{
  m[0][0] += a.m[0][0]; m[0][1] += a.m[0][1]; 
  m[1][0] += a.m[1][0]; m[1][1] += a.m[1][1]; 
}

/*****************************************************************************/

inline Matrix22 Matrix22::operator*(const VecFloat d)
{
  Matrix22 temp; 
  temp.m[0][0] = m[0][0] * d; temp.m[0][1] = m[0][1] * d; 
  temp.m[1][0] = m[1][0] * d; temp.m[1][1] = m[1][1] * d; 
  return temp; 
}

/*****************************************************************************/

inline Matrix22 Matrix22::operator*(const Matrix22& a)
{
  Matrix22 temp; 
  temp.m[0][0] = m[0][0] * a.m[0][0] + m[0][1] * a.m[1][0];
  temp.m[0][1] = m[0][0] * a.m[0][1] + m[0][1] * a.m[1][1];
  temp.m[1][0] = m[1][0] * a.m[0][0] + m[1][1] * a.m[1][0];
  temp.m[1][1] = m[1][0] * a.m[0][1] + m[1][1] * a.m[1][1];

  return temp; 
}

/*****************************************************************************/
inline void Matrix22::operator*=(const Matrix22& a)
{
  VecFloat temp00 = m[0][0] * a.m[0][0] + m[0][1] * a.m[1][0];
  VecFloat temp01 = m[0][0] * a.m[0][1] + m[0][1] * a.m[1][1];
  VecFloat temp10 = m[1][0] * a.m[0][0] + m[1][1] * a.m[1][0];
  VecFloat temp11 = m[1][0] * a.m[0][1] + m[1][1] * a.m[1][1];

  m[0][0] = temp00; m[0][1] = temp01; 
  m[1][0] = temp10; m[1][1] = temp11;
}

/*****************************************************************************/
inline void Matrix22::operator*=(const VecFloat d)
{
  m[0][0] *= d; m[0][1] *= d;
  m[1][0] *= d; m[1][1] *= d;
}

/*****************************************************************************/
inline void Matrix22::operator/=(const VecFloat d)
{
  if (d == 0)
    return;
  m[0][0] /= d; m[1][0] /= d;
  m[1][0] /= d; m[1][1] /= d;
}

/*****************************************************************************/
inline Matrix22& Matrix22::operator=(const Matrix22& a)
{
  m[0][0] = a.m[0][0]; m[0][1] = a.m[0][1];
  m[1][0] = a.m[1][0]; m[1][1] = a.m[1][1]; 
  return (*this);
}

inline Matrix22& Matrix22::operator=(const VecMatrix& M)
{
  if (M.getcolumn() != 2 || M.getrow() != 2)
    return (*this);
  for (int i = 0; i < 2; i++)
    for (int j = 0; j < 2; j++)
      m[i][j] = M.m[i][j];
  return (*this); 
}



/**************************************************************************/
inline Vector2 Matrix22::operator*(const Vector2& v)
{
  Vector2 temp; 
  temp.x = m[0][0]*v[0] + m[0][1]*v[1];
  temp.y = m[1][0]*v[0] + m[1][1]*v[1];
  return temp; 
}	

/**************************************************************************/

inline Vector2 operator*(const Vector2& v, const Matrix22& A)
{
  Vector2 temp; 
  temp.x = v.x * A.m[0][0] + v.y * A.m[1][0]; 
  temp.y = v.x * A.m[0][1] + v.y * A.m[1][1]; 
  return temp; 
}

inline Matrix22 transpose(const Matrix22& M)
{
  Matrix22 temp; 
  for (int i = 0; i < 2; i++)
    for (int j = 0; j < 2; j++)
      temp.m[i][j] = M.m[j][i]; 
  return temp;
}

