// ----------------------------------------------------------------------------
// 'The Sniper'  -  c't Asteroids bot  -  Thorsten Denhard, 2008
// ----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <winsock2.h>

#include "player.h"
#include "helper.h"
#include "constants.h"
#include "batchhelper.h"

#define HIT_RADIUS_MAX    10.0f
#define CRIT_THRESH_T      1.1f
#define FLY_COUNT_WEIGHT   0.3f
#define CALIB_WAIT_COUNT   5
#define CALIB_CHECK_COUNT 20

static unsigned long int sDesctructID = 1;
static bool              sFiveMinFlag = false;
static bool              sTenMinFlag  = false;

Player::Player (SOCKET socket, ADDRESS serverIP, bool exitFive, bool exitTen, bool batch, bool verbose) : 
  Viewer                     (socket, serverIP, batch),
  mExitFive                  (exitFive),
  mExitTen                   (exitTen),
  mDestructActionFinishedFFN (0),
  mDestructActionID          (0),
  mDestructActionWasUrgent   (false),
  mHyperspaceFlag            (0),
  mLastZeroObjectsFrame      (0),
  mFirstFrame                (0),
  mVerbose                   (verbose),
  mCalibrationFlag           (0),
  mCalibrationAngleSum       (0.0),
  mCalibrationAngleCount     (0.0)
{
}

Player::~Player ()
{
}

void 
Player::updateAction ()
{  
  Viewer::updateAction ();

  bool didAction = false;  
  {
    didAction = didAction || this->checkShipNotPresent ();
    didAction = didAction || this->processCalibration  ();
    didAction = didAction || this->checkViewDirection  ();
    didAction = didAction || this->bailoutCheck        ();
    didAction = didAction || this->checkTargets        ();
  }
}

bool 
Player::getDestructInfo (int index, int maxF, int& rotateCountRet, int& waitCountRet, int& flyCountRet)
{
  // What must be done to destroy object at index?
  // - Rotation to (rough) alignment
  // - Possibly wait some frames to bring good alignment (allowed deviation depending 
  //   on object size)
  // - Shoot
  // - Shot flies some frames until it hits
  // - Note that object will move while we rotate/wait, plus wraparound issues. 

  rotateCountRet = -1;
  waitCountRet   = -1;
  flyCountRet    = -1;

  int i = 0;
  int k = 0;
  
  int type = mGame.mSceneObjects[index].mType;

  if (type != TYPE_ASTEROID_S &&
      type != TYPE_ASTEROID_M &&
      type != TYPE_ASTEROID_L &&
      type != TYPE_SAUCER)
  {
    return false;
  }
  
  double sx = double (mGame.mShipX);
  double sy = double (mGame.mShipY);

  double shipVX = double (mGame.mShipVelocityX);
  double shipVY = double (mGame.mShipVelocityY);

  int    bestRotateCount = 0;
  double bestT           = -1.0f;
  double bestTX          = 0.0f;
  double bestTY          = 0.0f;
 
  int maxI = TARGET_LOOKAHEAD-5;

  if (maxF != -1)
  {
    maxI = maxF;
  }
  
  if (maxI > TARGET_LOOKAHEAD-5)
  {
    maxI = TARGET_LOOKAHEAD-5;
  }  

  for (k=-1; k<=1; k+=2)
  {
    for (i=0; i<maxI; ++i)
    {
      if (true != mGame.mSceneObjects[index].mHasTarget[i])
      {
        continue;
      }

      unsigned char dirByte   = this->getLeftRightBalance () + k*i;        
      double        viewAngle = DEG_TO_RAD * Constants::sAngles[dirByte];
      double        viewX     = 0.0f;
      double        viewY     = 0.0f;

      vecForAngle (viewAngle, viewX, viewY);  

      {        
        double tx = double (mGame.mSceneObjects[index].mTargetX[i]);
        double ty = double (mGame.mSceneObjects[index].mTargetY[i]);
                
        double ux = double (mGame.mSceneObjects[index].mTargetX[i+5]);
        double uy = double (mGame.mSceneObjects[index].mTargetY[i+5]);
       
        // Use speed of target point if valid (wraparound can make it unusable),
        // else resort to object speed
        
        double ovx = (ux - tx) / 5.0f * GAME_FPS;
        double ovy = (uy - ty) / 5.0f * GAME_FPS;
        
        if (fabs (ovx) > 500.0 ||
            fabs (ovy) > 500.0)
        {
          ovx = double (mGame.mSceneObjects[index].mVX);
          ovy = double (mGame.mSceneObjects[index].mVY);
        }
               
        double vs = ovx*ovx + ovy*ovy;
        double vl = mySqrt (vs);
        
        if (vl <= 0.0f)
        {
          return false;
        }
        
        double relVX = ovx - shipVX;
        double relVY = ovy - shipVY;

        double currentT    = 0.0f;
        double currentS    = 0.0f;
        int    currentTest = testIntersection (tx, ty, relVX, relVY, sx, sy, 
                                               viewX, viewY, currentT, currentS);
        
        if (currentT <= 0.0f)
        {
          continue;
        }
        
        if (currentS < SHOT_SOURCE_OFF-5)
        {
          continue;
        }

        currentS -= SHOT_SOURCE_OFF;        
        currentS /= SHOT_SPEED;
                 
        double maxS = SHOT_DURATION;

        if (currentS >= maxS)
        {
          continue;
        }

        // Special: check if actual deviation is within the tolerance defined
        // by the object's radius (we may hit some way off because we would
        // hit the non-singular sized object anyway). This reduces great
        // waiting periods when objects move toward us between two allowed angles...

        double r = 0.8f * double (mGame.mSceneObjects[index].mRadius);

        if (r > HIT_RADIUS_MAX)
        {
          r = HIT_RADIUS_MAX;
        } 
          
        double relVS = relVX*relVX + relVY*relVY;
        double relVL = mySqrt (relVS);

        if (relVL > 0.0f)
        {
          double relVXN = relVX / relVL;
          double relVYN = relVY / relVL;

          double scalar = relVXN * viewX + relVYN * viewY;
          double angle  = myAcos (fabs (scalar));
          double sine   = sin (angle);

          if (sine != 0.0f)
          {        
            double lOff = r / sine;            
            double tOff = lOff / relVL;

            double newT = currentT - tOff;
            
            if (newT < 1e-10f)
            {
              newT = 1e-10f;
            }

            double oldRX = tx + currentT * relVX;
            double oldRY = ty + currentT * relVY;
            
            double newRX = tx + newT * relVX;
            double newRY = ty + newT * relVY;

            double oldVecX = oldRX - sx;
            double oldVecY = oldRY - sy;
            double oldVecS = oldVecX*oldVecX + oldVecY*oldVecY;
            double oldVecL = mySqrt (oldVecS);

            double newVecX = newRX - sx;
            double newVecY = newRY - sy;
            double newVecS = newVecX*newVecX + newVecY*newVecY;
            double newVecL = mySqrt (newVecS);
            
            if (oldVecL > 0.0f &&
                newVecL > 0.0f)
            {
              double oldVecXN = oldVecX / oldVecL;
              double oldVecYN = oldVecY / oldVecL;
              double newVecXN = newVecX / newVecL;
              double newVecYN = newVecY / newVecL;
              
              double oldToNewScalar = oldVecXN * newVecXN + oldVecYN * newVecYN;

              if (oldToNewScalar > 0.0f)
              {              
                currentT = newT;
              }
            }          
          }
        }

        // Check if best  
        
        if (bestT    < 0.0f ||
            currentT < bestT)
        {
          bestT           = currentT;
          bestRotateCount = i*k;
          bestTX          = tx;
          bestTY          = ty;
        }
      }
    }
  }

  if (bestT < 0.0f)
  {
    return false;
  }

  double dx = bestTX - sx;
  double dy = bestTY - sy;

  double ds = dx*dx + dy*dy;
  double dl = mySqrt (ds);

  rotateCountRet = bestRotateCount;
  waitCountRet   = int (floor (bestT * GAME_FPS + 0.5f));
  flyCountRet    = int (floor (dl / SHOT_SPEED * GAME_FPS + 0.5f));  
  
  return true;
}

double 
Player::getCollisionTime (int sceneObjectIndex, 
                          double posX, double posY, 
                          double voffX, double voffY,
                          double roff, double toff)
{
  if (sceneObjectIndex < 0 ||
      sceneObjectIndex >= mGame.mSceneObjectCount)
  {
    return MAX_COLL_T;
  }
  
  double shipR  = SHIP_RADIUS + roff;
  double objR   = double (mGame.mSceneObjects[sceneObjectIndex].mRadius);
      
  double ovx = double (mGame.mSceneObjects[sceneObjectIndex].mVX);
  double ovy = double (mGame.mSceneObjects[sceneObjectIndex].mVY);

  if (mGame.mSceneObjects[sceneObjectIndex].mType == TYPE_SHOT)
  {
    // Shots may not have zero speed
    if (ovx == 0.0f &&
        ovy == 0.0f)
    {
      return MAX_COLL_T;
    }
  }
  
  double objVX = ovx - voffX;
  double objVY = ovy - voffY;
  
  if (objVX == 0.0 &&
      objVY == 0.0)
  {
    return MAX_COLL_T;
  }
  
  double objX1 = mGame.mSceneObjects[sceneObjectIndex].mX;
  double objY1 = mGame.mSceneObjects[sceneObjectIndex].mY;
  
  double ct1 = ::getCollisionTime (objX1, objY1, objVX, objVY,
                                   posX,  posY,  objR,  shipR);  
  
  if (toff <= 0.0f)
  {
    if (mGame.mSceneObjects[sceneObjectIndex].mType == TYPE_SHOT)
    {
      // Account for limited lifetime of shots
      if (ct1 > SHOT_DURATION)
      {
        ct1 = MAX_COLL_T;
      }      
    }

    return ct1;
  }
  
  double objX2 = objX1 + toff * objVX;
  double objY2 = objY1 + toff * objVY;
  

  double ct2 = ::getCollisionTime (objX2, objY2, objVX, objVY,
                                   posX,  posY,  objR,  shipR);  
  
    
  if (ct2 > ct1)
  {
    // Collision time at toff is better than at tnow. That
    // means the object has (will have) passed by in this interval,
    // and its collision there can be ignored. Nonetheless, we return
    // a collisiontime counting from tnow (thats the definied semantic 
    // of the toff-behaviour)

    ct2 += toff;

    if (ct2 > MAX_COLL_T)
    {
      ct2 = MAX_COLL_T;
    }
    
    if (mGame.mSceneObjects[sceneObjectIndex].mType == TYPE_SHOT)
    {
      // Account for limited lifetime of shots
      if (ct2 > SHOT_DURATION)
      {
        ct2 = MAX_COLL_T;
      }      
    }
  
    return ct2;
  }

  if (mGame.mSceneObjects[sceneObjectIndex].mType == TYPE_SHOT)
  {
    // Account for limited lifetime of shots
    if (ct1 > SHOT_DURATION)
    {
      ct1 = MAX_COLL_T;
    }      
  }
  
  return ct1;  
}

int    
Player::getCollisionObject (double& collisionTimeRet,
                            double posX, double posY, 
                            double voffX, double voffY,
                            bool allowShots,
                            double roff, double toff)
{
  collisionTimeRet = 0.0f;

  int    i        = 0;  
  double minT     = MAX_COLL_T;
  int    minIndex = -1;
  
  for (i=0; i<mGame.mSceneObjectCount; ++i)
  {   
    int  type  = mGame.mSceneObjects[i].mType;
    
    if (true == allowShots)
    {
      if (type != TYPE_ASTEROID_S &&
          type != TYPE_ASTEROID_M &&
          type != TYPE_ASTEROID_L &&
          type != TYPE_SAUCER     &&
          type != TYPE_SHOT)
      {
        continue;
      }
    }
    else
    {    
      if (type != TYPE_ASTEROID_S &&
          type != TYPE_ASTEROID_M &&
          type != TYPE_ASTEROID_L &&
          type != TYPE_SAUCER)
      {
        continue;
      }
    }

    double nearestT = this->getCollisionTime (i, posX, posY, voffX, voffY, roff, toff);

    if (nearestT >= minT)
    {
      continue;
    }

    minT     = nearestT;
    minIndex = i;
  }

  collisionTimeRet = minT;
  return minIndex;
}

void 
Player::reset ()
{
  this->removeScheduledActions ();

  mDestructActionFinishedFFN = 0;
  mDestructActionID          = 0;
  mDestructActionWasUrgent   = false;
  mHyperspaceFlag            = 0;
  mLastZeroObjectsFrame      = 0;
  mCalibrationFlag           = 0;
  mCalibrationAngleSum       = 0.0;
  mCalibrationAngleCount     = 0.0;
}

bool
Player::processCalibration ()
{
  int i = 0;
  
  if (0 == mCalibrationFlag)
  {
    return false;
  }

  if (1 == mCalibrationFlag)
  {
    // Just beginning, reset data
    mCalibrationAngleCount  = 0.0f;
    mCalibrationAngleSum    = 0.0f;
    mCalibrationFlag       += 1;
    return true;
  }
  
  if (mCalibrationFlag <= CALIB_WAIT_COUNT)
  {
    // Still waiting
    mCalibrationFlag += 1;
    return true;
  }

  if (mCalibrationFlag >= CALIB_WAIT_COUNT + CALIB_CHECK_COUNT)
  {
    // Finished checking shot angles. Calculate best matching orientation if possible.
    
    if (mCalibrationAngleCount > 0.5)
    {
      double realAngle = mCalibrationAngleSum / mCalibrationAngleCount;      
      
      if (true == mVerbose)
      {
        printf ("> Calibration: samples %f, angle %f\n", mCalibrationAngleCount, realAngle / DEG_TO_RAD);
      }
      
      int realDX = mGame.mShipDX;
      int realDY = mGame.mShipDY;

      double bestDiff  = 1000.0f;
      int    bestIndex = -1;
      
      for (i=0; i<256; ++i)
      {
        if (realDX != Constants::sShipDX[i] ||
            realDY != Constants::sShipDY[i])
        {
          continue;
        }
        
        double currentTableAngle = DEG_TO_RAD * Constants::sAngles[i];
        double currentDiff       = fabs (currentTableAngle - realAngle);

        if (currentDiff > M_PI)
        {
          currentDiff = 2.0f * M_PI - currentDiff;
        }

        if (currentDiff < bestDiff)
        {
          bestDiff  = currentDiff;
          bestIndex = i;
        }
      }

      if (bestIndex != -1)
      {
        this->forceLeftRightBalance (bestIndex);
      }
    }

    // Calibration process finished
    mCalibrationFlag = 0;
    return true;
  }

  // Calibration in progress

  mCalibrationFlag += 1;
  
  for (i=0; i<mGame.mSceneObjectCount; ++i)
  {   
    if (mGame.mSceneObjects[i].mType != TYPE_SHOT)
    {
      continue;
    }

    double vx = mGame.mSceneObjects[i].mVX;
    double vy = mGame.mSceneObjects[i].mVY;

    double ox = mGame.mSceneObjects[i].mRelX;
    double oy = mGame.mSceneObjects[i].mRelY;      

    double sx = mGame.mShipX;
    double sy = mGame.mShipY;

    double dx = ox-sx;
    double dy = oy-sy;

    double ds = dx*dx + dy*dy;
    double dl = mySqrt (ds) - SHOT_SOURCE_OFF;

    if (dl < 0.0f)
    {
      continue;
    }

    double dThresh = double (mCalibrationFlag - CALIB_WAIT_COUNT + 1) / GAME_FPS * SHOT_SPEED;
   
    if (dl > dThresh)
    {
      continue;
    }

    mCalibrationAngleSum   += angleForVec (vx, vy);
    mCalibrationAngleCount += 1.0f;
  }
  
  return true;
}

bool
Player::checkViewDirection ()
{
  int  i                = 0;
  bool saucerInterfere  = (mLastSaucerFullFrameNumber + SHOT_DURATION_F + 2 >= mGame.mFullFrameNumber);

  unsigned char currentIndex = this->getLastLeftRightBalance ();
  
  int tableDX = Constants::sShipDX[currentIndex];
  int tableDY = Constants::sShipDY[currentIndex];
    
  if (mGame.mShipDX == tableDX &&
      mGame.mShipDY == tableDY)
  {
    // Not out of sync
    return false;
  }

  if (true == mVerbose)
  {
    printf ("! Frame %d: Out of sync (%d with %d %d)\n", 
        mGame.mFullFrameNumber, currentIndex, mGame.mShipDX, mGame.mShipDY);
  }

  if (true == saucerInterfere)
  {
    return false;
  }
 
  // Start calibration process: Wait a few frames to become "stable", 
  // schedule shot and calibration (see processCalibration)
  
  this->removeScheduledActions ();

  for (i=0; i<CALIB_WAIT_COUNT; ++i)
  {
    this->scheduleAction (KeysPacket::NOP  ());
  }
  
  this->scheduleAction (KeysPacket::FIRE ());
  
  mCalibrationFlag = 1;

  return true;
}

bool 
Player::checkShipNotPresent ()
{
  unsigned long int frame = mGame.mFullFrameNumber-mFirstFrame;
    
  if (frame >= 18000 &&
      true  != sFiveMinFlag)
  {
    sFiveMinFlag = true;

    printf ("> FIVE MINUTES!\n");

    if (true == mExitFive)
    {
      if (true == mBatch)
      {
        BatchHelper::saveScreenAndExit ();
      }
      else
      {
        exit (0);
      }
    }
  }
  
  if (frame >= 36000 &&
      true  != sTenMinFlag)
  {
    sTenMinFlag = true;

    printf ("> TEN MINUTES!\n");

    if (true == mExitTen)
    {
      if (true == mBatch)
      {
        BatchHelper::saveScreenAndExit ();
      }
      else
      {
        exit (0);
      }
    }
  }

  if (mGame.mSceneObjectCount < 1)
  {
    mLastZeroObjectsFrame = mGame.mFullFrameNumber;
  }
  
  if (mHyperspaceFlag > 0)
  {
    mHyperspaceFlag -= 1;
    return true;
  }

  if (true != mGame.mShipPresent)
  {
    this->reset ();
    return true;
  }

  if (mFirstFrame == 0)
  {
    mFirstFrame = mGame.mFullFrameNumber;
  }
  
  if (true == mVerbose)
  {  
    if (mFrameDelay > 0)
    {
      printf ("! Frame %d: delay %d\n", mGame.mFullFrameNumber, mFrameDelay);
    }
    
    if (mFrameLoss > 0)
    {
      printf ("! Frame %d: loss %d\n", mGame.mFullFrameNumber, mFrameLoss);
    }
  }

  return false;
}

bool 
Player::bailoutCheck ()
{
  // Note: aggresive tweak from 5.0 to 1.0 
  double critT    = 1.0f / GAME_FPS;      
  double nearestT = MAX_COLL_T;

  bool checkShots = (mLastSaucerFullFrameNumber + SHOT_DURATION_F + 2 >= mGame.mFullFrameNumber);
  
  this->getCollisionObject (nearestT,
                            mGame.mShipX, mGame.mShipY, 
                            mGame.mShipVelocityX, mGame.mShipVelocityY, checkShots);
      
  if (nearestT >  1e-4f &&
      nearestT <= critT)
  {
    // Bailout

    this->removeScheduledActions ();
    this->scheduleAction (KeysPacket::HYPERSPACE ());
    mHyperspaceFlag = 10;
    return true;
  }

  return false;
}

bool 
Player::checkTargets ()
{        
  int i = 0;
  int k = 0;

  // Hack for "minor action loss" issue, see networkengine
  
  if (mFrameDelay                > 0 &&
      mDestructActionFinishedFFN > 0)
  {
    mDestructActionFinishedFFN += 2;
  }
  
  // Prevent initial confusion at level start
  
  if (mGame.mFullFrameNumber < mLastZeroObjectsFrame + 5)
  {
    return false;
  }

  // Ammo counting

  int outOfAmmoFrames = 0;

  do
  {
    int shotCount = 0;
    {
      for (i=0; i<mGame.mSceneObjectCount; ++i)
      {   
        if (mGame.mSceneObjects[i].mType == TYPE_SHOT)
        {
          shotCount += 1;
        }
      }
    }

    // Less than four shots in the game -> ok
    
    if (shotCount < 4)
    {
      break;
    }

    outOfAmmoFrames = int (SHOT_DURATION_F);

    // outOfAmmoFrames may be reduced if at least one shot
    // hits a target earlier
    
    double minT = MAX_COLL_T;
    
    for (i=0; i<mGame.mSceneObjectCount; ++i)
    {   
      if (mGame.mSceneObjects[i].mType != TYPE_SHOT)
      {
        continue;
      }

      double shotX = mGame.mSceneObjects[i].mX;
      double shotY = mGame.mSceneObjects[i].mY;

      double shotVX = mGame.mSceneObjects[i].mVX;
      double shotVY = mGame.mSceneObjects[i].mVY;

      double collT = MAX_COLL_T;
      int collID = this->getCollisionObject (collT, shotX, shotY, shotVX, shotVY, 
                                             false, 1 - SHIP_RADIUS, 0.0f);
      
      if (collID != -1 &&
          collT  < minT)
      {
        minT = collT;
      }      
    }

    if (minT < SHOT_DURATION)
    {
      // Shot will have died in minT seconds
      outOfAmmoFrames = int (minT * GAME_FPS) + 1;
    }
  }
  while (false);
  
  // Check if tagged saucer significantly changed velocity
  // -> remove tag
  
  {    
    for (i=0; i<mGame.mSceneObjectCount; ++i)
    {         
      if (mGame.mSceneObjects[i].mDestructID == 0)
      {
        continue;
      }

      if (mGame.mSceneObjects[i].mType != TYPE_SAUCER)
      {
        continue;
      }
      
      double cvx = mGame.mSceneObjects[i].mVX;
      double cvy = mGame.mSceneObjects[i].mVY;

      double ovx = mGame.mSceneObjects[i].mDestructVX;
      double ovy = mGame.mSceneObjects[i].mDestructVY;

      double cvs = cvx*cvx + cvy*cvy;
      double cvl = mySqrt (cvs);

      double ovs = ovx*ovx + ovy*ovy;
      double ovl = mySqrt (ovs);

      if (cvl <= 0.0f ||
          ovl <= 0.0f)
      {
        continue;
      }
      
      cvx /= cvl;
      cvy /= cvl;

      ovx /= ovl;
      ovy /= ovl;

      double scalar = cvx*ovx + cvy*ovy;

      if (scalar < 0.75f)
      {
        mGame.mSceneObjects[i].mDestructID    = 0;
        mGame.mSceneObjects[i].mDestructFrame = 0;
        mGame.mSceneObjects[i].mDestructVX    = 0.0f;
        mGame.mSceneObjects[i].mDestructVY    = 0.0f;
      }      
    }
  }

  // Check if we lost our target for a running operation
  
  if (mGame.mFullFrameNumber < mDestructActionFinishedFFN &&
      mDestructActionID != -1)
  {
    unsigned long int id      = mDestructActionID;    
    bool              foundID = false;
    
    for (i=0; i<mGame.mSceneObjectCount; ++i)
    {         
      if (mGame.mSceneObjects[i].mDestructID == id)
      {
        foundID = true;
        break;
      }
    }

    if (true != foundID)
    {
      // Note: do not removeScheduledActions here! We often "lose"
      // large nearby asteroids, but the remaining shots of the
      // burst must be used to hit its children.
      mDestructActionFinishedFFN = 0;
    }
  }
  
  // Check for urgent targets (collision imminent)

  double nearestT  = MAX_COLL_T;
  int    nearestID = this->getCollisionObject (nearestT, mGame.mShipX, mGame.mShipY, 0.0f, 0.0f, false);
  int    nearestF  = int (floor (nearestT * GAME_FPS));
    
  if (nearestT > CRIT_THRESH_T)
  {
    nearestID = -1;
  }

  // Determine best target

  double bestCount        = 1000.0f;
  int    bestIndex        = -1;
  int    bestRotateCount  = 0;
  int    bestWaitCount    = 0;
  int    bestFlyCount     = 0;
  bool   urgent           = false;

  int onTheFlyRotates[50];

  for (i=0; i<50; ++i)
  {
    onTheFlyRotates[i] = 0;
  }

  int onTheFlyRotatesIndex = 0;
  
  for (i=0; i<mGame.mSceneObjectCount; ++i)
  {   
    int type = mGame.mSceneObjects[i].mType;
      
    // Is object already doomed? Then ignore.

    if (mGame.mSceneObjects[i].mDestructFrame > mGame.mFullFrameNumber)
    {      
      continue;
    }   
    
    // Check time needed to destroy object

    int rotateCount = 0;
    int waitCount   = 0;
    int flyCount    = 0;
    int maxF        = -1;

    if (i == nearestID)
    {
      // Tell getDestructInfo not to search in a future where 
      // we already have been destroyed
      maxF = nearestF;
    }
    
    if (true != this->getDestructInfo (i, maxF, rotateCount, waitCount, flyCount))
    {      
      continue;
    }

    // Store possible occasions for "on-the-fly" shooting (s.b.)
    
    if (waitCount            < 3 &&
        onTheFlyRotatesIndex < 50)
    {    
      onTheFlyRotates[onTheFlyRotatesIndex] = rotateCount;
      onTheFlyRotatesIndex += 1;
    }
    
    // Get current "quality"
    // Note: it seemed a good idea to include the shot's fly-time into
    // this quality. However, a full +=flyCount did not really improve things.
    // We use the totally arbitrary weight which produces good results IMO :-)
    // Note: subtract outOfAmmoFrames from rotation-wait-period. This is done 
    // because we don't effectively lose anything while we cannot shoot anyway,
    // and can concentrate more on the fly-count quality.
               
    double totalCount = 0.0f;
    double rotateWait = double (abs (rotateCount)) + 
                        double (waitCount) - 
                        double (outOfAmmoFrames);

    if (rotateWait < 0.0f)
    {
      rotateWait = 0.0f;
    }
    
    totalCount += rotateWait;
    totalCount += double (flyCount) * FLY_COUNT_WEIGHT;  
    
    // Saucers have relative priority
    
    if (type == TYPE_SAUCER)
    {
      totalCount *= 0.5f;
    }

    if (i == nearestID)
    {
      // Urgent targets have priority

      bestCount       = totalCount;
      bestIndex       = i;
      bestRotateCount = rotateCount;
      bestWaitCount   = waitCount;
      bestFlyCount    = flyCount;
      urgent          = true;
      break;
    }

    // Check if best

    if (totalCount < bestCount)
    {
      bestCount       = totalCount;
      bestIndex       = i;
      bestRotateCount = rotateCount;
      bestWaitCount   = waitCount;
      bestFlyCount    = flyCount;
    }
  }

  // Dont schedule new destruction as long as old action still in
  // progress. Exception: urgent emergency.
  
  if (mGame.mFullFrameNumber < mDestructActionFinishedFFN)
  {
    if (! (true == urgent &&
           true != mDestructActionWasUrgent))
    {
      return true;
    }
  }

  // Check if no target found  

  if (bestIndex < 0)
  {
    return false;
  }  
  
  // Check if no immediate target found
  // In an ideal, bugfree program, we could check for >0 or
  // dont need the check at all, so this is rather obscure...

  if (bestWaitCount >= 10)
  {
    return false;
  }
    
  // Check if out of ammo (ignored if urgent)
    
  bool outOfAmmo = (outOfAmmoFrames > bestWaitCount + abs (bestRotateCount) + 1);
  
  if (true != urgent &&
      true == outOfAmmo)
  {
    // Do some rotation nonetheless

    this->removeScheduledActions ();
    
    for (i=0; i<abs (bestRotateCount); ++i)
    {
      if (bestRotateCount < 0)
      {
        this->scheduleAction (KeysPacket::RIGHT ());
      }
      else
      {
        this->scheduleAction (KeysPacket::LEFT ());
      }
    }
    
    mDestructActionFinishedFFN = mGame.mFullFrameNumber + abs (bestRotateCount);
    
    return true;
  }
    
  // Schedule actions for destruction of target
 
  int commandCount = 0; 
  {
    // Everything else gets removed

    this->removeScheduledActions ();

    // Rotate
    // Note: Additional shooting as a desparate attempt to possibly gain some points
    //       While rotating towards an urgent asteroid (probably coming from behind), 
    //       we might come across angles where we will hit a target without waiting. 
    //       Then shoot while rotating.

    int  fireCount = 0;
    bool fireFlag  = false;
    
    for (i=0; i<abs (bestRotateCount); ++i)
    {
      if (bestRotateCount < 0)
      {
        bool alsoFire  = false;        
        bool allowFire = (fireCount < 2) && (i != 0) && (true != fireFlag) && (urgent);

        if (bestWaitCount == 0 &&
            i             == abs (bestRotateCount)-1)
        {
          allowFire = false;
        }
        
        if (allowFire)
        {
          for (k=0; k<50; ++k)
          {
            if (onTheFlyRotates[k] == -(i+0))
            {
              alsoFire            = true;
              fireCount          += 1;
              onTheFlyRotates[k]  = 0;
              break;
            }
          }
        }

        fireFlag = alsoFire;
        
        this->scheduleAction (KeysPacket::RIGHT (alsoFire));
        commandCount += 1;
      }
      else
      {
        bool alsoFire  = false;        
        bool allowFire = (fireCount < 2) && (i != 0) && (true != fireFlag) && (urgent);

        if (bestWaitCount == 0 &&
            i             == abs (bestRotateCount)-1)
        {
          allowFire = false;
        }
        
        if (allowFire)
        {
          for (k=0; k<50; ++k)
          {
            if (onTheFlyRotates[k] == (i+0))
            {
              alsoFire            = true;
              fireCount          += 1;
              onTheFlyRotates[k]  = 0;
              break;
            }
          }
        }
        
        fireFlag = alsoFire;
        
        this->scheduleAction (KeysPacket::LEFT (alsoFire));
        commandCount += 1;
      }
    }

    // Wait

    for (i=0; i<bestWaitCount; ++i)
    {
      this->scheduleAction (KeysPacket::NOP ());
      commandCount += 1;
    }

    // Fire
    
    this->scheduleAction (KeysPacket::FIRE ());
    commandCount += 1;

    // Larger asteroids get a full burst of shots

    bool forbidBurst = false;

    if (true != urgent)
    {
      if (nearestT * GAME_FPS < abs (bestRotateCount) + bestWaitCount + bestFlyCount + 5)
      {
        forbidBurst = true;
      }
    }
    
    if (true != forbidBurst)
    {
      int type = mGame.mSceneObjects[bestIndex].mType;

      if (type == TYPE_ASTEROID_L)
      {
        this->scheduleAction (KeysPacket::NOP ());
        this->scheduleAction (KeysPacket::FIRE ());
        this->scheduleAction (KeysPacket::NOP ());
        this->scheduleAction (KeysPacket::FIRE ());
        this->scheduleAction (KeysPacket::NOP ());
        this->scheduleAction (KeysPacket::FIRE ());
        commandCount += 6;
      }
      else if (type == TYPE_ASTEROID_M)
      {
        this->scheduleAction (KeysPacket::NOP ());
        this->scheduleAction (KeysPacket::FIRE ());
        this->scheduleAction (KeysPacket::NOP ());
        this->scheduleAction (KeysPacket::FIRE ());
        commandCount += 4;
      }
    }
  }

  // Tag object

  if (true != outOfAmmo)
  { 
    unsigned long int destructFrame = mGame.mFullFrameNumber + commandCount + bestFlyCount + 1;
   
    mGame.mSceneObjects[bestIndex].mDestructFrame = destructFrame;

    // Remember other data

    unsigned long int desctructID = sDesctructID;
    sDesctructID += 1;
    
    mGame.mSceneObjects[bestIndex].mDestructID = desctructID;
    mGame.mSceneObjects[bestIndex].mDestructVX = mGame.mSceneObjects[bestIndex].mVX;
    mGame.mSceneObjects[bestIndex].mDestructVY = mGame.mSceneObjects[bestIndex].mVY;
    
    mDestructActionFinishedFFN = mGame.mFullFrameNumber + commandCount + 1;
    mDestructActionID          = desctructID;
    mDestructActionWasUrgent   = urgent;
  }
  
  return true;
}

