#ifndef LINKEDCLASS_H
#define LINKEDCLASS_H




/// \brief Template to extend any class with one (unique) double-linked list.

/**
 *  Defines a linked list for a given class.
 *  Instances of one class can be chained up.
 *
 *  New instances of must be added
 *  to the list explicitly, using @see listInsertBefore
 *  or @see listInsertAfter, otherwise they are not
 *  included in the list.
 *
 *  There is no direct access to start or end element, nor
 *  is there a counter for these.
 */


template <class T>
class LinkedInstance
{
public:
    T* listPrev;          ///< Previous element in global list.
    T* listNext;          ///< Next element in global list.



    /**
     *  Default constructor for safe initialization
     *  of fields. The derived instance is not included
     *  in the global list.
     */
    LinkedInstance()
    : listPrev(0), listNext(0)
    {}



    /**
     *  Destructor that eliminates a derived instance
     *  from the list, maintaining integrity.
     */
    ~LinkedInstance() {
        if (listPrev) listPrev->listNext = listNext;
        if (listNext) listNext->listPrev = listPrev;
    }



    virtual void listInsertBefore(T* o) {
        if (listPrev) {
            o->listPrev = listPrev;
            listPrev->listNext = o;
        };
        o->listNext = (T*)this;
        listPrev = o;
    }



    virtual void listInsertAfter(T* o) {
        if (listNext) {
            o->listNext = listNext;
            listNext->listPrev = o;
        };
        o->listPrev = (T*)this;
        listNext = o;
    }


};











/// \brief Template for a global double-linked list of a class.

/**
 *  Defines a global linked list for a given class.
 *  This is a lightweight replacement for similar
 *  C++STL containers.
 *
 *  New instances of derived classes must be added
 *  to the list explicitly, using @see listAddFirst
 *  or @see listAddLast, otherwise they are not
 *  included in the list.
 */
 
template <class T>
class LinkedClass : public LinkedInstance<T>
{
private:
    bool inList; ///< Whether derived class is linked into global list or not.
public:
    static int listCount; ///< Number of elements in global list.
    static T* listFirst;  ///< First element in global list.
    static T* listLast;   ///< Last element in global list.



    /**
     *  Default constructor for safe initialization
     *  of fields. The derived instance is not included
     *  in the global list.
     */

    LinkedClass()
    : LinkedInstance<T>(), inList(false)
    {}



    /**
     *  Destructor that eliminates a derived instance
     *  from the list, maintaining integrity.
     */

    ~LinkedClass() {
        if (inList) {
            if (this == listFirst) listFirst = LinkedInstance<T>::listNext;
            if (this == listLast) listLast = LinkedInstance<T>::listPrev;
            listCount--;
        }
    }


    /**
     *  Global clean up of the list. Any elements currently
     *  in the list will be deleted. Global counters are
     *  reset to zero.
     */

    static void listClear() {
        while (listFirst && listCount) delete listFirst;
        listFirst = 0;
        listLast = 0;
        listCount = 0;
    }



    /**
     *  Add an element to the end of the list.
     */

    static void listAddLast(T* o) {
        if (listLast) listLast->listNext = o;
        o->listPrev = listLast;
        listLast = o;
        listCount++;
        o->inList = true;
        if (!listFirst) listFirst = o;
    }



    void listInsertBefore(T* o) {
        LinkedInstance<T>::listInsertBefore(o);
        listCount++;
    }



    void listInsertAfter(T* o) {
        LinkedInstance<T>::listInsertAfter(o);
        listCount++;
    }



    /**
     *  Add an element to the beginning of the list.
     */

    static void listAddFirst(T* o) {
        if (listFirst) listFirst->listPrev = o;
        o->listNext = listFirst;
        listFirst = o;
        listCount++;
        o->inList = true;
        if (!listLast) listLast = o;
    }

};

#endif
