#ifndef SHIP_FICTIVESHOT_H
#define SHIP_FICTIVESHOT_H

#include <cmath>
#include "../KeysPacket.h"
#include "../FrameStatus/FrameStatus.h"
#include "Ship.h"


    /**
     *  A fictive shot is a special MovingObject, which does not
     *  appear in the video ram, but represents a fictive shot just fired
     *  by a Ship. This can be used for collision testing, i.e., 
     *  checking whether a shot at the present stage would yield
     *  a hit or not.
     *
     *  Ship velocity is taken into account by adding it to the shot
     *  velocity. The overall shot velocity is still limited by
     *  480 pixel/sec in each grid direction.
     */

    class Ship::FictiveShot : public MovingObject {

    public:
        Ship *ship;
        FictiveShot(Ship *ship) : ship(ship) {}
        virtual int getx() {
            int step = ship->step;
            if (FrameStatus* frame = ship->frame) 
            {
                if (frame->keys.getLeft()) step = (step+  1)&0xff;
                if (frame->keys.getRight()) step = (step+255)&0xff;
            }
            return ship->getx()+Ship::shotXOffs[step]
              +ship->getu()/60.*FICTIVE_SHOT_MOVE_SHIP
              -getu()/60.*FICTIVE_SHOT_MOVE_BACK
              ;
        }
        virtual int gety() {
            int step = ship->step;
            if (FrameStatus* frame = ship->frame) 
            {
                if (frame->keys.getLeft()) step = (step+  1)&0xff;
                if (frame->keys.getRight()) step = (step+255)&0xff;
            }
            return ship->gety()+Ship::shotYOffs[step]
              +ship->getv()/60.*FICTIVE_SHOT_MOVE_SHIP
              -getv()/60.*FICTIVE_SHOT_MOVE_BACK
              ;
        }
        virtual double getu() {
            int step = ship->step;
            if (FrameStatus* frame = ship->frame) 
            {
                if (frame->keys.getLeft()) step = (step+  1)&0xff;
                if (frame->keys.getRight()) step = (step+255)&0xff;
            }
            return limit480(Ship::shotUVel[step]+ship->getu());
        }
        virtual double getv() {
            int step = ship->step;
            if (FrameStatus* frame = ship->frame) 
            {
                if (frame->keys.getLeft()) step = (step+  1)&0xff;
                if (frame->keys.getRight()) step = (step+255)&0xff;
            }
            return limit480(Ship::shotVVel[step]+ship->getv());
        }
        virtual std::string className() {return "FictiveShot";}
    };

inline
    Ship::Ship(int x, int y, int dx, int dy)
    : dx(dx), dy(dy), MovingObject(0,x,y), fictiveShot(new FictiveShot(this))
    {}

inline
    Ship::Ship(FrameStatus *frame, int x, int y, int dx, int dy)
    : dx(dx), dy(dy), MovingObject(frame,x,y), fictiveShot(new FictiveShot(this))
    {}

inline
    Ship::Ship() : fictiveShot(new FictiveShot(this)) {}


#endif
