#ifndef FRAMESTATUS_EVASIONMEASURES_H
#define FRAMESTATUS_EVASIONMEASURES_H

#include "FrameStatus.h"

inline
void
FrameStatus::evasionMeasures(KeysPacket& keys)
{

            ///  Check if collisions might occur in the near future.
            ///  First, we test for collisions with asteroids.

            double colx = 0;
            double coly = 0;
            double colWeight = 0;

            int nasteroids = asteroids.size();

            for (int i=0; i<nasteroids; ++i)
            {

                double diam = asteroids[i]->diameter() + ship->diameter();

                Collision thisCollision(*ship,*(asteroids[i]));

                /// Whether a collision is dangerous is decided briefly
                /// by two numbers: Time to collision, which also takes
                /// the object diameters into account (as an approximation
                /// for the evasion distance), and the distance during
                /// the collision (because the collision event is not actually
                /// the event of collision, but the event of closest passing).

                if (  (thisCollision.time > diam/15.)
                    ||(thisCollision.offs > 1.5*diam)) continue;

                /// If the time to collision is below 5 frames,
                /// we flee to hyperspace.

                if (thisCollision.time < 5./60.) {
                    printf("hyperspace <- asteroid time\n");
                    keys.hyperspace(true);
                }

                int dx = asteroids[i]->getx() - ship->getx();
                int dy = asteroids[i]->gety() - ship->gety();
                FrameStatus::normalize(dx,dy);

                /// Also, if an object is too close (and we
                /// might not have accurate velocity data),
                /// Hyperspace is our choice.

                double dist = sqrt(dx*dx+dy*dy) - 10 -1.5*diam;
                if (dist<0.) {
                    printf("hyperspace <- asteroid dist\n");
                    keys.hyperspace(true);
                }

                /// Otherwise, we assemble the data to derive an
                /// evasion vector. (At this stage, it is a vector
                /// pointing to the direction of danger.) The length
                /// of the vector (or vector increment) is a weight
                /// for the danger level. Danger level is based on
                /// time-to-impact, and distance.

                double len = sqrt(dx*dx+dy*dy);
                double weight = 0.5/thisCollision.time
                              + diam/thisCollision.offs;
                colx += dx*weight/len;
                coly += dy*weight/len;
                colWeight = colWeight<weight?weight:colWeight;
            }

            ///  Shots are also tested for collision. Usually they are
            ///  to fast, but if there is a chance to avoid hyperspace,
            ///  we should go for it.

            int nshots = shots.size();

            for (int i=0; i<nshots; ++i)
            {
                if (! shots[i]->firedFromUfo) continue;

                Collision thisCollision(*ship,*(shots[i]));

                if (  (thisCollision.time > 1.0)
                    ||(thisCollision.offs > ship->diameter())) continue;

                if (thisCollision.time < 5./60.) {
                    printf("hyperspace <- shot time\n");
                    printf("  shot: %d\n",i);
                    printf("  offs: %10.5f\n",thisCollision.offs);
                    printf("  time: %10.5f\n",thisCollision.time);
                    printf("  shot x,y: %d %d\n",shots[i]->getx(),shots[i]->gety());
                    printf("  shot u,v: %10.5f %10.5f\n",shots[i]->getu(),shots[i]->getv());
                    keys.hyperspace(true);
                }

                int dx = shots[i]->getx() - ship->getx();
                int dy = shots[i]->gety() - ship->gety();
                FrameStatus::normalize(dx,dy);

                double len = sqrt(dx*dx+dy*dy);
                double weight = 0.5/thisCollision.time
                              + 10./thisCollision.offs;
                colx += dx*weight/len;
                coly += dy*weight/len;
                colWeight = colWeight<weight?weight:colWeight;
            }

            if (colWeight>0) {
                double a = atan2(colx,coly)
                         - Ship::stepAngle[ship->step]/180.*M_PI;
                while (a<-M_PI) a+=2*M_PI;
                while (a>=M_PI) a-=2*M_PI;
                if (a < -M_PI/2.) {
                    keys.thrust(true);
                    keys.left(true);
                } else if (a < -M_PI/4) {
                    keys.thrust(true);
                    keys.right(true);
                } else if (a < 0) {
                    keys.thrust(false);
                    keys.right(true);
                } else if (a < M_PI/4) {
                    keys.thrust(false);
                    keys.left(true);
                } else if (a < M_PI/2) {
                    keys.thrust(true);
                    keys.left(true);
                } else {
                    keys.thrust(true);
                    keys.right(true);
                }
            }

    if (ufo) {

        double dx = (double) ufo->getx() - ship->getx();
        double dy = (double) ufo->gety() - ship->gety();
        double dist = sqrt(dx*dx-dy*dy);
        if (dist < (ufo->diameter()+ship->diameter()+10)) {
                    printf("hyperspace <- ufo dist\n");
                    keys.hyperspace(true);
                }
    }

}
#endif
