#ifndef FRAMESTATUS_LINKFRAME_H
#define FRAMESTATUS_LINKFRAME_H

#include "FrameStatus.h"
#include "../MovingObjects/MovingObject.h"

    /**
     *  Put, after some analysis, the present frame
     *  on the global frame stack, stored for later
     *  reference.
     */

inline
void FrameStatus::linkFrame()
    {

        /// The first thing we do, is to compare the objects
        /// in the present frame to objects in the past frame,
        /// to see whether new ones occured or old ones
        /// disappeared. The whole check is useful to log the
        /// history of objects, which in turn is used to determine
        /// their trajectories, and to predict collisions and hits.

        if (listLast) {

            // assume number of frames dropped.
            int frameDrop = 0;

            /// Latency is computed as a forgetful average.
            latency = 0.5*listLast->latency + 0.5*latency;

            // establish a likeliness matrix for asteroids
            int nlast = listLast->asteroids.size();
            int ncurr =           asteroids.size();
            for (int ilast = 0;ilast<nlast;ilast++) 
            {
                Asteroid* alast = listLast->asteroids[ilast];
                for (int icurr = 0;icurr<ncurr;icurr++)
                {
                    Asteroid* acurr = asteroids[icurr];
                    int i = ilast+icurr*nlast;
                    if (alast->sf != acurr->sf) continue;
                    else if (alast->type != acurr->type) continue;
                    else {
                        double dx = alast->getx()
                                   -acurr->getx()
                                   +alast->getu()*(1.+frameDrop)/60.;
                        double dy = alast->gety()
                                   -acurr->gety()
                                   +alast->getv()*(1.+frameDrop)/60.;
                        normalize(dx,dy);

                        // normalize by 8, the max. pixel speed per frame
                        dx /= 8.;
                        dy /= 8.;

                        new ObjectLink<Asteroid>(alast,acurr,1. - dx*dx - dy*dy);
                    }
                }
            }

            // establish a likeliness matrix for shots
            nlast = listLast->shots.size();
            ncurr =           shots.size();
            for (int ilast = 0;ilast<nlast;ilast++) 
            {
                Shot* alast = listLast->shots[ilast];
                for (int icurr = 0;icurr<ncurr;icurr++)
                {
                    Shot* acurr = shots[icurr];
                    int i = ilast+icurr*nlast;

                    double dx = alast->getx()
                               -acurr->getx()
                               +alast->getu()*(1.+frameDrop)/60.;
                    double dy = alast->gety()
                               -acurr->gety()
                               +alast->getv()*(1.+frameDrop)/60.;
                    normalize(dx,dy);

                    // normalize by 8, the max. pixel speed per frame
                    dx /= 8.;
                    dy /= 8.;

                    new ObjectLink<Shot>(alast,acurr,1. - dx*dx - dy*dy);
                }
            }

            // Use obvious or very likely estimates to
            // identify link-to-previous objects.
            ObjectLink<Asteroid>::eliminateObvious(listLast->asteroids,asteroids);
            ObjectLink<Shot>::eliminateObvious(listLast->shots,shots);


            // Finally, keep only the most likely links.
            ObjectLink<Asteroid>::eliminateAll(listLast->asteroids,asteroids);
            ObjectLink<Shot>::eliminateAll(listLast->shots,shots);


            // Now recount, and see how many of the current objects
            // have none or multiple links to the past.
            ObjectLink<Asteroid>::balanceInfo(listLast->asteroids,asteroids);
            ObjectLink<Shot>::balanceInfo(listLast->shots,shots);


            /// Asteroids of previous and present frames are linked.
            for (std::vector<Asteroid*>::iterator i = asteroids.begin();
                 i != asteroids.end(); i++)
            {
                if ((*i)->linkToPrev.size()) {
                    ObjectLink<Asteroid>* l = *((*i)->linkToPrev.begin());
                    l->prev->listInsertAfter(l->curr);
                }
            }


            /// Shots of previous and present frames are linked.
            for (std::vector<Shot*>::iterator i = shots.begin();
                 i != shots.end(); i++)
            {
                if ((*i)->linkToPrev.size()) {
                    ObjectLink<Shot>* l = *((*i)->linkToPrev.begin());
                    l->prev->listInsertAfter(l->curr);
                } else {
                    /// If shots cannot be associated with previous shots,
                    /// they might just have been fired from the ship. Note
                    /// that apparently, a shot comes to life at about 20 pixels
                    /// in front of the ship, and not at the ship's location.
                    if (FrameStatus* past = FrameStatus::listLast) {
                        if (Ship* lastShip = past->ship) {
                            bool fired =    past->keys.getFire();
                            double dx = (*i)->getx() - lastShip->getx()-20*Ship::stepSin[lastShip->step];
                            double dy = (*i)->gety() - lastShip->gety()-20*Ship::stepCos[lastShip->step];
                            normalize(dx,dy);
                            fired = fired && fabs(dx)<=5 && fabs(dy)<=5;
                            if (fired) {

                                if (double shipShots = (double)FrameStatus::shipShots)
                                    printf("Score/Shot: %15.5f   Score: %d\n",
                                        FrameStatus::score/shipShots,score);

                                (*i)->firedFromShip = lastShip;
                                printf("Shot just fired %10.5f %10.5f\n",dx ,dy);
                                FrameStatus::shipShots++;
                            } else 
                            if (Ufo* lastUfo = past->ufo) {
                                int dx = (*i)->getx() - lastUfo->getx();
                                int dy = (*i)->gety() - lastUfo->gety();
                                normalize(dx,dy);
                                fired = abs(dx)<=25 && abs(dy)<=25;
                                if (fired) {
                                    (*i)->firedFromUfo = lastUfo;
                                    printf("Ufo just fired %d %d\n",dx ,dy);
                                } else {
                                    printf("Shot without origin occured %d %d\n",dx ,dy);
                                }
                            } else {
                                printf("Shot without origin occured %d %d\n",dx ,dy);
                            }
                        }
                    }
                }
            }



            /// Out of curiosity, we also look at the shots
            /// that ran out of life.
            if (FrameStatus::listLast) {
                std::vector<Shot*>* oldShots = &(FrameStatus::listLast->shots);
                for (std::vector<Shot*>::iterator i = oldShots->begin();
                    i != oldShots->end(); i++)
                {
//                    if (! (*i)->listNext) 
                    {
                        int age = 0;
                        Shot* o = *i;
                        while (o->listPrev) {
                            o = (Shot*)(o->listPrev);
                            age++;
                        }
                        if (age==41)
                        if (Ship* ship = o->firedFromShip) {
                            printf("ShotAge: %d ",age);
                            double u = ((*i)->getx() - o->getx())/41.*60.;
                            double v = ((*i)->gety() - o->gety())/41.*60.;
                            double uu = sqrt(u*u+v*v);
                            if (uu<465 || uu>495) continue;
                            int step = ship->step;
                            if (FrameStatus* frame = ship->frame)
                            if (frame = frame->listNext)
                             {
                                if (frame->keys.getLeft()) step = (step+  1)&0xff;
                                if (frame->keys.getRight()) step = (step+255)&0xff;
                            }
                            printf("ShotVel %10.5f %10.5f ",u,v);
                            printf("ShipStep %d ",step);
                            printf("ShotAngle %10.5f ",atan2(u,v)/M_PI*180);
                            printf("ShotOrigin %d %d ",o->getx(),o->gety());
                            //printf("ShipOrigin %d %d ",ship->getx(),ship->gety());
                            //printf("shipvel %10.5f %10.5f\n",ship->getu(),ship->getv());
                            printf("\n");
                        }
                    }
                }
            }

            /// If there is a ship in both the previous and the present frame,
            /// they are linked.
            if (ufo && listLast->ufo)
                listLast->ufo->listInsertAfter(ufo);


            /// If there is a ship in both the previous and the present frame,
            /// they are linked.
            if (ship && listLast->ship)
                listLast->ship->listInsertAfter(ship);


        } else {
            printf("No prev. frame\n");
            for (std::vector<Asteroid*>::iterator a = asteroids.begin();
                 a != asteroids.end(); a++)
                 (*a)->prev = 0;
        }



        /// The configured frame is placed at the end of the stack. If
        /// the number of stored elements on the stack grows above a
        /// given limit, frames are removed from the beginning.
        listAddLast(this);
        while (listCount > LINKED_FRAMES_SIZE) delete listFirst;



        /// Based on history, velocities of moving objects
        /// are deduced.
        if (ship) ship->updateVelocity();
        if (ufo) ufo->updateVelocity();
        for (std::vector<Asteroid*>::iterator i = asteroids.begin();
             i!=asteroids.end();i++)
                (*i)->updateVelocity();
        for (std::vector<Shot*>::iterator i = shots.begin();
             i!=shots.end();i++)
                (*i)->updateVelocity();


        /// Based on history, the information of the Ship,
        /// if present, can be made more accurate.
        if (ship) {

            /// First guess for the ship's orientation is
            /// the last known orientation.
            int step = Ship::lastStep;
            int dx = ship->dx;
            int dy = ship->dy;


            /// If previous keystrokes have been recorded, these
            /// might be taken into account. Experiments have shown
            /// that we should look at the keys stored with the
            /// previous frame.
            const int lookPastKeys = 1;
            if (listCount>=lookPastKeys) {
                FrameStatus* past = this;
                int i=lookPastKeys;
                while (i && past->listPrev) {
                    past = past->listPrev;
                    i--;
                }
                if (past->keys.getLeft()) step = (step+1) & 0xff;
                if (past->keys.getRight()) step = (step+255) & 0xff;
            }


            /// Only if the present step is not plausible,
            /// corrections are applied.
            int corr = 0;
            if (   Ship::stepDX[step]!=dx
                || Ship::stepDY[step]!=dy) {

                printf("corr: %d %d %d %d %d\n",step,dx,dy,Ship::stepDX[step],Ship::stepDY[step]);

                /// \todo In case of a bad restart, the orientation
                /// found below might be phase shifted. That should
                /// be checked against, otherwise a neverending chain
                /// of small corrections will be the consequence.

                /// For corrections, probe all adjacent steps, until
                /// a plausible pair dx, dy is found.
                for (int d=0;d<128;d++) {
                    int newStep = (step+256+d) & 0xff;
                    if (   Ship::stepDX[newStep]==dx
                        && Ship::stepDY[newStep]==dy) {
                        step = newStep;
                        corr = +d;
                        break;
                    }
                    newStep = (step+256-d) & 0xff;
                    if (   Ship::stepDX[newStep]==dx
                        && Ship::stepDY[newStep]==dy) {
                        step = newStep;
                        corr = -d;
                        break;
                    }
                }

                /// Because corrections are generally unexpected
                /// events, and should only occur in connection with
                /// dropped frames, time lag, or bad restart, we print
                /// out a short message to make the user aware.
                printf("correction: %d\n",corr);
            }

            ship->step = step;
            ship->lastStep = step;
        }
    }



#endif
