#ifndef SHIP_H
#define SHIP_H

#include <cmath>
#include "../KeysPacket.h"

const double SHIP_ACCEL = 224.2; // acceleration, pix/s^2
const double SHIP_VMAX  = 480.0; // max. speed, pix/s
const double SHIP_DEC   =  -0.00418; // exp. const of decay
const double SHIP_ROT   = 6*M_PI/256.;

const int SHOT_UPFRONT = 12;

inline int    limit480(int    x) {return x<480?x>-480?x:-480:480;}
inline double limit480(double x) {return x<480?x>-480?x:-480:480;}


inline int roundd(double x)
{
   return (int)(x > 0.0 ? x + 0.5 : x - 0.5);
}


/// \brief The player's vehicle.

/**
 *  The Ship is the player's vehicle. It manifests itself
 *  in the vector ram as a set of lines of brightness 12.
 */

class Ship : public MovingObject
{
public:
    int dx;  ///< Approximated orientation vector x-component.
    int dy;  ///< Approximated orientation vector x-component.
    int step;  ///< Precise orientation, counted in left-turns from initial orientation.
    static int lastStep;

    static const int stepDX[];
    static const int stepDY[];
    static const double stepAngle[];
    static const double stepSin[];
    static const double stepCos[];

    static const int shotYOffs[];
    static const int shotXOffs[];
    static const double shotVVel[];
    static const double shotUVel[];

    virtual std::string className() {return "Ship";}


    /// \brief Describes a ship control sequence (turn and thrust).
    /**
     *  A Move is a sequence of steering impulses. For
     *  simplicity, the basic move only accounts for
     *  a number of turns (performed instantly), thrust
     *  on after a certain frame count, and thrust of
     *  after another frame count.
     */

    class Move {
    public:

        int turns;         ///< Number of turns. Positive is left, negative is right.
        int start_thrust;  ///< Frame count at which to turn on the thruster.
        int end_thrust;    ///< Frame count at which to turn off the thruster.

        Move() : turns(0), start_thrust(0), end_thrust(0) {};
        Move(int t, int s, int e) : turns(t), start_thrust(s), end_thrust(e) {};

        /// Set game keys according to the current state of the Move.
        void applyToKeys(KeysPacket &keys) {
            if (turns>0) {
                turns--;
                keys.left(true);
            } else if (turns<0) {
                turns++;
                keys.right(true);
            }
            keys.thrust(start_thrust<=0 && end_thrust>0);
            start_thrust--;
            end_thrust--;
        }

        double mindist; ///< How close this move brings the ship to the destination.
        double mintime; ///< When this move brings the ship closest to the destination.

        /// A weight telling how good this Move is, set depending on
        /// minimum distance to target and time effort.
        double minwght() {return mindist*mindist/64.+0.01*mintime;}
    };

    /// \brief Fictive shot fired by player, for preliminary hit probability analysis.
    class FictiveShot;

    FictiveShot *fictiveShot;

    Ship();
    Ship(int x, int y, int dx, int dy);
    Ship(FrameStatus *frame, int x, int y, int dx, int dy);

    virtual double diameter() 
    {
        return 10.;
    }


    /**
     *   Return the orientation of the ship.
     *   Currently this uses the simple method
     *   from the example code, but might be
     *   replaced with a more accurate method
     *   in the future.
     *
     *   Pointing to the right returns Pi/2,
     *   pointing to the top returns 0.
     */
    double angle()
    { return stepAngle[step]/180.*M_PI; }



    /// \brief Evaluate a move against a destination.
    double testMove(Move &move, int dx, int dy);


    /// \brief Recompute velocity based on latest object history.
    virtual void updateVelocity();



    /**
     *   Compute the `Move' that brings the ship
     *   to a relative position dx, dy, based on
     *   current speed and orientation.
     */
    Move moveTo(int dx, int dy) {

        // This is the deviation between the current direction
        // and the destination.
        double da = angle() - atan2(dx,dy);
        while (da>=M_PI) da -= (2*M_PI);
        while (da<-M_PI) da += (2*M_PI);

        // Based on these simple figures, an initial guess is set
        // up: We turn towards the destination, and then fire the
        // enginges for a while.

        /// \todo Use a better rounding algorithm for turns.
        int turns = roundd(da/M_PI*180./4.21875);
        int start = abs(turns)-1;
            start = start<0?0:start;
        int end   = start+40;
        Move move(turns,start,end);

        // Now we `integrate' the resulting path, and iterate the
        // parameters until we detect an optimum path.
        int i;
        for(i=0;i<10;i++) {
            double f0 = testMove(move,dx,dy);
            Move moveAlt1(move); moveAlt1.turns--       ; double f1 = testMove(moveAlt1,dx,dy);
            Move moveAlt2(move); moveAlt2.turns++       ; double f2 = testMove(moveAlt2,dx,dy);
            Move moveAlt3(move); moveAlt3.start_thrust--; double f3 = testMove(moveAlt3,dx,dy);
            Move moveAlt4(move); moveAlt4.start_thrust++; double f4 = testMove(moveAlt4,dx,dy);
            Move moveAlt5(move); moveAlt5.end_thrust--  ; double f5 = testMove(moveAlt5,dx,dy);
            Move moveAlt6(move); moveAlt6.end_thrust++  ; double f6 = testMove(moveAlt6,dx,dy);

            /// \todo Find a better way of advancing the move.
            if (f0<f1&&f0<f2&&f0<f3&&f0<f4&&f0<f5&&f0<f6) break;
            if (f1<f0)      move.turns--;
            else if (f2<f0) move.turns++;
            if (f3<f0) move.start_thrust--;
            else if (f4<f0) move.end_thrust++;
            if (f5<f0) move.start_thrust--;
            else if (f6<f0) move.end_thrust++;
         }

        double u = getu();
        double v = getv();
        printf("move: %8.4f %8.4f %d %d %d %d\n",u,v,move.turns,move.start_thrust,move.end_thrust,i);

        return move;
    }

};

#include "Ship_testMove.h"
#include "Ship_updateVelocity.h"
#endif
