#ifndef SHIP_TESTMOVE_H
#define SHIP_TESTMOVE_H

#include "../FrameStatus/FrameStatus.h"
#include "Ship.h"


/**
 *   Integrate `move' and thus determine
 *   minimum time and distance to the target
 *   dx, dy.
 *
 *   @param move Move to test
 *   @param dx target x distance
 *   @param dy target y distance
 *   @return weighted error
 */
inline
double Ship::testMove(Move &move, int dx, int dy)
{
        double a = angle();
        double v[2] = {getu()/60.,getv()/60.};
        double mindist = dx*dx+dy*dy;
        double mintime = 0;
        double p[2] = {dx,dy};
        Move d(move);

        /// We derive the trajectory of the ship
        /// by integrating in time, for a given
        /// number of frames.
        const int FRAME_TOT=120;
        for(int f=0;f<FRAME_TOT;f++) {

            const double si = stepSin[step];
            const double co = stepCos[step];

            /// The integral takes into account the keys pressed.
            bool thrust, left, right;

            const int KEYS_PAST = 1;
            if (f<KEYS_PAST) {
                /// Due to a processing lag, the first keystrokes have
                /// already been sent at previous frames.
                FrameStatus *past = frame;
                for (int i=KEYS_PAST-1;i>f;i--)
                    if (past->listPrev) past = past->listPrev;
                thrust = past->keys.getThrust();
                left = past->keys.getLeft();
                right = past->keys.getRight();
            } else {
                /// After that, keystrokes are specified by the move.
                if (d.turns>0) {
                    d.turns--;
                    left = true;
                    right = false;
                } else if (d.turns<0) {
                    d.turns++;
                    left = false;
                    right = true;
                } else {
                    left = false;
                    right = false;
                }
                bool thrust = d.start_thrust<=0 && d.end_thrust>0;
                d.start_thrust--;
                d.end_thrust--;
            }

            /// We integrate using fractions of frames.
            /// This deviates from the game simulation,
            /// but gives a smoother result.
            const int FRAME_MULT=1;
            for(int i=0;i<FRAME_MULT;i++) {

                /// During time integration, the
                /// distance to the destination is
                /// checked at each round, in order
                /// to capture the minimum.
                double dist = p[0]*p[0]+p[1]*p[1];
                if (dist<mindist) {
                    mintime = f/(double)FRAME_MULT;
                    mindist = dist;
                };

                /// Position is derived by integrating
                /// the current velocity.
                p[0] -= v[0]/(double)FRAME_MULT;
                p[1] -= v[1]/(double)FRAME_MULT;

                /// Velocity is in any case reduced
                /// by an exponential decay.
                v[0] *= exp(SHIP_DEC/(double)FRAME_MULT);
                v[1] *= exp(SHIP_DEC/(double)FRAME_MULT);

                /// If thruster is on, velocity is additionally
                /// increased by constant acceleration in the
                /// direction of the ship's orientation.
                if (thrust) {
                    v[0] += SHIP_ACCEL/3600.*si/(double)FRAME_MULT;
                    v[1] += SHIP_ACCEL/3600.*co/(double)FRAME_MULT;
                }
            }
            // update orientation
            if (left) a -= SHIP_ROT;
            else if (right) a += SHIP_ROT;
            while (a>=M_PI) a -= (2*M_PI);
            while (a<-M_PI) a += (2*M_PI);
        }
        move.mintime = mintime;
        move.mindist = sqrt(mindist);
        return move.minwght();
}

#endif
