// game.h
#pragma once


//#define STATS						// if defined, statistics are enabled
#define STATE						// if defined, the game state is tracked
//#define LIMIT_PLAYTIME	5*60	// if defined, play time is limited (in seconds)
//#define USE_GUI					// if defined, gui is used

#include <assert.h>
#include <math.h>

typedef float speed_t;

#define MAX_ASTEROIDS	26
#define MAX_SHOTS		8

#define SPEED_UNKNOWN (speed_t)0			// Geschwindigkeit wurde noch nicht ermittelt

#define PI						3.14159
#define FIELD_WIDTH				1024	// screen size
#define FIELD_HEIGHT			768
#define SCREEN_CENTER_X			FIELD_WIDTH/2
#define SCREEN_CENTER_Y			FIELD_HEIGHT/2
//#define FPS						60	// frames per second

#define RADIUS_SHOT				1
#define RADIUS_SHIP				16
#define RADIUS_ASTEROID_SMALL	8
#define RADIUS_ASTEROID_MIDDLE	16
#define RADIUS_ASTEROID_BIG		33
#define SIZE_SAUCER_SMALL_DX	20
#define SIZE_SAUCER_SMALL_DY	12	
#define SIZE_SAUCER_BIG_DX		40
#define SIZE_SAUCER_BIG_DY		24

#define HIST_SIZE_SHIP			8	// size of histpry used for speed tracking of ship
#define HIST_SIZE_SAUCER		4	// size of histpry used for speed tracking of saucer

#define DEG2RAD(a)				((a)*PI/180)
#define SQ(v)					(v)*(v)

#define REL_COORD_NORMALIZE(dx, dy) { while (dx < -FIELD_WIDTH/2) dx += FIELD_WIDTH; \
			 while (dx >= FIELD_WIDTH/2) dx -= FIELD_WIDTH;   \
			 while (dy < -FIELD_HEIGHT/2) dy += FIELD_HEIGHT; \
			 while (dy >= FIELD_HEIGHT/2) dy -= FIELD_HEIGHT; }

/*struct vec {
	int x;
	int y;
};*/

static int distSq(const int x0, const int y0, const int x1, const int y1)
{
	int dx = x0 - x1;
	int dy = y0 - y1;
	REL_COORD_NORMALIZE(dx, dy);
	return SQ(dx)+SQ(dy);  // Quadrat des Abstands
}

static float distSqf(const int x0, const int y0, const float x1, const float y1)
{
	float dx = x0 - x1;
	float dy = y0 - y1;
	REL_COORD_NORMALIZE(dx, dy);
	return SQ(dx)+SQ(dy);  // Quadrat des Abstands
}

static float distSqff(const float x0, const float y0, const float x1, const float y1)
{
	float dx = x0 - x1;
	float dy = y0 - y1;
	REL_COORD_NORMALIZE(dx, dy);
	return SQ(dx)+SQ(dy);  // Quadrat des Abstands
}

struct GameObject
{
public:
	unsigned int id;
	int nTracked;	// number of tracked frames
	int tTracked;	// tracking time
	int x;    // Koordinaten des Mittelpunkts
	int y;
	speed_t vx;    // gemessene Geschwindigkeit (unbekannt, wenn vx==vy==SPEED_UNKNOWN)
	speed_t vy;
	void setSpeed(speed_t vx, speed_t vy);
	virtual bool equalType(const GameObject &o) const { return false; }
	virtual int getRadius() const { return 0; }
	int getSpeedSq() { return (int)(SQ(vx)+SQ(vy)); }
	float getSpeed() { return sqrt(SQ(vx)+SQ(vy)); }
	void getNextPos(int &xnew, int &ynew) const {
		xnew = x + (int)floorf(vx+0.5f); ynew = y + (int)floorf(vy+0.5f);
	}
	void getNextPos(float &xnew, float &ynew, float dt) const {
		xnew = x + vx*dt; ynew = y + vy*dt;
	}
	float getNextX(float dt) const {
		return x + vx*dt;
	}
	float getNextY(float dt) const {
		return y + vy*dt;
	}
	int getNextDist(const GameObject &o) const {
		int xnew, ynew, ox, oy;
		int radius = getRadius();

		getNextPos(xnew, ynew);

		if(o.nTracked>0)
			o.getNextPos(ox,oy);
		else { ox = o.x; oy = o.y; }

		return (int)sqrt((float)distSq(xnew, ynew, ox, oy)) - radius;
	}
	float getNextDist(const GameObject &o, float dt) const {
		float xnew, ynew, ox, oy;
		int radius = getRadius();

		getNextPos(xnew, ynew, dt);

		if(o.nTracked>0)
			o.getNextPos(ox,oy, dt);
		else { ox = (float)o.x; oy = (float)o.y; }

		return sqrt((float)distSqff(xnew, ynew, ox, oy)) - radius;
	}
	int getDist(const GameObject &o) const { return getDist(o.x, o.y); }
	int getDist(const int ox, const int oy) const { 
		int radius = getRadius();
		return (int)sqrt((float)distSq(x, y, ox, oy)) - radius;
	}
};

template<int size>
struct CircularHist
{
	int x[size];
	int y[size];
	int dt[size];
	short int idx;	//last index
	short int free; //free space

	int getSize() { return size; }
	void clear()
	{
		idx = 0;
		free = size;
	}
	void add(const GameObject &o, int adt)
	{
		if(idx==0) idx = size-1;
		else idx--;
		if(free>0) free--;
		x[idx] = o.x;
		y[idx] = o.y;
		dt[idx] = adt;
	}
	void getLatest(int &lx, int &ly, int &ldt)
	{
		short int idl = idx - free - 1;
		if(idl<0) idl += size;
		lx = x[idl];
		ly = y[idl];
		ldt = dt[idl];
	}
};

template<int size>
struct DynamicObject : public GameObject
{
public:
	bool present;
	CircularHist<size> hist;
};

struct Asteroid : public GameObject
{
public:
	int type; // 1 ... 4, uere Form
	int sf;   // scale factor: 0 = gro, 15 = mittel, 14 = klein

	void set(int x, int y, int type, int sf);
	bool equalType(const GameObject &o) const { return type==((Asteroid&)o).type && sf==((Asteroid&)o).sf; }
	int getRadius() const { 
		switch (sf)
		{	// Abstand um den ungefhren Radius des Asteroiden korrigieren
			case 0:  // groer Asteroid
				return RADIUS_ASTEROID_BIG;
			case 15: // mittlerer Asteroid
				return RADIUS_ASTEROID_MIDDLE;
			case 14: // kleiner Asteroid
				return RADIUS_ASTEROID_SMALL;
		}
		return 0;
	}
};


struct Saucer : public DynamicObject<HIST_SIZE_SAUCER>
{
public:
	int size;    // Gre: 15 = gro, 14 = klein

	void set(int x, int y, int size);
	bool equalType(const GameObject &o) const { return true; }
	int getRadius() const { 
		switch (size)
		{	// Abstand um den ungefhren Radius des UFOs korrigieren
		case 15: // groes UFO
			return (int)sqrtf((float)SIZE_SAUCER_BIG_DX/2*SIZE_SAUCER_BIG_DY/2);
		case 14: // kleines UFO
			return (int)sqrtf((float)SIZE_SAUCER_SMALL_DX/2*SIZE_SAUCER_SMALL_DY/2);
		}
		return 0;
	}
};

struct Ship : public DynamicObject<HIST_SIZE_SHIP>
{
public:
	int dx;        // Blickrichtung des Schiffes
	int dy;

	void set(int x, int y, int dx, int dy);
	bool equalType(const GameObject &o) const { return true; }
	int getRadius() const { return RADIUS_SHIP; }
};

struct Shot : public GameObject
{
public:
	void set(int x, int y);
	bool equalType(const GameObject &o) const { return true; }
	int getRadius() const { return RADIUS_SHOT; }
};

class GameStatus
{
public:
#ifdef STATE
   enum State
   {
      unknown = 1,   // noch nicht ermittelt
      standby = 2,   // Wartet auf Start-Button
      prepare = 3,   // Start-Button gedrckt, Spiel beginnt bald
      playing = 4,   // Aktiv im Spiel
      gameover = 5,  // Letztes Schiff verloren
      highscore = 6  // Highscore eingeben
   };
   State state;            // Zustand des Spieles

   unsigned int score;     // erreichte Punktzahl
   unsigned int lifes;     // noch zur Verfgung stehende Leben
   unsigned int level;     // aktuell gespieltes Level
#endif

    unsigned int maxId;	// max. object id
	int dt;			// delta time to last frame
	bool fired;
	char frameno;
	char latency;
	char framesLost;
	int dx, dy; // coordinate shift (ship centered)
	Ship ship;
	Saucer saucer;
	int nasteroids; // Anzahl Asteroiden
	Asteroid asteroids[MAX_ASTEROIDS];
	int nshots;     // Anzahl Schsse
	Shot shots[MAX_SHOTS];
	GameObject* focusedObj;

	void init(void);
	void clear(void);
	void copy(const GameStatus &game);
	unsigned int getNewId() { return ++maxId; }
};
