// player.cpp: Beispielspieler fr Asteroids
// Harald Bgeholz / c't

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#if defined(WINDOWS)
#include <winsock2.h>
#else
// 2 Includes fr socket()
#include <sys/types.h>
#include <sys/socket.h>
// 2 Includes fr inet_addr()
#include <netinet/in.h>
#include <arpa/inet.h>
// 2 Includes fr fcntl()
#include <unistd.h>
#include <fcntl.h>
// fr memset()
#define INVALID_SOCKET -1
#define WSAGetLastError() errno
#endif

#include "player.h"
#include <math.h>

#ifdef DEBUG
//#define DISABLE_FIRE	// if defined, ship does not shoot
#endif

#ifdef STATS
	#include <time.h>

	struct stats_t 
	{
		speed_t maxSpeedSq;
		speed_t vx,vy, maxVx, maxVy;
		void Update(const GameObject &o) {
			if(o.nTracked<5) return;

			speed_t v = SQ(o.vx)+SQ(o.vy);
			if(v>maxSpeedSq) 
			{
				maxSpeedSq = v;
				vx = o.vx;
				vy = o.vy;
			}
			if(fabs(o.vx)>fabs(maxVx)) maxVx = o.vx;
			if(fabs(o.vy)>fabs(maxVy)) maxVy = o.vy;
		}
	};

#else
	#define	stats_t	int
#endif

static stats_t gStatsAsteroid = {0};
static stats_t gStatsSaucer = {0};
static stats_t gStatsShip = {0};
static stats_t gStatsShot = {0};

#ifdef LIMIT_PLAYTIME
	static time_t  gStartTime;
#endif

#ifdef USE_GUI
static void GuiThreadProc( System::Object^ obj )
{
    asteroid::GUI^ pGui = (asteroid::GUI^)obj;

    if (pGui == nullptr)
		return;   // if pGui is not valid

	pGui->ShowDialog();

    return;   // thread completed successfully
}
#endif

void Player::Run(void)
{
	FramePacket frame;
	KeysPacket keys;
	GameStatus prevGame;
	GameStatus game;
	int t = -1;
	bool maxAsteroidsReached;

#ifdef USE_GUI
	System::Threading::Thread^ guiThread = gcnew System::Threading::Thread(gcnew System::Threading::ParameterizedThreadStart(GuiThreadProc));
	guiThread->Start(gui);
#endif

	game.init();

	// Warten auf Startbildschirm
	frame.ping = 1;
	keys.clear();
	while (game.state < GameStatus::standby)
	{
		keys.ping = frame.ping;       // Verluste egal
		SendPacket(keys);
		ReceivePacket(frame);
		InterpretScreen(frame, game);

		Sleep (50);
	}

   // Spiel starten, falls Startbildschirm
   if (game.state == GameStatus::standby)
      keys.start(true);

	#ifdef LIMIT_PLAYTIME
		bool selfDestruct = false;
		gStartTime = time(0);
	#endif

	while(game.state < GameStatus::gameover)
	{
		++t;         // Zeit
		++keys.ping; // jedes gesendete Pckchen erhlt eine individuelle Nummer zur Latenzmessung
		SendPacket(keys);
		ReceivePacket(frame);

		prevGame.copy(game);
		#ifdef USE_GUI
			gui->UpdateGameState(prevGame);
		#endif

		InterpretScreen(frame, game);

		#ifdef LIMIT_PLAYTIME
			if(selfDestruct) continue;

			if(time(0)-gStartTime > 5*60)
			{
				printf("5 Minutes finished: score = %d, lifes = %d\n", game.score>0 ? game.score: prevGame.score, game.lifes);
				selfDestruct = true;
			}
		#endif

		#ifdef STATE
			if(prevGame.nasteroids==0 && game.nasteroids>0)
				game.level = prevGame.level+1;
		#endif

		game.latency = keys.ping - frame.ping;
		game.framesLost = t>0 ? game.frameno - (prevGame.frameno+1) : 0;

		UpdateObjectTracking(prevGame, game);

		keys.clear();   // alle Tasten loslassen

		#ifdef STATS
			if(t>0 && t%500==0)
			{
				printf("maxSpeedAsteroid %.2f, gMaxSpeedSaucer %.2f, gMaxSpeedShip %.2f, gMaxSpeedShot %.2f\n", 
					sqrt(gStatsAsteroid.maxSpeedSq), sqrt(gStatsSaucer.maxSpeedSq), sqrt(gStatsShip.maxSpeedSq), sqrt(gStatsShot.maxSpeedSq));
			}

			if(t>0)// && t%10==0)
			{
				if(game.saucer.nTracked>0)
					printf("vSaucer %.2f, ", game.saucer.getSpeed());
				if(game.ship.nTracked>0)
					printf("vShip %.2f, ", game.ship.getSpeed());
				printf("nAst %d, lifes %d, level %d, score %d", game.nasteroids, game.lifes, game.level, game.score);

				if (game.framesLost!=0 || game.latency!=0)
				{
					printf(", lat %d, lost %d", game.latency, game.framesLost);
				}

				if(t%500==0)
					printf("\n");
				else
					printf("\r");
			}
		#endif

		if (!game.ship.present) continue;

		float dt = (float)game.latency + CONTROL_DELAY;
		maxAsteroidsReached = game.nasteroids >= MAX_ASTEROIDS;

#if 0	//DEBUG
		if(game.nasteroids<=4)
		{
			printf("x = %d, y = %d, vx = %f, vy = %f, v = %f\n", game.ship.x, game.ship.y, game.ship.vx, game.ship.vy, sqrt(SQ(game.ship.vx)+SQ(game.ship.vy)));
			continue;
		}
#endif
#if 0
		if(game.ship.nTracked>0)
			printf("vx = %f, vy = %f, v = %f\n", game.ship.vx, game.ship.vy, sqrt(SQ(game.ship.vx)+SQ(game.ship.vy)));
#endif
		// 1. Hyperspace, wenn Abstand zum nchsten Objekt zu gering
		// 2a. Objekt (auer Schuss) angreifen, wenn Kollision in den nchsten T_ATTACK_COLLI frames bevorsteht
		//    Objekt mit frhester Kollision zuerst angreifen.
		// 2b. Ufo angreifen, wenn Abstand zum Ufo kleiner als MAX_DIST_ATTACK_SAUCER ist
		// 2c. Objekt angreifen, wenn Abstand kleiner als MAX_DIST_ATTACK_ALL in den nchsten T_ATTACK_ALL frames.
		//    Geringsten Abstand innerhalb dieser Frames zuerst angreifen.
		//(2d). Objekt mit dem geringsten Abstand angreifen, wenn Abstand kleiner als MAX_DIST_ATTACK
		// 3. Schiff beschleunigen, wenn Abstand zum nchsten Objekt kleiner als MIN_DIST_THRUST ist

		//----------------------------------------------------------
		// 1. Hyperspace
		//----------------------------------------------------------

		int min_dist = 0x7fffffff, min_dist_SmallAsteroid = 0x7fffffff;
		//int min_dist_col;
		//int min_frames_col;
		//GameObject *minObjCol = 0;
		float min_t_hyp = 1e10;
		GameObject *minObj = 0, *minSmallAsteroid = 0;
		GameObject *o;

		for (int i=0; i<game.nasteroids; ++i)
		{   // nchstgelegenen Asteroiden suchen
			int dist = game.asteroids[i].getDist(game.ship);
			if (dist < min_dist)
			{
				min_dist = dist;
				minObj = &game.asteroids[i];
			}
			if (game.asteroids[i].sf==14 && dist < min_dist_SmallAsteroid)	// small asteroid
			{
				min_dist_SmallAsteroid = dist;
				minSmallAsteroid = &game.asteroids[i];
			}
		}
		if (game.saucer.present)
		{
			int dist = game.saucer.getDist(game.ship);
			if (dist < min_dist)
			{
				min_dist = dist;
				minObj = &game.saucer;
			}
		}

		//min_dist_col = min_dist;
		//minObjCol = minObj;

		for (int i=-1; i<game.nasteroids+game.nshots; ++i)
		{
			if(i>=game.nasteroids)
			{
				o = &game.shots[i-game.nasteroids];
				if(o->nTracked<MIN_TRACKS_SHOT)
					continue;
			}
			else if(i>=0)
				o = &game.asteroids[i];
			else
			{
				if(!game.saucer.present) continue;
				o = &game.saucer;
			}

			if(o->nTracked>=MIN_TRACKS)
			{
				int dCurSq = distSq(game.ship.x, game.ship.y, o->x, o->y);
				float xnew, ynew;
				o->getNextPos(xnew, ynew, 0.5f);
				float dNewSq = distSqf(game.ship.x, game.ship.y, xnew, ynew);
				if (dNewSq < (float)dCurSq)
				{
					float t;
					float dist = o->getNextDist(game.ship, dt);

					if (dist < RADIUS_SHIP + MIN_DIST_HYPERSPACE*2)
					{
						if(dist <= RADIUS_SHIP + MAX_DIST_COLLI)
							t = 1;
					#if T_HYPERSPACE>1
					#else
						else continue;
					#endif

						// choose object
						if(t<min_t_hyp)
						{
							min_t_hyp = t;
							//minObjCol = o;
						}
					}
				}
			}
		}

/*		for (int i=0; i<game.nshots; ++i)
			if(game.shots[i].nTracked>=MIN_TRACKS_SHOT)
			{
				int dCur = (int)sqrt((float)distSq(game.ship.x, game.ship.y, game.shots[i].x, game.shots[i].y));
				int xnew, ynew;
				game.shots[i].getNextPos(xnew, ynew);
				int dNew = (int)sqrt((float)distSq(game.ship.x, game.ship.y, xnew, ynew));
				if (dNew < dCur)
				{
					int dist = game.shots[i].getNextDist(game.ship);
					if (dist < min_dist_col)
					{
						min_dist_col = dist;
						minObjCol = &game.shots[i];
					}
				}
			}*/

		// Flucht, wenn Kollision unausweichlich
		if (min_t_hyp<=T_HYPERSPACE || 
			(min_dist < RADIUS_SHIP+MIN_DIST_HYPERSPACE && minObj->nTracked<MIN_TRACKS))
		{
			keys.hyperspace(true);
			continue;
		}

		//----------------------------------------------------------
		// 2. Attack
		//----------------------------------------------------------
		float min_t_col = 1e10;

		//----------------------------------------------------------
		// 2a. Collision
		//----------------------------------------------------------

		for (int i=-1; i<game.nasteroids; ++i)
		{
			if(i>=0)
				o = &game.asteroids[i];
			else
			{
				if(!game.saucer.present) continue;
				o = &game.saucer;
			}

			if(o->nTracked>=MIN_TRACKS)
			{
				int dx = o->x - game.ship.x;
				int dy = o->y - game.ship.y;
				REL_COORD_NORMALIZE(dx, dy);
				float dvx = o->vx - game.ship.vx;
				float dvy = o->vy - game.ship.vy;
				float dvSq = SQ(dvx) + SQ(dvy);
				float t = -(dx*dvx + dy*dvy) / dvSq;
				if(t<=0) continue;	// object is going away
				if(t>T_ATTACK_COLLI) continue;	// too far in future

				float dx_t = dx + dvx*t;
				float dy_t = dy + dvy*t;
				float d_tSq = SQ(dx_t) + SQ(dy_t);
				int dCol = o->getRadius() + game.ship.getRadius() + MAX_DIST_COLLI;
				if(d_tSq > SQ(dCol)) continue;	// object passes ship

				// choose object
				if(t<min_t_col)
				{
					min_t_col = t;
					game.focusedObj = o;
				}
			}
		}

		//----------------------------------------------------------
		// 2b. Saucer
		//----------------------------------------------------------
		if(!game.focusedObj && game.saucer.present)
		{
			int d = game.saucer.getDist(game.ship);
			if(d < MAX_DIST_ATTACK_SAUCER)
			{
				game.focusedObj = &game.saucer;
			}
		}

		//----------------------------------------------------------
		// 2c. All Objects
		//----------------------------------------------------------
		if(!game.focusedObj)
		{
			float min_d_obj = 1e10;

			for (int i=-1; i<game.nasteroids; ++i)
			{
				if(i>=0)
				{
					if(maxAsteroidsReached && game.asteroids[i].sf!=14)	//not small asteroid
						continue;

					o = &game.asteroids[i];
				}
				else
				{
					if(!game.saucer.present) continue;
					o = &game.saucer;
				}

				if(o->nTracked>=MIN_TRACKS)
				{
					int dx = o->x - game.ship.x;
					int dy = o->y - game.ship.y;
					REL_COORD_NORMALIZE(dx, dy);
					float dvx = o->vx - game.ship.vx;
					float dvy = o->vy - game.ship.vy;
					float dpdvx = dx*dvx;
					float dpdvy = dy*dvy;
					float dpdv = dpdvx + dpdvy;
					if(dpdv>0)
					{
						// TODO_________
						//if(dpdvx>0)
						//dpdv = 
					}
					float dvSq = SQ(dvx) + SQ(dvy);
					float t = -dpdv / dvSq;
					if(t<=0) continue;	// object is going away
					if(t>T_ATTACK_ALL) continue;	// too far in future

					float dx_t = dx + dvx*t;
					float dy_t = dy + dvy*t;
					float d_tSq = SQ(dx_t) + SQ(dy_t);
					int dCol = o->getRadius() + game.ship.getRadius() + MAX_DIST_ATTACK_ALL;
					if(d_tSq > SQ(dCol)) continue;	// object passes ship

					// choose object
					if(d_tSq<min_d_obj)
					{
						min_d_obj = d_tSq;
						game.focusedObj = o;
					}
				}
			}
		}

		//----------------------------------------------------------
		// 2d. 
		//----------------------------------------------------------
		if(!game.focusedObj)
		{
			if(maxAsteroidsReached && minSmallAsteroid)
				game.focusedObj = minSmallAsteroid;	//small asteroid
			else
				game.focusedObj = minObj;
		}

		// Attack selected object
		if(game.focusedObj)
		{
			float dx = game.focusedObj->getNextX(dt) - game.ship.getNextX(dt);
			float dy = game.focusedObj->getNextY(dt) - game.ship.getNextY(dt);
			REL_COORD_NORMALIZE(dx, dy);
			float d = sqrt((float)SQ(dx)+SQ(dy));

			// Schiff in Richtung auf das nchstgelegene Objekt drehen
			// mathematisch wird hier das Kreuzprodukt aus den Vektoren 
			// ship.dx/y/0 und min_dx/y/0 berechnet
			float angSO = game.ship.dx * dy - game.ship.dy * dx;
			float alpha = asin( (float)angSO / (d*sqrt((float)(SQ(game.ship.dx)+SQ(game.ship.dy)))) );
			
			if(game.focusedObj->nTracked>=MIN_TRACKS)
			{
				// Schiff zustzlich drehen, um die Geschwindigkeit des Objektes und des Schusses zu bercksichtigen
				float shipSpeedShot = SHIP_SPEED_SHOT;	// TODO: use real shot speed depending on ship speed
				alpha += asin((dx*game.focusedObj->vy - dy*game.focusedObj->vx) / (d*shipSpeedShot));
			}

			if(alpha>0)
				keys.left(true);
			else
				keys.right(true);

			float alphaAbs = fabsf(alpha);
			float maxAlpha = atan(game.focusedObj->getRadius() / d);
			if(!prevGame.fired || game.dt>1)
			{
				if ((alphaAbs<=maxAlpha*MAX_SHOOT_ANGLE_TOL || alphaAbs>=MIN_SHOOT_ANGLE_DEV))  // Feuerknopf drcken, wenn Objekt im Zielbereich
				{
#ifndef DISABLE_FIRE
					keys.fire(true);
					game.fired = true;
#endif
				}
				else	
				{
					// shoot if another object can be destroyed
					for (int i=-1; i<game.nasteroids; ++i)
					{
						if(i>=0)
						{
							if(maxAsteroidsReached && game.asteroids[i].sf!=14)	//not small asteroid
								continue;

							o = &game.asteroids[i];
						}
						else
						{
							if(!game.saucer.present) continue;
							o = &game.saucer;
						}

						if(o->nTracked>=MIN_TRACKS)
						{
							float dx = o->getNextX(dt) - game.ship.getNextX(dt);
							float dy = o->getNextY(dt) - game.ship.getNextY(dt);
							REL_COORD_NORMALIZE(dx, dy);
							float d = sqrt((float)SQ(dx)+SQ(dy));

							if(d<MAX_RANGE_SHOT*9/10)
							{
								// Schiff in Richtung auf das nchstgelegene Objekt drehen
								// mathematisch wird hier das Kreuzprodukt aus den Vektoren 
								// ship.dx/y/0 und min_dx/y/0 berechnet
								float angSO = game.ship.dx * dy - game.ship.dy * dx;
								float alpha = asin( (float)angSO / (d*sqrt((float)(SQ(game.ship.dx)+SQ(game.ship.dy)))) );
								
								// Schiff zustzlich drehen, um die Geschwindigkeit des Objektes und des Schusses zu bercksichtigen
								float shipSpeedShot = SHIP_SPEED_SHOT;	// TODO: use real shot speed depending on ship speed
								alpha += asin((dx*game.focusedObj->vy - dy*game.focusedObj->vx) / (d*shipSpeedShot));
								float maxAlpha = atan(o->getRadius() / d);
								if(fabsf(alpha)<=maxAlpha)
								{
#ifndef DISABLE_FIRE
									keys.fire(true);
									game.fired = true;
#endif
								}
							}
						}
					}
				}
			} // end of if(t % 2 == 0)
		}

		//----------------------------------------------------------
		// 3. Thrust
		//----------------------------------------------------------
		if(game.saucer.present || game.nasteroids>0)
		{
			if (min_dist > MIN_DIST_THRUST && (game.nasteroids>0 || game.saucer.present) 
					&& game.ship.getSpeedSq()<SQ(MAX_SPEED_THRUST)) // beschleunigen, wenn nichts in der Nhe
				keys.thrust(true);
		}
		else	// no objects available, go to screen center
		{
			// rotate to screen center
			int dx = SCREEN_CENTER_X - game.ship.x;
			int dy = SCREEN_CENTER_Y - game.ship.y;
			//REL_COORD_NORMALIZE(dx, dy);
			float d = sqrt((float)SQ(dx)+SQ(dy));
			int angSO = game.ship.dx * dy - game.ship.dy * dx;
			float alpha = asin( (float)angSO / (d*sqrt((float)(SQ(game.ship.dx)+SQ(game.ship.dy)))) );
			if(alpha>0)
				keys.left(true);
			else
				keys.right(true);

			// thrust to reach screen center
			float vShip = game.ship.getSpeed();
			float aShipX = SHIP_DECELERATON * game.ship.vx/vShip;
			float aShipY = SHIP_DECELERATON * game.ship.vy/vShip;
			float aShipSq = SQ(aShipX) + SQ(aShipY);
			float tStopSq = vShip*vShip / aShipSq;
			float tStop = sqrt(tStopSq);
			float dStopX = game.ship.vx*tStop + aShipX/2*tStopSq;
			float dStopY = game.ship.vy*tStop + aShipY/2*tStopSq;
			float dStop = sqrt(SQ(dStopX)+SQ(dStopY));
			if (d > 100 && fabsf(alpha)<DEG2RAD(60) && d > dStop && vShip<MAX_SPEED_THRUST) // beschleunigen
				keys.thrust(true);
		}
	}

   // falls Spiel durch Anzahl Frames abgebrochen wurde, auf Spielende warten
   if (game.state < GameStatus::gameover)
   {
      // Warteschleife, Spieler kann Selbstzerstrung einleiten
      while (game.state < GameStatus::gameover)
      {
         keys.ping = frame.ping;       // Verluste egal
         SendPacket(keys);
         ReceivePacket(frame);
         InterpretScreen(frame, game);

         keys.clear();   // alle Tasten loslassen
      }
   }

   // Warten auf Highscore-Bildschirm (bei standby dann aber abbrechen)
   keys.clear();
   while (game.state < GameStatus::highscore
      && game.state > GameStatus::standby)
   {
      keys.ping = frame.ping;       // Verluste egal
      SendPacket(keys);
      ReceivePacket(frame);
      InterpretScreen(frame, game);

      Sleep (50);
   }

   // Highscore schreiben
   if (game.state == GameStatus::highscore)
      WriteHighscore (PLAYER);

   // Warten bis anderer Bildschirm als Highscore
   keys.clear();
   while (game.state == GameStatus::highscore)
   {
      keys.ping = frame.ping;       // Verluste egal
      SendPacket(keys);
      ReceivePacket(frame);
      InterpretScreen(frame, game);

      Sleep (50);
   }

   while(true) 
	   Sleep (50);
}

#ifdef STATE
	#define ADDDIGIT(d) ++numofdigits, points = points*10 + d
	#define ADDCHAR(d) textstr[numofchars++] = d
#endif

void Player::InterpretScreen(FramePacket &packet, GameStatus& game)
{
	unsigned short vector_ram[512];
	int dx, dy, sf, vx, vy, vz, vs;
	int v1x = 0;
	int v1y = 0;
	int shipdetect = 0;

#ifdef STATE
    unsigned int points = 0;
    int numofdigits = 0;    // < 0 bedeutet fertig
    char textstr[64];
    int numofchars = 0;
#endif

	game.clear();

	game.frameno = packet.frameno;

	/* Vektor-RAM in 16-Bit-Worte konvertieren. War in der ersten Version mal ein sportlicher
	Typecast: unsigned short *vector_ram = (unsigned short*)packet.vectorram;
	Das klappt aber nur auf Little-Endian-Systemen, daher besser portabel: */
	for (int i=0; i<512; ++i)
		vector_ram[i] = (unsigned char)packet.vectorram[2*i] | (unsigned char)packet.vectorram[2*i+1] << 8;

	if (vector_ram[0] != 0xe001 && vector_ram[0] != 0xe201)
		return; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL

	int pc = 1;
	while (pc < 512)
	{
		int op = vector_ram[pc] >> 12;
		switch (op)
		{
		case 0xa: // LABS
			vy = vector_ram[pc] & 0x3ff;
			vx = vector_ram[pc+1] & 0x3ff;
			vs = vector_ram[pc+1] >> 12;
			break;
		case 0xb: // HALT
			return;
		case 0xc: // JSRL
			switch (vector_ram[pc] & 0xfff)
			{
			case 0x8f3:
				game.asteroids[game.nasteroids++].set(vx, vy, 1, vs);
				break;
			case 0x8ff:
				game.asteroids[game.nasteroids++].set(vx, vy, 2, vs);
				break;
			case 0x90d:
				game.asteroids[game.nasteroids++].set(vx, vy, 3, vs);
				break;
			case 0x91a:
				game.asteroids[game.nasteroids++].set(vx, vy, 4, vs);
				break;
			case 0x929:
				game.saucer.present = true;
				game.saucer.set(vx, vy, vs);
				break;
#ifdef STATE
           // Lebenerkennung
           case 0xa6d:
              ++game.lifes;
              break;
           // PUNKTEerkennung
           case 0xadd:    // Ziffer 0 oder O
              ADDCHAR ('O');
              if (numofdigits >= 0)
                 ADDDIGIT(0);
              break;
           case 0xb2e:    // Ziffer 1
              if (numofdigits >= 0)
                 ADDDIGIT(1);
              break;
           case 0xb32:    // Ziffer 2
              if (numofdigits >= 0)
                 ADDDIGIT(2);
              break;
           case 0xb3a:    // Ziffer 3
              if (numofdigits >= 0)
                 ADDDIGIT(3);
              break;
           case 0xb41:    // Ziffer 4
              if (numofdigits >= 0)
                 ADDDIGIT(4);
              break;
           case 0xb48:    // Ziffer 5
              if (numofdigits >= 0)
                 ADDDIGIT(5);
              break;
           case 0xb4f:    // Ziffer 6
              if (numofdigits >= 0)
                 ADDDIGIT(6);
              break;
           case 0xb56:    // Ziffer 7
              if (numofdigits >= 0)
                 ADDDIGIT(7);
              break;
           case 0xb5b:    // Ziffer 8
              if (numofdigits >= 0)
                 ADDDIGIT(8);
              break;
           case 0xb63:    // Ziffer 9
              if (numofdigits >= 0)
                 ADDDIGIT(9);
              break;
           //PUNKTEerkennung ende
           case 0xb2c:    // Leerzeichen (Punkte- oder Zustanderkennung)
           {
              unsigned __int16 value2 =
                 vector_ram[pc+2] | (vector_ram[pc+3] << 8);
              if (numofdigits > 0 && (value2 & 0xfff) == 0xb2c)
              //if (numofdigits > 0)
              {  // 2 Leerzeichen hintereinander --> Ende mit der Punktezahl
                 numofdigits = -numofdigits;
#ifdef STATE
				 game.score = points;
#endif
              }
              else if (numofchars > 0)
              {  // Leerzeichen beenden Wort --> Vergleiche fr Zustandserkennung
                 textstr[numofchars] = '\0';
                 if (strcmp (textstr, "STARTKNOEPFE") == 0)
                    game.state = GameStatus::standby;
                 else if (strcmp (textstr, "SPIELER") == 0)
                    game.state = GameStatus::prepare;
                 else if (strcmp (textstr, "SPIELENDE") == 0)
                    game.state = GameStatus::gameover;
                 else if (strcmp (textstr, "BUCHSTABENWAHL") == 0)
                    game.state = GameStatus::highscore;
                    // "ERGEBNIS" wird auch auf der Startseite angezeigt,
                    // wenn bereits ein highscore existiert
                    // Zeilenumbrche existieren nicht, daher "BESTENBITTE"
              }
              numofchars = 0;      // wegen 0 = O
              break;
           }

           // Worterkennung fr Zustnde
           case 0xA78: ADDCHAR ('A'); break;
           case 0xA80: ADDCHAR ('B'); break;
           case 0xA8D: ADDCHAR ('C'); break;
           case 0xA93: ADDCHAR ('D'); break;
           case 0xA9B: ADDCHAR ('E'); break;
           case 0xAA3: ADDCHAR ('F'); break;
           case 0xAAA: ADDCHAR ('G'); break;
           case 0xAB3: ADDCHAR ('H'); break;
           case 0xABA: ADDCHAR ('I'); break;
           case 0xAC1: ADDCHAR ('J'); break;
           case 0xAC7: ADDCHAR ('K'); break;
           case 0xACD: ADDCHAR ('L'); break;
           case 0xAD2: ADDCHAR ('M'); break;
           case 0xAD8: ADDCHAR ('N'); break;
//               case 0xADD: ADDCHAR ('O'); break;       siehe Ziffer 0
           case 0xAE3: ADDCHAR ('P'); break;
           case 0xAEA: ADDCHAR ('Q'); break;
           case 0xAF3: ADDCHAR ('R'); break;
           case 0xAFB: ADDCHAR ('S'); break;
           case 0xB02: ADDCHAR ('T'); break;
           case 0xB08: ADDCHAR ('U'); break;
           case 0xB0e: ADDCHAR ('V'); break;
           case 0xB13: ADDCHAR ('W'); break;
           case 0xB1a: ADDCHAR ('X'); break;
           case 0xB1f: ADDCHAR ('Y'); break;
           case 0xB26: ADDCHAR ('Z'); break;
           // Worterkennung fr Zustnde ende
#endif
			}  
			break;
		case 0xd: // RTSL
			return;
		case 0xe: // JMPL
			/*
			pc = vector_ram[pc] & 0xfff;
			break;
			*/
			return;
		case 0xf: // SVEC
			/*
			dy = vector_ram[pc] & 0x300;
			if ((vector_ram[pc] & 0x400) != 0)
				dy = -dy;
			dx = (vector_ram[pc] & 3) << 8;
			if ((vector_ram[pc] & 4) != 0)
				dx = -dx;
			sf = (((vector_ram[pc] & 8) >> 2) | ((vector_ram[pc] & 0x800) >> 11)) + 2;
			vz = (vector_ram[pc] & 0xf0) >> 4;
			*/
			break;
		default:
			dy = vector_ram[pc] & 0x3ff;
			if ((vector_ram[pc] & 0x400) != 0)
				dy = -dy;
			dx = vector_ram[pc+1] & 0x3ff;
			if ((vector_ram[pc+1] & 0x400) != 0)
				dx = -dx;
			sf = op;
			vz = vector_ram[pc+1] >> 12;
			if (dx == 0 && dy == 0 && vz == 15)
				game.shots[game.nshots++].set(vx, vy);
			if (op == 6 && vz == 12 && dx != 0 && dy != 0)
			{
				switch (shipdetect)
				{
				case 0:
					v1x = dx;
					v1y = dy;
					++shipdetect;
					break;
				case 1:
					game.ship.present = true;
					game.ship.set(vx, vy, v1x - dx, v1y - dy);
					++shipdetect;
					break;
				}
			}
			else if (shipdetect == 1)
				shipdetect = 0;

			break;
		}
		if (op <= 0xa)
			++pc;
		if (op != 0xe) // JMPL
			++pc;
	}   
}

static void TrackObjects(GameStatus& game, GameObject *prev0, int nprev, GameObject *cur0, int ncur, size_t objectSize, int maxSpeedXy, stats_t &stats)
{
	int i,j;
	int dx,dy,d;
	GameObject *prev, *cur;

	//check tracked asteroids
	for(i=0, prev=prev0; i<nprev; i++, prev=(GameObject*)(((char*)prev)+objectSize))
		if(prev->nTracked>0)
		{
			bool tracked = false;

			// estimated position
			int ex = prev->x + (int)(prev->vx * game.dt);
			int ey = prev->y + (int)(prev->vy * game.dt);

			for(j=0, cur=cur0; j<ncur; j++, cur=(GameObject*)(((char*)cur)+objectSize))
				if(cur->equalType(*prev) && cur->nTracked==0)
				{
					dx = cur->x - ex;
					dy = cur->y - ey;
					REL_COORD_NORMALIZE(dx, dy);

					d = dx*dx + dy*dy;
					if(d<=MAX_TRACKED_RADIUS_SQ)
					{
						tracked = true;
						cur->nTracked = prev->nTracked+1;
						cur->tTracked = prev->tTracked + game.dt;

						dx = cur->x - prev->x;
						dy = cur->y - prev->y;
						REL_COORD_NORMALIZE(dx, dy);

						speed_t vx = (dx + prev->vx*prev->tTracked) / cur->tTracked;
						speed_t vy = (dy + prev->vy*prev->tTracked) / cur->tTracked;
						cur->setSpeed(vx, vy);
						//cur->setSpeed(prev->vx, prev->vy);
						cur->id = prev->id;
						#ifdef STATS
							stats.Update(*cur);
						#endif

						break;
					}
				}

			if(!tracked)
				prev->nTracked = 0;
		}

	//check untracked asteroids
	for(j=0, cur=cur0; j<ncur; j++, cur=(GameObject*)(((char*)cur)+objectSize))
		if(cur->nTracked==0)
		{
			int mindSq = 2*SQ(maxSpeedXy)*game.dt + 1;
			GameObject *prevMin = NULL;

			for(i=0, prev=prev0; i<nprev; i++, prev=(GameObject*)(((char*)prev)+objectSize))
				if(cur->equalType(*prev) && prev->nTracked==0)
				{
					dx = cur->x - prev->x;
					dy = cur->y - prev->y;
					REL_COORD_NORMALIZE(dx, dy);

					d = dx*dx + dy*dy;
					if(d<mindSq)
					{
						mindSq = d;
						prevMin = prev;
					}
				}

			if(prevMin)
			{
				dx = cur->x - prevMin->x;
				dy = cur->y - prevMin->y;
				REL_COORD_NORMALIZE(dx, dy);
			}

			if(prevMin && max(abs(dx), abs(dy)) <= maxSpeedXy*game.dt)
			{
				speed_t vx = (speed_t)dx / game.dt;
				speed_t vy = (speed_t)dy / game.dt;
				cur->vx = prevMin->vx = vx;
				cur->vy = prevMin->vy = vy;
				cur->nTracked = 1;
				cur->tTracked = game.dt;
				cur->id = prev->id;
			}
			else
			{
				cur->id = game.getNewId();
			}
		}
}

template<int size>
static void TrackDynamicObject(GameStatus& game, DynamicObject<size> &prev, DynamicObject<size> &cur, stats_t &stats)
{
	//cur.hist = prev.hist;
	if(cur.present)
	{
		if(prev.present)
		{
			//if(prev.nTracked>0 && cur.x!=prev.x)
			//	prev.nTracked += 0;
			int x,y,dt;
			cur.hist.getLatest(x,y,dt);
			cur.tTracked = prev.tTracked + game.dt-dt;
			cur.hist.add(cur, game.dt);

			int dx = cur.x - x;
			int dy = cur.y - y;
			REL_COORD_NORMALIZE(dx, dy);

			speed_t vx = (speed_t)dx / cur.tTracked;
			speed_t vy = (speed_t)dy / cur.tTracked;

			cur.setSpeed(vx, vy);
			cur.nTracked = prev.nTracked+1;
			cur.id = prev.id;
			#ifdef STATS
				stats.Update(cur);
			#endif
		}
		else
		{
			cur.id = game.getNewId();
			cur.tTracked = 0;
			cur.nTracked = 0;
			cur.hist.add(cur, 0);
		}
	}
	else
	{
		cur.tTracked = 0;
		cur.nTracked = 0;
		cur.hist.clear();
	}
}

void Player::UpdateObjectTracking(GameStatus& prev, GameStatus& cur)
{
	cur.dt = cur.frameno - prev.frameno;	// delta time
	cur.maxId = prev.maxId;

	// correct time overflow
	if(cur.dt<=0) cur.dt += 256;

	if(cur.dt>TRACKING_MAX_DT)
	{
		int i;
		for(i=0; i<cur.nasteroids; i++)
			cur.asteroids[i].id = cur.getNewId();
		for(i=0; i<cur.nshots; i++)
			cur.shots[i].id = cur.getNewId();
		if(cur.saucer.present)
			cur.saucer.id = cur.getNewId();
		return;
	}

	// track asteroids
	TrackObjects(cur, prev.asteroids, prev.nasteroids, cur.asteroids, cur.nasteroids, sizeof(Asteroid), MAX_SPEED_ASTEROID_XY, gStatsAsteroid);

	// track shots
	TrackObjects(cur, prev.shots, prev.nshots, cur.shots, cur.nshots, sizeof(Shot), MAX_SPEED_SHOT, gStatsShot);

	// track saucer
	TrackDynamicObject<HIST_SIZE_SAUCER>(cur, prev.saucer, cur.saucer, gStatsSaucer);

	// track ship
	TrackDynamicObject<HIST_SIZE_SHIP>(cur, prev.ship, cur.ship, gStatsShip);
}

bool Player::InterpretHighscoreScreen(const FramePacket &packet, int pos, char* curchar)
{
   const unsigned char *vector_ram = (unsigned char*)&packet.vectorram[0];
   char textstr[64];
   int numofchars = 0;
   bool highscore = false;

   /* Bemerkung:
      unsigned __int16 value = vector_ram[i] | (vector_ram[i+1] << 8);
      statt
      unsigned __int16 value = reinterpret_cast<unsigned __int16*>(vector_ram)[i>>1];
      damit unabhngig von der Darstellung des Prozessors
      (Little- oder Big-Endian).
   */
   {
      unsigned __int16 value = vector_ram[0] | (vector_ram[1] << 8);
      if (value != 0xe001 && value != 0xe201)
         return false; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL
   }

   for (unsigned int pc = 2; pc < sizeof packet.vectorram;)
   {
      unsigned __int16 value = vector_ram[pc] | (vector_ram[pc+1] << 8);
      int op = value >> 12;
      switch (op)
      {
         case 0xb: // HALT
            goto end;
         case 0xc: // JSRL
         {
            switch (value & 0xfff)
            {
               case 0xb2c:    // Leerzeichen 
               {
                  if (numofchars > 0)
                  {  // Leerzeichen beenden Wort --> Vergleiche fr Zustandserkennung
                     textstr[numofchars] = '\0';
                     if (strcmp (textstr, "BUCHSTABENWAHL") == 0)
                        highscore = true;
                        // "ERGEBNIS" wird auch auf der Startseite angezeigt,
                        // wenn bereits ein highscore existiert
                        // Zeilenumbrche existieren nicht, daher "BESTENBITTE"
                     if (strncmp (textstr, "DRUECKEN", 8) == 0
                        && strcmp (textstr, "DRUECKENWENN") != 0)
                     {  // Die Eingabezeichen schliessen direkt an das Wort
                        // "DRUECKEN" an, aber nicht das Vorkommen "DRUECKENWENN"
                        textstr[numofchars++] = ' ';
                        if (numofchars >= pos+8)
                           goto end;
                        break;      // Zeichen nicht lschen
                     }
                     // OutputDebug (textstr);
                     numofchars = 0;
                  }
                  break;
               }

               case 0xA78: ADDCHAR ('A'); break;
               case 0xA80: ADDCHAR ('B'); break;
               case 0xA8D: ADDCHAR ('C'); break;
               case 0xA93: ADDCHAR ('D'); break;
               case 0xA9B: ADDCHAR ('E'); break;
               case 0xAA3: ADDCHAR ('F'); break;
               case 0xAAA: ADDCHAR ('G'); break;
               case 0xAB3: ADDCHAR ('H'); break;
               case 0xABA: ADDCHAR ('I'); break;
               case 0xAC1: ADDCHAR ('J'); break;
               case 0xAC7: ADDCHAR ('K'); break;
               case 0xACD: ADDCHAR ('L'); break;
               case 0xAD2: ADDCHAR ('M'); break;
               case 0xAD8: ADDCHAR ('N'); break;
               case 0xADD: ADDCHAR ('O'); break;
               case 0xAE3: ADDCHAR ('P'); break;
               case 0xAEA: ADDCHAR ('Q'); break;
               case 0xAF3: ADDCHAR ('R'); break;
               case 0xAFB: ADDCHAR ('S'); break;
               case 0xB02: ADDCHAR ('T'); break;
               case 0xB08: ADDCHAR ('U'); break;
               case 0xB0e: ADDCHAR ('V'); break;
               case 0xB13: ADDCHAR ('W'); break;
               case 0xB1a: ADDCHAR ('X'); break;
               case 0xB1f: ADDCHAR ('Y'); break;
               case 0xB26: ADDCHAR ('Z'); break;
            }
            break;
         }
         case 0xd: // RTSL
            goto end;
         case 0xe: // JMPL
            /*
            pc = (value & 0xfff) << 1;
            break;
            */
            goto end;
      }
      if (op <= 0xa)
         pc+=2;
      if (op != 0xe) // JMPL
         pc+=2;
   }

end:
   *curchar = numofchars > pos+8 ? textstr[pos+8] : '?';
   return highscore;
}

inline static int chardistance (char chr1, char chr2)
{
   if (chr1 == ' ')
      chr1 = '@';    // == 'A' - 1
   if (chr2 == ' ')
      chr2 = '@';    // == 'A' - 1

   if (chr2 >= chr1)
   {
      return (chr2 - chr1 < 27 + chr1 - chr2)
         ?  chr2 - chr1
         :  -(27 + chr1 - chr2);
   }

   // inverse
   return (chr1 - chr2 < 27 + chr2 - chr1)
      ?  -(chr1 - chr2)
      :  27 + chr2 - chr1;
}

void Player::WriteHighscore(const char *str)
{
   FramePacket frame;
   KeysPacket keys;
   keys.clear();
   for (unsigned int i = 0; i < 3 && str[i] != '\0'; ++i) // Hauptschleife, geht die Buchstaben durch
   {
      char cur='A';
      char chr = toupper(str[i]);
      if (chr < 'A' || chr > 'Z')
         chr = ' ';

      // Buchstaben auslesen, der angezeigt wird
      ReceivePacket(frame);
      while (1)
      {
         if (!InterpretHighscoreScreen(frame, i, &cur))
            return;     // Bildschirm hat gewechselt, kein Highscore mehr

         // passende Tasten drcken
         if (cur==chr)
         {  // richtiges Zeichen liegt vor, festmachen
            keys.clear();
            keys.hyperspace(true);
            keys.ping = frame.ping-1;
            SendPacket(keys);
            ReceivePacket(frame);
            Sleep (HIGHSCORESLEEPTIME);    // mind. 2 Frames warten
            keys.clear();
            keys.ping = frame.ping-1;
            SendPacket(keys);
            ReceivePacket(frame);
            Sleep (HIGHSCORESLEEPTIME);    // mind. 2 Frames warten
            break;   // raus aus while
         }
         else
         {  // noch nicht das richtige Zeichen ...
            keys.clear();
            int direction = chardistance (chr, cur);
            if (direction < 0)
               keys.left(true);
            else
               keys.right(true);

            keys.ping = frame.ping-1;
            SendPacket(keys);
            ReceivePacket(frame);
            Sleep (HIGHSCORESLEEPTIME);    // mind. 2 Frames warten

            keys.clear();
            keys.ping = frame.ping-1;
            SendPacket(keys);
            ReceivePacket(frame);
            Sleep (HIGHSCORESLEEPTIME);    // mind. 2 Frames warten
         }
      }  // while
   }  // for
}

KeysPacket::KeysPacket(void)
{
	signature[0] = 'c';
	signature[1] = 't';
	signature[2] = 'm';
	signature[3] = 'a';
	signature[4] = 'm';
	signature[5] = 'e';
	keys = '@';
	ping = 0;
}

void KeysPacket::clear(void)
{
	keys = '@';
}

void KeysPacket::start(bool b)
{
   if (b)
      keys |= KEY_START;
   else
      keys &= ~KEY_START;
}

void KeysPacket::hyperspace(bool b)
{
	if (b)
		keys |= KEY_HYPERSPACE;
	else
		keys &= ~KEY_HYPERSPACE;
}

void KeysPacket::fire(bool b)
{
	if (b)
		keys |= KEY_FIRE;
	else
		keys &= ~KEY_FIRE;
}

void KeysPacket::thrust(bool b)
{
	if (b)
		keys |= KEY_THRUST;
	else
		keys &= ~KEY_THRUST;
}

void KeysPacket::left(bool b)
{
	if (b)
	{
		keys |= KEY_LEFT;
		right(false);
	}
	else
		keys &= ~KEY_LEFT;
}

void KeysPacket::right(bool b)
{
	if (b)
	{
		keys |= KEY_RIGHT;
		left(false);
	}
	else
		keys &= ~KEY_RIGHT;
}

void Player::ReceivePacket(FramePacket &packet)
{
	sockaddr_in sender;
	int sender_size = sizeof sender;
	fd_set readfds, writefds, exceptfds;

	do
	{
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(sd, &readfds);
		FD_SET(sd, &exceptfds);
		select(sd+1, &readfds, &writefds, &exceptfds, NULL);
		int bytes_received = recv(sd, (char *)&packet, sizeof packet, 0);
		if (bytes_received != sizeof packet)
		{
			int err = WSAGetLastError();
			fprintf(stderr, "Fehler %d bei recvfrom().\n", err);
			exit(1);
		}
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(sd, &readfds);
		timeval zero;
		zero.tv_sec = zero.tv_usec = 0;
		select(sd+1, &readfds, &writefds, &exceptfds, &zero);
	} while(FD_ISSET(sd, &readfds));
}

void Player::SendPacket(KeysPacket &packet)
{
	sockaddr_in server;
	memset(&server, 0, sizeof server);
	server.sin_family = AF_INET;
	server.sin_port = htons(1979);
	server.sin_addr.s_addr = server_ip;
	if (sizeof packet != sendto(sd, (char *)&packet, sizeof packet, 0, (sockaddr*)&server, sizeof server))
	{
#if defined(WINDOWS)
		int err = WSAGetLastError();
		if (err != WSAEWOULDBLOCK)
		{
			fprintf(stderr, "Fehler %d bei sendto().\n", err);
			exit(1);
		}
#else
		if (errno != EAGAIN)
		{
			perror("Fehler bei sendto()");
			exit(1);
		}
#endif
	}
}
