// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2005  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.util.settings.IntegerPreferenceProperty;

import java.util.Locale;
import java.util.prefs.Preferences;

/**
 *  Editable integer preference property.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @version $Revision$
 */
public class SwingIntegerPreferenceProperty
        extends AbstractBasicSimpleEditablePreferenceProperty
        implements IntegerPreferenceProperty
{
  /** The minimal possible value. */
  private final int minValue;
  /** The maximal possible value. */
  private final int maxValue;
  /** The step used. */
  private final int step;
  /** Current value. */
  private int value;

  /**
   *  Create a integer preference property with maximum range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   */
  public SwingIntegerPreferenceProperty(String basicName, String baseTag, int value)
  {
    this(basicName, baseTag, value, 1);
  }

  /**
   *  Create a integer preference property with maximum range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param step       step to be used for value when stepping
   */
  public SwingIntegerPreferenceProperty(String basicName, String baseTag, int value, int step)
  {
    this(basicName, baseTag, value, Integer.MIN_VALUE, Integer.MAX_VALUE, step);
  }

  /**
   *  Create a integer preference property with the given range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   */
  public SwingIntegerPreferenceProperty(String basicName, String baseTag, int value, int minValue, int maxValue)
  {
    this(basicName, baseTag, value, minValue, maxValue, 1);
  }

  /**
   *  Create a integer preference property with the given range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param step       step to be used for value when stepping
   */
  public SwingIntegerPreferenceProperty(String basicName, String baseTag, int value, int minValue, int maxValue, int step)
  {
    super(basicName, baseTag);
    this.minValue = minValue;
    this.maxValue = maxValue;
    this.step = step;
    this.value = value;
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  public void readFrom(Preferences preferences)
  {
    setValue(preferences.getInt(getBasicName(), value));
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  public void storeTo(Preferences preferences)
  {
    preferences.putInt(getBasicName(), value);
  }

  /**
   * Get the editor components for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor components
   */
  public EditorProvider getEditorProvider(Locale l)
  {
    return new IntegerEditor(this, l);
  }

  /**
   *  Get the minimal allowed value.
   *  @return minimal value
   */
  public int getMinValue()
  {
    return minValue;
  }

  /**
   *  Get the maximal allowed value.
   *  @return maximal value
   */
  public int getMaxValue()
  {
    return maxValue;
  }

  /**
   *  Get the step used in steppers.
   *  @return step
   */
  public int getStep()
  {
    return step;
  }

  /**
   *  Get the properties value.
   *  @return property value
   */
  public int getValue()
  {
    return value;
  }

  /**
   *  Set the properties value.
   *  @param value value to set
   */
  public void setValue(int value)
  {
    int oldValue = this.value;
    if (value > maxValue) {
      this.value = maxValue;
    }
    else if (value < minValue) {
      this.value = minValue;
    }
    else {
      this.value = value;
    }
    if (oldValue != this.value) {
      fireValueChange(getBasicName(), oldValue, this.value);
    }
  }
}
