package de.heise.anniversary.contest.asteroids;

import java.util.ArrayList;
import java.util.List;

/**
 * The <code>GameStatus</code> class represents game status which
 * can be decoded from a <code>FramePacket</code>.
 * <p>
 * It describes the currently visible ship, saucer, asteroids and shots.
 */
public class GameStatus implements Cloneable {
	protected int ticks;
	protected Ship ship = null;
	protected Saucer saucer = null;
	protected List<Asteroid> asteroids = new ArrayList<Asteroid>();
	protected List<Shot> shots = new ArrayList<Shot>();

	private void clear() {
		ticks = 0;
		ship = null;
		saucer = null;
		asteroids.clear();
		shots.clear();
	}

	/**
	 * Fills this game status by interpreting the vector ram from <code>framePacket</code>.
	 *
	 * @param framePacket
	 * 			the <code>FramePacket</code> to interpret.
	 * @param ticks
	 * 			the current game-time for this game status
	 */
	public void interpretScreen(FramePacket framePacket, int ticks) {
		int dx = 0, dy = 0, vx = 0, vy = 0, vz = 0, vs = 0;
		int v1x = 0;
		int v1y = 0;
		int shipdetect = 0;

		this.clear();
		this.ticks = ticks;

		final int initialJump = framePacket.readVectorRAMWord(0);
		if (initialJump != 0xe001 && initialJump != 0xe201)
			return; // should not occur, first instruction is always a jump

		for (int pc = 1; true; ) {
			final int currentWord = framePacket.readVectorRAMWord(pc);
			int op = currentWord >> 12;
			switch (op) {
			case 0xa: { // LABS
				final int nextWord = framePacket.readVectorRAMWord(pc + 1);
				vy = currentWord & 0x3ff;
				vx = nextWord & 0x3ff;
				vs = nextWord >> 12;
				break;
			}
			case 0xb: // HALT
				return;
			case 0xc: // JSRL
				Asteroid.Size saucerSize = Asteroid.Size.BIG;
				if (vs == 15) saucerSize = Asteroid.Size.MIDDLE;
				else if (vs == 14) saucerSize = Asteroid.Size.SMALL;

				switch (currentWord & 0xfff) {
				case 0x8f3:
					asteroids.add(new Asteroid(vx, vy, Asteroid.Type.TYPE_1, saucerSize));
					break;
				case 0x8ff:
					asteroids.add(new Asteroid(vx, vy, Asteroid.Type.TYPE_2, saucerSize));
					break;
				case 0x90d:
					asteroids.add(new Asteroid(vx, vy, Asteroid.Type.TYPE_3, saucerSize));
					break;
				case 0x91a:
					asteroids.add(new Asteroid(vx, vy, Asteroid.Type.TYPE_4, saucerSize));
					break;
				case 0x929:
					saucer = new Saucer(vx, vy, vs == 15? Saucer.Size.BIG: Saucer.Size.SMALL);
					break;
				}
				break;
			case 0xd: // RTSL
				return;
			case 0xe: // JMPL
				// pc = currentWord & 0xfff;
				// break;
				return;
			case 0xf: // SVEC
				if (false) {
					dy = currentWord & 0x300;
					if ((currentWord & 0x400) != 0)	dy = -dy;
					dx = (currentWord & 3) << 8;
					if ((currentWord & 4) != 0)	dx = -dx;
					//sf = (((currentWord & 8) >> 2) | ((currentWord & 0x800) >> 11)) + 2;
					vz = (currentWord & 0xf0) >> 4;
				}
				break;
			default:
				{
					final int nextWord = framePacket.readVectorRAMWord(pc + 1);
					dy = currentWord & 0x3ff;
					if ((currentWord & 0x400) != 0)	dy = -dy;
					dx = nextWord & 0x3ff;
					if ((nextWord & 0x400) != 0) dx = -dx;
					//sf = op;
					vz = nextWord >> 12;
					if (dx == 0 && dy == 0 && vz == 15)
						shots.add(new Shot(vx, vy));
					if (op == 6 && vz == 12 && dx != 0 && dy != 0) {
						switch (shipdetect) {
						case 0:
							v1x = dx;
							v1y = dy;
							++shipdetect;
							break;
						case 1:
							ship = new Ship(vx, vy, v1x - dx, v1y - dy);
							++shipdetect;
							break;
						}
					} else if (shipdetect == 1)
						shipdetect = 0;
					break;
				}
			}
			if (op <= 0xa) ++pc; // two word instruction
			if (op != 0xe) ++pc; // not JMPL
		}
	}

	@Override
	public String toString() {
		StringBuilder stringBuilder = new StringBuilder();
		if (ship != null) {
			stringBuilder.append("ship:")
				.append(ship.getX()).append(",")
				.append(ship.getY()).append(";")
				.append(ship.getOrientation().getX()).append(",")
				.append(ship.getOrientation().getY()).append("\n");
		}
		if (saucer != null) {
			stringBuilder.append("saucer:")
				.append(saucer.getX()).append(",")
				.append(saucer.getY()).append(";")
				.append(saucer.getSize()).append("\n");
		}
		for (Asteroid asteroid: asteroids) {
			stringBuilder.append("asteroid:")
				.append(asteroid.getX()).append(",")
				.append(asteroid.getY()).append(";")
				.append(asteroid.getType()).append(";")
				.append(asteroid.getSize()).append("\n");
		}
		for (Shot shot: shots) {
			stringBuilder.append("shot:")
				.append(shot.getX()).append(",")
				.append(shot.getY()).append("\n");
		}
		return stringBuilder.toString();
	}

	/**
	 * Returns the current game-time.
	 *
	 * @return the current game-time
	 */
	public int getTicks() {
		return ticks;
	}

	/**
	 * Returns the current ship.
	 *
	 * @return the current ship, or <code>null</code> if none is found.
	 */
	public Ship getShip() {
		return ship;
	}

	/**
	 * Returns the current saucer.
	 *
	 * @return the current saucer, or <code>null</code> if none is found.
	 */
	public Saucer getSaucer() {
		return saucer;
	}

	/**
	 * Returns the current list of asteroids.
	 *
	 * @return the current list of asteroids.
	 */
	public List<Asteroid> getAsteroids() {
		return asteroids;
	}

	/**
	 * Returns the current list of shots.
	 *
	 * @return the current list of shots.
	 */
	public List<Shot> getShots() {
		return shots;
	}
}
