#include <math.h>
#include <stdio.h>
#include <strings.h>
#include "GameState.h"
#include "Globals.h"
#include "KeysPacket.h"

#define V_TRACK_SHOT 0
#define V_NEW_SHOT 0
#define V_TRACK_AST 0
#define V_NEW_AST 0
#define V_CALIB_AB 0
#define V_HYP_NEAREST 0
#define V_HYP 1
#define V_ANGLEBYTE 0
#define V_ANGLEBYTE_OK 1
#define V_ANGLEBYTE_CALIB_OK 1
#define V_NEXT_OBJ 1
#define V_TURN 0
#define V_WAIT_AND_FIRE 0
#define V_WAIT_FOR_TARGET 0


#define MIN_FRAMES_FOR_EXACT_SPEED 8
#define MAX_SHOT_DIST (700*8)

#define SCR_MIN_X 0
#define SCR_MAX_X (SCR_MIN_X + SCR_DX - 1)
#define SCR_DX (1024*8)
#define SCR_HALF_DX (SCR_DX >> 1)

#define SCR_MIN_Y (128*8)
#define SCR_MAX_Y (SCR_MIN_Y + SCR_DY - 1)
#define SCR_DY (768*8)
#define SCR_HALF_DY (SCR_DY >> 1)

#define MIN_SCORE (-1000000000)

// sinus tables for anglebyte
int angle2astsin[256];
int angle2astcos[256];

// initialize sinus table once at startup
class angle2ast_Initializer
{
public:
    angle2ast_Initializer()
    {
	for (int angle = 0; angle < 256; angle++)
	{
	    angle2astsin[angle] = round(127.0 * sin(double(angle) / 128.0 * M_PI));
	    angle2astcos[angle] = round(127.0 * cos(double(angle) / 128.0 * M_PI));
	}
    }
} angle2ast_Initializer_inst;


inline int abs(int i)
{
    return i < 0 ? -i : i;
}


inline void normalizeX(int &x)
{
    while (x < SCR_MIN_X)
	x += SCR_DX;
    while (x > SCR_MAX_X)
	x -= SCR_DX;
}


inline void normalizeY(int &y)
{
    while (y < SCR_MIN_Y)
	y += SCR_DY;
    while (y > SCR_MAX_Y)
	y -= SCR_DY;
}


// screen coordinates to internal cordinates
inline int f2i(int x)
{
    return x * 8;
}


// get distance vector between two points
// return square of distance between two points
inline int getDistVecSqr(int x1, int y1, int x2, int y2, int &dx, int &dy)
{
    dx = x2 - x1;
    dy = y2 - y1;
    dx += SCR_HALF_DX;
    dy += SCR_HALF_DY;
    normalizeX(dx);
    normalizeY(dy);
    dx -= SCR_HALF_DX;
    dy -= SCR_HALF_DY;
    return dx * dx + dy * dy;
}

inline int getDistVec(int x1, int y1, int x2, int y2)
{
    int dx = 0, dy = 0;
    return round(sqrt(getDistVecSqr(x1, y1, x2, y2, dx, dy)));
}


// return true if anglebyte could be determined,
// shift is the number of bits the astsin/cos values need to be shifted right before they match the vx/vy domain
bool getNearestAnglebyte(int vx, int vy, int shift, int &anglebyte_out)
{
    // find nearest angle 
    int nearestAngle = 0;
    int nearestDist = 0x7fffffff;
    for (int angle = 0; angle < 256; angle++)
    {
	// calc dist
	int dx = vx - (angle2astcos[angle] >> shift);
	int dy = vy - (angle2astsin[angle] >> shift);
	int dist = dx * dx + dy * dy;
	
	// remember nearest
	if (dist < nearestDist)
	{
	    nearestDist = dist;
	    nearestAngle = angle;
	}
    }
    
    // return anglebyte if valid
    if (nearestDist <= 4) // I only ever see 0 as dist
    {
#if V_TRACK_SHOT
	if (verbose)
	    printf("getNearestAnglebyte(%d,%d,%d): nearestDist=%d, nearestAngle=%d: dist OK\n",
		   vx, vy, shift, nearestDist, nearestAngle);
#endif	
	anglebyte_out = nearestAngle;
	return true;
    }
    else
    {
	printf("internal error: getNearestAnglebyte(%d,%d,%d): nearestDist=%d, nearestAngle=%d: dist too big\n",
	       vx, vy, shift, nearestDist, nearestAngle);
	return false;
    }
}


// predict one step
void GSObject::predictStep(uint32_t timestamp)
{
    if (valid && vxvyValid && (timestamp >= starttime))
    {
	cur_x += vx;
	normalizeX(cur_x);
	cur_y += vy;
	normalizeY(cur_y);
    }
}


GameState::GameState():
nextAsteroidIndex(0),
nextShotIndex(0),
fireActive(false),
doNotFire(false),
targetAnglebyte(0),
turnDir(3),
targetWait(0),
target(0)
{
    state = CALIBRATE_ANGLEBYTE_WAIT_FOR_SHIP;
    saucer.isSaucer = true;
    char buf[100];
    for (int i = 0; i < MAX_ASTEROIDS; i++)
    {
	sprintf(buf, "asteroid%02d", i);
	asteroids[i].name = buf;
    }
    saucer.name = "saucer";
}
    
 
void GameState::predict(uint32_t newtimestep)
{
    // catch up with time
    while (timestamp < newtimestep)
	predict();
}

void GameState::predict()
{
    // predict a single step
    predictShots();
    predictAsteroids();
    predictShip();
    predictSaucer();
    
    timestamp++;
}


void GameState::predictShip()
{
    ship.predictStep(timestamp);
}


void GameState::predictSaucer()
{
    saucer.predictStep(timestamp);
}


void GameState::predictShots()
{
    for (int i = 0; i < MAX_SHOTS; i++)
	shots[i].predictStep(timestamp);
}


void GameState::predictAsteroids()
{
    for (int i = 0; i < MAX_ASTEROIDS; i++)
	asteroids[i].predictStep(timestamp);
}



void GameState::updateFromFrameData(const FrameData &frameData)
{
    updateShip(frameData);
    updateSaucer(frameData);
    updateShots(frameData);
    updateAsteroids(frameData);
}


void GameState::updateShots(const FrameData &frameData)
{
    // remember which shots in the frame have already been identified
    bool frameShotsDone[MAX_SHOTS];
    memset(frameShotsDone, 0, sizeof(frameShotsDone));    
    
    // for all shots in GameState
    for (int i = 0; i < MAX_SHOTS; ++i)
    {
	if (shots[i].valid && (timestamp >= shots[i].starttime))
	{
	    int nearest = -1;
	    int nearest_dx = 0;
	    int nearest_dy = 0;
	    int mindist = 0x7fffffff;
	    // find nearest shot in frame
	    for (int j = 0; j < frameData.nshots; ++j)
	    {
		if (frameShotsDone[j])
		    continue;
		int dx = 0, dy = 0;
		int dist = getDistVecSqr(shots[i].cur_x, shots[i].cur_y, f2i(frameData.shots[j].x), f2i(frameData.shots[j].y), dx, dy);
		if (dist < mindist)
		{
		    mindist = dist;
		    nearest = j;
		    nearest_dx = dx;
		    nearest_dy = dy;
		}
	    }
	    
	    // check for tight match
	    int age = timestamp - shots[i].starttime;
	    
	    if (mindist <= shots[i].catchdist)
	    {
		// found shot: update shot and mark as done
		frameShotsDone[nearest] = true;
		// update speed vector
		if (!shots[i].vxvyExact)
		{
		    shots[i].cur_x = f2i(frameData.shots[nearest].x);
		    shots[i].cur_y = f2i(frameData.shots[nearest].y);
		    int dx = 0, dy = 0;
		    getDistVecSqr(shots[i].orig_x, shots[i].orig_y, shots[i].cur_x, shots[i].cur_y, dx, dy);
		    shots[i].vx = dx / age;
		    shots[i].vy = dy / age;
		    shots[i].vxvyValid = true;
		    shots[i].catchdist = 10*10*64;
		    if (age >= MIN_FRAMES_FOR_EXACT_SPEED)
		    {
			shots[i].vxvyExact = true;
			shots[i].catchdist = 2*2*64;
			
			int ship_anglebyte = -1;
			if (!shots[i].fromSaucer)
			{
			    if (shotAngles.empty())
			    {
				printf("error: not having anglebyte for shot in fifo, calibrating ...\n");
				state = CALIBRATE_ANGLEBYTE_WAIT_FOR_SHIP;
			    }
			    else
			    {
				ship_anglebyte = shotAngles.front();
				shotAngles.pop_front();
			    }
			}

			// determine anglebyte
			if ((abs(shots[i].vx) <= 103) && (abs(shots[i].vy) <= 103) && (!shots[i].fromSaucer))
			{
			    int vx = shots[i].vx - shots[i].ship_vx;
			    int vy = shots[i].vy - shots[i].ship_vy;
			    shots[i].anglebyteValid = getNearestAnglebyte(vx, vy, 1, shots[i].anglebyte);
			    if (ship_anglebyte >= 0)
			    {
				if (shots[i].anglebyte != ship_anglebyte)
				{
				    printf("error: anglebyte out of sync: ship=%3d, shot=%3d, calibrating ...\n",
					   ship_anglebyte, shots[i].anglebyte);
				    state = CALIBRATE_ANGLEBYTE_WAIT_FOR_SHIP;
				}
				else
				{
#if V_ANGLEBYTE_OK
				    if (verbose)
					printf("GameState::updateShots(): shot #%d: a=%3d, anglebyte OK\n", i, shots[i].anglebyte);
#endif				
				}
			    }
				
#if V_ANGLEBYTE
			    if (verbose)
				printf("GameState::updateShots(): shot #%d: a=%3d, ship.a=%d\n", i, shots[i].anglebyte, ship.anglebyte);
#endif
			}
		    }
		}
#if V_TRACK_SHOT		
		if (verbose)
		    printf("GameState::updateShots(): shot #%d: tracking %4d/%4d dx=%3d, dy=%3d, age=%3d, vx=%3d, vy=%3d, %d %d, mindist=%d, catchdist=%d\n",
			   i, shots[i].cur_x, shots[i].cur_y, nearest_dx, nearest_dy, age, shots[i].vx, shots[i].vy, shots[i].vxvyValid, shots[i].vxvyExact,
			   mindist, shots[i].catchdist);
#endif
	    }
	    else
	    {
		// did not find shot: remove
#if V_NEW_SHOT		
		if (verbose)
		    printf("GameState::updateShots(): shot #%d disappeared at %d,%d, mindist=%d, catchdist=%d, age=%d\n", 
			   i, shots[i].cur_x, shots[i].cur_y, mindist, shots[i].catchdist, age);
#endif		
		shots[i].valid = false;
	    }
	}
    }

    // for all unidentified shots in frame data
    for (int j = 0; j < frameData.nshots; j++)
    {
	if (frameShotsDone[j])
	    continue;

	// find free shot index
	int k;
	for (k = 0; (k < MAX_SHOTS) && shots[nextShotIndex % MAX_SHOTS].valid; k++, nextShotIndex++);
	if (k >= MAX_SHOTS)
	{
	    printf("internal error: GameState::updateShots(): no free shot slot found\n");
	    break;
	}
	int i = nextShotIndex++;
	nextShotIndex %= MAX_SHOTS;

	// create new shot
	shots[i].clear();
	shots[i].cur_x = f2i(frameData.shots[j].x);
	shots[i].cur_y = f2i(frameData.shots[j].y);
	shots[i].orig_x = shots[i].cur_x;
	shots[i].orig_y = shots[i].cur_y;
	shots[i].catchdist = 118*118;
	shots[i].starttime = timestamp;
	shots[i].valid = true;
	
	// check if shot is from saucer
	if (frameData.saucer_present)
	{
	    int dx = 0, dy = 0;
	    int distSaucer = getDistVecSqr(shots[i].cur_x, shots[i].cur_y, f2i(frameData.saucer_x), f2i(frameData.saucer_y), dx, dy);
	    int distShip = getDistVecSqr(shots[i].cur_x, shots[i].cur_y, f2i(frameData.ship_x), f2i(frameData.ship_y), dx, dy);
	    if (distSaucer < distShip)
		shots[i].fromSaucer = true;
	}
	
	// assign ship speed if not from saucer
	if ((!shots[i].fromSaucer) && ship.vxvyValid)
	{
	    shots[i].ship_vx = ship.vx;
	    shots[i].ship_vy = ship.vy;
	}

#if V_NEW_SHOT		
	if (verbose)
	    printf("GameState::updateShots(): new shot appeared: assigning slot #%d (at %d,%d), saucer=%d\n", i, shots[i].cur_x, shots[i].cur_y, shots[i].fromSaucer);
#endif	
    }
}


void GameState::updateAsteroids(const FrameData &frameData)
{
    // remember which asteroids in the frame have already been identified
    bool frameAsteroidsDone[MAX_ASTEROIDS];
    memset(frameAsteroidsDone, 0, sizeof(frameAsteroidsDone));
    
    // for all asteroids in GameState
    for (int i = 0; i < MAX_ASTEROIDS; ++i)
    {
	if (asteroids[i].valid && (timestamp >= asteroids[i].starttime))
	{
	    int nearest = -1;
	    int nearest_dx = 0;
	    int nearest_dy = 0;
	    int mindist = 0x7fffffff;
	    // find nearest asteroid in frame
	    for (int j = 0; j < frameData.nasteroids; ++j)
	    {
		if (frameAsteroidsDone[j])
		    continue;
		int dx = 0, dy = 0;
		int dist = getDistVecSqr(asteroids[i].cur_x, asteroids[i].cur_y, f2i(frameData.asteroids[j].x), f2i(frameData.asteroids[j].y), dx, dy);
		if (dist < mindist)
		{
		    mindist = dist;
		    nearest = j;
		    nearest_dx = dx;
		    nearest_dy = dy;
		}
	    }
	    
	    // check for tight match
	    int age = timestamp - asteroids[i].starttime;
	    
	    if (mindist <= asteroids[i].catchdist)
	    {
		// found asteroid: update asteroid and mark as done
		frameAsteroidsDone[nearest] = true;
		// update speed vector
		if (!asteroids[i].vxvyExact)
		{
		    asteroids[i].cur_x = f2i(frameData.asteroids[nearest].x);
		    asteroids[i].cur_y = f2i(frameData.asteroids[nearest].y);
		    int dx = 0, dy = 0;
		    getDistVecSqr(asteroids[i].orig_x, asteroids[i].orig_y, asteroids[i].cur_x, asteroids[i].cur_y, dx, dy);
		    asteroids[i].vx = dx / age;
		    asteroids[i].vy = dy / age;
		    asteroids[i].vxvyValid = true;
		    asteroids[i].catchdist = 10*10*64;
		    if (age >= MIN_FRAMES_FOR_EXACT_SPEED)
		    {
			asteroids[i].vxvyExact = true;
			asteroids[i].catchdist = 2*2*64;
#if 0			
			// determine anglebyte and make vx and vy even more exact
			if ((abs(asteroids[i].vx) <= 103) && (abs(asteroids[i].vy) <= 103) && (!asteroids[i].fromSaucer))
			{
			    int vx = asteroids[i].vx - asteroids[i].ship_vx;
			    int vy = asteroids[i].vy - asteroids[i].ship_vy;
			    asteroids[i].anglebyteValid = getNearestAnglebyte(vx, vy, 1, asteroids[i].anglebyte);
			}
#endif			
		    }
		}
#if V_TRACK_AST		
		if (verbose)
		    printf("GameState::updateAsteroids(): asteroid #%d: tracking %4d/%4d dx=%3d, dy=%3d, age=%3d, vx=%3d, vy=%3d, %d %d, mindist=%d, catchdist=%d\n",
			   i, asteroids[i].cur_x, asteroids[i].cur_y, nearest_dx, nearest_dy, age, asteroids[i].vx, asteroids[i].vy, asteroids[i].vxvyValid, asteroids[i].vxvyExact,
			   mindist, asteroids[i].catchdist);
#endif
	    }
	    else
	    {
		// did not find asteroid: remove
#if V_NEW_AST
		if (verbose)
		    printf("GameState::updateAsteroids(): asteroid #%d disappeared at %d,%d, mindist=%d, catchdist=%d, age=%d\n",
			   i, asteroids[i].cur_x, asteroids[i].cur_y, mindist, asteroids[i].catchdist, age);
#endif		
		asteroids[i].valid = false;
	    }
	}
    }

    // for all unidentified asteroids in frame data
    for (int j = 0; j < frameData.nasteroids; j++)
    {
	if (frameAsteroidsDone[j])
	    continue;

	// find free asteroid index
	int k;
	for (k = 0; (k < MAX_ASTEROIDS) && asteroids[nextAsteroidIndex % MAX_ASTEROIDS].valid; k++, nextAsteroidIndex++);
	if (k >= MAX_ASTEROIDS)
	{
	    printf("internal error: GameState::updateAsteroids(): no free asteroid slot found\n");
	    break;
	}
	int i = nextAsteroidIndex++;
	nextAsteroidIndex %= MAX_ASTEROIDS;

	// create new asteroid
	asteroids[i].clear();
	asteroids[i].cur_x = f2i(frameData.asteroids[j].x);
	asteroids[i].cur_y = f2i(frameData.asteroids[j].y);
	asteroids[i].orig_x = asteroids[i].cur_x;
	asteroids[i].orig_y = asteroids[i].cur_y;
	asteroids[i].catchdist = 118*118;
	asteroids[i].starttime = timestamp;
	asteroids[i].type = frameData.asteroids[j].type;
	asteroids[i].sf = frameData.asteroids[j].sf;
	asteroids[i].valid = true;
	
#if V_NEW_AST
	if (verbose)
	    printf("GameState::updateAsteroids(): new asteroid appeared: assigning slot #%d (at %d,%d), saucer=%d\n", i, asteroids[i].cur_x, asteroids[i].cur_y, asteroids[i].fromSaucer);
#endif	
    }
}


void GameState::updateShip(const FrameData &frameData)
{
    ship.valid = frameData.ship_present;
    ship.cur_x = f2i(frameData.ship_x);
    ship.cur_y = f2i(frameData.ship_y);
}


void GameState::updateSaucer(const FrameData &frameData)
{
    saucer.valid = frameData.saucer_present;
    saucer.cur_x = f2i(frameData.saucer_x);
    saucer.cur_y = f2i(frameData.saucer_y);
    saucer.sf = frameData.saucer_size;
}


void GameState::getKeys(KeysPacket &keysPacket)
{
    // handle fire
    if (doNotFire)
	doNotFire = false;
    if (fireActive)
    {
	// set fire low for at least one frame
	doNotFire = true;
	fireActive = false;
    }

    // check whetehr ship is present
    if (!ship.valid)
    {
	// reset statemachine and wait for ship
	state = CALIBRATE_ANGLEBYTE_WAIT_FOR_SHIP;
    }
    else
    {
	// check whether we need hyperspace
	if (needHyperspace())
	    keysPacket.hyperspace(true);
    }
    
dostate:
    switch (state)
    {
    case CALIBRATE_ANGLEBYTE_WAIT_FOR_SHIP:
	shotAngles.clear();
	if (ship.valid)
	{
#if V_CALIB_AB	    
	    if (verbose)
		printf("GameState::getKeys(): calibrate anglebyte: detected ship\n");
#endif	    
	    state = CALIBRATE_ANGLEBYTE_WAIT_FOR_NO_SHOT;
	    goto dostate;
	}
	break;
	
    case CALIBRATE_ANGLEBYTE_WAIT_FOR_NO_SHOT:
	{
	    int numValidShots = 0;
	    for (int i = 0; i < MAX_SHOTS; ++i)
	    {
		if (shots[i].valid && (!shots[i].fromSaucer))
		{
		    ++numValidShots;
		}
	    }
	    if (numValidShots == 0)
	    {
#if V_CALIB_AB	    
		if (verbose)
		    printf("GameState::getKeys(): calibrate anglebyte: no shots visible: firing single shot ...\n");
#endif		
		state = CALIBRATE_ANGLEBYTE_FIRE;
		goto dostate;
	    }
	}
	break;

    case CALIBRATE_ANGLEBYTE_FIRE:
	keysPacket.fire(true);
	fireActive = true;
	shotAngles.push_back(-1);
	state = CALIBRATE_ANGLEBYTE_WAIT_FOR_SHOT;
#if V_CALIB_AB	    
	if (verbose)
	    printf("GameState::getKeys(): calibrate anglebyte: firing ...\n");
#endif	
	timeout = timestamp + 5;
	break;
	
    case CALIBRATE_ANGLEBYTE_WAIT_FOR_SHOT:
	// check if we have a shot
	for (int i = 0; i < MAX_SHOTS; ++i)
	{
	    if (shots[i].valid && (!shots[i].fromSaucer))
	    {
		timeout = timestamp + 20;
		state = CALIBRATE_ANGLEBYTE_TRACK;
#if V_CALIB_AB	    
		if (verbose)
		    printf("GameState::getKeys(): calibrate anglebyte: found shot, tracking ...\n");
#endif		
		goto dostate;
	    }	    
	}
	if (timestamp >= timeout)
	{
	    printf("GameState::getKeys(): error: did not see shot after 5 frames, shooting again\n");
	    state = CALIBRATE_ANGLEBYTE_FIRE;
	}
	break;

    case CALIBRATE_ANGLEBYTE_TRACK:
	// check if shot has a valid anglebyte
	{
	    int numValidShots = 0;
	    bool foundAngleByte = false;
	    for (int i = 0; i < MAX_SHOTS; ++i)
	    {
		if (shots[i].valid && (!shots[i].fromSaucer))
		{
		    ++numValidShots;
		    if (shots[i].anglebyteValid)
		    {
#if V_CALIB_AB	    
			if (verbose)
			    printf("GameState::getKeys(): calibrate anglebyte: predicted/current anglebyte %d / %d\n",				   
				   ship.anglebyte, shots[i].anglebyte);
#endif			
			ship.anglebyte = shots[i].anglebyte;
			foundAngleByte = true;
		    }
		}
	    }	    
	    if (numValidShots == 0)
	    {
		// no shot visible: fire again
		printf("GameState::getKeys(): recoverable error: lost shot before determining anglebyte, shooting again\n");
		state = CALIBRATE_ANGLEBYTE_FIRE;
	    }
	    else if (numValidShots > 1)
	    {
		// too many visible shots: wait until they are all gone
		printf("GameState::getKeys(): recoverable error: too many shoots around, waiting until they are gone ...\n");
		state = CALIBRATE_ANGLEBYTE_WAIT_FOR_NO_SHOT;
	    }
	    else if (numValidShots == 1)
	    {
		if (foundAngleByte)
		{
#if V_ANGLEBYTE_CALIB_OK
		    if (verbose)
			printf("GameState::getKeys(): anglebyte calibrated OK: %3d\n", ship.anglebyte);
#endif				
		    targetAnglebyte = ship.anglebyte;
		    setupNext();
#if V_CALIB_AB	    
		    if (verbose)
			printf("GameState::getKeys(): calibrate anglebyte: found valid anglebyte %d, now fight ...\n", ship.anglebyte);
#endif		    
		    goto dostate;
		}
	    }
	}
	if (timestamp >= timeout)
	{
	    printf("GameState::getKeys(): error: stalled in CALIBRATE_ANGLEBYTE_TRACK\n");
	    state = CALIBRATE_ANGLEBYTE_WAIT_FOR_SHIP;
	}
	break;
	
    case TURN:
#if V_TURN	
	if (verbose)
	    printf("TURN %d %d\n", ship.anglebyte, targetAnglebyte);
#endif
	if (ship.anglebyte == targetAnglebyte)
	{
	    state = WAIT_AND_FIRE;
//	    goto dostate;
	}
	else
	{	    
	    if (turnDir > 0)
	    {
		keysPacket.left(true);
		ship.anglebyte += 3;
	    }
	    else
	    {
		keysPacket.right(true);
		ship.anglebyte -= 3;
	    }
	    ship.anglebyte &= 255;
	}
	break;
	
    case WAIT_AND_FIRE:
#if V_WAIT_AND_FIRE
	if (verbose)
	    printf("WAIT_AND_FIRE %d %d %d\n", targetWait, doNotFire, getFreeShots());
#endif
	if (targetWait == 0)
	{
	    if (doNotFire || (getFreeShots() == 0))
	    {
	    }
	    else
	    {		
		keysPacket.fire(true);
		fireActive = true;
		shotAngles.push_back(ship.anglebyte);
		setupNext();
		goto dostate;
	    }
	}
	else
	{
	    targetWait--;
	}
	break;

	// no target is visible, nothing to do
    case WAIT_FOR_TARGET:
#if V_WAIT_FOR_TARGET
	if (verbose)
	    printf("WAIT_FOR_TARGET\n");
#endif
	setupNext();
	break;

    case FIGHT:
	keysPacket.left(true);
	state = CALIBRATE_ANGLEBYTE_WAIT_FOR_NO_SHOT;
	ship.anglebyte += 3;
	ship.anglebyte &= 255;
	break;
	
    default:
	printf("GameState::getKeys(): error, unknown state %d\n", state);
	break;
    }
}


bool GameState::needHyperspace()
{
    int mindist = 0x7fffffff;
    const char *reason = "";
    
    // calc minimum distance for all asteroids
    for (int i = 0; i < MAX_ASTEROIDS; i++)
    {
	if (!asteroids[i].valid)
	    continue;
	
	// get distance of center
	int dist = getDistVec(ship.cur_x, ship.cur_y, asteroids[i].cur_x, asteroids[i].cur_y);
	// subtract radius of object
	switch (asteroids[i].sf)
	{	
	default:
	case 0:  // big
	    dist -= 40*8;
	    break;
	case 15: // medium
	    dist -= 20*8;
	    break;
	case 14: // small
	    dist -= 8*8;
	    break;
	}
	// get minimum
	if (dist < mindist)
	{
	    mindist = dist;
	    reason = "asteroid";
	}
    }
    
    // calc minimum distance for all saucer shots
    for (int i = 0; i < MAX_SHOTS; i++)
    {
	if (!shots[i].valid)
	    continue;
	if (!shots[i].fromSaucer)
	    continue;
	
	// get distance of center
	int dist = getDistVec(ship.cur_x, ship.cur_y, shots[i].cur_x, shots[i].cur_y);
	// subtract radius of object
	// (shots are fast so assume some real radius)
	dist -= 8*8;
	// get minimum
	if (dist < mindist)
	{
	    mindist = dist;
	    reason = "saucer shot";
	}
    }
    
    // saucer
    if (saucer.valid)
    {
	// get distance of center
	int dist = getDistVec(ship.cur_x, ship.cur_y, saucer.cur_x, saucer.cur_y);
	// subtract radius of object
	// (shots are fast so assume some real radius)
	if (saucer.sf == 15)
	    dist -= 20*8;	
	else 
	    dist -= 10*8;
	// get minimum
	if (dist < mindist)
	{
	    mindist = dist;
	    reason = "saucer";
	}
    }
    
#if V_HYP_NEAREST
    if (mindist < 50*8)
    {
	if (verbose)	
	    printf("hyperspace: nearest object: dist=%4d %s\n", mindist/8, reason);
    }
#endif

    // return true on collision
    if (mindist < 40*8) // 27 too tight?
    {
#if V_HYP
	if (verbose)
	    printf("hyperspace because of near %s\n", reason);
#endif
	return true;
    }
    
    return false;
}


int GameState::getFreeShots()
{
    int numShots = 0;
    for (int i = 0; i < MAX_SHOTS; i++)
    {
	if (shots[i].valid && (!shots[i].fromSaucer))
	    numShots++;
    }
    if (numShots < 4)
	return 4 - numShots;
    return 0;
}


void GameState::setupNext()
{
    // find next target
    target = findNextTarget();

    if (target)
    {
	// calc parameter for target
	// set targetAnglebyte and targetWait
	if (!calcTarget())
	{
	    // no targetreachable
	    state = WAIT_FOR_TARGET;
	}
	else
	{
	    state = TURN;
	    target->ignoreUntil = timestamp + 0;
	}
    }
    else
    {
	// no object visible
	state = WAIT_FOR_TARGET;
    }    
}


GSObject *GameState::findNextTarget()
{
    // assign scores to asteroids
    for (int i = 0; i < MAX_ASTEROIDS; i++)
    {
	asteroids[i].score = MIN_SCORE;
	if (!asteroids[i].valid)
	    continue;
	if (timestamp <= asteroids[i].ignoreUntil)
	    continue;
	// use distance as negative score
	int dist = getDistVec(ship.cur_x, ship.cur_y, asteroids[i].cur_x, asteroids[i].cur_y);
	if (dist < MAX_SHOT_DIST)
	    asteroids[i].score = -dist;
    }

    // assign score to saucer
    saucer.score = MIN_SCORE;
    if (saucer.valid && (timestamp > saucer.ignoreUntil))
    {
	int dist = getDistVec(ship.cur_x, ship.cur_y, saucer.cur_x, saucer.cur_y);
	if (dist < MAX_SHOT_DIST)
	    saucer.score = 10;
    }
    
    // find object with highest score
    GSObject *obj = 0;
    int maxscore = MIN_SCORE;
    for (int i = 0; i < MAX_ASTEROIDS; i++)
    {
	if (asteroids[i].score > maxscore)
	{
	    maxscore = asteroids[i].score;
	    obj = &asteroids[i];
	}
    }
    if (saucer.score > maxscore)
    {
	maxscore = saucer.score;
	obj = &saucer;
    }
    
#if V_NEXT_OBJ
    if (verbose)
    {
	if (obj)
	{
	    printf("GameState::findNextObject(): targeting %s at score %d\n",
		   obj->name.c_str(), obj->score);
	}
	else
	{
	    printf("GameState::findNextObject(): no object to target\n");
	}
    }
#endif			
    return obj;
}


static inline void normalizeAngle(double a)
{
    while (a < M_PI) a += 2 * M_PI;
    while (a > M_PI) a -= 2 * M_PI;
}


static void getNearestAngle(int in, int ref, int &nearest_out, int &dir, int &steps)
{
    int diff = (in - ref) & 255;
    if (diff < 128)
    {
	dir = +3;
	diff = (diff / 3) * 3;
	steps = diff / 3;
	nearest_out = (ref + diff) & 255;
    }
    else
    {
	dir = -3;
	diff = 256 - diff;
	diff = ((diff + 2) / 3) * 3;
	steps = diff / 3;
	nearest_out = (ref - diff) & 255;
    }
}
    


bool GameState::calcTarget()
{
    // calc dist vec
    int dx = 0, dy = 0;
    int dist = getDistVecSqr(ship.cur_x, ship.cur_y, target->cur_x, target->cur_y, dx, dy);
    dist = round(sqrt(dist));
    
    // calc angles
    double exactAngle = atan2(dy, dx);
    // get next timestep position to determine angle
    int dx2 = 0, dy2 = 0;
    int t_x2 = target->cur_x + target->vx;
    int t_y2 = target->cur_y + target->vy;
    int dist2 = getDistVecSqr(ship.cur_x, ship.cur_y, t_x2, t_y2, dx2, dy2);
    double exactAngle2 = atan2(dy2, dx2);
    double exactAngleDiff = exactAngle2 - exactAngle;
    normalizeAngle(exactAngleDiff);

    int ta = round(exactAngle * 128.0 / M_PI);
    int tDir = 0;
    int tSteps = 0;
    getNearestAngle(ta, ship.anglebyte, ta, tDir, tSteps);
    printf("ta-1=%d %g\n", ta, exactAngle);
   
    for(int n = 0; n < 5; n++)
    {
	// calc time
	int time = tSteps + (dist - 95) / 63;
    
	// advance object
	GSObject obj = *target;
	for (int t = 0; t < time; t++)
	    obj.predictStep();
	
	dist = getDistVecSqr(ship.cur_x, ship.cur_y, obj.cur_x, obj.cur_y, dx, dy);
	dist = round(sqrt(dist));
	exactAngle = atan2(dy, dx);
	
	ta = round(exactAngle * 128.0 / M_PI);
	tDir = 0;
	tSteps = 0;
	getNearestAngle(ta, ship.anglebyte, ta, tDir, tSteps);
	printf("ta%d=%d %d %d %g\n", n, ta, tDir, tSteps, exactAngle);
    }
    
    // refine
    int mindist = +1000000;
    int minwait = 0;
    int minda = 0;
    int minangle = 0;
    for (int da = -2; da <= +2; da ++)
    {
	for (int wait = 0; wait <= 20; wait += 5)
	{
	    int angle = (ta + (3*da)) & 255;
	    int steps = tSteps;
	    if (tDir > 0)
		steps += da;
	    else
		steps -= da;
	    int dist = simDist(angle, wait, *target, steps);
	    if (dist < mindist)
	    {
		mindist = dist;
		minda = da;
		minwait = wait;
		minangle = angle;
	    }
	}
    }

    
    
    targetAnglebyte = minangle;
    // recalc dir
    int diff = (targetAnglebyte - ship.anglebyte) & 255;
    turnDir = (diff < 128) ? +3 : -3;
    targetAnglebyte &= 255;
    targetWait = minwait;
    printf("AAA targetAnglebyte=%d, ship.anglebyte=%d, wait=%d, dir=%d\n", targetAnglebyte, ship.anglebyte, targetWait, turnDir);
    return true;
}


int GameState::simDist(int angle, int wait, const GSObject &obj_, int steps)
{
    // move object to time of shot start
    GSObject obj = obj_;
    for (int i = 0; i < steps + wait; i++)
	obj.predictStep();
    
    // start prjectile and monitor distance
    int mindist = 0x7fffffff;
    GSObject shot;
    shot.valid = true;
    shot.vxvyValid = true;
    int ac = angle2astcos[angle & 255];
    int as = angle2astsin[angle & 255];
    shot.cur_x = ship.cur_x + (ac >> 1) + (ac >> 2);
    shot.cur_y = ship.cur_y + (as >> 1) + (as >> 2);
    shot.vx = ac >> 1;
    shot.vy = as >> 1;
    for(;;)
    {
	int dist = getDistVec(shot.cur_x, shot.cur_y, obj.cur_x, obj.cur_y);
	if (dist < mindist)
	{
	    mindist = dist;
	    obj.predictStep();
	    shot.predictStep();
	}
	else
	{
	    break;
	}
    }
    return mindist;
}
