// This file is part of "Omniroid", an Asteroids bot written for the 2008 c't anniversary contest
// Omniroid was written by Vladimir "CyberShadow" Panteleev <thecybershadow@gmail.com>
// This file is written in the D Programming Language ( http://digitalmars.com/d/ )

/// Initial state calculation.
module initsearch;

import display;
import network;
import std.stdio;
import asteroids;
import vgadebug;

enum { ASTEROID_SAMPLES = 16 }

struct AsteroidCoord
{
	short[ASTEROID_SAMPLES] sx;
	ubyte found;
	short[8] x0;
	short vx;
	short limit;

	void compute(ubyte[] frameSequence)
	{
		found = 0;
		short x0e = (sx[0]+3)*8+7;
		for (short x0=(sx[0]-3)*8;x0<=x0e;x0++)
			vxLoop:
			for (short vx=-16;vx<=15;vx++)
				if (vx<=-6 || vx>=6)
				{
					for (int i=0;i<frameSequence.length;i++)
					{
						if (((x0 + frameSequence[i]*vx + limit)%limit)/8 != sx[i])
							continue vxLoop;
					}
					// x0,vx match current data
					this.x0[found] = x0%(limit);
					if (found == 0)
						this.vx = vx;
					else
						assert(vx == this.vx);
					found++;
				}
		/+writefln(frameSequence);
		writefln(sx);
		writefln(x0s, " / ", vx);
		writefln(); // +/
		if (found==0)
			throw new Exception("Initial asteroid position sieve ran out of results");
	}

	short[] x0s()
	{
		return x0[0..found];
	}

	bool hasZero()
	{
		foreach(x;x0s)
			if(x==0)
				return true;
		return false;
	}
}

union AsteroidState
{
	struct
	{
		AsteroidCoord x, y;
	}
	union
	{
		AsteroidCoord[2] coords;
	}
}

class InitialSearch
{
	DisplayParser parser;
	int phase;
	ubyte startFrame, fireFrame, thrustFrame, thrustEndFrame;
	ubyte[] frameSequence;
	AsteroidState[4] asteroids;
	ubyte[] placement;
	ubyte startAngle, startFrame8;
	bool[] thrustSequence;
	FramePacket[256] displays;
	short[10] pingFrames;
	
	this()
	{
		parser = new DisplayParser;
		pingFrames[] = -1;
		foreach (ref asteroid;asteroids)
			asteroid.x.limit = 1024*8,
			asteroid.y.limit =  768*8;
	}
	
	/// Find the initial game state from given display data.
	final Game[] search(ref FramePacket frame, ref KeysData keys)
	{
		parser.interpret(frame.display);
		if (pingFrames[frame.ping]==-1)
			pingFrames[frame.ping] = frame.frameno;
		if (phase!=0 && frame.frameno == startFrame)
			throw new Exception("Frame counter overflow - something went wrong during initialization!");
		switch (phase)
		{
			case 0:
				if (parser.nasteroids==0 || !parser.ship_present)
					break;
				startFrame = frame.frameno-1;
				writefln("Commencing initial unknown search procedure...");
				phase++;
				// fall through
			case 1:
				assert(parser.nasteroids == 4);
				int frameDelta = frame.frameno - startFrame;
				if (frameDelta<0) frameDelta += 0x100; // account for frame counter wrap-around
				int sampleIndex = frameSequence.length;
				frameSequence ~= frameDelta;
				foreach (i,ref asteroid;asteroids)
				{
					asteroid.x.sx[sampleIndex] = x2logic(parser.asteroids[i].x);
					asteroid.y.sx[sampleIndex] = y2logic(parser.asteroids[i].y);
				}
				if (frameSequence.length < ASTEROID_SAMPLES)
					break;
				foreach (ref asteroid;asteroids)
					foreach (ref coord;asteroid.coords)
					{
						coord.compute(frameSequence);
						foreach(x;coord.x0s[1..$])
							assert((x >> 8) == (coord.x0[0] >> 8));
					}
				writefln("Initial asteroids coordinates and velocities determined!");
				
				foreach (ref asteroid;asteroids)
				{
					foreach (ref x;asteroid.x.x0s) if (x >= 1024*8) x -= 1024*8;
					foreach (ref y;asteroid.y.x0s) if (y >=  768*8) y -=  768*8;
				}

				// reduce uncertain values for the border coordinate
				foreach (ref asteroid;asteroids)
				{
					bool zx = asteroid.x.hasZero();
					bool zy = asteroid.y.hasZero();
					if (!zx && !zy) throw new Exception("Asteroid is not aligned to boundary on start (packet loss?)");
					if (zx && !zy && asteroid.x.found>1) 
						{ asteroid.x.x0[0] = 0; asteroid.x.found = 1; }
					else
					if (!zx && zy && asteroid.y.found>1) 
						{ asteroid.y.x0[0] = 0; asteroid.y.found = 1; }
				}
				
				foreach_reverse (ref asteroid;parser.asteroids[0..4])
					placement ~= asteroid.type;
				foreach_reverse (ref asteroid;asteroids)
					placement ~= asteroid.x.x0[0] >> 8;
				foreach_reverse (ref asteroid;asteroids)
					placement ~= asteroid.y.x0[0] >> 8;
				foreach_reverse (ref asteroid;asteroids)
					placement ~= asteroid.x.vx;
				foreach_reverse (ref asteroid;asteroids)
					placement ~= asteroid.y.vx;
				
				phase++;
				// fall through
			case 2:
				fireFrame = frame.frameno;
				keys.fire = true;
				keys.ping = phase;
				phase++;
				break;
			case 3:
				if (frame.ping < 2) // fire command not yet received
				{
					fireFrame = frame.frameno; // resend
					break;
				}
				keys.fire = false;
				keys.ping = phase;
				if (parser.nshots==0)
					break;
				uint p = (parser.shots[0].x<<16) | parser.shots[0].y;
				//writefln("Angle: %d, %d", parser.shots[0].x, parser.shots[0].y);
				auto bp = p in bulletPositions;
				assert(bp, "Initial angle not in angle lookup table");
				if (*bp == -1)
					break;
				startAngle = *bp;
				writefln("Initial angle determined!");
				phase++;
				// fall through
			case 4:
				thrustFrame = frame.frameno;
				keys.thrust = true;
				keys.ping = phase;
				frameSequence = null;
				thrustSequence = null;
				phase++;
				break;
			case 5:
				// wait for our packet to arrive
				if (frameSequence.length==0 && !parser.ship_thrust) break;
				phase++;
				//break;
			case 6:
				int frameDelta = frame.frameno - thrustFrame;
				if (frameDelta<0) frameDelta += 0x100; // account for frame counter wrap-around
				frameSequence ~= frameDelta;
				thrustSequence ~= parser.ship_thrust;
				int n = 0;
				for (int sf=0;sf<8;sf++)
				{
					bool ok = true;
					foreach (i,f;frameSequence)
						if (((sf + f) & 4) / 4 != thrustSequence[i])
						{
							ok = false;
							break;
						}
					if (ok)
					{
						n++;
						if (n>1)
							break;
						startFrame8 = sf;
					}
				}
				//writefln(frameSequence);
				//writefln(cast(ubyte[])thrustSequence);
				if (n==0) throw new Exception("Frame number sieve ran out of results");
				if (n>1) break;
				keys.thrust = false;
				keys.ping = phase;
				thrustEndFrame = frame.frameno;
				writefln("Start frame determined!");
				// wait for the last packet with "thrust" to go through
				phase++;
				break;
			case 7:
				if (frame.ping < 6) // "release thrust" command not yet received
				{
					thrustEndFrame = frame.frameno; // resend
					break;
				}
				// create a post-placement template game
				Game base;
				base.Initialize();
				base.ObjAngle[0] = startAngle;
				base.Input_Start = true;
				base.Option_UnknownFrame = true;
				base.Option_UnknownUFOY = true;
				do base.Step(); while (base.Asteroid_Spawn_Timeout);
				
				//writefln(placement);
				auto seeds = placement in startSeeds;
				if (seeds is null) throw new Exception("Initial placements not in seed lookup table");
				writefln("Initial placement matches %d seeds", seeds.length);
				Game[] pool = new Game[seeds.length];
				foreach (i, seed; *seeds)
				{
					pool[i] = base;
					pool[i].RandomSeed = seed;
				}
				
				void stepAll()
				{
					foreach (ref game; pool)
						game.Step();
				}

				stepAll();
				foreach(ref game; pool)
					assert(game.AsteroidCount==4 && game.ObjType[OBJECT_LAST_ASTEROID]!=0);

				// set asteroid micro-positions (and variatons)
				int variationCount = 1;
				foreach(ref asteroid;asteroids)
					foreach(ref coord;asteroid.coords)
						variationCount *= coord.found;
				writefln("Initial position variations: %d possibilities in total", variationCount);
				Game[] newPool = new Game[pool.length * variationCount];
				int newPoolIndex;

				foreach(ref game; pool)
				{
					for (int i=0; i<4; i++)
					{
						assert(game.ObjX[OBJECT_LAST_ASTEROID-i] >> 8 == asteroids[i].x.x0[0] >> 8);
						assert(game.ObjY[OBJECT_LAST_ASTEROID-i] >> 8 == asteroids[i].y.x0[0] >> 8);
						assert(game.ObjSpeedX[OBJECT_LAST_ASTEROID-i] == asteroids[i].x.vx);
						assert(game.ObjSpeedY[OBJECT_LAST_ASTEROID-i] == asteroids[i].y.vx);
					}
					foreach(x0;asteroids[0].x.x0s)
					foreach(y0;asteroids[0].y.x0s)
					foreach(x1;asteroids[1].x.x0s)
					foreach(y1;asteroids[1].y.x0s)
					foreach(x2;asteroids[2].x.x0s)
					foreach(y2;asteroids[2].y.x0s)
					foreach(x3;asteroids[3].x.x0s)
					foreach(y3;asteroids[3].y.x0s)
					{
						auto newGame = &newPool[newPoolIndex];
						*newGame = game;
						newGame.ObjX[OBJECT_LAST_ASTEROID-0] = x0;
						newGame.ObjY[OBJECT_LAST_ASTEROID-0] = y0;
						newGame.ObjX[OBJECT_LAST_ASTEROID-1] = x1;
						newGame.ObjY[OBJECT_LAST_ASTEROID-1] = y1;
						newGame.ObjX[OBJECT_LAST_ASTEROID-2] = x2;
						newGame.ObjY[OBJECT_LAST_ASTEROID-2] = y2;
						newGame.ObjX[OBJECT_LAST_ASTEROID-3] = x3;
						newGame.ObjY[OBJECT_LAST_ASTEROID-3] = y3;
						newPoolIndex++;
					}
				}
				delete pool;
				pool = newPool;

				static ubyte add(ubyte a, int b) { return a+b; } // wrap to 8 bits

				ubyte f = startFrame;
				
				//debug writefln("fireFrame=%d thrustFrame=%d thrustEndFrame=%d", add(fireFrame, -startFrame), add(thrustFrame, -startFrame), add(thrustEndFrame, -startFrame));
				void verifyDisplay(ref FramePacket frame)
				{
					//debug writefln("Verifying frame %d", add(f, -startFrame));
					foreach(game;pool)
					{
						if (!displaysMatch(game.ActiveVideoPage, frame.display))
						{
							dumpScreen("ourgame.txt", game.ActiveVideoPage);
							dumpScreen("realgame.txt", frame.display);
							throw new Exception("Display mismatch");
						}
					}
				}

				foreach(ref game; pool)
					game.FrameCount = add(startFrame8, startFrame-thrustFrame);
				
				while (f != frame.frameno)
				{
					if (f == /*add(fireFrame, 1)*/pingFrames[2])
						foreach(ref game; pool) game.Input_Fire = true;
					if (f == /*add(fireFrame, 2)*/pingFrames[3])
						foreach(ref game; pool) game.Input_Fire = false;
					if (f == /*add(thrustFrame, 1)*/pingFrames[4])
						foreach(ref game; pool) game.Input_Thrust = true;
					if (f == /*add(thrustEndFrame, 1)*/pingFrames[6])
						foreach(ref game; pool) game.Input_Thrust = false;

					if (f != startFrame)
						if (displays[f].frameno == f)
							verifyDisplay(displays[f]);

					stepAll(); f++;
				}

				verifyDisplay(frame);
				
				writefln("Initial search finished with a resulting pool of %d games.", pool.length);
				return pool;
		}
		if (phase)
			displays[frame.frameno] = frame;
		return null;
	}
}

private:

int x2logic(int x)
{
	//return x < 1024/2 ? x : x-1024;
	return x;
}

int y2logic(int y)
{
	y -= 0x400 / 8;
	//return y <  768/2 ? y : y- 768;
	return y;
}

ushort[][ubyte[]] startSeeds;
bool[0x10000] startSeedVisited;
int[uint] bulletPositions;

static this()
{
	writef("Precomputing starting asteroid position tables... ");
	Game game;
	game.Initialize();
	int max;
	while (!startSeedVisited[game.RandomSeed])
	{
		//if (game.RandomSeed==0x64FB) writef("<!>");
		startSeedVisited[game.RandomSeed] = true;
		Game clone = game;
		clone.Random(); // simulate the Random() call in the main loop, that happens before PlaceAsteroids()
		//auto placementRandomSeed = clone.RandomSeed;
		clone.PlaceAsteroids();
		ubyte[] placement;
		//foreach (t;clone.ObjType[MAX_ASTEROIDS-4..MAX_ASTEROIDS])
		//	assert((t&0b100) == 0b100 && (t&0b11100011) == 0);
		foreach (t;clone.ObjType[MAX_ASTEROIDS-4..MAX_ASTEROIDS])
			placement ~= t >> 3;
		foreach (x;clone.ObjX[MAX_ASTEROIDS-4..MAX_ASTEROIDS])
			placement ~= x >> 8;
		foreach (y;clone.ObjY[MAX_ASTEROIDS-4..MAX_ASTEROIDS])
			placement ~= y >> 8;
		foreach (x;clone.ObjSpeedX[MAX_ASTEROIDS-4..MAX_ASTEROIDS])
			placement ~= x;
		foreach (y;clone.ObjSpeedY[MAX_ASTEROIDS-4..MAX_ASTEROIDS])
			placement ~= y;
		if (placement in startSeeds)
			startSeeds[placement] ~= game.RandomSeed;
		else
			startSeeds[placement] = [game.RandomSeed];
		
		game.Random();
	}
	writefln("Done (%d total positions).", startSeeds.length);

	writef("Precomputing starting angle tables... ");
	game.MakeNewGame();
	DisplayParser parser = new DisplayParser;
	for (int a=0; a<0x100; a++)
	{
		Game clone = game;
		clone.ObjAngle[0] = a;
		clone.Input_Fire = true;
		clone.Step();
		clone.Input_Fire = false;
		for (int n=0;n<16;n++)
		{
			parser.interpret(clone.ActiveVideoPage);
			if (parser.nshots)
			{
				uint p = (parser.shots[0].x<<16) | parser.shots[0].y;
				if (p in bulletPositions && bulletPositions[p] != a)
					bulletPositions[p] = -1;
				else
					bulletPositions[p] = a;
			}
			else
				assert(clone.ObjType[OBJECT_LAST_SHIP_BULLET]==0);
			clone.Step();
		}
	}
	int n;
	foreach(k,v;bulletPositions)
		if(v!=-1)
			n++;
	
	writefln("Done (%d/%d coordinates).", n, bulletPositions.length);
}
