// This file is part of "Omniroid", an Asteroids bot written for the 2008 c't anniversary contest
// Omniroid was written by Vladimir "CyberShadow" Panteleev <thecybershadow@gmail.com>
// This file is written in the D Programming Language ( http://digitalmars.com/d/ )

/// Base synchronized bot class.
module syncbot;

import std.c.time;
import std.stdio;
import std.gc;
import basebot;
import network;
import asteroids;
import initsearch;
import display;
import vgadebug, textoutput;
debug import utils, std.file;

/// A base class for bots that need to maintain perfect synchronization with the emulator.
class SyncBot : Bot
{
	InitialSearch initsrc;
	KeysData result;
	KeysData[256] keyHistory;
	ubyte[] keyQueue;
	Game[] pool;
	ubyte frameno, ping, lastReceivedPing;
	debug FILE* dbgout;

	this(NetworkClient network)
	{
		super(network);
		initsrc = new InitialSearch();
		debug dbgout = fopen("dbgout.txt", "wt");
	}

	~this()
	{
		debug fclose(dbgout);
	}

	/// Get input to send (should return quickly).
	KeysData getInput(ref FramePacket frame)
	{
		if (pool is null)
		{
			pool = initsrc.search(frame, result);
			if (pool) // games are all hatched up and ready to leave the incubator :P
			{
				writefln("Entering synchronized mode.");
				stepAllUsing(result);
				frameno = frame.frameno;
				lastReceivedPing = cast(ubyte)-1;
				keyHistory[result.ping] = result;
				keyQueue ~= result.ping;
			}
		}	
		else
		{
			ubyte lostFrames = frame.frameno - frameno - 1; // wrap
			if (lostFrames) // handle lost display frames
			{
				writefln("+ [%04X] %d lost frame packets... ", pool[0].FrameCount+lostFrames+1, lostFrames);
				if (frame.ping == lastReceivedPing)
				{
					writefln("  Key packets were lost (no pool inflation)");
					for (int i=0; i<lostFrames; i++)
						stepAllUsing(keyHistory[lastReceivedPing]);
				}
				else
				{
					ubyte[] lostKeys;
					foreach (i, key; keyQueue)
						if (key == frame.ping)
						{
							lostKeys = [lastReceivedPing] ~ keyQueue[0..i+1];
							keyQueue = keyQueue[i..$];
							break;
						}
					assert(lostKeys, "Unknown key!");
					assert(lostKeys.length >= 2);
					for (int k=lostKeys.length-1; k>0; k--)
						if (keyHistory[lostKeys[k]].keys == keyHistory[lostKeys[k-1]].keys) // identical keys - collapse
						{
							lostKeys = lostKeys[0..k-1] ~ lostKeys[k..$];
						}

					writefln("  %d key packets to arrange...", lostKeys.length);
					
					if (lostKeys.length == 1) // there have been no varying keys sent during the dropped frames
					{
						writefln("  No variations.");
						assert(keyHistory[lostKeys[0]].keys == keyHistory[lastReceivedPing].keys);
						for (int i=0; i<lostFrames; i++)
							stepAllUsing(keyHistory[lastReceivedPing]);
					}
					else
					//if (pool.length * lostKeys.length * lostFrames > 256) // this would inflate the game pool many times
					//	throw new Exception("Too much packet loss - check your network connection");
					//else
					{
						// enumerate any combination of the sequence that keys might have arrived in (maintaining order)
						bool[Game] newPool;
						ubyte[] currentKeys = new ubyte[lostFrames];
					
						void backtrack(int keyIndex, int startPos)
						{
							if (keyIndex == lostKeys.length)
							{
								Game[] ourSet = pool.dup; 
								for (int i = 0; i < lostFrames; i++)
								{
									stepUsing(keyHistory[currentKeys[i]], ourSet, true);
								}
								foreach (ref game;ourSet)
									if (!(game in newPool))
										newPool[game] = true;
							}
							else
								for (int i = startPos; i <= lostFrames; i++)
								{
									backtrack(keyIndex+1, i);
									if (i < lostFrames)
										currentKeys[i] = lostKeys[keyIndex];
								}
						}

						backtrack(0, 0);

						writefln("  Backtracking finished: pool grew from %d to %d", pool.length, newPool.length);
						pool = newPool.keys;
						
						hasNoPointers(pool.ptr);
						foreach(k,v;newPool)
							newPool.remove(k);
					}
				}
				frameno += lostFrames;
			}

			frameno++;
			verifyDisplay(frame);
				
			if (frame.ping == lastReceivedPing)
			{
				writefln("+ [%04X] Key packet is late", pool[0].FrameCount+1);
				stepAllUsing(keyHistory[lastReceivedPing]); // the emulator will re-use the last packet it received
			}
			else
			{
				if (keyQueue.length > 1)
				{
					auto oldLength = keyQueue.length;
					while (keyQueue[0] != frame.ping) // omitted key frames
					{
						keyQueue = keyQueue[1..$];
						assert(keyQueue.length, "Unknown key!");
					}
					writef("+ [%04X] %d key packets omitted", pool[0].FrameCount+1, oldLength-keyQueue.length);
					if (keyQueue.length > 1)
						writefln(" (%d still in queue)", keyQueue.length-1);
					else
						writefln();
				}
				stepAllUsing(keyHistory[keyQueue[0]]);
				keyQueue = keyQueue[1..$]; // the key that was received
			}

			lastReceivedPing = frame.ping;

			result = getInput();
			ping++;
			result.ping = ping;
			keyHistory[ping] = result;
			keyQueue ~= ping;
		}
		return result;
	}

	abstract KeysData getInput();

protected:
final:
	//debug void logState(GameType)(GameType[] pool) // templated due to a bug?
	debug void logState(Game[] pool) // templated due to a bug?
	{
		static int lastFrame;
		static CumulativeText dbg;
		if (pool is null || lastFrame != pool[0].FrameCount)
		{
			fwritef(dbgout, "%s", dbg.flush()); fflush(dbgout); 
			if (pool)
				write(std.string.format("gameframes/%04X.txt", pool[0].FrameCount), gameDump(pool[0])); 
		}
		if (pool)
		{
			foreach (ref game; pool)
			{
				dbg.add(gameDebugInfo(game, scrollLockStatus()));
				//scr.add(gameDump(game));
			}
			lastFrame = pool[0].FrameCount;
		}
	}

	void stepAllUsing(KeysData data)
	{
		stepUsing(data, pool, true);
	}

	static void stepOneUsing(GameType)(KeysData data, ref GameType game)
	{
		game.Input_HyperSpace       = data.hyperspace;
		game.Input_Fire             = data.fire;
		game.Input_Thrust           = data.thrust;
		game.Input_CounterClockwise = data.left;
		game.Input_Clockwise        = data.right;
		game.Step();
	}

	static void stepOneUsingN(GameType)(KeysData data, ref GameType game, int count)
	{
		game.Input_HyperSpace       = data.hyperspace;
		game.Input_Fire             = data.fire;
		game.Input_Thrust           = data.thrust;
		game.Input_CounterClockwise = data.left;
		game.Input_Clockwise        = data.right;
		for (int i=0; i<count; i++)
			game.Step();
	}

	void stepUsing(GameType)(KeysData data, ref GameType[] pool, bool realGames = false)
	{
		debug if (realGames)
			if (is (GameType : Game))
				logState(cast(Game[])pool);
		
		// UFO turning
		uint[] toDup;
		foreach (i, ref game; pool)
			if (game.Option_UnknownFrame && game.ObjType[OBJECT_UFO]>0 && (game.FrameCount&7)==7)
				toDup ~= i;
		if (toDup)
		{
			auto oldLength = pool.length;
			auto newLength = pool.length + toDup.length;
			if (is (GameType : Game))
				if ((cast(Game[])pool).ptr == this.pool.ptr)
					writefln("Testing UFO turning for %d games (from %d to %d)", toDup.length, oldLength, newLength);
			pool.length = newLength;
			foreach (i, n; toDup)
			{
				auto newGame = &pool[oldLength + i];
				*newGame = pool[n];
				newGame.FrameCount = 0x40FF;
				newGame.Option_UnknownFrame = false;
				// if the turn happens, the old games will be dropped; if it doesn't, the new ones will (maybe not immediately)
			}
		}

		// UFO subpositions
		toDup = null;
		foreach (i, ref game; pool)
			if (game.Option_UnknownUFOY)
				if (game.UFO_Timeout==1 && ((game.FrameCount+1)&3)==0 && game.ObjType[OBJECT_UFO]==0 &&    // an UFO will spawn?
					(game.Idle_Timer<=1 || (game.AsteroidCount>0 && game.AsteroidCount<game.UFO_AsteroidThreshhold)))
					toDup ~= i;
		if (toDup)
		{
			auto unduped = pool.length - toDup.length;
			auto oldLength = pool.length;
			auto newLength = unduped + toDup.length*32;
			GameType[] newPool = new GameType[newLength];
			if (is (GameType : Game))
				if ((cast(Game[])pool).ptr == this.pool.ptr)
					writefln("[%04X] Testing UFO subposition for %d games (from %d to %d)", pool[0].FrameCount, toDup.length, oldLength, newLength);
			
			// copy unaffected games, if any
			bool[] dupMap = new bool[pool.length];
			foreach (n; toDup) dupMap[n] = true;
			int m;
			foreach (i, ref game; pool)
				if (!dupMap[i])
					newPool[m++] = game;
			assert(m==unduped);

			foreach (i, n; toDup)
				for (int y=0; y<32; y++)
				{
					auto newGame = &newPool[unduped + i*32 + y];
					*newGame = pool[n];
					newGame.ObjY[OBJECT_UFO] = y << 3;
					newGame.Option_UnknownUFOY = false;
					// the next turn, only 8 of each 32 will remain; those will be filtered away as the UFO fires
				}
			pool = newPool;
		}

		foreach(ref game;pool)
		{
			game.Input_HyperSpace       = data.hyperspace;
			game.Input_Fire             = data.fire;
			game.Input_Thrust           = data.thrust;
			game.Input_CounterClockwise = data.left;
			game.Input_Clockwise        = data.right;
			game.Step();
		}
	}

	void verifyDisplay(ref FramePacket frame)
	{
		//debug writefln("Verifying frame %d", add(f, -startFrame));
		int[] badGames;
		foreach (i,ref game;pool)
		{
			if (!displaysMatch(game.ActiveVideoPage, frame.display))
				badGames ~= i;
		}
		if (badGames.length)
		{
			writefln("Reducing game pool from %d to %d", pool.length, pool.length-badGames.length);
			if (pool.length == badGames.length)
			{
				dumpScreen("ourgame.txt", pool[0].ActiveVideoPage);
				dumpScreen("realgame.txt", frame.display);
				debug 
				{ 
					Game[] nullPool = null;
					logState(pool); logState(nullPool); 
					foreach (i, ref game; pool)
						dumpGame(std.string.format("pool/%04X.txt", i), game);
				}
				throw new Exception("Complete desynchronization!");
			}
			Game[] newPool = new Game[pool.length - badGames.length];
			int n;
			foreach (i,ref game;pool)
				if (badGames.length==0 || i != badGames[0])
					newPool[n++] = game;
				else
					badGames = badGames[1..$];	
			pool = newPool;
		}
	}

	FastGame[] getFastPool()
	{
		return cast(FastGame[])(pool.dup);
	}

	void getFastGame(ref FastGame game)
	{
		game = (cast(FastGame[])pool)[0];
	}
}
