// This file is part of "Omniroid", an Asteroids bot written for the 2008 c't anniversary contest
// Omniroid was written by Vladimir "CyberShadow" Panteleev <thecybershadow@gmail.com>
// This file is written in the D Programming Language ( http://digitalmars.com/d/ )

/// Module to draw the state of the game to a console window.
module textoutput;

import std.stdio;
import asteroids;

/// Draw a game's state to a text screen
string[] renderGame(ref Game game, int w, int h)
{
	string[] screen = new string[h];
	foreach(ref s;screen)
	{
		s = new char[w];
		s[] = ' ';
	}
	foreach (i,type;game.ObjType)
		if (type)
		{
			char c = '?';
			if (i <= OBJECT_LAST_ASTEROID)
				if (type&1)
					c = '+';
				else
				if (type&2)
					c = '*';
				else
					c = '#';
			else
			if (i == OBJECT_SHIP)
				if (type&1)
					c = 'A';
				else
					c = 'a';
			else
			if (i == OBJECT_UFO)
				if (type&1)
					c = 'u';
				else
					c = 'U';
			else
				c = '.';
			screen[(0x17FF-game.ObjY[i])*h/0x1800][game.ObjX[i]*w/0x2000] = c;
		}
	if (game.PlayerCount == 0 && (game.FrameCount & 0x20))
	{
		string s = "PRESS START";
		int x = (w-s.length)/2;
		screen[h/3][x..x+s.length] = s[];
	}
	if (game.PlayerPrompt_Timeout)
	{
		string s = "PLAYER 1";
		int x = (w-s.length)/2;
		screen[h/3][x..x+s.length] = s[];
	}
	return screen;
}

/// A function to dump a game's state to easily readable and comparable text
string gameDebugInfo(ref Game game, bool verbose)
{
	with (game)
	{
		string result = format("Frame %04X: PC=%02X Score=%4d0 Lives=%d SafeJump=%02X PPT=%02X Random=%02X%02X FH=%02X AC=%02X UFOT=%02X UFOL=%02X IT=%02X RT=%02X AST=%02X UFOAT=%02X\n", 
                                FrameCount, 
                                            PlayerCount, 
                                                    (Score % 100_000)/10, 
                                                               Lives,
                                                                        SafeJump,
                                                                                      PlayerPrompt_Timeout,
                                                                                               RandomSeed1, RandomSeed2, 
                                                                                                               FireHistory,
                                                                                                                       AsteroidCount,
        /*o o*/                                                                                                                UFO_Timeout, 
                                                                                                                                         UFO_Level,
        /+___+/                                                                                                                                    Idle_Timer,
                                                                                                                                                           Respawn_Timeout,
                                                                                                                                                                   Asteroid_Spawn_Timeout,
                                                                                                                                                                            UFO_AsteroidThreshhold);
		if (verbose)
		{
			for (int i=0;i<MAX_OBJECTS;i++)
			{
				string name;
				if (i<MAX_ASTEROIDS)                name = "Asteroid";
				else if (i==OBJECT_SHIP)            name = "Ship";
				else if (i==OBJECT_UFO)             name = "UFO";
				else if (i<=OBJECT_LAST_UFO_BULLET) name = "UFO Bullet";
				else                                name = "Ship Bullet";
				result ~= format("            %11s #%2d T=%02X XY=[%04X,%04X] V->", name, i,
					ObjType[i],
					ObjX[i],
					ObjY[i]);
				if (i==OBJECT_SHIP)
					result ~= format("(%04X,%04X) ",
						ShipSpeedX,
						ShipSpeedY);
				else
					result ~= format("(%02X  ,%02X  ) ",
						ObjSpeedX[i], 
						ObjSpeedY[i]);
				if (i==OBJECT_SHIP || i==OBJECT_UFO)
					result ~= format("A=%02X ", ObjAngle[i-OBJECT_SHIP]);
				result ~= \n;
			}
		}
		return result;
	}
}

/// A combination of the above for output to files.
string gameDump(ref Game game)
{
	return join(renderGame(game, 64, 32), \r\n) ~ \r\n ~ gameDebugInfo(game, true);
}

version(Windows)
{
	import std.c.windows.windows;

	extern(Windows) HANDLE GetStdHandle(DWORD);
	extern(Windows) byte GetKeyState(DWORD);

	void printGame(ref Game game)
	{
		SetConsoleCursorPosition(GetStdHandle(STD_OUTPUT_HANDLE), COORD(0, 0));
		CONSOLE_SCREEN_BUFFER_INFO csbi;
		GetConsoleScreenBufferInfo(GetStdHandle(STD_OUTPUT_HANDLE), &csbi);
		auto screen = renderGame(game, csbi.dwSize.X, csbi.dwSize.Y-1);
		foreach(line;screen)
			writef("%s", line);
	}
}

import std.file;
import std.string;

void dumpGame(string fileName, ref Game game)
{
	write(fileName, gameDump(game));
}

void dumpGameN(ref Game game, string prefix="screens/")
{
	static int[string] frameNr;
	if (!(prefix in frameNr)) frameNr[prefix]=0;
	dumpGame(format("%s%04X.txt", prefix, frameNr[prefix]++), game);
}

