// This file is part of "Omniroid", an Asteroids bot written for the 2008 c't anniversary contest
// Omniroid was written by Vladimir "CyberShadow" Panteleev <thecybershadow@gmail.com>
// This file is written in the D Programming Language ( http://digitalmars.com/d/ )

/// Network connection code.
module network;

import std.socket;
import display;

version(Windows) pragma(lib, "wsock32");

align(1)
struct FramePacket
{
    DisplayData display;
	ubyte frameno;
	ubyte ping;

	static assert(this.sizeof == 1026);
}

align(1)
struct KeysData
{
private:
	enum
	{
		KEY_HYPERSPACE = 1,
		KEY_FIRE = 2,
		KEY_THRUST = 4,
		KEY_RIGHT = 8,
		KEY_LEFT = 0x10
	}

	void setKey(ubyte key, bool value)
	{
		if (value)
			keys |= key;
		else
			keys &= ~key;
	}

	bool getKey(ubyte key)
	{
		return (key & keys) != 0;
	}

public:
	ubyte keys = 0x40; // '@'
	ubyte ping;

	void clear() { keys = 0x40; }

	void hyperspace(bool b) { setKey(KEY_HYPERSPACE, b); }
	void fire(bool b)       { setKey(KEY_FIRE, b); }
	void thrust(bool b)     { setKey(KEY_THRUST, b); }
	void left(bool b)       { setKey(KEY_LEFT , b); if (b) right(false); }
	void right(bool b)      { setKey(KEY_RIGHT, b); if (b) left (false); }

	bool hyperspace()       { return getKey(KEY_HYPERSPACE); }
	bool fire()             { return getKey(KEY_FIRE); }
	bool thrust()           { return getKey(KEY_THRUST); }
	bool left()             { return getKey(KEY_LEFT); }
	bool right()            { return getKey(KEY_RIGHT); }

	static assert(this.sizeof == 2);
}

align(1)
struct KeysPacket
{
private:
	char signature[6] = "ctmame";
public:
	KeysData data;

	static assert(this.sizeof == 8);
}

align(1)
struct NamePacket
{
private:
	char signature[6] = "ctname";
public:
    char[32] name;
	static assert(this.sizeof == 38);
}

class NetworkClient
{
	abstract void waitForPacket();
	abstract bool receivePacket(ref FramePacket packet); // non-modal!
	abstract void sendPacket(ref KeysPacket packet);
	abstract void sendName(string name);
	final void sendKeys(KeysData data)
	{
		KeysPacket packet;
		packet.data = data;
		sendPacket(packet);
	}
}

class UdpNetworkClient : NetworkClient
{
	this(string ip)
	{
		host = new InternetHost();
		if (!host.getHostByName(ip))
			throw new Exception("Invalid IP address: '" ~ ip ~ "'");

		socket = new Socket(AddressFamily.INET, SocketType.DGRAM);
		socket.blocking = false;
		socket.bind(new InternetAddress("0.0.0.0", 0));
		int[] size = [FramePacket.sizeof * 16];
		socket.setOption(SocketOptionLevel.SOCKET, SocketOption.RCVBUF, size);
	}

	void waitForPacket()
	{
		SocketSet readfds = new SocketSet(1);
		readfds.reset();
		readfds.add(socket);
		Socket.select(readfds, null, null, null);
	}

	bool receivePacket(ref FramePacket packet) // non-modal!
	{
		SocketSet readfds = new SocketSet(1);
		readfds.reset();
		readfds.add(socket);
		timeval zero;
		zero.tv_sec = zero.tv_usec = 0;
		Socket.select(readfds, null, null, &zero);
		if (readfds.isSet(socket))
		{
			int bytes_received = socket.receive((&packet)[0..1]);
			if (bytes_received != packet.sizeof)
				if (bytes_received > 0)
					throw new Exception((cast(string)((&packet)[0..1]))[0..bytes_received]); // error message from server
				else
					throw new Exception("Error with recvfrom().");
			return true;
		}
		else
			return false;
	}

	void sendPacket(ref KeysPacket packet)
	{
		InternetAddress server = new InternetAddress(host.name, 1979);
		if (packet.sizeof != socket.sendTo((&packet)[0..1], server))
			throw new Exception("Error with sendto().");
	}

	void sendName(string name)
	{
		NamePacket packet;
		packet.name[] = 0;
		packet.name[0..name.length] = name[];
		InternetAddress server = new InternetAddress(host.name, 1979);
		if (packet.sizeof != socket.sendTo((&packet)[0..1], server))
			throw new Exception("Error with sendto().");
	}

	InternetHost host;
	Socket socket;
}
