// This file is part of "Omniroid", an Asteroids bot written for the 2008 c't anniversary contest
// Omniroid was written by Vladimir "CyberShadow" Panteleev <thecybershadow@gmail.com>
// This file is written in the D Programming Language ( http://digitalmars.com/d/ )

/// VectorRAM Disassembler (based on the Perl version by Arne Binder)
module vgadebug;

import std.string;
import std.file;

void dumpScreen(string fileName, ushort[] data)
{
	write(fileName, join(decompileScreen(data), \r\n));
}

void dumpScreenN(ushort[] data, string prefix="frames/")
{
	static int[string] frameNr;
	if (!(prefix in frameNr)) frameNr[prefix]=0;
	dumpScreen(format("%s%04X.txt", prefix, frameNr[prefix]++), data);
}

string[] decompileScreen(ushort[] data)
{
	string[] result;

	ushort op = 0;
	bool end;
	while (!end)
	{
		int x, y, z, s, addr;
		string txt, comment;
		ubyte len;
		switch (data[op] >> 12)
		{
			case 0xA:
				x = data[op+1] & 0x03FF;
				y = data[op] & 0x03FF;
				s = (data[op+1] & 0xF000) >> 12;
				txt = format("LABS (%d, %d), s%d", x, y, s);
				len = 2; comment = clabs();
				break;
			case 0xB:
				txt = format("HALT");
				len = 1; comment = "Program Ended"; end = 1;
				break;
			case 0xC: 
				addr = data[op] & 0x0FFF;
				txt = format("JSRL %03X", addr); 
				len = 1; comment = cjmpaddr(addr);
				break;
			case 0xD:
				txt = format("RTSL");
				len = 1; comment = "Return From Subroutine";
				break;
			case 0xE:
				addr = data[op] & 0x0FFF;
				txt = format("JMPL %03X", addr);
				len = 1; comment = cjmpaddr(addr);
				break;
			case 0xF:
				x = ((data[op] & 0x0004)?-1:1)*(data[op] & 0x0003)*0x0100;
				y = ((data[op] & 0x0400)?-1:1)*(data[op] & 0x0300);
				s = ((data[op] & 0x0008) >> 2)	| ((data[op] & 0x0800) >> 11);
				z = (data[op] & 0x00F0) >> 4;
				txt = format("SVEC (%d, %d), s%d, z%d", x, y, s, z);
				len = 1; comment = csvec();
				break;
			default:
				x = ((data[op+1] & 0x0400)?-1:1)*(data[op+1] & 0x03FF);
				y = ((data[op] & 0x0400)?-1:1)*(data[op] & 0x03FF);
				s = (data[op] & 0xF000) >> 12;
				z = (data[op+1] & 0xF000) >> 12;
				txt = format("VCTR (%d, %d), s%d, z%d", x, y, s, z);
				len = 2; comment = cvctr(x, y, z);
				break;
		}
		string output;
		if (len==1)
			output = format("%03X %04X      %-23s", op, data[op], txt);
		else
			output = format("%03X %04X %04X %-23s", op, data[op], data[op+1], txt);
		if (comment)
			output ~= "; "~comment;
		result ~= output;
		op += len;
		// hard end
		if (op > data.length-2)
			end = true;
	}
	return result;
}

private:
// ***********************
// * Kommentar Datenbank *
// ***********************

// Lange Vektoren
string cvctr(int x, int y, int z) {
	if (x == 0 && y == 0 && z == 0) { return "pause"; }
 return "";
}

// Positionen
string clabs() {
	return "";
}

// Kurze Vektoren
string csvec() {
	return "";
}

// Sprungadressen
string cjmpaddr(ushort addr) {
	if (addr == 0x852) { return "Copyright Notice"; }
	if (addr == 0x880) { return "Large Explosion (3)"; }
	if (addr == 0x896) { return "Medium Explosion (2)"; }
	if (addr == 0x8B5) { return "Medium Explosion (1)"; }
	if (addr == 0x8D0) { return "Small Explosion (0)"; }
	if (addr == 0x8F3) { return "Asteroid Type 1"; }
	if (addr == 0x8FF) { return "Asteroid Type 2"; }
	if (addr == 0x90D) { return "Asteroid Type 3"; }
	if (addr == 0x91A) { return "Asteroid Type 4"; }
	if (addr == 0x929) { return "UFO"; }
	if (addr == 0xA6D) { return "Ship"; }
	if (addr == 0xA78) { return "Letter A"; }
	if (addr == 0xA80) { return "Letter B"; }
	if (addr == 0xA8D) { return "Letter C"; }
	if (addr == 0xA93) { return "Letter D"; }
	if (addr == 0xA9B) { return "Letter E"; }
	if (addr == 0xAA3) { return "Letter F"; }
	if (addr == 0xAAA) { return "Letter G"; }
	if (addr == 0xAB3) { return "Letter H"; }
	if (addr == 0xABA) { return "Letter I"; }
	if (addr == 0xAC1) { return "Letter J"; }
	if (addr == 0xAC7) { return "Letter K"; }
	if (addr == 0xACD) { return "Letter L"; }
	if (addr == 0xAD2) { return "Letter M"; }
	if (addr == 0xAD8) { return "Letter N"; }
	if (addr == 0xADD) { return "Letter O / Number 0"; }
	if (addr == 0xAE3) { return "Letter P"; }
	if (addr == 0xAEA) { return "Letter Q"; }
	if (addr == 0xAF3) { return "Letter R"; }
	if (addr == 0xAFB) { return "Letter S"; }
	if (addr == 0xB02) { return "Letter T"; }
	if (addr == 0xB08) { return "Letter U"; }
	if (addr == 0xB0E) { return "Letter V"; }
	if (addr == 0xB13) { return "Letter W"; }
	if (addr == 0xB1A) { return "Letter X"; }
	if (addr == 0xB1F) { return "Letter Y"; }
	if (addr == 0xB26) { return "Letter Z"; }
	if (addr == 0xB2C) { return "Space"; }
	if (addr == 0xB2E) { return "Number 1"; }
	if (addr == 0xB32) { return "Number 2"; }
	if (addr == 0xB3A) { return "Number 3"; }
	if (addr == 0xB41) { return "Number 4"; }
	if (addr == 0xB48) { return "Number 5"; }
	if (addr == 0xB4F) { return "Number 6"; }
	if (addr == 0xB56) { return "Number 7"; }
	if (addr == 0xB5B) { return "Number 8"; }
	if (addr == 0xB63) { return "Number 9"; }
	if (addr == 0xEEE) { return "* Wildcard"; }  // used internally
	return "";
}
