
#include "world.h"

#include "QUnit.hpp"
#include <iostream>
#include "world.h"
#include "screen.h"

class WorldCoordsTest
{

    QUnit::UnitTest qunit;

    void test_mod()
    {
        QUNIT_IS_EQUAL(1,  mod(1, 7));
        QUNIT_IS_EQUAL(1,  mod(8, 7));
        QUNIT_IS_EQUAL(1,  mod(15, 7));
        QUNIT_IS_EQUAL(1,  mod(-6, 7));
        QUNIT_IS_EQUAL(1,  mod(-13, 7));
        QUNIT_IS_EQUAL(0,  mod(0, 7));
        QUNIT_IS_EQUAL(0,  mod(7, 7));
        QUNIT_IS_EQUAL(0,  mod(14, 7));
        QUNIT_IS_EQUAL(0,  mod(-7, 7));
        QUNIT_IS_EQUAL(0,  mod(-14, 7));
    
    }
    
    void test_absdist()
    {
        QUNIT_IS_EQUAL(1,  (unsigned int)absdist(23, 24));
        QUNIT_IS_EQUAL(1,  (unsigned int)absdist(24, 23));
        QUNIT_IS_EQUAL(11, (unsigned int)absdist(250, 5));
        QUNIT_IS_EQUAL(11, (unsigned int)absdist(5, 250));
    }

    void testAssignByPlus()
    {
        WorldCoords c(20, 30);
        WorldDisp8 d(2, 3);
        c += d;
        QUNIT_IS_EQUAL(22, c.x);
        QUNIT_IS_EQUAL(33, c.y);

        WorldCoords c2(0x1f00, 0x1700);
        WorldDisp16 d2(0x1f1, 0x1f2);
        c2 += d2;
        QUNIT_IS_EQUAL(0xf1, c2.x);
        QUNIT_IS_EQUAL(0xf2, c2.y);

        WorldCoords c3(0x1234, 0x1432);
        WorldDisp8 d3(-1, -2);
        c3 += d3;
        QUNIT_IS_EQUAL(0x1233, c3.x);
        QUNIT_IS_EQUAL(0x1430, c3.y);

        WorldCoords c4(0x000c, 0x000d);
        WorldDisp8 d4(-14, -16);
        c4 += d4;
        QUNIT_IS_EQUAL(0x1ffe, c4.x);
        QUNIT_IS_EQUAL(0x17fd, c4.y);
    }

    void testAssignByMinus()
    {
        WorldCoords c(20, 30);
        WorldDisp8 d(2, 3);
        c -= d;
        QUNIT_IS_EQUAL(18, c.x);
        QUNIT_IS_EQUAL(27, c.y);

        WorldCoords c2(0xf2, 0xf5);
        WorldDisp16 d2(0x1f1, 0x1f3);
        c2 -= d2;
        QUNIT_IS_EQUAL(0x1f01, c2.x);
        QUNIT_IS_EQUAL(0x1702, c2.y);
    }

    void testSubstractWc()
    {
        WorldCoords c0(20, 30);
        WorldCoords c1(4, 6);

        WorldDisp16 d = c0-c1;
        QUNIT_IS_EQUAL(16, d.dx);
        QUNIT_IS_EQUAL(24, d.dy);

        WorldDisp16 d2 = c1-c0;
        QUNIT_IS_EQUAL(-16, d2.dx);
        QUNIT_IS_EQUAL(-24, d2.dy);

        WorldCoords c2(0x1f00, 0x1700);
        WorldCoords c3(0x00ff, 0x0100);

        WorldDisp16 d3 = c2-c3;
        QUNIT_IS_EQUAL(-0x1ff, d3.dx);
        QUNIT_IS_EQUAL(-0x200, d3.dy);

        WorldDisp16 d4 = c3-c2;
        QUNIT_IS_EQUAL(0x1ff, d4.dx);
        QUNIT_IS_EQUAL(0x200, d4.dy);
    }
    
    void test_refinePos()
    {
    //rock554   84    p(2654, 5030)+-(  5,   5)       d(-22,  -6)+-(  1,   1) * 1573  X 1878 by    1
    //    ASTEROID(332, 756)->(2660, 5028)        sf14    T2
    //    ASTEROID(332, 397)->(2660, 2156)        sf14    T2
    
        Asteroid rock554(554, 1573, WorldCoords(2654, 5030), 84);
        rock554.p_error = WorldDisp16(5,5);
        rock554.d = WorldDisp8(-22, -6);
        rock554.d_error = WorldDisp8(1,1);
        ScnCoords sc1(332, 756);  
        ScnCoords sc2(332, 397);
        QUNIT_IS_TRUE((rock554.inRange(sc1)));
        QUNIT_IS_FALSE((rock554.inRange(sc2)));
    }   

    // uint16_t _id, uint16_t _size, const WorldCoords& pos, const WorldDisp8& _d, uint8_t _d_error )
    void test_collide()
    {
        Object o0(0, 1, UINT32_MAX, 100, WorldCoords(0,0), WorldDisp8(1,1), 0);
        Object o1(1, 1, UINT32_MAX, 100, WorldCoords(1000,800), WorldDisp8(0,0), 0);
        QUNIT_IS_EQUAL(1800,  collide(o0, o1, 1000, 2000, false));

        Object o2(2, 1, UINT32_MAX, 100, WorldCoords(1000,1000), WorldDisp8(0,0), 0);
        QUNIT_IS_EQUAL(1850,  collide(o0, o2, 1000, 2000, false));
        
        Object o3(3, 1, UINT32_MAX, 100, WorldCoords(0,0), WorldDisp8(1,24), 0);
        QUNIT_IS_EQUAL(1800,  collide(o3, o1, 1000, 2000, false));
        
        Shot s(16, 188, WorldCoords(4349, 3168));
        s.d = WorldDisp8(63, 0);
        Asteroid a(13, 188, WorldCoords(8040, 3496), 264);
        a.d = WorldDisp8(-6, -6);
        QUNIT_IS_EQUAL(239,  collide(s, a, 188, 500, false));
        QUNIT_IS_EQUAL(239,  collide(a, s, 188, 500, false));
    }
           void test_collide_ship2_rock81()
    {
/*
//Time:395
//ship2     56    p(4192, 3168)+-  0      d(  0,   0)+-  0        X  --- by    0  [1e,1e]
//rock80    84    p(4465, 3580)+-  2      d( 20,   6)+-  0        X  --- by    0
//rock4    144    p(6971,  240)+-  4      d( -8,  27)+-  0        X  --- by    0
//rock81    84    p(4478, 3348)+-  1      d( 26, -21)+-  0        X  --- by    0
//rock5    144    p(1876, 5056)+-  1      d( -6,  -6)+-  0        X  --- by    0
//rock6    144    p(4068, 2563)+-  4      d(  6, -16)+-  0        X  --- by    0
//rock43    84    p(3984, 1442)+-  1      d(-10,   6)+-  0        X  --- by    0
//rock7     84    p(4237, 2102)+-  2      d( 20,   6)+-  0        X  --- by    0
//rock8     84    p(1498, 3633)+-  2      d(  6, -20)+-  0        X  --- by    0
//rock9     84    p(5449, 3677)+-  2      d(-13, -20)+-  0        X  --- by    0
//
//Collision obj81<-->obj2 in 2147483647
//
//        But then...
//Our Model: Time:695
//sshot102   0    p(4664, 3168)+-  0      d( 63,   0)+-  0        X  758 by    0  ttl:16
//sshot103   0    p(4538, 3168)+-  0      d( 63,   0)+-  0        X  762 by    0  ttl:17
//sshot101   0    p(6554, 3318)+-  0      d( 63,   4)+-  0        X  730 by    0  ttl:9
//ship2     56    p(4192, 3168)+-  0      d(  0,   0)+-  0        X  --- by    0  [9,9]
//rock4    144    p(4571, 2196)+-  4      d( -8,  27)+-  0        X  --- by    0
//rock81    84    p(4086, 3192)+-  1      d( 26, -21)+-  0        X  --- by    0
//rock5    144    p(  76, 3256)+-  1      d( -6,  -6)+-  0        X  --- by    0
//rock43    84    p( 984, 3242)+-  1      d(-10,   6)+-  0        X  --- by    0
//rock7     84    p(2045, 3902)+-  2      d( 20,   6)+-  0        X  --- by    0
//rock8     84    p(3298, 3777)+-  2      d(  6, -20)+-  0        X  --- by    0
//rock9     84    p(1549, 3821)+-  2      d(-13, -20)+-  0        X  --- by    0
//
//ctFrame:171
//Event:ship2 DISAPPEAR
//Event:rock81 DISAPPEAR
  */      
        Ship ship2(2, 395, WorldCoords(4192, 3168), Interval<uint8_t>(0x1e));
        Asteroid rock81(81, 395, WorldCoords(4478, 3348), 84);
        rock81.d = WorldDisp8(26, -21);
        QUNIT_IS_EQUAL(695,  collide(rock81, ship2, 395, 800, true));
    }

    void test_sshot82()
    {
//    Time:222
//    sshot82    0    p(4349, 3143)+-  0      d( 63, -10)+-  0
//    rock65    84    p(6808, 3898)+-  1      d( 18, -22)+-  0
                   
        Shot s(82, 222, WorldCoords(4349, 3143));
        s.d = WorldDisp8(63, -10);
        Asteroid a(65, 222, WorldCoords(6809, 3898), 84);
        a.d = WorldDisp8(18, -22);
        QUNIT_IS_EQUAL(UINT32_MAX,  collide(s, a, 222, 500, false));

        a.p.x = 6808;
        a.p_error = WorldDisp16(1, 0);
        QUNIT_IS_EQUAL(UINT32_MAX,  collide(s, a, 222, 500, false));
    }
               
    void test_angle2ticks()
    {
        QUNIT_IS_EQUAL(-128, (int)(angle2ticks(128))); // 1.5 turns for reverse
        QUNIT_IS_EQUAL(9, (int)(angle2ticks(27)));
        QUNIT_IS_EQUAL(-2, (int)(angle2ticks(250)));
    }

    void test_ast_sin()
    {
        QUNIT_IS_EQUAL( 123, (int)(ast_sin(0x4a)));
        QUNIT_IS_EQUAL( -31, (int)(ast_cos(0x4a)));
        //QUNIT_IS_EQUAL( -16, (int)(ast_cos(0x4a)/2));
        QUNIT_IS_EQUAL( -16, (int)(ast_cos(0x4a)>>1));
    }

    void test_ast_tan()
    {
        QUNIT_IS_EQUAL( 0, (int)(ast_tan(0)));
        QUNIT_IS_EQUAL( 1, (int)(ast_tan(0x20)));
        QUNIT_IS_EQUAL(-1, (int)(ast_tan(0x60)));
        QUNIT_IS_EQUAL( 1, (int)(ast_tan(0xa0)));
        QUNIT_IS_EQUAL(-1, (int)(ast_tan(0xe0)));
    }

    void test_ast_atan2()
    {
        QUNIT_IS_EQUAL(0x00, (int)(ast_atan2( 0,  1)));
        QUNIT_IS_EQUAL(0x20, (int)(ast_atan2( 1,  1)));
        QUNIT_IS_EQUAL(0x60, (int)(ast_atan2( 1, -1)));
        QUNIT_IS_EQUAL(0xa0, (int)(ast_atan2(-1, -1)));
        QUNIT_IS_EQUAL(0xe0, (int)(ast_atan2(-1,  1)));
        
        QUNIT_IS_EQUAL(0x20, (int)(ast_atan2( 1001,  1000)));
        QUNIT_IS_EQUAL(0x20, (int)(ast_atan2(  999,  1000)));
    }
    
    void test_decelerate()
    {
        QUNIT_IS_EQUAL(16256, decelerate(16383));
        QUNIT_IS_EQUAL(-16255, decelerate(-16383));
        
        Ship ship(2, 395, WorldCoords(4192, 3168), Interval<uint8_t>(0x1e));
        ship.d.dx = 0x3f;
        ship.dx_friction = 0xff;
        frameOffset = 0;
        ship.advance(1);
        
        QUNIT_IS_EQUAL(0x3f, (int)(ship.d.dx));
        QUNIT_IS_EQUAL(0x80, (int)(ship.dx_friction));
        
        ship.d.dx = 0xc0;
        ship.dx_friction = 0x01;
        ship.advance(1);
        
        QUNIT_IS_EQUAL(-64, (int)(ship.d.dx));
        QUNIT_IS_EQUAL(0x81, (int)(ship.dx_friction));
    }

    void test_getTimeOfDeath()
    {
        Shot s(0, 1, WorldCoords());
        frameOffset = 1;
        s.ttl = 16;
        QUNIT_IS_EQUAL(272, s.getTimeOfDeath(208));
        frameOffset = 0;
        
        s.ttl = Shot::TTL;
        QUNIT_IS_EQUAL(10690, s.getTimeOfDeath(1000));
        frameOffset = 1;
        QUNIT_IS_EQUAL(10690, s.getTimeOfDeath(1000));
        frameOffset = 2;
        QUNIT_IS_EQUAL(10690, s.getTimeOfDeath(1000));
        frameOffset = 3;
        QUNIT_IS_EQUAL(10690, s.getTimeOfDeath(1000));
    }
    
    void test_boundRockSpeed()
    {
        QUNIT_IS_EQUAL(-31, (int)Asteroid::boundSpeed(-32));
        QUNIT_IS_EQUAL(-17, (int)Asteroid::boundSpeed(-17));
        QUNIT_IS_EQUAL(-6, (int)Asteroid::boundSpeed(-5));
        QUNIT_IS_EQUAL(6, (int)Asteroid::boundSpeed(5));
        QUNIT_IS_EQUAL(17, (int)Asteroid::boundSpeed(17));
        QUNIT_IS_EQUAL(31, (int)Asteroid::boundSpeed(32));
    }
public:

    WorldCoordsTest(std::ostream & out, int verbose_level)
            : qunit(out, verbose_level)
    {}

    int run()
    {
        test_boundRockSpeed();
        test_decelerate();
        test_sshot82();
        test_refinePos();
        test_collide_ship2_rock81();
        test_mod();
        test_absdist();
        testAssignByPlus();
        testAssignByMinus();
        testSubstractWc();
        test_collide();
        test_angle2ticks();
        test_ast_sin();
        test_ast_tan();
        test_ast_atan2();
        test_getTimeOfDeath();
        return qunit.errors();
    }
};


int main(int argc, char** argv)
{
    return WorldCoordsTest(std::cerr, QUnit::verbose).run();
}
