#include "tracking.h"
#include "event.h"

using std::vector;
using std::list;
using std::set;
using std::bitset;
using std::cout;
using std::cerr;
using std::dec;
using std::hex;
using std::max;
using std::endl;
using std::make_pair;
using std::min;
using observation::Observation;
Interval<uint8_t> getHeadings(int16_t heading_dx, int16_t heading_dy);
uint16_t getHeadingHash(int16_t dx, int16_t dy);

const uint8_t TrackedAsteroid::SF_SMALL;
const uint8_t TrackedAsteroid::SF_MEDIUM;

WorldCoords scn2world(const ScnCoords& c)
{
    return WorldCoords(((c.x) << 3) + 4, ((c.y - 128) << 3) + 4);
}

ScnCoords world2scn(const WorldCoords& c)
{
    return ScnCoords((c.x) >> 3, ((c.y) >> 3) + 128);
}

bool Object::inRange(const ScnCoords& c) const {
    
//    uint16_t xmin((c.x) << 3);
//    uint16_t ymax((c.y - 128) << 3);
//    uint16_t xmax(xmin+7);
//    uint16_t ymax(ymin+7);
//    
//    WorldDisp16 dist(pos - p);
    
    ScnCoords topRight(world2scn(p+p_error));
    ScnCoords bottomLeft(world2scn(p-p_error));
    
//    return bottomLeft.x <= c.x && c.x <= topRight.x
//            && bottomLeft.y <= c.y && c.y <= topRight.y;
        return mod(topRight.x - c.x, 1024) <= mod(topRight.x - bottomLeft.x, 1024)
            && mod(topRight.y - c.y, 768) <= mod(topRight.y - bottomLeft.y, 768);

    //int16_t dist = (_pos - p).cheb();

    //cout << "dist:" << dist << endl;
    //return dist <= p_error;
    //return abs(dist.dx) <= p_error.dx &&  abs(dist.dy) <= p_error.dy;

}

TrackedPositions::TrackedPositions()
: lastEntry(0), nextCheck(1) {
    clear();
}

TrackedPositions::TrackedPositions(uint32_t time, const ScnCoords& pos)
: lastEntry(0), nextCheck(1) {
    clear();
    setPos(time, pos);
}

void TrackedPositions::clear() {
    for (size_t i = 0; i < MAX_OBSERVATIONS; ++i) {
        scncoords[i] = ScnCoords();
        obspresent[i] = false;
    }
}

#if 0

TrackedCoordinates::TrackedCoordinates()
: TrackedPositions(), {
}

TrackedCoordinates::TrackedCoordinates(uint32_t time, const ScnCoords& pos)
: firstSighted(time), nextCheck(1) {
}
#endif


uint16_t TrackedUfo::sf2size(uint8_t sf)
{
    uint16_t size;
    switch(sf) {
        case SF_SMALL:
            size = SIZE_SMALL;
            break;
        case SF_LARGE:
            size = SIZE_LARGE;
            break;
        default:
            size = SIZE_SMALL;
            assert(false);
    }
    return size;
}

uint16_t TrackedAsteroid::sf2size(uint8_t sf)
{
    uint16_t size;
    switch(sf) {
        case SF_SMALL:
            size = SIZE_SMALL;
            break;
        case SF_MEDIUM:
            size = SIZE_MEDIUM;
            break;
        case SF_LARGE:
            size = SIZE_LARGE;
            break;
        default:
            size = SIZE_SMALL;
            assert(false);
    }
    return size;
}


bool TrackedPositions::refinePosition(uint32_t time, const WorldDisp8& d, WorldCoords& new_p, WorldDisp16& new_p_error) const {
    const WorldDisp16 d16(d.dx, d.dy);
    const ScnCoords& pnow(getPos(time));
    set<int16_t> xs, ys;
    for (int i = 0; i < 8; ++i) {
        xs.insert(((pnow.x) << 3) + i);
        ys.insert(((pnow.y - 128) << 3) + i);
    }

    for (int i = 1; i < 8; ++i) {
        if (!isObserved(time-i))
            return false;
        const ScnCoords& pi(getPos(time - i));
        WorldDisp16 wd(i * d16);
        for (set<int16_t>::iterator x = xs.begin(); x != xs.end();) {
            if ((mod((*x) - wd.dx, 0x2000) >> 3) == pi.x)
                ++x;
            else
                xs.erase(x++);
        }
        for (set<int16_t>::iterator y = ys.begin(); y != ys.end();) {
            if (((mod((*y) - wd.dy, 0x1800) >> 3) + 128) == pi.y)
                ++y;
            else
                ys.erase(y++);
        }
    }

    uint16_t minx = *min_element(xs.begin(), xs.end());
    uint16_t maxx = *max_element(xs.begin(), xs.end());
    uint16_t miny = *min_element(ys.begin(), ys.end());
    uint16_t maxy = *max_element(ys.begin(), ys.end());
    new_p.x = minx + (maxx - minx) / 2;
    new_p.y = miny + (maxy - miny) / 2;
    new_p_error.dx = max(maxx-new_p.x, new_p.x-minx);
    new_p_error.dy = max(maxy-new_p.y, new_p.y-miny);
    assert (!xs.empty());
    assert (!ys.empty());
    return true;
}

/******************************************************************************/


//bool TrackedObject::refine(Object& o) {
//    bool modified = false;
//    int fc = observations.currentFrame;
//    int f0 = max((int) (o.firstSighted), fc - MAX_OBSERVATIONS);
//    int d = fc - f0;
//    if (d < 9 && (d == 1 || d == 2 || d == 4 || d == 8)) {
//        Observation& oc = observations.getSlot(fc);
//        Observation& o0 = observations.getSlot(f0);
//        const observation::Shot& o0 = o0.shots[o0.shotIds[o.id]];
//        const observation::Shot& oc = oc.shots[oc.shotIds[o.id]];
//        WorldCoords p0(scn2world(o0.pos));
//        WorldCoords pc(scn2world(oc.pos));
//        WorldDisp8 new_d((pc.x - p0.x) / d, (pc.y - p0.y) / d);
//        if (new_d != o.d)
//        {
//            o.d = new_d;
//            modified = true;
//        }
//        o.d_error = 4 / d;
//
//        const ScnCoords p(world2scn(o.pos));
//        if (sc.pos != p) {
//            o.pos = scn2world(sc.pos);
//            o.p_error = 4;
//            modified = true;
//        }
//    }
//    return modified;
//}

uint8_t TrackedShot::refine(uint32_t time, const ScnCoords& pos) {
    uint8_t modified(0);
    setPos(time, pos);
    WorldCoords new_p(p);
    WorldDisp16 new_p_error(p_error);

   if (p_error.cheb() > 4 || d_error.cheb() > 0 || pos != getExpScnCoords()) {
        new_p = scn2world(pos);
        new_p_error = WorldDisp16(4, 4);
    }

    if (nextCheck && isObserved(time - nextCheck)) {
        WorldDisp16 dif = scn2world(pos) - scn2world(getPos(time - nextCheck));
        WorldDisp8 new_d(dif.dx / nextCheck, dif.dy / nextCheck);
        if (new_d != d) {
            d = new_d;
            modified |= Event::REFINE_SPEED;
        }
        WorldDisp8 new_d_error((8 / nextCheck) - 1, (8 / nextCheck) - 1);
        if (new_d_error != d_error) {
            d_error = new_d_error;
            modified |= Event::REFINE_ERROR;
        }
        if ((nextCheck <<= 1) > 8)
            nextCheck = 0;

        if (p_error != WorldDisp16() && d_error == WorldDisp8()) {
            if (!refinePosition(time, d, new_p, new_p_error))
                nextCheck = 8;
        }
    }

        if (new_p != p) {
            p = new_p;
            modified |= Event::REFINE_POS;
        }
        if (new_p_error != p_error) {
            p_error = new_p_error;
            modified |= Event::REFINE_ERROR;
        }
    
    return modified;
}

uint8_t TrackedUfo::refine(uint32_t time, const ScnCoords& pos) {
    uint8_t modified(0);
    setPos(time, pos);
    WorldCoords new_p = p;
    WorldDisp16 new_p_error(p_error);
    bool pixelMiss = world2scn(new_p) != pos;
    if (pixelMiss)
        nextCheck = 1;
    if (nextCheck && isObserved(time - nextCheck)) {
        new_p -= d;
        //new_p_error -= d_error;
        new_p_error.dx -= d_error.dx;
        new_p_error.dy -= d_error.dy;

        WorldDisp16 dif = scn2world(pos) - scn2world(getPos(time - nextCheck));
        WorldDisp8 new_d(dif.dx > 0 ? 16 : -16, dif.dy == 0 ? 0 : dif.dy < 0 ? -16 : 16);
        const WorldDisp8 new_d_error;

        new_p += new_d;

        pixelMiss = world2scn(new_p) != pos;
        if (new_d != d) {
            d = new_d;
            modified |= Event::REFINE_SPEED;
        }
        if (new_d_error != d_error) {
            d_error = new_d_error;
            modified |= Event::REFINE_ERROR;
        }
        nextCheck = 0;
    }

    if (pixelMiss) {
        new_p = scn2world(pos);
        new_p_error = WorldDisp16(0,4);
    }

    if (new_p != p) {
        p = new_p;
        modified |= Event::REFINE_POS;
    }
    return modified;
}

uint8_t TrackedShip::refine(uint32_t time, const ScnCoords& pos, uint16_t headingHash) {
    //    const uint32_t fc = observations.currentTime;
    //    const Observation* o0;
    uint8_t modified(0);
    setPos(time, pos);
    //    uint32_t f0;
    WorldCoords new_p(p);
    WorldDisp16 new_p_error(p_error);
    bool pixelMiss = world2scn(new_p) != pos;

    //    if (pixelMiss)
    //        nextCheck = 1;
    //    if ( nextCheck && (f0 = fc - nextCheck) >= firstSighted && (o0 = observationget(f0))) {
    //        new_p -= d;
    //        new_p_error -= d_error;
    //
    //        WorldDisp16 dif = scn2world(pos) - scn2world(o0->ship.pos);
    //        WorldDisp8 new_d(dif.dx, dif.dy);
    //        const uint8_t new_d_error = 0;
    //
    //        new_p += new_d;
    //
    //        pixelMiss = world2scn(new_p) != pos;
    //        if (new_d != d) {
    //            d = new_d;
    //            modified |= Event::REFINE_SPEED;
    //        }
    //        if (new_d_error != d_error) {
    //            d_error = new_d_error;
    //            modified |= Event::REFINE_ERROR;
    //        }
    //        nextCheck = 0;
    //    }

    Interval<uint8_t> new_heading = heading;
    Interval<uint8_t> possible_headings = getHeadings(headingHash);
    new_heading &= possible_headings;
    if (new_heading.empty()) {
        //assert(false);
        cerr << "error computing heading, resetting" << endl;
        new_heading = possible_headings;
    }

    if (new_heading != heading) {
        heading = new_heading;
        modified |= Event::REFINE_HEADING;
    }

    if (pixelMiss) {
        new_p = scn2world(pos);
        new_p_error = WorldDisp16(4, 4);
    }

    if (new_p != p) {
        p = new_p;
        modified |= Event::REFINE_POS;
    }
    if (new_p_error != p_error) {
        p_error = new_p_error;
        modified |= Event::REFINE_ERROR;
    }
    return modified;
}

uint8_t TrackedAsteroid::refine(uint32_t time, const ScnCoords& pos) {
    uint8_t modified(0);
    setPos(time, pos);
    WorldCoords new_p(p);
    WorldDisp16 new_p_error(p_error);

    if (p_error.cheb() > 4 || d_error.cheb() > 0 || pos != getExpScnCoords()) {
        new_p = scn2world(pos);
        new_p_error = WorldDisp16(4, 4);
    }

    if (nextCheck && isObserved(time - nextCheck)) {
        WorldDisp16 dif = scn2world(pos) - scn2world(getPos(time - nextCheck));
        WorldDisp8 new_d(dif.dx / nextCheck, dif.dy / nextCheck);
        if (new_d != d) {
            d = new_d;
            modified |= Event::REFINE_SPEED;
        }
        WorldDisp8 new_d_error((8 / nextCheck) - 1, (8 / nextCheck) - 1);
        if (new_d_error != d_error) {
            d_error = new_d_error;
            modified |= Event::REFINE_ERROR;
        }
        if ((nextCheck <<= 1) > 8) {
            nextCheck = 0;

            if (p_error!=WorldDisp16() && d_error==WorldDisp8()) {
                if (!refinePosition(time, d, new_p, new_p_error))
                    nextCheck = 8;
            }
        }
    }
    if (new_p != p) {
        p = new_p;
        modified |= Event::REFINE_POS;
    }
    if (new_p_error != p_error) {
        p_error = new_p_error;
        modified |= Event::REFINE_ERROR;
    }

    return modified;
}

/******************************************************************************/

void TrackedGameState::reconcileShots(const Observation& obs, vector<Event>& events) {
    static const int MAXPASS = 3;
    static const int accus[MAXPASS + 1] = {0, 4, 16, INT_MAX
    };
    list<size_t> obs_todo;
    const uint8_t slot(TrackedShot::getSlot(time));
    assert(obs.shots.size() <= GameState::MAX_SHOTS_SHIP + GameState::MAX_SHOTS_UFO);
    for (size_t i = 0; i < obs.shots.size(); ++i)
        obs_todo.push_back(i);

    for (list<TrackedShot>::iterator sshot = sshots.begin(); sshot != sshots.end(); ++sshot)
        sshot->clearObservation(slot);
    for (list<TrackedShot>::iterator ushot = ushots.begin(); ushot != ushots.end(); ++ushot)
        ushot->clearObservation(slot);
    
    for (int pass = 0; pass < MAXPASS; ++pass) {
        size_t mi(0);
        for (list<TrackedShot>::iterator sshot = sshots.begin(); sshot != sshots.end();) {
            if (sshot->isObserved(slot)) {
                ++sshot;
                continue;
            }
            int accu = sshot->p_error.cheb();
            if (!(accus[pass] <= accu && accu < accus[pass + 1])) {
                ++sshot;
                continue;
            }
            bool found = false;
            for (list<size_t>::iterator i = obs_todo.begin(); i != obs_todo.end(); ++i) {
                const observation::Shot& oshot = obs.shots[*i];
                if (oshot.pos == sshot->getExpScnCoords() || sshot->inRange(oshot.pos)) {
                    uint8_t modified = sshot->refine(time, oshot.pos);
                    if (modified)
                        events.push_back(Event(sshot->id, Event::TYPE_SSHOT | modified));
                    found = true;
                    if (mi > *i)
                        moveTo(sshots, sshot++, *i);
                    else
                        ++sshot;
                    ++mi;
                    obs_todo.erase(i);
                    break;
                }
            }
            if (!found) {
                uint16_t id = sshot->id;
                sshot = sshots.erase(sshot);
                events.push_back(Event(id, Event::TYPE_SSHOT | Event::DISAPPEAR));
            }
        }

        mi = 0;
        for (list<TrackedShot>::iterator ushot = ushots.begin(); ushot != ushots.end();) {
            if (ushot->isObserved(slot)) {
                ++ushot;
                continue;
            }
            int accu = ushot->p_error.cheb();
            if (!(accus[pass] <= accu && accu < accus[pass + 1])) {
                ++ushot;
                continue;
            }
            bool found = false;
            for (list<size_t>::iterator i = obs_todo.begin(); i != obs_todo.end(); ++i) {
                const observation::Shot& oshot = obs.shots[*i];
                if (oshot.pos == ushot->getExpScnCoords() || ushot->inRange(oshot.pos)) {
                    uint8_t modified = ushot->refine(time, oshot.pos);
                    if (modified)
                        events.push_back(Event(ushot->id, Event::TYPE_USHOT | modified));
                    found = true;
                    if (mi > *i)
                        moveTo(ushots, ushot++, *i);
                    else
                        ++ushot;
                    obs_todo.erase(i);
                    break;
                }
            }
            if (!found) {
                uint16_t id = ushot->id;
                ushot = ushots.erase(ushot);
                events.push_back(Event(id, Event::TYPE_USHOT | Event::DISAPPEAR));
            }
        }
    }

    for (list<size_t>::iterator i = obs_todo.begin(); i != obs_todo.end(); ++i) {
        uint16_t id = ++nextId;
        TrackedShot s(id, time, obs.shots[*i].pos);
        if (*i + MAX_SHOTS_UFO < obs.shots.size()) {
            list<TrackedShot>::iterator idx = sshots.begin();
            for (size_t j=0; idx!=sshots.end() && j<*i; ++j) ++idx;
            sshots.insert(idx, s);
            events.push_back(Event(id, Event::TYPE_SSHOT | Event::APPEAR));
        } else {
            list<TrackedShot>::iterator idx = ushots.begin();
            for (size_t j=0; idx!=ushots.end() && j<MAX_SHOTS_UFO-(obs.shots.size()-*i); ++j) ++idx;
//            s.size = 1;
            ushots.insert(idx, s);
            events.push_back(Event(id, Event::TYPE_USHOT | Event::APPEAR));
        }
    }
}

#if 0

void TrackedGameState::reconcileShots(vector<Event>& events) {
    Observation& obs = observations.getCurrent();
    int minidx, maxidx;
    list<TrackedShot>::iterator sshot;
    //    model.ushots.reverse();
    reverse(model.ushots.begin(), model.ushots.end());

    minidx = 0;
    maxidx = min((int) (obs.shots.size()), GameState::MAX_SHOTS_SHIP) - 1;

    for (sshot = model.sshots.begin(); sshot != model.sshots.end();) {
        const ScnCoords p(world2scn(sshot->pos));
        bool found = false;
        for (int i = minidx; i <= maxidx; ++i) {
            observation::Shot& oshot = obs.shots[i];
            const bool exact = (oshot.pos == p);
            if (exact || sshot->inRange(oshot.pos)) {
                obs.shotIds[sshot->id] = i;
                uint8_t modified = refineShot(*sshot, !exact);
                if (modified)
                    events.push_back(Event(sshot->id, Event::TYPE_SSHOT | modified));

                for (; minidx < i; ++minidx) {
                    uint16_t id = ++nextId;
                    model.sshots.insert(sshot, TrackedShot(id, obs.time, scn2world(obs.shots[maxidx].pos)));
                    obs.shotIds[id] = minidx;
                    events.push_back(Event(id, Event::TYPE_SSHOT | Event::APPEAR));
                }
                ++minidx;
                found = true;
                break;
            }
        }
        if (!found) {
            uint16_t id = sshot->id;
            sshot = model.sshots.erase(sshot);
            events.push_back(Event(id, Event::TYPE_SSHOT | Event::DISAPPEAR));
        } else
            ++sshot;
    }

    // add definitive sshots
    for (; minidx < (int) (obs.shots.size()) - GameState::MAX_SHOTS_UFO; ++minidx) {
        uint16_t id = ++nextId;
        model.sshots.insert(sshot, TrackedShot(id, obs.time, scn2world(obs.shots[minidx].pos)));
        obs.shotIds[id] = minidx;
        events.push_back(Event(id, Event::TYPE_SSHOT | Event::APPEAR));
    }


    maxidx = obs.shots.size() - 1;
    //    minidx = max(0, (int)(obs.shots.size())-GameState::MAX_SHOTS_UFO);

    for (list<TrackedShot>::iterator ushot = model.ushots.begin(); ushot != model.ushots.end();) {
        const ScnCoords p(world2scn(ushot->pos));
        bool found = false;
        for (int i = maxidx; i >= minidx; --i) {
            observation::Shot & oshot(obs.shots[i]);
            const bool exact = (oshot.pos == p);
            if (exact || ushot->inRange(oshot.pos)) {
                obs.shotIds[ushot->id] = i;
                uint8_t modified = refineShot(*ushot, !exact);
                if (modified)
                    events.push_back(Event(ushot->id, Event::TYPE_USHOT | modified));

                for (; maxidx > i; --maxidx) {
                    uint16_t id = ++nextId;
                    model.ushots.insert(ushot, TrackedShot(id, obs.time, scn2world(obs.shots[maxidx].pos)));
                    obs.shotIds[id] = maxidx;
                    events.push_back(Event(id, Event::TYPE_USHOT | Event::APPEAR));
                }
                --maxidx;
                found = true;
                break;
            }
        }
        if (!found) {
            uint16_t id = ushot->id;
            ushot = model.sshots.erase(ushot);
            events.push_back(Event(id, Event::TYPE_USHOT | Event::DISAPPEAR));
        } else
            ++ushot;
    }

    //reverse(model.ushots.begin(), model.ushots.end());
    model.ushots.reverse();

    // add definitive ushots
    for (; maxidx >= GameState::MAX_SHOTS_SHIP; --maxidx) {
        uint16_t id = ++nextId;
        model.ushots.insert(model.ushots.begin(), TrackedShot(id, obs.time, scn2world(obs.shots[maxidx].pos)));
        obs.shotIds[id] = maxidx;
        events.push_back(Event(id, Event::TYPE_USHOT | Event::APPEAR));
    }

    //assert(minidx >= (int) (obs.shots.size()) - GameState::MAX_SHOTS_SHIP);
    // distribute rest
    for (int i = maxidx; i >= minidx; --i) {
        uint16_t id = ++nextId;
        TrackedShot s(id, obs.time, scn2world(obs.shots[i].pos));
        if ((model.ship.pos - s.pos).euclid() < (model.ufo.pos - s.pos).euclid()) {
            sshot = model.sshots.insert(sshot, s);
            obs.shotIds[id] = i;
            events.push_back(Event(id, Event::TYPE_SSHOT | Event::APPEAR));
        } else {
            model.ushots.insert(model.ushots.begin(), s);
            obs.shotIds[id] = i;
            events.push_back(Event(id, Event::TYPE_USHOT | Event::APPEAR));
        }
    }

    //model.ushots.reverse();
}
#endif

void TrackedGameState::reconcileUFO(const Observation& obs, vector<Event>& events) {
    if (obs.ufo.present) {
        if (!ufo.present) {
            uint16_t id = ++nextId;
            WorldCoords p = scn2world(obs.ufo.pos);
            WorldDisp8 d;
            ufo = TrackedUfo(id, obs.ufo.sf, obs.time, obs.ufo.pos);
            events.push_back(Event(id, Event::TYPE_UFO | Event::APPEAR));
        } else {
            uint8_t modified = ufo.refine(time, obs.ufo.pos);
            if (modified)
                events.push_back(Event(ufo.id, Event::TYPE_UFO | modified));
        }

    } else {
        if (ufo.present) {
            uint16_t id = ufo.id;
            ufo.present = false;
            events.push_back(Event(id, Event::TYPE_UFO | Event::DISAPPEAR));
        }
    }
}

void TrackedGameState::reconcileShip(const Observation& obs, vector<Event>& events) {
    if (obs.ship.present) {
        uint16_t heading = getHeadingHash(obs.ship.heading_dx, obs.ship.heading_dy);
        if (!ship.present) {
            uint16_t id = ++nextId;
            ship = TrackedShip(id, obs.time, obs.ship.pos, heading, ship.p);
            events.push_back(Event(id, Event::TYPE_SHIP | Event::APPEAR));
        } else {
            uint8_t modified = ship.refine(time, obs.ship.pos, heading);
            if (modified)
                events.push_back(Event(ship.id, Event::TYPE_SHIP | modified));
        }
    } else {
        if (ship.present) {
            if (!ship.activeKeys.hyperspace())
                cout << "Ship crashed?" << endl;
            uint16_t id = ship.id;
            ship.present = false;
            events.push_back(Event(id, Event::TYPE_SHIP | Event::DISAPPEAR));
        }
    }
}

#if 0
void TrackedGameState::reconcileAsteroids(const Observation& obs, vector<Event>& events) {
    int minidx(0), maxidx;
    list<TrackedAsteroid>::iterator ast;
    for (ast = asteroids.begin(); ast != asteroids.end();) {
        const ScnCoords p(world2scn(ast->p));
        bool found = false;
        maxidx = obs.asteroids.size() - 1;
        for (int i = minidx; i <= maxidx; ++i) {
            const observation::Asteroid & oast(obs.asteroids[i]);
            const bool exact = (oast.pos == p);
            if (exact || ast->inRange(oast.pos)) {
                uint8_t modified = ast->refine(time, oast.pos);
                if (modified)
                    events.push_back(Event(ast->id, Event::TYPE_ROCK | modified));

                for (; minidx < i; ++minidx) {
                    uint16_t id = ++nextId;
                    asteroids.insert(ast, TrackedAsteroid(id, obs.time, obs.asteroids[minidx].pos, oast.size, oast.type));
                    events.push_back(Event(id, Event::TYPE_ROCK | Event::APPEAR));
                }
                ++minidx;
                found = true;
                break;
            }
        }
        if (!found) {
            uint16_t id = ast->id;
            ast = asteroids.erase(ast);
            events.push_back(Event(id, Event::TYPE_ROCK | Event::DISAPPEAR));
        } else
            ++ast;
    }

    for (; minidx < (int) obs.asteroids.size(); ++minidx) {
        uint16_t id = ++nextId;
        const observation::Asteroid & oast(obs.asteroids[minidx]);
        asteroids.insert(ast, TrackedAsteroid(id, obs.time, oast.pos, oast.size, oast.type));
        events.push_back(Event(id, Event::TYPE_ROCK | Event::APPEAR));
    }
}
#endif

void TrackedGameState::reconcileAsteroids(const Observation& obs, vector<Event>& events) {
    static const int MAXPASS = 3;
    static const int accus[MAXPASS + 1] = {0, 4, 16, INT_MAX
    }; // accuracies
    list<size_t> obs_todo;
    const uint8_t slot(TrackedShot::getSlot(time));
    assert(obs.asteroids.size() <= GameState::MAX_ASTEROIDS);
    for (size_t i = 0; i < obs.asteroids.size(); ++i)
        obs_todo.push_back(i);

    for (list<TrackedAsteroid>::iterator ast = asteroids.begin(); ast != asteroids.end(); ++ast)
        ast->clearObservation(slot);
    
    for (int pass = 0; pass < MAXPASS; ++pass) {
        size_t mi(0);
        for (list<TrackedAsteroid>::iterator ast = asteroids.begin(); ast != asteroids.end();) {
            if (ast->isObserved(slot) || ast->birth > time) {
                ++ast;
                continue;
            }
            int accu = ast->p_error.cheb();
            if (!(accus[pass] <= accu && accu < accus[pass + 1])) {
                ++ast;
                continue;
            }
            bool found = false;
            for (list<size_t>::iterator i = obs_todo.begin(); i != obs_todo.end(); ++i) {
                const observation::Asteroid& oast = obs.asteroids[*i];
                if (oast.sf == ast->sf && (oast.type == ast->type || !ast->type) 
                        && ( oast.pos == ast->getExpScnCoords() || ast->inRange(oast.pos) ) ) {
                    uint8_t modified = ast->refine(time, oast.pos);
                    if (modified)
                        events.push_back(Event(ast->id, Event::TYPE_ROCK | modified));
                    found = true;
                    if (mi > *i)
                        moveTo(asteroids, ast++, *i);
                    else
                        ++ast;
                    ++mi;
                    obs_todo.erase(i);
                    break;
                }
            }
            if (!found) {
                uint16_t id = ast->id;
                ast = asteroids.erase(ast);
                events.push_back(Event(id, Event::TYPE_ROCK | Event::DISAPPEAR));
            }
        }

//        mi = 0;
//        for (list<TrackedShot>::iterator ushot = ushots.begin(); ushot != ushots.end();) {
//            if (ushot->isObserved(slot)) {
//                ++ushot;
//                continue;
//            }
//            int accu = ushot->p_error;
//            if (!(accus[pass] <= accu && accu < accus[pass + 1])) {
//                ++ushot;
//                continue;
//            }
//            bool found = false;
//            for (list<size_t>::iterator i = obs_todo.begin(); i != obs_todo.end(); ++i) {
//                const observation::Shot& oshot = obs.shots[*i];
//                if (oshot.pos == ushot->getExpScnCoords() || ushot->inRange(oshot.pos)) {
//                    obs_todo.erase(i);
//                    uint8_t modified = ushot->refine(time, oshot.pos);
//                    if (modified)
//                        events.push_back(Event(ushot->id, Event::TYPE_USHOT | modified));
//                    found = true;
//                    if (mi > *i)
//                        moveTo(ushots, ushot++, *i);
//                    else
//                        ++ushot;
//                    break;
//                }
//            }
//            if (!found) {
//                uint16_t id = ushot->id;
//                ushot = ushots.erase(ushot);
//                events.push_back(Event(id, Event::TYPE_USHOT | Event::DISAPPEAR));
//            }
//        }
    }

    for (list<size_t>::iterator i = obs_todo.begin(); i != obs_todo.end(); ++i) {
        uint16_t id = ++nextId;
        TrackedAsteroid s(id, time, obs.asteroids[*i].pos, obs.asteroids[*i].sf, obs.asteroids[*i].type);
        list<TrackedAsteroid>::iterator idx = asteroids.begin();
        for (size_t j = 0; idx != asteroids.end() && j<*i; ++j)++idx;
        asteroids.insert(idx, s);
        events.push_back(Event(id, Event::TYPE_ROCK | Event::APPEAR));
    }
}


void TrackedGameState::update(const observation::Observation& obs, std::vector<Event>& events) {
    reconcileShots(obs, events);
    reconcileUFO(obs, events);
    reconcileShip(obs, events);
    reconcileAsteroids(obs, events);
    
    bool disshot(false);
    bool disother(false);
    for (vector<Event>::const_iterator i = events.begin(); i != events.end(); ++i) {
        if (i->flags & Event::DISAPPEAR) {
            if (((i->flags & 7) == Event::TYPE_USHOT)||((i->flags & 7) == Event::TYPE_SSHOT) )
                disshot = true;
            else
                disother = true;
        }
    }
    if (disshot && !disother) {
        cout << "time:" << time << " frameOffset:" << frameOffset << "time & frameoffset:" << ((time+frameOffset)&3) << endl;
        unsigned int nfo = 5 - (time&3) % 4;
        if (nfo != frameOffset) {
            cout << "adjusting FrameOffset to "<<nfo <<endl;
            frameOffset = nfo;
           cout << "time:" << time << " frameOffset:" << frameOffset << "time & frameoffset:" << ((time+frameOffset)&3) << endl;
     }
    }

    for (vector<Event>::const_iterator i = events.begin(); i != events.end(); ++i)
        cout << *i << endl;
    cout << endl;

}

void TrackedGameState::updateCollisions(const std::vector<Event>& events)
{
    set<uint16_t> objs;

    bool disappearance(false);
    for (vector<Event>::const_iterator e = events.begin(); e != events.end(); ++e) {
        if (e->flags & Event::DISAPPEAR) {
            clearCollision(ObjectDeathByMatcher(e->id), false, objs);
            disappearance = true;
        } else if ((e->flags & Event::APPEAR) == 0)
            objs.insert(e->id);
    }

    for (Asteroids_t::iterator a = asteroids.begin(); a != asteroids.end();) {
        if (a->birth > time) {
            bool found(false);
            for (Asteroids_t::iterator ast = asteroids.begin(); ast != asteroids.end(); ++ast)
                if (ast->death == a->birth && ast->size > a->size) { // TODO use id ?
                    found = true;
                    break;
                }
            if (!found) {
                cout << "Clearing leftover rock" << a->id << endl;
                eraseChildren(*a, objs);
                if (a->deathBy)
                    objs.insert(a->deathBy);
                asteroids.erase(a++);
                continue;
            }
        }
        ++a;
    }
    
    
    if (ship.death < time)
        objs.insert(ship.id);

    if (ufo.death < time)
        objs.insert(ufo.id);

    for (Asteroids_t::iterator ast = asteroids.begin(); ast != asteroids.end(); ++ast)
        if (ast->death < time)
            objs.insert(ast->id);

    for (Shots_t::iterator sshot = sshots.begin(); sshot != sshots.end(); ++sshot)
        if (sshot->death < time)
            objs.insert(sshot->id);

    for (Shots_t::iterator ushot = ushots.begin(); ushot != ushots.end(); ++ushot)
        if (ushot->death < time)
            objs.insert(ushot->id);

    for (vector<Event>::const_iterator e = events.begin(); e != events.end(); ++e) {
        if (e->flags & Event::APPEAR)
            objs.insert(e->id);
    }

    assert (objs.find(0) == objs.end());
    clearCollisions(objs, true);
    for (set<uint16_t>::const_iterator o = objs.begin(); o != objs.end(); ++o) {
        collideObject(*o);
    }

}

bool TrackedGameState::findTarget(Target& target) const
{
    // target collider <50
    const Asteroid* collider(0);
    if (ship.deathBy)
        for (list<TrackedAsteroid>::const_iterator a = asteroids.begin();
                a != asteroids.end(); ++a) {
            if (a->id == ship.deathBy) {
                collider = &*a;
                if (ship.death - time < 50) {
                    cout << "Targeting urgent collider rock" << collider->id << endl; 
                    if (targetObject(*collider, target, time+60)) {
                        target.targetId = collider->id;
                        return true;
                    }
                }
            }
        }


    // target ufo
    if (ufo.present && !hitByMe(ufo)) {
            cout << "Targeting UFO" << endl; 
            if (targetObject(ufo, target, time+100)) {
            target.targetId = ufo.id;
                        return true;
        }
    }

    // target collider < 500
    if (collider && ship.death - time < 500) {
        cout << "Targeting collider rock" << collider->id << endl;
        if (targetObject(*collider, target, 60)) {
            target.targetId = collider->id;
                        return true;

        }
    }

    list<Object> targets;
    for (list<TrackedAsteroid>::const_iterator a = asteroids.begin();
            a != asteroids.end(); ++a) {
        if (!hitByMe(*a)) {
            targets.push_back(Object(*a));
        }
    }
    if (targets.empty())
        return false;

    const uint8_t initial_heading(ship.heading.start + ship.heading.n / 2);
    bitset<256> possible_headings;
    possible_headings.reset();
    possible_headings.set(initial_heading);
    const uint32_t tfmin(getTimeOfShotAvailable());
    double slope(0.0);
    const static double SLOPES[5] = { 0.0, 0.0, 1.0, 2.0, 3.0 };
    Ship m_ship(ship);
    target.targetId=0;
    target.thit = UINT32_MAX;
    target.thit = time + 200; // TODO
    for (uint32_t n = 1; true; ++n) {
        const uint32_t tn(time + n);
        uint32_t ththresh = target.thit - (int)((tn - target.tfire) * slope);
        if (ththresh <= tn) {
            cout << "Aborting search at "<<tn << endl;
            break;
        }
        possible_headings.set((uint8_t) (initial_heading + (n-1) * 3));
        possible_headings.set((uint8_t) (initial_heading - (n-1) * 3));

        m_ship.advance((tn+frameOffset) & 1);
        for (list<Object>::iterator tgt = targets.begin(); tgt != targets.end(); ) {
            tgt->advance(tn);
            if (tgt->p_error.cheb() > tgt->size) {
                tgt = targets.erase(tgt);
                continue;
            }
            if (tn >= tfmin) {
                Mark mark;
#ifndef NDEBUG                
                cout << "Targeting rock"<<tgt->id<< " at " << tn<<  " :  ";
#endif
                if (m_ship.targetObject(tn, *tgt, ththresh, possible_headings, mark)) {

                    target.targetId = tgt->id;
                    target.alpha = mark.alpha;
                    target.tfire = tn;
                    target.thit = mark.thit;
#ifndef NDEBUG                     
                    cout << "Found target at " << tn << " : rock" << target.targetId 
                         << " alpha:" << hex << (int)(target.alpha) << dec 
                         << " tfire:" << target.tfire
                         << " thit:" << target.thit << endl;
#endif
                    slope = SLOPES[getShotsAvailableAt(target.tfire)];
                    ththresh = mark.thit;
                }
            }
        ++tgt;
        }
    }
    return target.targetId ? true : false;
}


/******************************************************************************/


// Compute ship heading from dx/dy

namespace {

    static const uint16_t keys[] = {
        0xc000, 0xbf13, 0xbc25, 0xb837, 0xb149, 0xaa5a, 0xa06b, 0x957a,
        0x8888, 0x7a95, 0x6ba0, 0x5aaa, 0x49b1, 0x37b8, 0x25bc, 0x13bf,
        0x00c0, 0xedbf, 0xdbbc, 0xc9b8, 0xb7b1, 0xa6aa, 0x95a0, 0x8695,
        0x7888, 0x6b7a, 0x606b, 0x565a, 0x4f49, 0x4837, 0x4425, 0x4113,
        0x4000, 0x41ed, 0x44db, 0x48c9, 0x4fb7, 0x56a6, 0x6095, 0x6b86,
        0x7878, 0x866b, 0x9560, 0xa656, 0xb74f, 0xc948, 0xdb44, 0xed41,
        0x0040, 0x1341, 0x2544, 0x3748, 0x494f, 0x5a56, 0x6b60, 0x7a6b,
        0x8878, 0x9586, 0xa095, 0xaaa6, 0xb1b7, 0xb8c9, 0xbcdb, 0xbfed
    };

    static const uint8_t upper_bounds[] = {
        3, 7, 11, 15, 19, 23, 27, 31, 35, 39, 43, 47, 51, 55, 59, 63,
        64, 68, 72, 76, 80, 84, 88, 92, 96, 100, 104, 108, 112, 116, 120, 124,
        131, 135, 139, 143, 147, 151, 155, 159, 163, 167, 171, 175, 179, 183, 187, 191,
        192, 196, 200, 204, 208, 212, 216, 220, 224, 228, 232, 236, 240, 244, 248, 252
    };

    std::map<uint16_t, Interval<uint8_t> > buildHeadingMap() {
        std::map<uint16_t, Interval<uint8_t> > res;
        uint8_t dir0 = 253;
        for (size_t i = 0; i < 64; ++i) {
            uint8_t dir1 = upper_bounds[i];
            res.insert(make_pair(keys[i], Interval<uint8_t > (dir0, dir1)));
            dir0 = dir1 + 1;
        }
        return res;
    }

    static const std::map<uint16_t, Interval<uint8_t> > headingHash2direction = buildHeadingMap();

}

uint16_t getHeadingHash(int16_t dx, int16_t dy) {
    int8_t dx8(dx >> 3);
    int8_t dy8(dy >> 3);
    uint16_t hash((((uint8_t) (dx8)) << 8) | ((uint8_t) (dy8)));
    return hash;
}

Interval<uint8_t> getHeadings(uint16_t heading_hash) {
    std::map<uint16_t, Interval<uint8_t> >::const_iterator i
            = headingHash2direction.find(heading_hash);
    if (i != headingHash2direction.end()) {
        return i->second;
    }
    cerr << "Can't find ship heading" << hex << (int) (heading_hash) << dec << endl;
    assert(false);
    //    return Interval<uint8_t>(0, 255);
    return Interval<uint8_t > ();
}

Interval<uint8_t> getHeadings(int16_t heading_dx, int16_t heading_dy) {
    return getHeadings(getHeadingHash(heading_dx, heading_dy));
    //cerr << "Can't find ship heading" << (int)(heading_dx) << " " << (int)(heading_dy) << "==" << hex << getHeadingHash(heading_dx, heading_dy) << dec << endl;
}

