#ifndef TRACKING_H
#define TRACKING_H
#include "world.h"
#include "screen.h"

WorldCoords scn2world(const ScnCoords& c);
ScnCoords world2scn(const WorldCoords& c);
Interval<uint8_t> getHeadings(uint16_t heading_hash);

class Event;

class TrackedPositions
{
public:
    static const unsigned int MAX_OBSERVATIONS = 15;

private:
    ScnCoords scncoords[MAX_OBSERVATIONS];
    bool obspresent[MAX_OBSERVATIONS];
    uint32_t lastEntry;

public:
    uint8_t nextCheck;
    TrackedPositions();
    TrackedPositions(uint32_t time, const ScnCoords& pos);
    void clear();
    bool isObserved(uint32_t time) const;
    const ScnCoords& getPos(uint32_t time) const;
     bool refinePosition(uint32_t time, const WorldDisp8& d, WorldCoords& new_p, WorldDisp16& new_p_error) const;
protected:
    void setPos(uint32_t time, const ScnCoords& pos);
private:
   inline void clearObservation(uint8_t slot)
    { scncoords[slot] = ScnCoords(); obspresent[slot] = false; }
    bool isObserved(uint8_t slot) const;
    const ScnCoords& getPos(uint8_t slot) const;
    inline static uint8_t getSlot(uint32_t time)
    { return time%MAX_OBSERVATIONS; }
    friend class TrackedGameState;
};

#if 0

struct ShipHeading
{
    int16_t heading_dx;
    int16_t heading_dy;
};

class TrackedCoordinates : public TrackedPositions
{
public:
    ShipHeading headings[TrackedPositions::MAX_OBSERVATIONS];
    TrackedCoordinates();
    TrackedCoordinates(uint32_t time, const ScnCoords& pos);
};

#endif

class TrackedShot : public Shot, public TrackedPositions
{
public:
    inline TrackedShot(uint16_t _id, uint32_t _firstSighted, const ScnCoords& _pos)
            : Shot(_id, _firstSighted, scn2world(_pos)), TrackedPositions(_firstSighted, _pos)
    {}
    inline TrackedShot( uint16_t _id, uint32_t _time, const Ship& ship)
    : Shot(_id, _time, ship), TrackedPositions()
    {
        if (p_error.cheb() == 0)
            nextCheck = 0;
    }

    uint8_t refine(uint32_t time, const ScnCoords& pos);
    inline ScnCoords getExpScnCoords() const { return world2scn(p); }
};

class TrackedAsteroid : public Asteroid, public TrackedPositions
{
public:
    static const uint8_t SF_SMALL = 14;
    static const uint8_t SF_MEDIUM = 15;
    static const uint8_t SF_LARGE = 0;

    uint8_t sf; // scale factor: 0 = large, 15 = medium, 14 = small
    uint8_t type; // 1 ... 4

    inline TrackedAsteroid(uint16_t _id, uint32_t _firstSighted, const ScnCoords& _pos, uint8_t _sf, uint8_t _type)
            : Asteroid(_id, _firstSighted, scn2world(_pos), sf2size(_sf)), TrackedPositions(_firstSighted, _pos), sf(_sf), type(_type)
    {}
    inline TrackedAsteroid( uint16_t _id, const TrackedAsteroid& breaking)
    : Asteroid(_id, breaking), TrackedPositions(), sf(breaking.sf == SF_LARGE ? SF_MEDIUM : SF_SMALL), type(0) {}
    uint8_t refine(uint32_t time, const ScnCoords& pos);
    inline ScnCoords getExpScnCoords() const { return world2scn(p); }

//private:
    static uint16_t sf2size(uint8_t);
};

class TrackedUfo : public Ufo, public TrackedPositions
{
public:
    static const uint8_t SF_SMALL = 14;
    static const uint8_t SF_LARGE = 15;
    
    inline TrackedUfo(uint16_t _id, uint8_t _sf, uint32_t _firstSighted, const ScnCoords& _pos)
            : Ufo(_id, _firstSighted, sf2size(_sf)), TrackedPositions(_firstSighted, _pos)
    {
        p = scn2world(_pos);
        switch (_pos.x) {
            case 2:
                p.x = 16;
                d.dx = 16;
                break;
            case 1021:
                p.x = 8175;
                d.dx = -16;
                break;
        }
    }

    inline TrackedUfo()
            : Ufo(), TrackedPositions()
{}
    uint8_t refine(uint32_t time, const ScnCoords& pos);
    inline ScnCoords getExpScnCoords() const { return world2scn(p); }

private:
    static uint16_t sf2size(uint8_t);
};


class TrackedShip : public Ship, public TrackedPositions
{
public:
    inline TrackedShip()
            : Ship(), TrackedPositions()
    {}
    inline TrackedShip(uint16_t _id, uint32_t _firstSighted, const ScnCoords& _pos, uint16_t _heading, const WorldCoords& oldc)
            : Ship(_id, _firstSighted, scn2world(_pos), getHeadings(_heading)), TrackedPositions(_firstSighted, _pos)
    {
                if (_pos == ScnCoords(524, 524)) {
                    p = WorldCoords(4192, 3168);
                    p_error = WorldDisp16();
                } else {
                    WorldCoords n(p.x & 0xff00 | oldc.x & 0x00ff, p.y & 0xff00 | oldc.y & 0x00ff);
                    if (world2scn(n) == _pos) {
                        std::cout << "Reusing low position low bytes" << std::endl;
                        p = n;
                        p_error = WorldDisp16();
                        d = WorldDisp8();
                        d_error = WorldDisp8();
                    }
                }
    }
    uint8_t refine(uint32_t time, const ScnCoords& pos, uint16_t heading);
    inline ScnCoords getExpScnCoords() const { return world2scn(p); }
};

typedef _GameState<TrackedShot, TrackedUfo, TrackedShip, TrackedAsteroid, std::list<TrackedShot>, std::list<TrackedAsteroid> > TrackedGameStateBase;

#if 0
class TrackedGameState
{
public:
    static const int MAX_ASTEROIDS = 26;
    static const int MAX_SHOTS_SHIP = 4;
    static const int MAX_SHOTS_UFO = 2;

    uint32_t time;
    uint8_t frameOffset;
    std::list<TrackedShot> sshots;
    std::list<TrackedShot> ushots;
    TrackedUfo ufo;
    TrackedShip ship;
    std::list<TrackedAsteroid> asteroids;

    inline TrackedGameState() : time(0), frameOffset(0), sshots(), ushots(), ufo(), ship(), asteroids()
    {}
    void advance();
    bool targetObject(const Object& obj, Target& res) const;
    void checkCollisions();
    void print(std::ostream& os) const;
};
#endif

class TrackedGameState : public TrackedGameStateBase
{
public:
    typedef std::list<TrackedAsteroid> Asteroids_t;
    typedef std::list<TrackedShot> Shots_t;
    void update(const observation::Observation& frame, std::vector<Event>& events);
    void reconcileShots(const observation::Observation& frame, std::vector<Event>& events);
    void reconcileUFO(const observation::Observation& frame, std::vector<Event>& events);
    void reconcileShip(const observation::Observation& frame, std::vector<Event>& events);
    void reconcileAsteroids(const observation::Observation& frame, std::vector<Event>& events);
    void updateCollisions(const std::vector<Event>& events);
    bool findTarget(Target& target) const;
};

inline bool TrackedPositions::isObserved(uint32_t time) const
{
    return lastEntry-time < MAX_OBSERVATIONS
            && isObserved(getSlot(time));
}

inline bool TrackedPositions::isObserved(uint8_t slot) const
{
    assert(slot <= MAX_OBSERVATIONS);
    return obspresent[slot];
}

inline void TrackedPositions::setPos(uint32_t time, const ScnCoords& pos)
{
    assert (time>lastEntry);
    uint32_t clearMax = std::min(time, lastEntry+MAX_OBSERVATIONS);
    for (uint32_t t=lastEntry+1; t<=clearMax; ++t) {
        clearObservation(getSlot(t));
    }
        
    const uint8_t slot(getSlot(time));
    scncoords[slot] = pos;
    obspresent[slot] = true;
    lastEntry = time;
}


inline const ScnCoords& TrackedPositions::getPos(uint32_t time) const
{
    assert (isObserved(time));
    return scncoords[getSlot(time)];
}

template<typename C>
typename C::iterator moveTo(C& c, const typename C::iterator& i, size_t pos)
{
    typename C::value_type tmp = *i;
    c.erase(i);

    typename C::iterator dest = c.begin();
    for (size_t i=0; dest!=c.end() && i<pos; ++i)
        ++dest;
    return c.insert(dest, tmp);
}

#endif // !defined TRACKING_H
