
#include "world.h"

#include "world.h"
#include "keys.h"
#include <sstream>
using std::ostream;
using std::cout;
using std::endl;
using std::hex;
using std::bitset;
using std::dec;
using std::setw;
using std::left;
using std::right;
using std::string;
using std::ostringstream;
using std::list;
using std::min;
using std::max;

//const int GameState<Shot, Ufo, Ship, Asteroid>::MAX_SHOTS_UFO;
//const int GameState::MAX_SHOTS_SHIP;
uint16_t nextId(0);
uint32_t frameOffset(0);

void breakpoint()
{
    cout << "Breakpoint reached" << endl;
}

const char* Command::types[] =
    { "NOP", "TURN", "FIRE", "ACCEL"
    };

ostream& operator<<(ostream& os, const Command& c)
{
    os << c.start << ": " << Command::types[c.type] << " " << hex << (unsigned int)(c.turn_to) << dec;
    return os;
}

ostream& operator<<(ostream& os, const Maneuver& m)
{
    os << m.when << ": " << "[" << hex << (unsigned int)(m.angle) << dec << "] " << (unsigned int)(m.accel_duration);
    return os;
}

ostream& operator<<(ostream& os, Ship::mission_result_t r)
{
    static const char* mission_result_names[] = { "MISSION_IMPOSSIBLE", "MISSION_PROCEEDING", "MISSION_COMPLETED" };
    static const char* COMMA = ", ";
    const char* comma = &COMMA[2]; // ""

    if (r == Ship::MISSION_IMPOSSIBLE) {
        os << mission_result_names[Ship::MISSION_IMPOSSIBLE];
        comma = COMMA;
    }
    if ((r & Ship::MISSION_PROCEEDING) == Ship::MISSION_PROCEEDING) {
        os << comma << mission_result_names[Ship::MISSION_PROCEEDING];
        comma = COMMA;
    }
    if ((r & Ship::MISSION_COMPLETED) == Ship::MISSION_COMPLETED) {
        os << comma << mission_result_names[Ship::MISSION_COMPLETED];
        comma = COMMA;
    }
    return os;
}


bool Object::inRange(const WorldCoords& pos) const
{
    WorldDisp16 dist(pos - p);
            
    //int16_t dist = (_pos - p).cheb();

    //cout << "dist:" << dist << endl;
    //return dist <= p_error;
    return abs(dist.dx) <= p_error.dx &&  abs(dist.dy) <= p_error.dy;
    
}


void Object::advance(uint32_t time)
{
    p += d;
//    p_error += d_error;
    if (time>=birth) {
        p_error.dx = min(4095, p_error.dx + d_error.dx);
        p_error.dy = min(4095, p_error.dy + d_error.dy);
    }
}

void Object::advance(uint32_t from, uint32_t to)
{
    assert (to>=from);
    int32_t n(to-from);    
    p += n*d;
//    p_error += d_error;
    int32_t errors(min(to-from, to-birth));
    p_error.dx = min(4095, p_error.dx + errors*d_error.dx);
    p_error.dy = min(4095, p_error.dy + errors*d_error.dy);
}


Shot::Shot(uint16_t _id, uint32_t time, const Ship& ship)
        : Object( _id, time, getTimeOfDeath(time, TTL), SIZE, ship.p, ship.d, 0 /* TODO */ ), ttl(TTL)
{
    const uint8_t heading(ship.heading.start+ship.heading.n/2);
    const int8_t dy(ast_sin(heading)>>1); // this is not the same as /2
    const int8_t dx(ast_cos(heading)>>1);
    p.x += dx + (dx>>1);
    p.y += dy + (dy>>1);
    d.dx += dx;
    d.dy += dy;
    p_error = ship.p_error;
    d_error = ship.d_error;
}

void Shot::advance(uint32_t time, bool updateTTL)
{
    if (updateTTL && ttl) {
        --ttl;
    }

    Object::advance(time);
}

void Shot::advance(uint32_t time)
{
    const bool updateTTL( (( time+frameOffset ) & 3) == 0 );
    advance(time, updateTTL);
}


Asteroid::Asteroid( uint16_t _id, const Asteroid& breaking)
 : Object(_id, breaking.death, UINT32_MAX, breaking.size == SIZE_LARGE ? SIZE_MEDIUM : SIZE_SMALL, breaking.p, breaking.d, 0 ) 
{
    const int8_t dxmax = boundSpeed(breaking.d.dx + breaking.d_error.dx + 15);
    const int8_t dxmin = boundSpeed(breaking.d.dx - breaking.d_error.dx - 16);
    d.dx = (dxmin + dxmax)/2;
    d_error.dx = max(dxmax-d.dx, d.dx-dxmin);

    const int8_t dymax = boundSpeed(breaking.d.dy + breaking.d_error.dy + 15);
    const int8_t dymin = boundSpeed(breaking.d.dy - breaking.d_error.dy - 16);
    d.dy = (dymin + dymax)/2;
    d_error.dy = max(dymax-d.dy, d.dy-dymin);
}

inline int16_t toInt16(uint8_t lo, uint8_t hi)
{
    return (((int16_t)hi)<<8)+lo;
}

int16_t decelerate(int16_t speed)
{
               
//                speed -= (d.dx<<1) + 1;
//                speed -= (speed >> 7);
                return speed - (speed/128) + (speed < 0 ? 1 : 0);
}

void Ship::advance(uint32_t time)
{
    const bool updateSpeed(( time+frameOffset ) & 1);
    if (activeKeys.left())
        heading += 3;
    if (activeKeys.right())
        heading -= 3;

    if (updateSpeed) {
        if (activeKeys.thrust()) { // accelerate
            //int8_t cos0 = ast_cos(heading.start); // TODO
            int8_t cos1 = ast_cos(heading.start + heading.n / 2);
            //int8_t cos2 = ast_cos(heading.start + heading.n);

            if (cos1 > 0) {
                int16_t dingsx = dx_friction + 2 * cos1;
                dx_friction = dingsx;
                if (dingsx > 255)
                    ++d.dx;
                if (d.dx > VMAX) {
                    d.dx = VMAX;
                    dx_friction = 255;
                }
            } else {
                int16_t dingsx = dx_friction + 2 * cos1;
                dx_friction = dingsx;
                if (dingsx < 0)
                    --d.dx;
                if (d.dx < -VMAX) {
                    d.dx = -VMAX;
                    dx_friction = 1;
                }
            }

            //int8_t sin0 = ast_sin(heading.start);
            int8_t sin1 = ast_sin(heading.start + heading.n / 2);
            //int8_t sin2 = ast_sin(heading.start + heading.n);

            if (sin1 > 0) {
                int16_t dingsy = dy_friction + 2 * sin1;
                dy_friction = dingsy;
                if (dingsy > 255)
                    ++d.dy;
                if (d.dy > VMAX) {
                    d.dy = VMAX;
                    dy_friction = 255;
                }
            } else {
                int16_t dingsy = dy_friction + 2 * sin1;
                dy_friction = dingsy;
                if (dingsy < 0)
                    --d.dy;
                if (d.dy < -VMAX) {
                    d.dy = -VMAX;
                    dy_friction = 1;
                }
            }

            //d.dx
            // find max and min sin and cos of headings
            // add mean to d and dx_friction and error to d_error
            // bound

        } else { // decelerate

            if (d.dx || dx_friction) {
//                int8_t dingsx = d.dx * 2 + 1;
//                uint8_t subx = (dingsx > dx_friction) ? 1 : 0;
//                dx_friction -= dingsx;
//                d.dx -= subx;
                
                int16_t speed = decelerate(toInt16(dx_friction, d.dx));
//                speed -= (d.dx<<1) + 1;
                //speed -= (speed >> 7);
                dx_friction = speed & 0xff;
                d.dx = speed >> 8;
//                d.dx = speed / 256;
            }
            if (d.dy || dy_friction) {
                int8_t dingsy = d.dy * 2 + 1;
                uint8_t suby = (dingsy > dy_friction) ? 1 : 0;
                dy_friction -= dingsy;
                d.dy -= suby;
            }
        }
    }

    Object::advance(time);
}

int8_t alter(uint8_t i) {
    ++i;
    return (i&1) ? -(i>>1) : (i>>1);
}




Ship::mission_result_t Ship::deriveKeys(uint32_t time, Keys& keys) const {
    Maneuver c(activeCommand);
    --(c.when);

    // determine if we can turn to angle before "when"
    // if exact heading is unknown check if we possibly can

    if (c.isFire()) {
    if (heading.n == 1) { // exact heading known
        int8_t dist = c.angle - heading.start;
        if (dist) {
            int8_t ticks = angle2ticks(dist);
            if (time + abs(ticks) + (c.isFire() ? 0 : 0) > c.when)
                return MISSION_IMPOSSIBLE;
            if (ticks > 0)
                keys.left(true);
            else
                keys.right(true);
        } else {
            if (c.isFire()) {
                if (time >= c.when) {
                    if (activeKeys.fire() || time > c.when) {
                        return MISSION_IMPOSSIBLE;
                    } else {
                        keys.fire(true);
                        return MISSION_COMPLETED; //  TODO: MISSION_CAN_PIPELINE
                    }
                }
            } else {
                if (time - c.when < c.accel_duration) {
                    keys.thrust(true);
                } else if (time - c.when == c.accel_duration) {
                    return MISSION_COMPLETED;
                } else {
                    return MISSION_IMPOSSIBLE;
                }
            }
        }
    } else {
        // exact heading not known
        // if already turning keep turning in same direction to learn heading
        if (activeKeys.left())
            keys.left(true);
        else if (activeKeys.right())
            keys.right(true);
        else { // not turning yet => start turning in general direction
            // hoping and pretending we can still reach our goal
            int8_t dist = c.angle - heading.start;
            if (dist > 0)
                keys.left(true);
            else
                keys.right(true);
        }
    }
    return MISSION_PROCEEDING;
    } else {
        int8_t dist = c.angle - getHeading();
        if (dist > 0)
            keys.left(true);
        else if (dist < 0)
            keys.right(true);

        if (dist == 0 || c.accel_duration == 20) {
            if (time - c.when < c.accel_duration) {
                keys.thrust(true);
            } else if (dist == 0) {
                if (time - c.when == c.accel_duration) {
                    return MISSION_COMPLETED;
                } else {
                    return MISSION_IMPOSSIBLE;
                }

            }

        }
                return MISSION_PROCEEDING;

    }
}

inline uint8_t select(int16_t s, uint8_t plus, uint8_t minus)
{
    return s < 0 ? minus : s > 0 ? plus : 0;
}


class CollisionDetector
{
public:
    const int16_t a;
    const int16_t a1_5;

    CollisionDetector(int16_t r_2)
            : a(r_2), a1_5(r_2+r_2/2)
    {}

    bool checkCollision(const WorldCoords& c1, const WorldCoords& c2)
    {
        int16_t dx = (c2.x - c1.x)/2;
        if (dx<0) dx = -dx-1;
        int16_t dy = (c2.y - c1.y)/2;
        if (dy<0) dy = -dy-1;
        return dx<=a && dy<=a && dx+dy<=a1_5;
    }
};

class CollisionDetector2
{
public:
    const int16_t a;
    const int16_t a1_5;
    const WorldDisp16 d;

    CollisionDetector2(int16_t r_2, WorldDisp16 _d)
            : a(r_2), a1_5(r_2+r_2/2), d(_d)
    {}

    int checkCollision(const WorldCoords& c1, const WorldCoords& c2)
    {
        //cout << c1 << "<-->" << c2 << endl;
        bool negx(false), negy(false);
        int16_t dx = (c2.x - c1.x) >> 1;
        if (dx<0) {
            dx = -dx-1;
            negx = true;
        }
        if (dx>a)
            return -1;
        int16_t dy = (c2.y - c1.y) >> 1;
        if (dy<0) {
            dy = -dy-1;
            negy = true;
        }
        if (dy>a)
            return -1;
        int16_t gap=dx+dy-a1_5;
        if (gap<=0)
            return 0;
        gap *= 2;
        int16_t dxy = (negx ? -d.dx : d.dx) + (negy ? -d.dy : d.dy);
        if (dxy<=0)
            return -1;
        int16_t n = (gap + dxy-1) / dxy;
        assert (n > 0);
        return n;
    }
};

class CollisionDetector3
{
public:
    const int16_t ax;
    const int16_t ay;
    const int16_t a1_5;
    const WorldDisp16 d;

    CollisionDetector3(int16_t rx_2, int16_t ry_2, WorldDisp16 _d)
            : ax(rx_2), ay(ry_2), a1_5(max(ax, ay)+min(ax,ay)/2), d(_d)
    {}

    int checkCollision(const WorldCoords& c1, const WorldCoords& c2)
    {
        //cout << c1 << "<-->" << c2 << endl;
        bool negx(false), negy(false);
        int16_t dx = (c2.x - c1.x) >> 1;
        if (dx<0) {
            dx = -dx-1;
            negx = true;
        }
        if (dx>ax)
            return -1;
        int16_t dy = (c2.y - c1.y) >> 1;
        if (dy<0) {
            dy = -dy-1;
            negy = true;
        }
        if (dy>ay)
            return -1;
        int16_t gap=dx+dy-a1_5;
        if (gap<=0)
            return 0;
        gap *= 2;
        int16_t dxy = (negx ? -d.dx : d.dx) + (negy ? -d.dy : d.dy);
        if (dxy<=0)
            return -1;
        int16_t n = (gap + dxy-1) / dxy;
        assert (n > 0);
        return n;
    }
};


struct Octagon3
{
//    const int a;
//    const int a1_5;
    const int ax2;
    const int ay2;
//    const int a3;
//    const int _a3;
//    const int a7;
    const WorldCoords p;
    const WorldCoords A;
    const WorldCoords B;
    const WorldCoords C;
    const WorldCoords D;
    const WorldCoords E;
    const WorldCoords F;
    const WorldCoords G;
    const WorldCoords H;
    const uint16_t modC_Gx;
    const uint16_t modE_Ay;
    Octagon3(const WorldCoords& p, int ax, int ay)
            : ax2(ax*2), ay2(ay*2),
//        a(l), a1_5(a+a/2) a2(a*2), a3(a*3), _a3(-a3), a7(a*7),
//            p(_p),
     A(p+WorldDisp16(-ax, -ay2)),
     B(p+WorldDisp16(ax, -ay2)),
     C(p+WorldDisp16(ax2, -ay)),
     D(p+WorldDisp16(ax2, ay)),
     E(p+WorldDisp16(ax, ay2)),
     F(p+WorldDisp16(-ax, ay2)),
     G(p+WorldDisp16(-ax2, ay)),
     H(p+WorldDisp16(-ax2, -ay)),
     modC_Gx(mod(C.x-G.x, 0x2000)),
     modE_Ay(mod(E.y-A.y, 0x1800))
    {
    }
    
};

struct Octagon
{
//    const int a;
//    const int a1_5;
    const int a2;
//    const int a3;
//    const int _a3;
//    const int a7;
    const WorldCoords p;
    const WorldCoords A;
    const WorldCoords B;
    const WorldCoords C;
    const WorldCoords D;
    const WorldCoords E;
    const WorldCoords F;
    const WorldCoords G;
    const WorldCoords H;
    const uint16_t modC_Gx;
    const uint16_t modE_Ay;
    Octagon(const WorldCoords& p, int a)
            : a2(a*2),
//        a(l), a1_5(a+a/2) a2(a*2), a3(a*3), _a3(-a3), a7(a*7),
//            p(_p),
     A(p+WorldDisp16(-a, -a2)),
     B(p+WorldDisp16(a, -a2)),
     C(p+WorldDisp16(a2, -a)),
     D(p+WorldDisp16(a2, a)),
     E(p+WorldDisp16(a, a2)),
     F(p+WorldDisp16(-a, a2)),
     G(p+WorldDisp16(-a2, a)),
     H(p+WorldDisp16(-a2, -a)),
     modC_Gx(mod(C.x-G.x, 0x2000)),
     modE_Ay(mod(E.y-A.y, 0x1800))
    {
    }
    
};



int32_t collide(const Object& o0, const Object& o1, int32_t nmax, bool optimistic)
{
    if (nmax == 0) return INT32_MAX; else --nmax;
#ifndef NDEBUG
    cout << "Collision obj" << o0.id << "<-->obj"<< o1.id << " before " << nmax << "... ";
#endif
    assert (nmax >= 0 && nmax < 18000);
    typedef enum { EDGE_A = 1, EDGE_B = 2, EDGE_C = 4, EDGE_D = 8, EDGE_E = 16, EDGE_F = 32, EDGE_G = 64, EDGE_H = 128 }; 
    uint8_t edges(0);

    const WorldDisp16 pd = o1.p - o0.p;
    const WorldCoords p(pd.dx, pd.dy);
    const WorldDisp8 d = o0.d - o1.d;

    const int a = (o0.size + o1.size)/2;
    const int ax = (o0.size + o1.size + (optimistic?1 :-1)*(o0.p_error.dx+o1.p_error.dx))/2;
    const int ay = (o0.size + o1.size + (optimistic?1 :-1)*(o0.p_error.dy+o1.p_error.dy))/2;

    Octagon o(p, a);
//    Octagon3 o(p, ax, ay);
    

    WorldDisp32 d0(o0.d.dx, o0.d.dy), d1(o1.d.dx, o1.d.dy);
//    CollisionDetector2 cd(a, WorldDisp16(d.dx, d.dy));
    CollisionDetector3 cd(ax, ay, WorldDisp16(d.dx, d.dy));
    
    edges |= select(d.dx,      EDGE_G, EDGE_C);
    edges |= select(d.dy,      EDGE_A, EDGE_E);
    edges |= select(d.dx+d.dy, EDGE_H, EDGE_D);
    edges |= select(d.dx-d.dy, EDGE_F, EDGE_B);
    
    int32_t hitmin(INT32_MAX);
    if (cd.checkCollision(o0.p, o1.p)==0) {
        hitmin = 0;
    } else if (edges) {
    
    int32_t k=0;
    int32_t nmin;
    do {
        nmin = INT32_MAX;
        
        if (edges & EDGE_A) {
            int32_t n = (o.A.y +k*0x1800 +d.dy-1)/d.dy;
            nmin = min(nmin, n);
//            Object ot0(o0);
//            o0.advance(time, time+n);
//            Object ot1(o1);
//            o1.advance(time, time+n);
            int16_t nd = cd.checkCollision(o0.p + n*d0, o1.p + n*d1);
            if (nd>0) {
                n += nd;
                nd = cd.checkCollision(o0.p + n*d0, o1.p + n*d1);
            }
            if (!nd && (n < hitmin) && (n < nmax)) {
                    hitmin = n;
            }
        } else if (edges & EDGE_E) {
            int32_t n = (-0x1800+o.E.y -k*0x1800+d.dy-1)/d.dy;
            nmin = min(nmin, n);
            int16_t nd = cd.checkCollision(o0.p + n*d0, o1.p + n*d1);
            if (nd>0) {
                n += nd;
                nd = cd.checkCollision(o0.p + n*d0, o1.p + n*d1);
            }
            if (!nd && (n < hitmin) && (n < nmax)) {
                    hitmin = n;
            }
        }
        if (edges & EDGE_C) {
            int32_t n = (-0x2000+o.C.x -k*0x2000 +d.dx-1)/d.dx;
            nmin = min(nmin, n);
            int16_t nd = cd.checkCollision(o0.p + n*d0, o1.p + n*d1);
            if (nd>0) {
                n += nd;
                nd = cd.checkCollision(o0.p + n*d0, o1.p + n*d1);
            }
            if (!nd && (n < hitmin) && (n < nmax)) {
                    hitmin = n;
            }
        } else if (edges & EDGE_G) {
            int32_t n = (o.G.x +k*0x2000 +d.dx-1)/d.dx;
            nmin = min(nmin, n);
            int16_t nd = cd.checkCollision(o0.p + n*d0, o1.p + n*d1);
            if (nd>0) {
                n += nd;
                nd = cd.checkCollision(o0.p + n*d0, o1.p + n*d1);
            }
            if (!nd && (n < hitmin) && (n < nmax)) {
                    hitmin = n;
            }
        }
                 
        assert (nmin >= 0);
        assert (hitmin >= 0);
        ++k;
    } while ( nmin <= nmax && nmin <= hitmin);
    }
   // cout << "Collision obj" << o0.id << "<-->obj"<< o1.id <<" in " << hitmin << endl;
//    if (hitmin != INT32_MAX)
//    {
//        cout << "Collision obj" << o0.id << "<-->obj"<< o1.id<<" in " << hitmin << endl;
//    }
#ifndef NDEBUG
    cout << hitmin << endl;
#endif
    return hitmin;
}

uint32_t collide(const Object& o0, const Object& o1, uint32_t time, uint32_t tmax, bool optimistic)
{
    if (tmax < time)
        return UINT32_MAX;
    int32_t in = collide(o0, o1, tmax-time, optimistic);
    uint32_t res;
    if (in == INT32_MAX) {
        res = UINT32_MAX;
    } else {
        res = time+in;
//        if (res <= o0.birth || res <= o1.birth)
        if (res < o0.birth || res < o1.birth) // TODO
            res = UINT32_MAX;
    }
    return res;
}

uint32_t Object::collide(Object& o, uint32_t tmax, bool optimistic)
{
    return ::collide(*this, o, tmax, optimistic);
}

bool Ship::targetObject(const Object& obj, int32_t thrange, Interval<uint8_t>& aperture, double& a) const
{
    assert (thrange >= 0);
    const int vix = d.dx;
    const int viy = d.dy;
    const int vax = obj.d.dx;
    const int vay = obj.d.dy;
    WorldDisp16 d = obj.p - p;
    
    // Transform
    const int pex = d.dx;
    const int pey = d.dy;
    const int vex = vax - vix;
    const int vey = vay - viy;

    const int osize = obj.size - obj.p_error.cheb(); 
    
//    const double t155 = 63.5*63.5 ;
//    const double pexXpex = pex*pex ;
//    const double peyXpey = pey*pey ;
//    const double vexXvex = vex*vex ;
//    const double veyXvey = vey*vey ;
//    const double th = -((sqrt(t155*pexXpex + t155*peyXpey - pexXpex*veyXvey - peyXpey*vexXvex
//                              + 2.0*pex*pey*vex*vey) + pex*vex + pey*vey)/(-t155 + vexXvex + veyXvey));
//    
//    

    const static double s = 63.5;
    const static double q = 2.5*s;
    const double vex2 = vex*vex ;
    const double vey2 = vey*vey ;
    const static double s2 = s*s ;
    const double pex2 = pex*pex ;
    const double pey2 = pey*pey ;
    const static double q2 = q*q ;
    const double th = -((q + (s*sqrt(-(vex2*pey2) - vey2*pex2 + vex2*q2 + s2*pex2 + vey2*q2 + s2*
pey2 - 2.0*pex*q*s*vex - 2.0*pey*q*s*vey + 2.0*pex*pey*vex*vey) - q*vex2
 - q*vey2 + pex*s*vex + pey*s*vey)/(vex2 + vey2 - s2))/s);
    
   
    
    
    //const int th = (int)(thd);
#ifndef NDEBUG                
    cout << "th:+" << setw(5) << th << " thrange:+" << thrange;
#endif
    
    if (th <= 0 || th > thrange) {
#ifndef NDEBUG                
        cout << " Aborting" << endl;
#endif
        return false;
    }

    //const uint8_t a = (uint8_t)(atan2(pey + th*vey, pex + th*vex) * (128.0 / M_PI));
    
    a = atan2(pey + th*vey, pex + th*vex) * (128.0 / M_PI);
//    const uint8_t a = ast_atan2(mod(pey + th*vey, 0x1800), mod(pex + th*vex, 0x2000));

    
    
    const double distance = sqrt(pex2 + pey2) - q;

//    if (speed > 2*osize)
//        return false;
    
    const double elbow = asin(osize/distance)*(128.0/M_PI);
    
#ifndef NDEBUG
    const double speed = distance/th; // impact speed of projectile
    const double eff_size = sqrt(osize*osize-speed*speed/4.0);  // need to hit this sphere of asteroid
    const double selbow = asin(eff_size/distance)*(128.0/M_PI);
#endif
    //res.aperture = Interval<uint8_t>(a-elbow, a+elbow);
    uint8_t floor((uint8_t)ceil(a-elbow));
    uint8_t ceil((uint8_t)(a+elbow));
 
#ifndef NDEBUG                
    cout <<" alpha: " << a << "["<< hex << (int)((uint8_t)a) << dec 
    << "] size:" << osize << " elbow:" << elbow << ", speed:" << speed << " eff_size:" << eff_size << " selbow:" << selbow
    << " ["<<hex<<(int)(floor)<<", "<<(int)(ceil)<< dec << "]";
#endif
    
    if (ceil-floor>127) {
#ifndef NDEBUG                
        cout << " Aborting" << endl;
#endif
        return false;
    }
    aperture = Interval<uint8_t>(floor, ceil);
    return true;
}

bool Ship::targetObject(uint32_t time, const Object& obj, uint32_t thmax, const bitset<256>& possible_headings, Interval<uint8_t>& aperture, Mark& res) const
{
    assert (thmax >= time);
    double a;
    if (targetObject(obj, min(thmax - time, (uint32_t) (INT32_MAX)), aperture, a)) {
        uint8_t ai((uint8_t) (a + .5));
        int8_t sign(((uint8_t) a) < ai ? -1 : 1);
        int out(0);
//        
//        
        Ship tmpShip(*this);
//        tmpShip.advance(time);
        tmpShip.heading.n = 1;

#ifndef NDEBUG                
        cout << "\n";
#endif
        for (int i = 0; out < 2; ++i) {
            int8_t d = sign * alter(i);
            uint8_t b = ai + d;
            if (!aperture.contains(b)) {
                ++out;
                continue;
            }
#ifndef NDEBUG                
            cout << "Choosing " << hex << (int) (b) << dec << " bd:" << (int) (d) << " avail:" << possible_headings.test(b) << endl;
#endif
            if (!possible_headings.test(b))
                continue;
            tmpShip.heading.start = b;
            Shot tmpShot(0, time, tmpShip);
            tmpShot.advance(time);
#ifndef NDEBUG                
            printObject(cout, tmpShot, "tmpshot"); cout << "\n";
            printObject(cout, obj, "tmpobj"); cout << endl;
#endif
            uint32_t col = ::collide(tmpShot, obj, time, tmpShot.getTimeOfDeath(time), false);
            
            
            if (col != UINT32_MAX) {
                Object tmpObj(obj);
                tmpObj.advance(time, col);
                if (tmpObj.p_error.cheb() < tmpObj.size) {
                res.alpha = b;
                res.thit = col;
                return true;
                }
            }
        }
    }
    return false;
    
//    uint8_t u((uint8_t)a);
//    uint8_t v(u+1);
//    uint8_t ubound(floor);
//    uint8_t vbound(ceil);
//    int8_t du(-1);
//    int8_t dv(+1);
//    
//    
//    
//    res.thit = th < 256 ? (uint8_t)(th) : 255;
//    
//    
//    const WorldDisp16 t(pex + th*vex, pey + th*vey);
//    const WorldDisp16 t2(ast_cos(a)/4+th*(ast_cos(a)/2), ast_sin(a)/4+th*(ast_sin(a)/2));
//    cout << "\tt:"<< setw(5) << t << "\tt2:" << setw(5) << t2; 
//    
//    Ship tmpShip(*this);
//    tmpShip.heading.start = a;
//    tmpShip.heading.n = 1;
//    Shot tmpShot(0, 0, tmpShip);
//    for (int i=0; i<th; ++i)
//        tmpShot.advance(false);
//    cout << "\tadvShot:" << tmpShot.p;
//    Object tmpObj(obj);
//    for (int i=0; i<th; ++i)
//        tmpObj.advance();
//    cout << "\tadvObj:" << tmpObj.p;
////         << " speed:"<< speed << "\tsize:" << obj.size;
//    
//    return true;
}

bool Ship::targetObject(uint32_t time, const Object& obj, uint32_t thmax, const bitset<256>& possible_headings, Mark& res) const
{
    Interval<uint8_t> aperture;
    return targetObject(time, obj, thmax, possible_headings, aperture, res);
}

ostream& printObject(ostream& os, const Object& o, const char* type)
{
    ostringstream name;
    name << type << o.id;
    os << setw(8) << left << name.str() << right << " " << setw(3) << o.size
    << "\tp" << o.p << "+-"<<o.p_error
    << "\td" << o.d  << "+-"<<o.d_error
    << "\t*" << setw(5) << o.birth
    << "\tX" << setw(5);
    
    if (o.death == UINT32_MAX)
        os << "---";
    else
        os << o.death;
    os << " by " << setw(4) << o.deathBy;
    return os;
}


uint8_t ast_atan2(int16_t y, int16_t x) {
    double v=((double)(y))/x;
//    cout << "ast_tan[0]:" << _ast_tan[0] << endl;
//    cout << "ast_tan[126]:" << _ast_tan[126] << endl;
//    cout << "v:" << v << endl;
    double* p = std::lower_bound(&_ast_tan[0], &_ast_tan[127], v);
    if (p != &_ast_tan[0] && v-*(p-1) < *p-v)
        --p;
    uint8_t n = p - _ast_tan;
//    cout << "n:" << hex << (int)(n) << dec << endl;
    uint8_t i = n - 0x3f + ((x < 0) ? 0x80 : 0);
//    cout << "A[" << hex << (int)(i) << dec <<  "] == " << ast_tan(i) << endl;
    return i;
}


int8_t _ast_sin[256];
double _ast_tan[127];
namespace
{
uint8_t sinq1[65]={
                      0x00,0x03,0x06,0x09,0x0c,0x10,0x13,0x16,0x19,0x1c,0x1f,0x22,0x25,0x28,0x2b,0x2e,0x31,0x33,0x36,0x39,
                      0x3c,0x3f,0x41,0x44,0x47,0x49,0x4c,0x4e,0x51,0x53,0x55,0x58,0x5a,0x5c,0x5e,0x60,0x62,0x64,0x66,0x68,
                      0x6a,0x6b,0x6d,0x6f,0x70,0x71,0x73,0x74,0x75,0x76,0x78,0x79,0x7a,0x7a,0x7b,0x7c,0x7d,0x7d,0x7e,0x7e,
                      0x7e,0x7f,0x7f,0x7f,0x7f};


bool initTrig()
{
    for(uint8_t i=0x00; i<0x41; ++i)
        _ast_sin[i] = sinq1[i];
    for(uint8_t i=0x01; i<0x41; ++i)
        _ast_sin[0x40+i] = sinq1[0x40-i];
    for(uint8_t i=0x01; i<0x80; ++i)
        _ast_sin[256-i] = -_ast_sin[i];

    for(size_t i=0x00; i<0x7f; ++i) {
        uint8_t a = i - 0x3f;
        _ast_tan[i] = (double)(ast_sin(a)) / ast_cos(a);
//        cout << "A[" << hex << (int)(i) << "=>"  << (int)(a) << dec <<  "] == " << _ast_tan[i] << endl;
    }
    
//    for(size_t i=0x00; i<=0xff; ++i) {
//        cout << i << ", " << (int)(ast_sin(i)) << ", " << (int)(ast_cos(i)) << ", " << (int)(128.0*sin(i*((M_PI*2.0)/256.0))) << endl;
//    }
//    exit(1);
    return true;
}



static const bool sin_initialized = initTrig();
}
