#ifndef WORLD_H
#define WORLD_H

#include <vector>
#include <list>
#include <set>
//#include <iostream>
#include <iomanip>
#include <cmath>
#include <bitset>
#include "Interval.h"
#include "keys.h"

//#include <stdint.h>
#define INT32_MAX              (2147483647)
#define UINT32_MAX             (4294967295U)

//namespace std {
//    class ostream;
//}

struct ScnCoords;

class Object;
std::ostream& printObject( std::ostream& os, const Object& o, const char* type );
uint32_t collide(const Object& o0, const Object& o1, uint32_t time, uint32_t tmax, bool optimistic);
//uint32_t collide(const Object& o0, const Object& o1, uint32_t tmax);

extern uint16_t nextId;
extern uint32_t frameOffset;

void breakpoint();

int16_t decelerate(int16_t speed);

template<typename inttype>
inline inttype mod(inttype a, inttype m)
{
    assert (m>0);
        return a>=0 ? a % m : a - (a-m+1)/m*m;
  //      return a>=0 ? a % m : a - a/m*m;
}

// abs(result) == number of turn operations to cover angle_diff
// result > 0 ? left turns : right turns
inline int8_t angle2ticks(int8_t angle_diff)
{
    static const uint8_t INV3MOD256 = 171; // inverse of 3 mod 256
    return INV3MOD256 * angle_diff; // implicit modulo 256
}

template<typename size=int16_t>
class WorldDisp
{
public:
    typedef size dsize_t;
    dsize_t dx;
    dsize_t dy;
    WorldDisp( dsize_t dx, dsize_t dy );
    WorldDisp();
    bool operator==( const WorldDisp& o ) const;
    bool operator!=( const WorldDisp& o ) const;
    dsize_t cheb() const;
    double euclid() const;
};

template<typename size>
WorldDisp<size> operator-( const WorldDisp<size>& d0, const WorldDisp<size>& d1 );

template<typename size>
inline WorldDisp<size> operator*( const WorldDisp<size>& d, int32_t n)
{ return WorldDisp<size>(n*d.dx, n*d.dy); }

template<typename size>
inline WorldDisp<size> operator*( int32_t n, const WorldDisp<size>& d)
{ return WorldDisp<size>(n*d.dx, n*d.dy); }

//template<typename size>
//inline bool operator==(const WorldDisp<size>& d1, const WorldDisp<size>& d2)
//{ return d1.dx == d2.dx && d1.dy == d2.dy; }
//
//template<typename size>
//inline bool operator!=(const WorldDisp<size>& d1, const WorldDisp<size>& d2)
//{ return d1.dx != d1.dx && d1.dy != d2.dy; }

typedef WorldDisp<int8_t> WorldDisp8;

typedef WorldDisp<int16_t> WorldDisp16;

typedef WorldDisp<int32_t> WorldDisp32;

class WorldCoords
{
public:
    uint16_t x;
    uint16_t y;
    bool operator==( const WorldCoords& o ) const;
    bool operator!=( const WorldCoords& o ) const;
    template<typename size>
    WorldCoords& operator+=( const WorldDisp<size>& wd );
    template<typename size>
    WorldCoords& operator-=( const WorldDisp<size>& wd );
//    WorldCoords& operator+=( const WorldDisp<size>& wd );
//    WorldCoords& operator-=( const WorldDisp<size>& wd );
//    template<typename size1, typename size2>
//    inline WorldCoords( size1 _x, size2 _y )
//            : x(mod(_x, 0x2000)), y(mod(_y, 0x1800))
//    {}
    inline WorldCoords( int32_t _x, int32_t _y )
            : x(mod<int32_t>(_x, 0x2000)), y(mod<int32_t>(_y, 0x1800))
    {}
    inline WorldCoords( int16_t _x, int16_t _y )
            : x(mod<int16_t>(_x, 0x2000)), y(mod<int16_t>(_y, 0x1800))
    {}
 
//    inline WorldCoords( uint16_t _x, uint16_t _y )
//            : x(mod(_x, 0x2000), y(mod(_y, 0x1800))
//    {}

    inline WorldCoords()
            : x( 0 ), y( 0 )
    {}
};

template<typename size>
inline WorldCoords operator+(const WorldCoords& wc, const WorldDisp<size>& wd)
{ return WorldCoords(wc.x+wd.dx, wc.y+wd.dy); }

template<typename size>
inline WorldCoords operator-(const WorldCoords& wc, const WorldDisp<size>& wd)
{ return WorldCoords(wc.x-wd.dx, wc.y-wd.dy); }


WorldDisp16 operator-( const WorldCoords& wc0, const WorldCoords& wc1 );

// euclidean distance
//double dist2(const WorldCoords& p1, const WorldCoords& p2);
// infinity norm (Chebyshev) distance
//double distinf(const WorldCoords& p1, const WorldCoords& p2);

class Object
{
public:
    static const int8_t VMAX = 111;
    uint16_t id;
    uint16_t size; // radius
    WorldCoords p; // position
    WorldDisp8 d;  // velocity
    WorldDisp16 p_error;
    WorldDisp8 d_error;
    uint32_t birth;
    uint32_t death;
    uint16_t deathBy;

    bool inRange( const WorldCoords& pos ) const;
    bool inRange( const ScnCoords& pos ) const;
    void advance(uint32_t time);
    void advance(uint32_t from, uint32_t to);
    uint32_t collide(Object& o, uint32_t tmax, bool optimistic);
protected:
    inline Object()
            : id( 0 ), size( 0 ), p(), d(), p_error( 4, 4 ), d_error( VMAX, VMAX ), birth(0), death(0), deathBy(0)
    {}

    inline Object( uint16_t _id, uint32_t _birth, uint32_t _death, uint16_t _size, const WorldCoords& pos, const WorldDisp8& _d, uint8_t _d_error )
            : id( _id ), size( _size ), p( pos ), d( _d ), p_error(4, 4), d_error(_d_error, _d_error), birth(_birth), death(_death), deathBy(0)
    {}
};

class Ship;

class Shot : public Object
{
public:
    static const int16_t SIZE = 0;
    static const int8_t VMAX = 111;
    //static const int MAX_SHOT_FRAMES = 18*4; // 18 decremented every 4th frame
    static const int TTL = 18;
    uint8_t ttl;
    void advance(uint32_t time, bool updateTTL);
    void advance(uint32_t time);
    inline Shot( uint16_t _id, uint32_t _birth, const WorldCoords& _pos )
            : Object( _id, _birth, getTimeOfDeath(_birth, TTL), SIZE,  _pos, WorldDisp8(), VMAX ), ttl(TTL)
    {}
    Shot( uint16_t _id, uint32_t time, const Ship& ship);
    inline static uint32_t getTimeOfDeath(uint32_t time, uint8_t ttl)
    {
        return time + ((4-(time+frameOffset+0)%4)%4) + (ttl-1)*4 +0; // TODO
    }
   inline uint32_t getTimeOfDeath(uint32_t time) const
    { return getTimeOfDeath(time, ttl); }
};

class Asteroid : public Object
{
public:
    static const int16_t SIZE_SMALL = 0x2A << 1;
    static const int16_t SIZE_MEDIUM = 0x48 << 1;
    static const int16_t SIZE_LARGE = 0x84 << 1;
    static const int8_t VMAX = 31;
    inline Asteroid( uint16_t _id, uint32_t _birth, const WorldCoords& _pos, uint16_t _size )
            : Object( _id, _birth, UINT32_MAX, _size, _pos, WorldDisp8(), VMAX )
    {}
    Asteroid( uint16_t _id, const Asteroid& breaking);
    inline static int8_t boundSpeed(int8_t speed) {
    return (speed < 0) ? ((speed <= -6) ? ((speed < -31) ? -31 : speed) : -6) : ((speed >= 6) ? ((speed > 31) ? 31 : speed) : 6 );
}

};

class Mark
{
public:
    uint8_t alpha;
    uint32_t thit; // absolute hit time
    inline Mark() : alpha(0), thit(0)
    {}
};

class Target
{
public:
    uint8_t alpha;
    uint32_t tfire; // shoot time
    uint32_t thit;
    uint16_t targetId;
    inline Target() : alpha(0), tfire(0), thit(0), targetId(0)
    {}
};

class Maneuver
{
public:
uint32_t when;
uint8_t angle;
//typedef enum { FIRE, ACCEL } type_t;
uint8_t accel_duration;
inline Maneuver()
: when(0), angle(0), accel_duration(0) {}
inline Maneuver(uint8_t _angle, uint32_t _when, uint8_t duration=0)
: when(_when), angle(_angle), accel_duration(duration) {}
bool operator==(const Maneuver& m) const
{ return when == m.when && angle == m.angle && accel_duration == m.accel_duration; }
bool operator!=(const Maneuver& m) const
{ return when != m.when || angle != m.angle || accel_duration != m.accel_duration; }
inline bool isFire() const
{ return accel_duration == 255; }

};
std::ostream& operator<<( std::ostream& os, const Maneuver& m );

class Command
{
public:
    static const char* types[];
    uint32_t start;

    typedef enum { NOP, TURN, FIRE, ACCEL } type_t;
    type_t type;
    union {
        uint8_t turn_to;
        uint8_t accel_for;
    };

    inline Command()
            : start( 0 ), type( NOP ), turn_to( 0 )
    {}

    inline Command( uint32_t _start, type_t _type, uint8_t data = 0 )
            : start( _start ), type( _type ), turn_to( data )
    {}

};
std::ostream& operator<<( std::ostream& os, const Command& c );


class Ship : public Object
{
public:
    typedef enum { MISSION_IMPOSSIBLE, MISSION_PROCEEDING, MISSION_COMPLETED } mission_result_t;
//    static const int16_t SIZE = 0x1C << 1;
    static const int16_t SIZE = (0x1E << 1)+10;
    static const int8_t VMAX = 63;

    bool present;
    //uint8_t heading;
    Interval<uint8_t> heading;
    uint8_t dx_friction;
    uint8_t dy_friction;
    Keys sentKeys;
    Keys activeKeys;
//    Command activeCommand;
    Maneuver activeCommand;
    void advance(uint32_t time);
    bool targetObject(const Object& obj, int32_t thrange, Interval<uint8_t>& aperture, double& a) const;
    bool targetObject(uint32_t time, const Object& obj, uint32_t thmax, const std::bitset<256>& possible_headings, Interval<uint8_t>& aperture, Mark& res) const;
    bool targetObject(uint32_t time, const Object& obj, uint32_t thmax, const std::bitset<256>& possible_headings, Mark& res) const;
    //char deriveKeys(uint32_t time) const;
    mission_result_t deriveKeys(uint32_t time, Keys& keys) const;

    inline Ship() : Object(), present( false ), heading( 0,255 ),
            dx_friction( 0 ), dy_friction( 0 ), sentKeys( 0 ), activeKeys( 0 ), activeCommand()
    {}

    inline Ship( uint16_t _id, uint32_t _birth, const WorldCoords& _pos, const Interval<uint8_t>& _heading )
            : Object( _id, _birth, UINT32_MAX, SIZE, _pos, WorldDisp8(), 0 ),
            present( true ), heading( _heading ), dx_friction( 0 ), dy_friction( 0 ), sentKeys( 0 ), activeKeys( 0 ), activeCommand()
    {}
    inline uint8_t getHeading() const
    { return heading.start + heading.n/2; }

};
std::ostream& operator<<( std::ostream& os, Ship::mission_result_t r );

class Ufo : public Object
{
public:
    static const int16_t SIZE_SMALL = 0x12 << 1;
    static const int16_t SIZE_LARGE = (0x12 + 0x12) << 1;
    static const int8_t VMAX = 16;
    static const uint16_t TTL = 0x2000/VMAX; 
    bool present;

//inline Ufo(uint16_t _id, uint8_t _size, const WorldCoords& _pos, const WorldDisp8& _d)
//    : Object(_id, _size, _pos, _d, VMAX), present(true)
//{}

    inline Ufo( uint16_t _id, uint32_t _birth, uint16_t _size )
            : Object( _id, _birth, _birth + TTL, _size, WorldCoords(), WorldDisp8(), VMAX ), present( true )
    {}

    inline Ufo()
            : Object(), present( false )
    {}
    inline static uint32_t getTimeOfNaturalDeath(uint32_t birth)
    { return birth + TTL; }
             
    inline uint32_t getTimeOfNaturalDeath() const
    { return getTimeOfNaturalDeath(birth); }
};

class Event;

struct ObjectIdMatcher
{
    const uint16_t id;
    inline ObjectIdMatcher(uint16_t _id) : id(_id) {}
    inline bool matches(const Object& o) const
    { return o.id == id; }
};

struct ObjectDeathByMatcher
{
    const uint16_t db;
    inline ObjectDeathByMatcher(uint16_t _db) : db(_db) {}
    inline bool matches(const Object& o) const
    { return o.deathBy == db; }
};


template<class Shot_t=Shot, class Ufo_t=Ufo, class Ship_t=Ship, class Asteroid_t=Asteroid,
        class Shots_t=std::vector<Shot_t>, class Asteroids_t=std::vector<Asteroid_t> >
class _GameState
{
public:
    static const unsigned MAX_ASTEROIDS = 26;
    static const unsigned MAX_SHOTS_SHIP = 4;
    static const int MAX_SHOTS_UFO = 2;

    typedef Shots_t shots_t;
    typedef Asteroids_t asteroids_t;
    
    uint32_t time;
    Shots_t sshots;
    Shots_t ushots;
    Ufo_t ufo;
    Ship_t ship;
    Asteroids_t asteroids;

    inline _GameState() : time( 0 ), sshots(), ushots(), ufo(), ship(), asteroids()
    {}

    void advance();
    bool targetObject(const Object& obj, Target& res, uint32_t thmax) const;
//    void checkCollisions();
    void clearCollisions(std::set<uint16_t>& objects, bool transitive);
    template<class Matcher>
    void clearCollision(const Matcher& matcher, bool transitive, std::set<uint16_t>& cleared);
    inline void clearCollision(uint16_t id, bool transitive, std::set<uint16_t>& cleared)
    { clearCollision(ObjectIdMatcher(id), transitive, cleared);}

    void eraseChildren(Asteroid& ast, std::set<uint16_t>& cleared);
    void collideObject(uint16_t id);
//    void clearNonTransAndCollideObject(uint16_t id);
    void clearAndCollideObjects(std::set<uint16_t>& ids, bool transitive);
    
    template<class Vessel_t>
    void collideShot(Shot_t& shot, Vessel_t& vessel, bool optimistic);
    template<class Vessel1_t, class Vessel2_t>
    void collideVessel(Vessel1_t& vessel1, Vessel2_t& vessel2, Shots_t& v2shots, bool optimistic);

    void collideSShot(Shot_t& sshot);
    void collideUShot(Shot_t& ushot);
    void collideShip();
    void collideAsteroid(Asteroid_t& ast);
    void collideUfo();
    bool hitByMe(const Object& o) const;
    void print( std::ostream& os ) const;
    uint32_t getTimeOfShotAvailable() const;
    uint8_t getShotsAvailableAt(uint32_t time) const;
    template<class Matcher>
    Object* findObject(const Matcher& matcher);
};

class GameState : public _GameState<Shot, Ufo, Ship, Asteroid>
{
public:
    inline const uint32_t g() const
    { return time; }
    uint32_t h() const;
    inline const uint32_t f() const
    { return g() + h(); }
    
};

/******************************************************************************/
// INLINES

extern int8_t _ast_sin[256];
extern double _ast_tan[127];
inline int8_t ast_sin( uint8_t angle )
{
    return _ast_sin[angle];
}

inline int8_t ast_cos( uint8_t angle )
{
    angle += 0x40;
    return ast_sin( angle );
}

inline double ast_tan( uint8_t angle )
{
    return _ast_tan[(angle+0x3f)%0x80];
}

uint8_t ast_atan2(int16_t y, int16_t x);
// WorldDisp

template<typename size>
inline WorldDisp<size>::WorldDisp( size _dx, size _dy )
        : dx( _dx ), dy( _dy )
{}

template<typename size>
inline WorldDisp<size>::WorldDisp()
        : dx( 0 ), dy( 0 )
{}

template<typename size>
inline bool WorldDisp<size>::operator==( const WorldDisp& o ) const
{
    return dx==o.dx && dy==o.dy;
}

template<typename size>
inline bool WorldDisp<size>::operator!=( const WorldDisp& o ) const
{
    return dx!=o.dx || dy!=o.dy;
}

template<typename size>
inline size WorldDisp<size>::cheb() const
{
    return std::max( abs( dx ), abs( dy ) );
}

template<typename size>
inline double WorldDisp<size>::euclid() const
{
    return sqrt( dx*dx + dy*dy );
}

template<typename size>
inline WorldDisp<size> operator-( const WorldDisp<size>& d0, const WorldDisp<size>& d1 )
{ return WorldDisp<size>(d0.dx-d1.dx, d0.dy-d1.dy); }

template<typename size>
std::ostream& operator<<( std::ostream& os, const WorldDisp<size>& d )
{
    os << "("<<std::setw(3)<<( int )( d.dx )<<", "<<std::setw(3)<<( int )( d.dy )<<")";
    return os;
}

//inline double dist2(const WorldCoords& p1, const WorldCoords& p2)
//{
//int16_t dx = p1.x-p2.x;
//int16_t dy = p1.y-p2.y;
//return sqrt(dx*dx+dy*dy);
//}
//
//inline double distinf(const WorldCoords& p1, const WorldCoords& p2)
//{
//return std::max(abs(p1.x-p2.x), abs(p1.y-p2.y));
//}

inline bool WorldCoords::operator==( const WorldCoords& o ) const
{
    return x==o.x && y==o.y;
}

inline bool WorldCoords::operator!=( const WorldCoords& o ) const
{
    return x!=o.x || y!=o.y;
}

template<typename size>
inline WorldCoords& WorldCoords::operator+=( const WorldDisp<size>& o )
{
//    x = ( uint16_t )( x+o.dx ) % 0x2000;
//    int16_t ny = y+o.dy;
//    y = ny > 0 ? ny % 0x1800 : 0x1800+ny;
    x = mod(x+o.dx, 0x2000);
    y = mod((int)(y)+o.dy, 0x1800);
    return *this;
}

template<typename size>
inline WorldCoords& WorldCoords::operator-=( const WorldDisp<size>& o )
{
    x = mod(x-o.dx, 0x2000);
    y = mod((int)(y)-o.dy, 0x1800);
    return *this;
}

inline std::ostream& operator<<( std::ostream& os, const WorldCoords& c )
{
    os << "("<< std::setw(4)<<c.x<<", "<<std::setw(4)<<c.y<<")";
    return os;
}

inline uint8_t absdist( uint8_t a, uint8_t b )
{
    uint8_t res = a-b;
    return res > 127 ? 256 - res : res;
}

inline WorldDisp16 operator-( const WorldCoords& c0, const WorldCoords& c1 )
{
    int16_t x = c0.x-c1.x;

    if ( x>0x1000 )
        x -= 0x2000;
    else if ( x<-0x1000 )
        x += 0x2000;

    int16_t y = c0.y-c1.y;

    if ( y>0x0c00 )
        y -= 0x1800;
    else if ( y<-0xc00 )
        y += 0x1800;

    return WorldDisp16( x,y );
}


template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::advance()
{
    ++time;
    const bool updateTTL( (( time+frameOffset ) & 3) == 0 );

    for (typename std::list<Shot_t>::iterator s = sshots.begin(); s != sshots.end();) {
//        if (!s->ttl) {
        if (time>s->death) {
            std::cout << "predicting death of sshot" << s->id << std::endl;
            s = sshots.erase(s);
        } else {
            ++s;
        }
    }

    for (typename std::list<Shot_t>::iterator s = ushots.begin(); s != ushots.end();) {
//        if (!s->ttl) {
        if (time>s->death) {
            std::cout << "predicting death of ushot" << s->id << std::endl;
            s = ushots.erase(s);
        } else {
            ++s;
        }
    }
    for ( typename std::list<Asteroid_t>::iterator a = asteroids.begin(); a != asteroids.end();) {
        if (time>a->death) {
            std::cout << "predicting death of rock" << a->id << std::endl;
            a = asteroids.erase(a);
        } else {
            ++a;
        }
    }

    if (ship.activeKeys.fire() && sshots.size() < MAX_SHOTS_SHIP) {
        uint16_t id = ++nextId;
        std::cout << "predicting sshot" << id << std::endl;
        sshots.insert(sshots.end(), Shot_t(id, time, ship));
//        breakpoint();
        collideSShot(sshots.back());
    }

    for (typename std::list<Shot_t>::iterator s = sshots.begin(); s != sshots.end(); ++s)
        //if (s->birth < time)
            s->advance(time, updateTTL);

    for (typename std::list<Shot_t>::iterator s = ushots.begin(); s != ushots.end(); ++s)
        //if (s->birth < time)
            s->advance(time, updateTTL);
    
    //if (ufo.birth < time)
        ufo.advance(time);

    //if (ship.birth < time)
        ship.advance(time);

    for ( typename std::list<Asteroid_t>::iterator a = asteroids.begin(); a != asteroids.end(); ++a )
        //if (a->birth < time)
            a->advance(time);
}


template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
bool _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::targetObject( const Object& obj, Target& tgt, uint32_t thmax ) const
{
    const uint32_t tfmin(getTimeOfShotAvailable());
    const uint8_t initial_heading(ship.heading.start + ship.heading.n / 2);
    std::bitset<256> l_headings;
    l_headings.reset();
    l_headings.set(initial_heading);
    Ship_t l_ship = ship;
    l_ship.activeKeys = 0;
    Object l_obj = obj;
    uint32_t l = time;
    uint32_t r = thmax;
    Mark mark;
    
//    for (size_t i = l; i <= r; ++i) {
//        possible_headings.set((uint8_t) (initial_heading + (i-time) * 3));
//        possible_headings.set((uint8_t) (initial_heading - (i-time) * 3));
//    }

    while (l < tfmin) {
        ++l;
        l_headings.set((uint8_t) (initial_heading + (l-time-1) * 3));
        l_headings.set((uint8_t) (initial_heading - (l-time-1) * 3));
        l_ship.advance(l);
        l_obj.advance(l);
    }
    uint32_t m = l;
    
    while (true) {
        if (l > r)
            return false;
        Ship m_ship = l_ship;
        Object m_obj = l_obj;
        std::bitset<256> m_headings(l_headings);
#ifndef NDEBUG        
        std::cout << "l:" << std::setw(5) << l << "\tm:" << std::setw(5) << m << "\tr:" << std::setw(5) << r << "\t";
#endif
        for (size_t i = l; i < m; ) {
            ++i;
            m_ship.advance(i);
            m_obj.advance(i);
            m_headings.set((uint8_t) (initial_heading + (i-time-1) * 3));
            m_headings.set((uint8_t) (initial_heading - (i-time-1) * 3));
        }
        if (m_obj.p_error.cheb() > m_obj.size) {
            r = m-1;
            m = r;
            continue;
        }
        
        uint32_t turnt;
        Interval<uint8_t> aperture;
          uint8_t ship_heading = ship.heading.start + ship.heading.n / 2;
        if (m_ship.targetObject(time, m_obj, time+100, m_headings, mark)) { // TODO
//        Interval<uint8_t> aperture;
//        double a;
//        if (m_ship.targetObject(m_obj, 100, aperture, a)) { // TODO
//
//        uint8_t ship_heading = ship.heading.start + ship.heading.n / 2;
//        uint8_t ticks(255);
//        for (uint8_t a = aperture.start; a != aperture.end(); ++a) {
//            uint8_t t = abs(angle2ticks(a - ship_heading));
//            if (t < ticks) {
//                angle = a;
//                ticks = t;
//            }
//        }
        uint8_t ticks(abs(angle2ticks(mark.alpha - ship_heading)));
//        turnt = time + ticks + 1;
        turnt = time + ticks;
        }
        else {
            if (aperture.empty())
            turnt = m+1; // TODO
            else {
        uint8_t ticks(255);
        for (uint8_t a = aperture.start; a != aperture.end(); ++a) {
            uint8_t t = abs(angle2ticks(a - ship_heading));
            if (t < ticks) {
//                angle = a;
                ticks = t;
            }
        }
        turnt = time + ticks;

        }
        }
        std::cout << "\tturnt:" << std::setw(5) << turnt << std::endl;

        if (turnt < m) {
            if (m - turnt <= 2)
                break;

            if (m == r) {
                if (l == m)
                    break;

                m = l;
                continue;
            }

            r = m;
            m = std::max(l, turnt);

            if (r - m <= 1)
                break;
        } else if (turnt > m) {
            if (m == l) {
                ++l;
                l_ship.advance(l);
                l_obj.advance(l);
                l_headings.set((uint8_t) (initial_heading + (l-time-1) * 3));
                l_headings.set((uint8_t) (initial_heading - (l-time-1) * 3));

                if (turnt >= r)
                    m = l;
                else
                    m = std::min(turnt, r);

                continue;
            } else if (turnt < r) {
                m = turnt;
            } else if (m == r) {
                --r;
                m = l;
            } else {
                //m = r;
                m = l;
            }
        } else // turnt == m
            break;
    }

    tgt.tfire = m;
    tgt.thit = mark.thit;
    tgt.alpha = mark.alpha;

    // TODO check collision time <-> torpedo range
    return true;
}



                
inline void markCollision(Object& o1, Object& o2, uint32_t time, std::set<uint16_t>& displaced)
{
    uint16_t dp;
    if ((dp = o1.deathBy))
        displaced.insert(dp);
    //assert(!displaced1); we may be called explicitly but second after update
    if ((dp = o2.deathBy))
        displaced.insert(dp);
    o1.death = time;
    o1.deathBy = o2.id;
    o2.death = time;
    o2.deathBy = o1.id;
}

template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
template<class Vessel_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::collideShot(Shot_t& shot, Vessel_t& vessel, bool optimistic)
{
    uint32_t tmin = shot.getTimeOfDeath(time);
    tmin = std::min(tmin, shot.death);
    uint16_t& deathBy(shot.deathBy);
    
    bool hitvessel(false);
    if (vessel.present && vessel.id != deathBy) {
        uint32_t t = collide(shot, vessel, time, min(tmin, vessel.death), optimistic);
        if (t != 0xffffffff) {
            tmin = t;
            hitvessel = true;
        }
    }        
    
    typename Asteroids_t::iterator hitast(asteroids.end());
    for ( typename Asteroids_t::iterator ast = asteroids.begin(); ast != asteroids.end(); ++ast ) {
        if (ast->id == deathBy) continue;
        uint32_t t = collide(shot, *ast, time, min(tmin, ast->death), false);
        if (t != 0xffffffff) {
            tmin = t;
            hitast = ast;
            hitvessel = false;
        }
    }

    std::set<uint16_t> displaced;
    if (hitvessel) {
        markCollision(shot, vessel, tmin, displaced);
    } else if (hitast != asteroids.end()) {
        if (hitast->size > Asteroid::SIZE_SMALL) {
            eraseChildren(*hitast, displaced);
            markCollision(shot, *hitast, tmin, displaced);
            uint16_t id = ++nextId;
            asteroids.insert(asteroids.end(), Asteroid_t(id, *hitast));
            id = ++nextId;
            asteroids.insert(asteroids.end(), Asteroid_t(id, *hitast));
        } else {
            markCollision(shot, *hitast, tmin, displaced);
        }
    }
    if (!shot.deathBy)
        shot.death=shot.getTimeOfDeath(time);

    clearAndCollideObjects(displaced, false);
}

template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
template<class Vessel1_t, class Vessel2_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::collideVessel(Vessel1_t& vessel1, Vessel2_t& vessel2, Shots_t& v2shots, bool optimistic)
{
    if (!vessel1.present)
        return;
    
    uint32_t tmin = vessel1.death;
    uint16_t& deathBy(vessel1.deathBy);
    
    bool hitvessel2(false);
    if (vessel2.present && vessel2.id != deathBy) {
        uint32_t t = collide(vessel1, vessel2, time, min(tmin, vessel2.death), true);
        if (t != 0xffffffff) {
            tmin = t;
            hitvessel2 = true;
        }
    }        
    
    typename Asteroids_t::iterator hitast(asteroids.end());
    for ( typename Asteroids_t::iterator ast = asteroids.begin(); ast != asteroids.end(); ++ast ) {
        if (ast->id == deathBy) continue;
        uint32_t t = collide(vessel1, *ast, time, min(min(tmin, ast->death), time+5000), optimistic); // TODO!
        if (t != 0xffffffff) {
            tmin = t;
            hitast = ast;
            hitvessel2 = false;
        }
    }

    typename Shots_t::iterator hitshot(v2shots.end());
    for ( typename Shots_t::iterator shot = v2shots.begin(); shot != v2shots.end(); ++shot ) {
        if (shot->id == deathBy) continue;
        uint32_t till = min(tmin, shot->death);
        uint32_t t = collide(vessel1, *shot, time, min(till, shot->getTimeOfDeath(time)), optimistic);
        if (t != 0xffffffff) {
            tmin = t;
            hitshot = shot;
            hitvessel2 = false;
            hitast = asteroids.end();
        }
    }

    std::set<uint16_t> displaced;
    if (hitvessel2) {
        markCollision(vessel1, vessel2, tmin, displaced);
    } else if ( hitast != asteroids.end()) {
        if (hitast->size > Asteroid::SIZE_SMALL) {
           eraseChildren(*hitast, displaced);
           markCollision(vessel1, *hitast, tmin, displaced);
           uint16_t id = ++nextId;
           asteroids.insert(asteroids.end(), Asteroid_t(id, *hitast));
           id = ++nextId;
           asteroids.insert(asteroids.end(), Asteroid_t(id, *hitast));    
        } else {
           markCollision(vessel1, *hitast, tmin, displaced);
        }
    } else if ( hitshot != v2shots.end()) {
        markCollision(vessel1, *hitshot, tmin, displaced);
    }
    
    clearAndCollideObjects(displaced, false);
};


template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::collideSShot(Shot_t& sshot)
{
    collideShot(sshot, ufo, false);
}

template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::collideUShot(Shot_t& ushot)
{
    collideShot(ushot, ship, true);
}

template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::collideUfo()
{
    collideVessel(ufo, ship, sshots, false);
    if (!ufo.deathBy)
        ufo.death=ufo.getTimeOfNaturalDeath();
}

template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::collideShip()
{
    collideVessel(ship, ufo, ushots, true);
}

template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::collideAsteroid(Asteroid_t& ast)
{
    uint32_t tmin = ast.death;
    uint16_t& deathBy = ast.deathBy;
    
    bool hitship(false);
    if (ship.present && ship.id != deathBy) {
        uint32_t t = collide(ast, ship, time, min(min(tmin, ship.death), time+5000), true); // TODO
        if (t != 0xffffffff) {
            tmin = t;
            hitship = true;
        }
    }        
 
    bool hitufo(false);
    if (ufo.present && ufo.id != deathBy) {
        uint32_t t = collide(ast, ufo, time, min(tmin, ufo.death), false);
        if (t != 0xffffffff) {
            tmin = t;
            hitship = false;
            hitufo = true;
        }
    }        
 
    typename Shots_t::iterator hitsshot(sshots.end());
    for ( typename Shots_t::iterator sshot = sshots.begin(); sshot != sshots.end(); ++sshot ) {
        if (sshot->id == deathBy) continue;
        uint32_t till = min(tmin, sshot->death);
        till = min(till, sshot->getTimeOfDeath(time));
        uint32_t t = collide(ast, *sshot, time, till, false);
        if (t != 0xffffffff) {
            tmin = t;
            hitsshot = sshot;
            hitship = false;
            hitufo = false;
        }
    }

    typename Shots_t::iterator hitushot(ushots.end());
    for ( typename Shots_t::iterator ushot = ushots.begin(); ushot != ushots.end(); ++ushot ) {
        if (ushot->id == deathBy) continue;
        uint32_t till = min(tmin, ushot->death);
        till = min(till, ushot->getTimeOfDeath(time));
        uint32_t t = collide(ast, *ushot, time, till, false);
        if (t != 0xffffffff) {
            tmin = t;
            hitushot = ushot;
            hitship = false;
            hitufo = false;
            hitsshot = sshots.end();
        }
    }
    
    
    std::set<uint16_t> displaced;
    const bool mindChildren((hitship||hitufo||(hitsshot!=sshots.end())||(hitushot!=ushots.end()))
                            && (ast.size > Asteroid::SIZE_SMALL));
    if (mindChildren)
        eraseChildren(ast, displaced);
        
    if (hitship) {
        markCollision(ast, ship, tmin, displaced);
    } else if (hitufo) {
        markCollision(ast, ufo, tmin, displaced);
    } else if ( hitsshot != sshots.end()) {
        markCollision(ast, *hitsshot, tmin, displaced);
    } else if ( hitushot != ushots.end()) {
        markCollision(ast, *hitushot, tmin, displaced);
    }       
    if (mindChildren) {
           uint16_t id = ++nextId;
           asteroids.insert(asteroids.end(), Asteroid_t(id, ast));
           id = ++nextId;
           asteroids.insert(asteroids.end(), Asteroid_t(id, ast));    
    }
    
    clearAndCollideObjects(displaced, false);
}


template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::eraseChildren(Asteroid& ast, std::set<uint16_t>& cleared)
{
    for (typename Asteroids_t::iterator a = asteroids.begin(); a != asteroids.end();) {
        if (a->birth == ast.death && a->size < ast.size ) {  // TODO use id?
            if (a->deathBy) {
                eraseChildren(*a, cleared);
                clearCollision(a->deathBy, false, cleared);
            }
            a = asteroids.erase(a);
        } else {
            ++a;
        }
    }
}

inline std::ostream& operator<<( std::ostream& os, const ObjectIdMatcher& m )
{
    os << "id==" << m.id;
    return os;
}

inline std::ostream& operator<<( std::ostream& os, const ObjectDeathByMatcher& m )
{
    os << "deathBy==" << m.db;
    return os;
}

template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
template<class Matcher>
Object* _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::findObject(const Matcher& matcher)
{
    if (matcher.matches(ship))
        return &ship;

    if (matcher.matches(ufo))
        return &ufo;

    for (typename Asteroids_t::iterator ast = asteroids.begin(); ast != asteroids.end(); ++ast)
        if (matcher.matches(*ast))
            return &*ast;


    for (typename Shots_t::iterator sshot = sshots.begin(); sshot != sshots.end(); ++sshot)
        if (matcher.matches(*sshot))
            return &*sshot;

    for (typename Shots_t::iterator ushot = ushots.begin(); ushot != ushots.end(); ++ushot)
        if (matcher.matches(*ushot))
            return &*ushot;

    return 0;
};


template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
template<class Matcher>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::clearCollision(const Matcher& matcher, bool transitive, std::set<uint16_t>& cleared) {
    uint16_t db;
#ifndef NDEBUG
    std::cout << "Clearing collision with " << matcher << " trans:" << transitive << std::endl;
#endif
    if (matcher.matches(ship)) {
        if ((db = ship.deathBy)) {
            ship.deathBy = 0;
            ship.death = 0xffffffff;
            cleared.insert(ship.id);
            if (transitive)
                clearCollision(db, false, cleared);
        }
        return;
    }

    if (matcher.matches(ufo)) {
        if ((db = ufo.deathBy)) {
            ufo.deathBy = 0;
            ufo.death = ufo.getTimeOfNaturalDeath();
            cleared.insert(ufo.id);
            if (transitive)
                clearCollision(db, false, cleared);
        }
        return;
    }

    for (typename Asteroids_t::iterator ast = asteroids.begin(); ast != asteroids.end(); ++ast) {
        if (matcher.matches(*ast)) {
            if ((db = ast->deathBy)) {
                if (ast->size > Asteroid::SIZE_SMALL)
                    eraseChildren(*ast, cleared);
                ast->deathBy = 0;
                ast->death = 0xffffffff;
                cleared.insert(ast->id);
                if (transitive)
                    clearCollision(db, false, cleared);
            }
            return;
        }
    }


    for (typename Shots_t::iterator sshot = sshots.begin(); sshot != sshots.end(); ++sshot) {
        if (matcher.matches(*sshot)) {
            if ((db = sshot->deathBy)) {
                sshot->deathBy = 0;
                sshot->death = sshot->getTimeOfDeath(time);
                cleared.insert(sshot->id);
                if (transitive)
                    clearCollision(db, false, cleared);
            }
            return;
        }
    }

    for (typename Shots_t::iterator ushot = ushots.begin(); ushot != ushots.end(); ++ushot) {
        if (matcher.matches(*ushot)) {
            if ((db = ushot->deathBy)) {
                ushot->deathBy = 0;
                ushot->death = ushot->getTimeOfDeath(time);
                cleared.insert(ushot->id);
                if (transitive)
                    clearCollision(db, false, cleared);
            }
            return;
        }
    }
}

template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::clearCollisions(std::set<uint16_t>& objects, bool transitive)
{
    std::set<uint16_t> neu;
    for (std::set<uint16_t>::const_iterator o = objects.begin(); o != objects.end(); ++o) {
        clearCollision(*o, transitive, neu);
    }
    objects.insert(neu.begin(), neu.end());
}
       
template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
         class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::collideObject(uint16_t id) {
//    uint16_t displaced(0);
    if (id == ship.id) {
        collideShip();
    } else if (id == ufo.id) {
        collideUfo();
    } else {
        for (typename Asteroids_t::iterator ast = asteroids.begin(); ast != asteroids.end(); ++ast) {
            if (ast->id == id) {
                collideAsteroid(*ast);
                return;
            }
        }
    }
    for (typename Shots_t::iterator sshot = sshots.begin(); sshot != sshots.end(); ++sshot) {
        if (sshot->id == id) {
            collideSShot(*sshot);
            return;
        }
    }
    for (typename Shots_t::iterator ushot = ushots.begin(); ushot != ushots.end(); ++ushot) {
        if (ushot->id == id) {
            collideUShot(*ushot);
            return;
        }
    }
}


template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
         class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::clearAndCollideObjects(std::set<uint16_t>& ids, bool transitive) {
    clearCollisions(ids, transitive);
    for (std::set<uint16_t>::const_iterator o = ids.begin(); o != ids.end(); ++o) {
        collideObject(*o);
}
}


template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
                class Shots_t, class Asteroids_t>  
bool _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::hitByMe(const Object& o) const
{
    const uint16_t& hitBy(o.deathBy);
    if (hitBy)
        for (typename Shots_t::const_iterator s = sshots.begin(); s != sshots.end(); ++s)
            if (s->id == hitBy)
                return true;
    return false;
}


template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
void _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::print( std::ostream& os ) const
{
    using std::cout;
    using std::endl;
    using std::list;
    using std::hex;
    using std::dec;
    cout << "Time:" << time << endl;
    for ( typename list<Shot_t>::const_iterator shot = sshots.begin(); shot != sshots.end(); ++shot ) {
        printObject( cout, *shot, "sshot" ) << "\tttl:" << (int)(shot->ttl) << endl;
    }

    for ( typename list<Shot_t>::const_iterator shot = ushots.begin(); shot != ushots.end(); ++shot ) {
        printObject( cout, *shot, "ushot" ) << "\tttl:" << (int)(shot->ttl) << endl;
    }

    if ( ufo.present ) {
        printObject( cout, ufo, "ufo" ) << endl;
    }

    if ( ship.present ) {
        printObject( cout, ship, "ship" )
        << "\t" << hex << ship.heading << dec << endl;
    }

    for ( typename list<Asteroid_t>::const_iterator ast = asteroids.begin(); ast != asteroids.end(); ++ast ) {
        printObject( cout, *ast, "rock" ) << endl;
    }
}

template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
uint32_t _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::getTimeOfShotAvailable() const
{
    uint32_t res;
    if (sshots.size() < MAX_SHOTS_SHIP) {
        res = time+1;
    } else {
        res = UINT32_MAX;
        for (typename Shots_t::const_iterator shot = sshots.begin(); shot != sshots.end(); ++shot)
            res = min(res, shot->death+1);
    }
    if ((res == time+1) && ship.activeKeys.fire())
        ++res;

    std::cout << "Next shot available at " << res << std::endl;
    assert (res > time);
    return res;
};

template<class Shot_t, class Ufo_t, class Ship_t, class Asteroid_t,
        class Shots_t, class Asteroids_t>
uint8_t _GameState<Shot_t, Ufo_t, Ship_t, Asteroid_t, Shots_t, Asteroids_t>::getShotsAvailableAt(uint32_t t) const
{
    uint8_t res(MAX_SHOTS_SHIP);
     for ( typename Shots_t::const_iterator shot = sshots.begin(); shot != sshots.end(); ++shot ) {
        if (shot->death >= t)
            --res;
    }
    assert (res <= 4);
    std::cout << "Shots available at " << t << " : " << (int)(res) << std::endl;
    return res;
};

#endif // !defined WORLD_H
