// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import de.caff.util.Pair;

import java.util.*;

/**
 *  Helper class which tries to add velocities to game objects and lifetime to bullets.
 *  This is done in a too simple-minded way by just comparing the last two screens and connecting nearest objects
 *  (using the velocities calculated for the previous frame).
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class SimpleVelocityPreparer
        implements FramePreparer
{
  /** Maximum squared distance we assume an asteroid may move between frames. */
  private static final int MAX_SQUARED_ASTEROID_VELOCITY = 6*6;
  /** Maximum squared distance we assume a bullet may move between frames. */
  private static final int MAX_SQUARED_BULLET_VELOCITY = 16*16;

  /**
   *  Helper class to presort asteroids to allow easier compare between frames.
   */
  private static class AsteroidSelector
  {

    /**
     *  Helper class: key for mapping asteroids.
     */
    private static class Key
    {
      /** The asteroid's type. */
      private final int type;
      /** The asteroids size. */
      private final int size;

      /**
       * Constrictor.
       * @param ast  asteroid
       */
      private Key(Asteroid ast)
      {
        this.type = ast.getType();
        this.size = ast.getSize();
      }

      public boolean equals(Object o)
      {
        if (this == o) {
          return true;
        }
        if (o == null || getClass() != o.getClass()) {
          return false;
        }

        Key key = (Key)o;

        if (size != key.size) {
          return false;
        }
        if (type != key.type) {
          return false;
        }

        return true;
      }

      public int hashCode()
      {
        int result;
        result = type;
        result = 31 * result + size;
        return result;
      }
    }
    /** Presorted mapping of asteroids. */
    private Map<Key, Collection<Asteroid>> sorted = new HashMap<Key, Collection<Asteroid>>();

    /**
     *  Constructor.
     *  @param asteroids asteroids to be presorted
     */
    private AsteroidSelector(Collection<Asteroid> asteroids)
    {
      for (Asteroid ast: asteroids) {
        Key key = new Key(ast);
        Collection<Asteroid> list = sorted.get(key);
        if (list == null) {
          list = new LinkedList<Asteroid>();
          sorted.put(key, list);
        }
        list.add(ast);
      }
    }

    /**
     *  Get the best matching asteroid assuming the keys collected here are from the previous
     *  frame while the given asteroid is from the current frame.
     *  @param asteroid current frame astreroid
     *  @return best match or <code>null</code>
     */
    public Asteroid getBestMatch(Asteroid asteroid)
    {
      Key key = new Key(asteroid);
      Collection<Asteroid> list = sorted.get(key);
      SortedMap<Integer, Asteroid> result = new TreeMap<Integer, Asteroid>();
      if (list != null) {
        int futureX = asteroid.getX() + asteroid.getVelocityX();
        int futureY = asteroid.getY() + asteroid.getVelocityY();
        for (Asteroid a: list) {
          int dist2 = a.getSquaredDistance(futureX, futureY);
          if (dist2 < MAX_SQUARED_ASTEROID_VELOCITY) {
            result.put(dist2, a);
          }
        }
      }
      return result.isEmpty() ? null : result.values().iterator().next();
    }
  }

  /**
   * Prepare the frame(s).
   *
   * @param frameInfos the collected frame infos
   */
  public void prepareFrames(LinkedList<FrameInfo> frameInfos)
  {
    if (frameInfos.size() >= 2) {
      FrameInfo lastFrame = frameInfos.getLast();
      FrameInfo prevFrame = frameInfos.get(frameInfos.size() - 2);
      Ufo ufo = lastFrame.getUfo();
      if (ufo != null) {
        ufo.setVelocityFromDelta(prevFrame.getUfo());
        //ufo.setVelocityDoubles();
      }
      SpaceShip ship = lastFrame.getSpaceShip();
      if (ship != null) {
        ship.setVelocityFromDelta(prevFrame.getSpaceShip());
        //ship.setVelocityDoubles();
      }

      // Asteroids
      AsteroidSelector selector = new AsteroidSelector(lastFrame.getAsteroids());
      for (Asteroid ast: prevFrame.getAsteroids()) {
        Asteroid candidate = selector.getBestMatch(ast);
        
        if (candidate != null) {
          //System.out.println(candidate.x+":"+candidate.y+"-"+ast.x+":"+ast.y);
          candidate.setVelocityFromDelta(ast);
          candidate.setVelocityDoubles();
        }
      }

      // Bullets
      for (Bullet oldBullet: prevFrame.getBullets()) {
        int futureX = oldBullet.getX() + oldBullet.getVelocityX();
        int futureY = oldBullet.getY() + oldBullet.getVelocityY();

        SortedMap<Integer, Pair<Bullet>> result = new TreeMap<Integer, Pair<Bullet>>();
        for (Bullet bullet: lastFrame.getBullets()) {
          int dist2 = bullet.getSquaredDistance(futureX, futureY);
          if (dist2 < MAX_SQUARED_BULLET_VELOCITY) {
            result.put(dist2, new Pair<Bullet>(oldBullet, bullet));
          }
        }
        LinkedList<Pair<Bullet>> pairs = new LinkedList<Pair<Bullet>>(result.values());
        while (!pairs.isEmpty()) {
          Pair<Bullet> pair = pairs.remove(0);
          pair.second.setVelocityFromDelta(pair.first);
          pair.second.setVelocityDoubles();
          pair.second.setLifetime(pair.first.getLifetime() + 1);
          for (ListIterator<Pair<Bullet>> it = pairs.listIterator();  it.hasNext();  ) {
            Pair<Bullet> p = it.next();
            if (p.first.equals(pair.first)  ||  p.second.equals(pair.second)) {
              it.remove();
            }
          }
        }
      }
    }
    else if (frameInfos.size() > 1) {
      // set bullet lifetimes to 1 to indicate lifetimes are calculated
      for (Bullet bullet: frameInfos.getLast().getBullets()) {
        bullet.setLifetime(1);
      }
    }
  }
}
