import gui.MainFrame;
import gui.MainPanel;
import helpers.CombatHeuristics;
import helpers.CombatMessages;
import helpers.CombatStatistics;
import io.CombatAI;
import io.CombatConnection;
import io.CombatControl;
import io.CombatState;

import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.LinkedList;

import javax.swing.JOptionPane;

import objects.Asteroid;
import objects.CombatObject;
import objects.PreAsteroid;
import objects.Saucer;
import objects.Ship;
import objects.Shot;
import objects.TargetObject;
import structures.TargetConfiguration;
import structures.TargetInformation;
import utils.Tables;
import utils.Utils;

// Title: EarthDefenseBot 0.5.1
// Autor: Thomas Berndt
// Last Modified: 06/29/2008
// Revision: 40
public class EarthDefenseBot implements CombatAI {
	
	public static final String WINDOW_TITLE = "Earth Defense Bot v.0.5.1";
	
	public static void main(String[] arguments)
	{
		EarthDefenseBot edb = new EarthDefenseBot(arguments);	
		edb.show();
	}
	
	//--- GUI ---
	private MainFrame mainFrame;
	
	private MainPanel mainPanel;
	
	//--- Data ---
	private CombatConnection connection;
	
	private Thread updateGUIThread;
	
	private TargetConfiguration curTarget = new TargetConfiguration();
	
	private LinkedList<CombatObject> vicinity = new LinkedList<CombatObject>();
	
	private LinkedList<CombatObject> colliding = new LinkedList<CombatObject>();
	
	private long lastShotTime = Long.MAX_VALUE;
	
	private boolean isShipLost = false; 
	
	private double maxEvasionSpeed = 8.0;
	
	private boolean isStartingAutonomous = true;
	
	private boolean isStoppingAutonomous = true;
	
	private boolean isActivated = false;	
	
	private boolean isDataDirty = false;	
	
	private boolean forceTargetSwitch = false; 
	
	private Point[] strategicPositions = 
	{
		new Point(148, 276),
		new Point(875, 276),
		new Point(148, 747),
		new Point(875, 747)
	};
	
	private Runnable updateGUICycle = new Runnable()
	{
		public void run()
		{
			while(isActivated)
			{
				try {
					Thread.sleep(500);
				} catch (InterruptedException e) {
				}
				updateStatus();
				mainPanel.repaint();
				isDataDirty = true;
			}
		}
	};

	public ActionListener controlListener = new ActionListener()
	{
		public void actionPerformed(ActionEvent e)
		{
			if(mainPanel.controlPanel.isServerButtonClicked(e))
			{
				Object newServer = JOptionPane.showInputDialog(
					mainFrame, 					
					"Please enter the new server IP: ", 
					"Select Server", 
					JOptionPane.QUESTION_MESSAGE, 
					null, null, 
					connection);
				if(newServer != null)
				{
					setActivated(false);
					connection.selectHost(newServer.toString());					
				}
			} else if(mainPanel.controlPanel.isStartButtonClicked(e))
			{
				toggleActivated();
			} else if(mainPanel.controlPanel.isAutoStartBoxToggled(e)) {
				isStartingAutonomous = !isStartingAutonomous;
			} else if(mainPanel.controlPanel.isAutoStopBoxToggled(e)) {
				isStoppingAutonomous = !isStoppingAutonomous;
			}
		}
	};
	
	public EarthDefenseBot(String[] args) {
		// Init Tables
		Utils.init();
		
		// Init UI
		mainPanel = new MainPanel(controlListener);
		mainFrame = new MainFrame("/icon.gif", WINDOW_TITLE, mainPanel);
		
		// Init message forwarding to GUI
		CombatMessages.setGUIOutput(mainPanel.combatPanel);
		
		// Read server address from parameters
		String host = CombatConnection.DEFAULT_HOST;
		if(args.length > 0)
		{
			host = args[0];
		}
		
		// Read port from parameters
		int port = CombatConnection.DEFAULT_PORT;
		if(args.length > 1)
		{
			port = Integer.parseInt(args[1]);
		}	
		
		// Init connection 
		connection = new CombatConnection(host, port, this);
		
		// Init connection status forwarding to GUI
		connection.setGUIOutput(mainPanel.statusPanel);
		
		// Start communication with mame
		connection.startCommunication();
	}
	
	public boolean approximateTarget(TargetConfiguration config, CombatState state, CombatControl control)
	{
		calculateTurnTime(config, config.info, control);
		config.shootTime = 0;
		config.postWaitTime = 0;
		int lastAngleIndex;
		int angleIndexDif;
		int[] timeTaken = new int[10];
		int i = 0;
		int invalidCount = 0;
		boolean isConverging = true;
		boolean isFinished = false;
		
		// Approximate optimum shot angle by iterating the appropriate formulas
		do
		{	
			lastAngleIndex = config.angleIndex;
			timeTaken[i] = config.getTimeTaken();
			config.preCalculate(timeTaken[i], state, control, false);
			calculateTurnTime(config, config.preCalc, control);
			calculateShootTime(config, config.preCalc, control);
			angleIndexDif = Utils.getAngleIndexDif(config.angleIndex, lastAngleIndex);
			
			// Approximation might not always converge
			for(int j = i - 2; j > 0; j--)
			{
				if(timeTaken[i] == timeTaken[j])
				{
					isConverging = false;
					break;
				}
			}		
			if(isConverging == false)
			{
				// Non converging: Choose average then
				int timeSum = timeTaken[i];
				int numTimes = 1;
				for(int j = i - 1; j > 0 && timeTaken[j] != timeTaken[i]; j--)
				{
					timeSum += timeTaken[j];
					++numTimes;
				}
				
				if(timeSum % numTimes == 0)
				{
					timeTaken[i] = timeSum / numTimes;
				} else {
					double crossProduct = config.info.difX * config.object.getMovementY() - config.info.difY * config.object.getMovementX();
					if(Utils.getAngleDif(state.getShip().getMovementAngle(), config.info.angle) < 0)
					{
						crossProduct *= -1;
					}
					if(crossProduct < 0)
					{
						// Wait for target to approach 
						timeTaken[i] = timeSum / numTimes + 1;
					} else {
						// Get ahead of target and wait for approach
						timeTaken[i] = timeSum / numTimes;
					}
				}
				config.preCalculate(timeTaken[i], state, control, false);
				calculateTurnTime(config, config.preCalc, control);
				calculateShootTime(config, config.preCalc, control);				
				timeTaken[i] = config.getTimeTaken();
				config.preCalculate(timeTaken[i], state, control, true);
			}
			
			// Approximation might end with impossible value
			if(config.isValid == false)
			{
				++invalidCount;
			} else {
				invalidCount = 0;
			}
			
			// Detect approximation termination
			isFinished = 
				invalidCount == 2 
				|| i == 9
				|| (i > 0 && angleIndexDif == 0 && timeTaken[i] == timeTaken[i - 1]);
			++i;
		}while(!isFinished && isConverging);		

		// Add possible waiting time to calculation (up to 10 frames)
		if(config.isValid)
		{
			
			if(config.object.isMovementPrecise() && !state.getShip().isMoving())
			{	
				i = 0;				
				config.preCalc.angleDif = Utils.getAngleDif(Utils.TABLES.getAngle(config.angleIndex), config.preCalc.angle);
				if(!CombatHeuristics.isTargetHittable(config, config.preCalc))
				{
					config.postWaitTime = config.turnTime - config.preWaitTime;
					do
					{
						++config.postWaitTime;
						config.preCalculate(config.getWaitTime() + config.shootTime, state, control, false);
						calculateTurnTime(config, config.preCalc, control);
						calculateShootTime(config, config.preCalc, control);
						config.preCalc.angleDif = Utils.getAngleDif(Utils.TABLES.getAngle(config.angleIndex), config.preCalc.angle);
						++i;					
					} while(i < 10 && !CombatHeuristics.isTargetHittable(config, config.preCalc));					
				}				
				config.preCalculate(Math.max(config.getWaitTime(), config.turnTime) + config.shootTime, state, control, true);			
				if(i < 10)
				{
					config.isDifficult = false;
				} else {
					config.isDifficult = true;
				} 
			}
			return true;
		} 
		return false;
	}
	
	public void aquireTarget(CombatState state, CombatControl control, boolean firstInterpretation)
	{	
		// Aquire a new target to aim for
		Ship ship = state.getShip();
		Saucer saucer = state.getSaucer();	
		double bestRating = Double.MAX_VALUE;
		TargetConfiguration config = new TargetConfiguration();
		curTarget.object = null;
		curTarget.isDifficult = false;
		forceTargetSwitch = false;
		int asteroidNum = state.getAsteroidNumber();
		
		// Iterate through the asteroids and select the most suitable one
		for (Asteroid curAsteroid : state.getAsteroids())
		{
			config.calculate(ship, curAsteroid);			
			if(firstInterpretation)
			{
				checkCollisionStatus(config, false);
			}
			if(curAsteroid.getTargetedStatus() == 0) 
			{	
				config.preWaitTime = CombatHeuristics.getPrecisionDelay(config.info.distance, curAsteroid) + 1;
				if(approximateTarget(config, state, control))
				{						
					config.prioritization = CombatHeuristics.getCollidingPrioritization(config);
					if(config.isBetterThan(bestRating))
					{
						bestRating = config.getRating();	
						curTarget.setConfig(config);
					}
				}
			} else if(curAsteroid.hasChild())
			{
				PreAsteroid child = curAsteroid.getChild();				
				config.calculate(ship, child);
				config.preWaitTime = child.getTimeToSpawn() + 1;
				if(approximateTarget(config, state, control) && config.isBetterThan(bestRating))
				{
					bestRating = config.getRating();	
					curTarget.setConfig(config);
				}
			}
		}
		
		// Check if there's a saucer available
		if(state.isSaucerAlive())
		{
			config.calculate(ship, saucer);
			if(firstInterpretation)
			{
				checkCollisionStatus(config, saucer.getScale() == CombatObject.SMALL_SCALE);
			}
			if(saucer.getTargetedStatus() == 0)
			{
				if(approximateTarget(config, state, control))
				{
					config.prioritization = CombatHeuristics.getSaucerPrioritization(state, config, saucer, asteroidNum);
					if(config.isBetterThan(bestRating))
					{
						bestRating = config.getRating();
						curTarget.setConfig(config);
					}
				}
			}
		}
	}
	
	public void calculateShootTime(TargetConfiguration c, TargetInformation i, CombatControl control)
	{
		// Calculate approximated shoot time
		c.shootTime = (int)Math.round(i.distance / Utils.TABLES.getShotSpeed(c.angleIndex));		
		if(control.willFire())
		{
			c.shootTime += 2;
		} else {
			c.shootTime += Math.max(2 - Math.max(c.getWaitTime(), c.turnTime), 1);		
			if(control.isFiring())
			{	
				++c.shootTime;
			}
		}
	}
	
	public void calculateTurnTime(TargetConfiguration c, TargetInformation i, CombatControl control)
	{
		// Calculate approximated turn time
		c.turnTime = 0;
		int startIndex = c.ship.getAngleIndex();
		if(control.isTurningLeft()) 
		{
			++c.turnTime;
			startIndex = Utils.incByteAngle(startIndex);
		} else if(control.isTurningRight()) 
		{
			++c.turnTime;
			startIndex = Utils.decByteAngle(startIndex);	
		} 
		i.angleDif = Utils.getAngleDif(Utils.TABLES.getAngle(startIndex), i.angle);
		int nextIndex = startIndex;
		if(i.angleDif > 0)
		{
			// Counter-clockwise search for angle
			if(i.angle < 0)
			{
				while(Utils.TABLES.getAngle(nextIndex) > 0)
				{
					++c.turnTime;
					nextIndex = Utils.incByteAngle(nextIndex);
				}
			} else  {
				while(Utils.TABLES.getAngle(nextIndex) < 0)
				{
					++c.turnTime;
					nextIndex = Utils.incByteAngle(nextIndex);
				}	
			}			
			while(!Utils.isAngleReachedCCW(Utils.TABLES.getAngle(nextIndex), i.angle))
			{
				++c.turnTime;				
				nextIndex = Utils.incByteAngle(nextIndex);
			}
			// Found -> select nearest
			double difNext = Math.abs(Utils.getAngleDif(Utils.TABLES.getAngle(nextIndex), i.angle));
			c.angleIndex = Utils.decByteAngle(nextIndex);
			double difLast = Math.abs(Utils.getAngleDif(Utils.TABLES.getAngle(c.angleIndex), i.angle));
			if(difNext > difLast)
			{ 
				--c.turnTime;
			} else {				
				c.angleIndex = nextIndex;
			}
			// Check whether a faster direction still suits
			if(c.angleIndex != startIndex)
			{
				nextIndex = Utils.decByteAngle(c.angleIndex);
				difLast = Math.abs(Utils.getAngleDif(Utils.TABLES.getAngle(nextIndex), i.angle));
				double hitZone = CombatHeuristics.getImprovedHitZone(c.object);			
				double deviationLast = Utils.TABLES.sin(difLast) * i.distance;
				if(deviationLast < hitZone)
				{
					--c.turnTime;
					c.angleIndex = nextIndex;
				}
			}
		} else {
			// Clockwise search for angle
			if(i.angle > 0)
			{
				while(Utils.TABLES.getAngle(nextIndex) < 0)
				{
					++c.turnTime;
					nextIndex = Utils.decByteAngle(nextIndex);
				}				
			} else  {
				while(Utils.TABLES.getAngle(nextIndex) > 0)
				{
					++c.turnTime;
					nextIndex = Utils.decByteAngle(nextIndex);
				}	
			}
			while(!Utils.isAngleReachedCW(Utils.TABLES.getAngle(nextIndex), i.angle))
			{
				++c.turnTime;
				nextIndex = Utils.decByteAngle(nextIndex);
			}
			// Found -> select nearest
			double difNext = Math.abs(Utils.getAngleDif(Utils.TABLES.getAngle(nextIndex), i.angle));
			c.angleIndex = Utils.incByteAngle(nextIndex);
			double difLast = Math.abs(Utils.getAngleDif(Utils.TABLES.getAngle(c.angleIndex), i.angle));
			if(difNext > difLast)
			{
				--c.turnTime;
			} else {				
				c.angleIndex = nextIndex;
			}
			// Check whether a faster direction still suits
			if(c.angleIndex != startIndex)
			{
				nextIndex = Utils.incByteAngle(c.angleIndex);
				difLast = Math.abs(Utils.getAngleDif(Utils.TABLES.getAngle(nextIndex), i.angle));
				double hitZone = CombatHeuristics.getImprovedHitZone(c.object);			
				double deviationLast = Utils.TABLES.sin(difLast) * i.distance;
				if(deviationLast < hitZone)
				{
					--c.turnTime;
					c.angleIndex = nextIndex;
				}
			}
		}
		// Add internal game latency	
		if(c.turnTime > 0) {
			if(!control.isTurningLeft() && !control.isTurningRight()) {
				++c.turnTime;
			}
		}
	}
	
	public void checkCollisionStatus(TargetConfiguration config, boolean isSmallSaucer)
	{
		// Create a list of all targets in closer vicinity 
		if(config.info.distance - config.object.getRadius() < Utils.VICINITY_RANGE)
		{
			vicinity.add(config.object);
		}
		
		// Create a list of all targets which will collide soon
		config.collisionTime = Utils.willCollideWithin(config.ship, config.object, Utils.EVASION_MIN_TIME);
		if(isSmallSaucer == false)
		{
			if(config.object.getTargetedStatus() == 0 
				&& config.collisionTime < Utils.EVASION_MIN_TIME)				
			{
				if(Math.abs(config.info.angleDif) > Utils.EVASION_MIN_ANGLE_ASTEROID)
				{
					colliding.add(config.object);
					maxEvasionSpeed = Math.max(maxEvasionSpeed, (config.info.angleDif / (double)Utils.EVASION_MIN_ANGLE_ASTEROID) * 8.0);
				}
			}
		}
	}
	
	public void followTarget(CombatState state, TargetObject targetObject, CombatControl control, boolean firstInterpretation)
	{
		// Follow the currently selected target (but allow swapping)
		double bestRating = curTarget.getRating();
		curTarget.reCalculate();
		TargetConfiguration config = new TargetConfiguration();
		Ship ship = state.getShip();
		Saucer saucer = state.getSaucer();
		boolean newTarget = false;
		int asteroidNum = state.getAsteroidNumber();
		
		// Iterate through the asteroids and check for better ones
		for (Asteroid curAsteroid : state.getAsteroids())
		{  
			config.calculate(ship, curAsteroid);
			if(firstInterpretation)
			{
				checkCollisionStatus(config, false);
			}
			if(curAsteroid.getTargetedStatus() == 0) 
			{				
				if(CombatHeuristics.isTargetSwitchReasonable(targetObject, config))
				{					
					config.preWaitTime = CombatHeuristics.getPrecisionDelay(config.info.distance, curAsteroid) + 1;
					if(approximateTarget(config, state, control))
					{	
						config.prioritization = CombatHeuristics.getCollidingPrioritization(config);
						// Markup: "|| curTarget.object.getID() == -1" only needed, because children usage was altered
						if(config.isBetterThan(bestRating) || targetObject.getID() == -1)
						{
							bestRating = config.getRating();	
							curTarget.setConfig(config);
							newTarget = true;						
						}
					}
				}
			}
		}	
		
		// Check if there's a saucer available to swap to
		if(state.isSaucerAlive())
		{
			config.calculate(ship, saucer);
			if(firstInterpretation)
			{
				checkCollisionStatus(config, saucer.getScale() == CombatObject.SMALL_SCALE);
			}
			if(saucer.getTargetedStatus() == 0)
			{
				if(approximateTarget(config, state, control))
				{					
					config.prioritization = CombatHeuristics.getSaucerPrioritization(state, config, saucer, asteroidNum);
					if(config.isBetterThan(bestRating))
					{
						bestRating = config.getRating();
						curTarget.setConfig(config);
						newTarget = true;
					}
				}
			}
		}	
		
		// Target has been swapped
		if(newTarget)
		{
			++CombatStatistics.TARGET_COUNT;
			lastShotTime = Long.MAX_VALUE;
		}
	}
	
	public void handleState(CombatState state, CombatControl control)
	{
		// Handle the currently received combat state
		if(state.isIngame())
		{
			Ship ship = state.getShip();
			updateStatistics(state);			
			colliding.clear();
			vicinity.clear();
			if(ship.isAlive() && !control.isWarping())
			{
				// Detect ship / target loss
				if(isShipLost)
				{
					CombatMessages.verbose(CombatMessages.SHIP_NEW);
					isShipLost = false;
				}
				if(curTarget.object == null || curTarget.object.isAlive() == false)
				{
					forceTargetSwitch = true;
				}
				boolean reinterprete = false;
				
				// Aquire a new target or follow the current one
				if(forceTargetSwitch || CombatHeuristics.isMaxShotCountReached(curTarget.object))					
				{	
					if(CombatHeuristics.willSpawnChildren(curTarget.object, state))
					{
						((Asteroid)curTarget.object).createChild(curTarget.preCalc.distance, (int)Math.round(curTarget.preCalc.distance / Utils.TABLES.getShotSpeed(curTarget.angleIndex)));
					}
					aquireTarget(state, control, true);
					if(curTarget.object != null)
					{
						++CombatStatistics.TARGET_COUNT;
						if(curTarget.object instanceof Saucer)
						{
							CombatMessages.verbose(CombatMessages.SAUCER_FIRE);
						} else if(!state.isSaucerAlive()) {
							CombatMessages.verbose(CombatMessages.ASTEROIDS);
						}	
						lastShotTime = Long.MAX_VALUE;
					}
				} else {
					curTarget.passTime(state.getPacketDif());
					followTarget(state, curTarget.object, control, true);
				}
				
				// Approximate current target and perform appropriate actions
				if(curTarget.object != null)
				{
					if(approximateTarget(curTarget, state, control))
					{
						// Turn to current target
						if(curTarget.turnTime > 1)
						{
							if(curTarget.preCalc.angleDif > 0) {
								control.turnLeft();
							} else {
								control.turnRight();
							}
						// Fire on current target
						} else if(ship.isTorpedoReady() && curTarget.isReadyToFire(state)){	
							if(!control.willFire() && !control.isFiring() && CombatHeuristics.isTargetHittable(curTarget, curTarget.preCalc))
							{
								control.fire();
								lastShotTime = CombatStatistics.TIME;
								state.fireFlag(curTarget.object, 2 + (int)Math.round(curTarget.preCalc.distance / Utils.TABLES.getShotSpeed(curTarget.angleIndex)));
								if(CombatHeuristics.isMaxShotCountReached(curTarget.object))
								{
									reinterprete = true;
								}
							}
						} else if(CombatStatistics.TIME - lastShotTime > 3) {
							forceTargetSwitch = true;
							reinterprete = true;
						}
						if(CombatStatistics.TIME - lastShotTime > 6) {
							forceTargetSwitch = true;
							reinterprete = true;
						}
					} else {
						forceTargetSwitch = true;
						reinterprete = true;
					}
					
					// Targeting sequence finished, reinterprete state again
					if(reinterprete)
					{
						reinterpretState(state, control);
					}
				} else {
					// Turn into direction of spawning Asteroids
					prepareForNextSector(ship, control);
				}
				
				// Check if hyperspace is needed
				if(checkEmergencyStatus(state, control))
				{
					if(!control.isWarping())
					{
						CombatMessages.verbose(CombatMessages.EMERGENCY);
						control.hyperspace();
						forceTargetSwitch = true;
						++CombatStatistics.ESCAPES;
					}
				}
				
				// Apply movement if possible
				if(state.isMovementPhase())
				{
					move(state, control);
				}
			} else {	
				// Ship is in hyperspace or dead
				forceTargetSwitch = true;
				if(!isShipLost && ship.isLost())
				{
					CombatMessages.verbose(CombatMessages.SHIP_LOST);					
					isShipLost = true;
					++CombatStatistics.DEATHS;
					CombatStatistics.exaggerateSaucerDelay();
				}
				CombatStatistics.increaseSaucerDelay();
				++CombatStatistics.IDLE_TIME;
			}
			
			// Automatic stop
			if(isStoppingAutonomous && CombatStatistics.TIME > Utils.TIME_LIMIT - 1)
			{	
				setActivated(false);
			}	
		} else if(isStartingAutonomous) {
			
			// Automatic start
			if(!control.isStarting())
			{
				control.start();
			}
		}	
	}
	
	public synchronized boolean isActivated(){
		return isActivated;
	}
	
	public boolean checkEmergencyStatus(CombatState state, CombatControl control)
	{
		// Iterate through the shots and add them to colliding list / vicinity
		Ship ship = state.getShip();
		for(Shot curShot : state.getShots())
		{
			if(curShot.getOwner() != ship)
			{
				int difX = Utils.getXDif(ship.getPosX(), curShot.getPosX());
				int difY = Utils.getYDif(ship.getPosY(), curShot.getPosY());
				double distance = Utils.TABLES.dist(difX, difY);
				if(distance - curShot.getRadius() < Utils.VICINITY_RANGE)
				{
					CombatMessages.verbose(CombatMessages.SAUCER_TAKE_FIRE);
					vicinity.add(curShot);
				}
				double angle = Utils.TABLES.atan2(difY, difX);
				double angleDif = Math.abs(Utils.getAngleDif(ship.getMovementAngle(), angle));
				int collisionTime = Utils.willCollideWithin(ship, curShot, Utils.EVASION_MIN_TIME);
				if(collisionTime < Utils.EVASION_MIN_TIME &&
					Math.abs(angleDif - Utils.EVASION_OPTIMUM_ANGLE_SHOT) < Utils.EVASION_MAX_DEVIATION_SHOT)				
				{
					colliding.add(curShot);
				}
			}
		}	
		
		// Search the vicinity for dangerous objects
		for(CombatObject curObject : vicinity)
		{
			if(curObject instanceof TargetObject)
			{
				TargetObject curTargetObject = (TargetObject)curObject;
				if((control.isFiring() && curTargetObject.getTargetedStatus() > 0)
					|| curTargetObject.getFiredUponStatus() > 0)
				{
					continue;
				}
			}
			if(Utils.willCollideIn(ship, curObject, 2)) {
				return true;
			}
		}
		return false;
	}
	
	public void move(CombatState state, CombatControl control)
	{
		Ship ship = state.getShip();
		int shipSpeed = (int)Math.ceil(ship.getSpeed() * 8);
		
		// Movement to screen corner
		if(!control.isAccelerating())
		{
			double minDist = Double.MAX_VALUE;
			int minX = 0;
			int minY = 0;
			for(int i = 0; i < 4; i++)
			{
				int x = Utils.getXDif(ship.getPosX(), strategicPositions[i].x);
				int y = Utils.getYDif(ship.getPosY(), strategicPositions[i].y);
				if(Utils.TABLES.dist(x, y) < minDist)
				{
					minDist = Utils.TABLES.dist(x, y);
					minX = x;
					minY = y;
				}
			}
			double goalAngle = Utils.TABLES.atan2(minY, minX);
			int futureShipAngle = ship.getAngleIndex();
			if(control.isTurningLeft())
			{
				futureShipAngle = Utils.incByteAngle(futureShipAngle);
			} else if(control.isTurningRight())
			{
				futureShipAngle = Utils.decByteAngle(futureShipAngle);
			}
			double angleDif = Math.abs(Utils.getAngleDif(Utils.TABLES.getAngle(futureShipAngle), goalAngle));		
			if(shipSpeed < 8 && minDist > Tables.SPEED_BORDERS[shipSpeed] && angleDif < Math.PI/10.5)
			{
				control.accelerate();
			}
		}
		
		// Movement to avoid collision
		if(!control.isAccelerating())
		{
			colliding.remove(curTarget.object);
			if(shipSpeed < maxEvasionSpeed && !colliding.isEmpty())
			{
				control.accelerate();
			}
			maxEvasionSpeed = 8.0;
		}
		
		// Movement to avoid target delay
		if(!control.isAccelerating())
		{
			if(curTarget.isDifficult)
			{
				if(shipSpeed < 2 && Math.abs(curTarget.preCalc.angleDif) > Math.PI / 4.0)
				{
					control.accelerate();
				}
			}
		}
	}
	
	public void prepareForNextSector(Ship ship, CombatControl control)
	{
		// Turn to face the screen border
		double turnAngle = Utils.TABLES.getPreparationDirection(ship.getPosX(), ship.getPosY());
		double angleDif = Utils.getAngleDif(ship.getMovementAngle(), turnAngle);
		double fuzzy = Math.PI / 20.0;
		if(angleDif > fuzzy)
		{
			control.turnLeft();
		} else if(angleDif < -fuzzy)
		{
			control.turnRight();
		}
	}
	
	public void reinterpretState(CombatState state, CombatControl control)
	{
		// Detect target loss
		if(curTarget.object == null || curTarget.object.isAlive() == false)
		{
			forceTargetSwitch = true;
		}
		Ship ship = state.getShip();
		
		// Aquire a new target or follow the current one
		if(forceTargetSwitch || CombatHeuristics.isMaxShotCountReached(curTarget.object))					
		{
			if(CombatHeuristics.willSpawnChildren(curTarget.object, state))
			{
				((Asteroid)curTarget.object).createChild(curTarget.preCalc.distance, (int)Math.round(curTarget.preCalc.distance / Utils.TABLES.getShotSpeed(curTarget.angleIndex)));
			}
			aquireTarget(state, control, false);
			if(curTarget.object != null)
			{
				++CombatStatistics.TARGET_COUNT;			
				lastShotTime = Long.MAX_VALUE;
			}
		} else {
			followTarget(state, curTarget.object, control, false);
		}	
		
		// Approximate current target and perform appropriate actions
		if(curTarget.object != null)
		{				
			if(approximateTarget(curTarget, state, control))
			{
				// Turn to current target
				if(curTarget.turnTime > 1)
				{
					if(curTarget.preCalc.angleDif > 0) {
						control.turnLeft();
					} else {
						control.turnRight();
					}
				// Fire on current target
				} else if(ship.isTorpedoReady() && curTarget.isReadyToFire(state)){	
					if(!control.willFire() && !control.isFiring() && CombatHeuristics.isTargetHittable(curTarget, curTarget.preCalc))
					{					
						control.fire();
						lastShotTime = CombatStatistics.TIME;
						state.fireFlag(curTarget.object, 2 + (int)Math.round(curTarget.preCalc.distance / Utils.TABLES.getShotSpeed(curTarget.angleIndex)));
					}
				}  else if(CombatStatistics.TIME - lastShotTime > 3) {
					forceTargetSwitch = true;
				}
				if(CombatStatistics.TIME - lastShotTime > 6) {
					forceTargetSwitch = true;
				}
			} else {
				forceTargetSwitch = true;
			}
		} else {
			prepareForNextSector(ship, control);
		}
	}
	
	public void reset()
	{
		setActivated(false);
	}
	
	public synchronized void setActivated(boolean activated)
	{
		if(activated != isActivated)
		{	
			isActivated = activated;
			mainFrame.setIgnoreRepaint(isActivated);
			mainPanel.controlPanel.updateButtons(activated);
			connection.updateStatus();				
			if(activated)
			{	
				isDataDirty = true;
				lastShotTime = Long.MAX_VALUE;
				isShipLost = false;
				maxEvasionSpeed = 8.0;
				forceTargetSwitch = true;
				curTarget.object = null;
				CombatStatistics.reset();
				CombatMessages.reset();
				mainPanel.combatPanel.resetCombatEventList();
				updateGUIThread = new Thread(updateGUICycle);
				updateGUIThread.setName("Time Thread");
				updateGUIThread.setPriority(Thread.MIN_PRIORITY);
				updateGUIThread.start();
			} else {
				try {
					updateGUIThread.interrupt();
					updateGUIThread.join();
				} catch (InterruptedException e1) {
					e1.printStackTrace();
				}
				CombatStatistics.dumpToFile();
				System.gc();
			}			
		}
	}
	
	public void show()
	{
		mainFrame.pack();	
		mainFrame.centerOnScreen();
		mainFrame.setVisible(true);		
	}
	
	public synchronized void toggleActivated()
	{
		setActivated(!isActivated);
	}
	
	public void updateStatistics(CombatState state)
	{
		CombatStatistics.TIME += state.getPacketDif();
		if(isDataDirty)
		{
			mainPanel.statisticsPanel.updateStatus(CombatStatistics.getScore(), CombatStatistics.SPACE_SECTOR, state.getAsteroidNumber());
			mainPanel.statisticsPanel.updateActions(CombatStatistics.SHOTS_FIRED, CombatStatistics.DEATHS, CombatStatistics.ESCAPES, CombatStatistics.getGlobalSaucerCount());
			mainPanel.statisticsPanel.updateRatings(CombatStatistics.getTargetRate(), CombatStatistics.getShootingRate(), CombatStatistics.getHitRate(), CombatStatistics.getActivityRate());
			isDataDirty = false;
		}
	}
	
	public void updateStatus()
	{
		mainPanel.statusPanel.updateTime(CombatStatistics.TIME);
		mainPanel.statusPanel.updateFrame(CombatStatistics.TIME);
		mainPanel.statusPanel.updateLoss(CombatStatistics.PACKET_LOSS);	
	}

	public void watchState(CombatState state, CombatControl control)
	{
		if(isStartingAutonomous) {
			if(!state.isIngame())
			{
				setActivated(true);
			}
		}		
	}
}
