package io;

import helpers.CombatStatistics;



public class CombatControl {

	public static final int BUFFER_SIZE = 10;
	
	public static final byte NONE = 0x00;
	
	public static final byte HYPERSPACE = 0x01;
	
	public static final byte FIRE = 0x02;
	
	public static final byte ACCELLERATE = 0x04;
	
	public static final byte RIGHT = 0x08;
	
	public static final byte LEFT = 0x10;
	
	public static final byte START = 0x20;
	
	private byte control = NONE;
	
	private byte[] buffer = new byte[BUFFER_SIZE]; 
	
	private int curBuf = -1;
	
	private int packetCount;
	
	private int packetDif = 0;
	
	private int lastPacketDif = 0;
	
	public CombatControl()
	{
		reset();
	}
	
	public void accelerate()
	{
		control |= ACCELLERATE;
	}
	
	public void fire()
	{
		control |= FIRE;
	}
	
	byte getControl()
	{
		return control;
	}
	
	byte getPacketCount()
	{
		return (byte)packetCount;
	}
	
	public boolean hasAccelerated()
	{
		int i = curBuf - (lastPacketDif + 1);
		while(i < 0)
		{
			i += BUFFER_SIZE;
		}
		return (buffer[i] & ACCELLERATE) == ACCELLERATE;
	}
	
	public boolean hasFired()
	{
		int i = curBuf - (lastPacketDif + 1);
		while(i < 0)
		{
			i += BUFFER_SIZE;
		}
		return (buffer[i] & FIRE) == FIRE;
	}
	
	public boolean hasTurnedLeft()
	{
		int i = curBuf - (lastPacketDif + 1);
		while(i < 0)
		{
			i += BUFFER_SIZE;
		}
		return (buffer[i] & LEFT) == LEFT;
	}
	
	public boolean hasTurnedRight()
	{
		int i = curBuf - (lastPacketDif + 1);
		while(i < 0)
		{
			i += BUFFER_SIZE;
		}
		return (buffer[i] & RIGHT) == RIGHT;
	}
	
	public boolean isAccelerating()
	{
		int i = curBuf - packetDif;
		while(i < 0)
		{
			i += BUFFER_SIZE;
		}
		return (buffer[i] & ACCELLERATE) == ACCELLERATE;
	}
	
	public boolean isFiring()
	{
		int i = curBuf - packetDif;
		while(i < 0)
		{
			i += BUFFER_SIZE;
		}
		return (buffer[i] & FIRE) == FIRE;
	}
	
	public boolean isStarting()
	{
		int i = curBuf - packetDif;
		while(i < 0)
		{
			i += BUFFER_SIZE;
		}
		return (buffer[i] & START) == START;
	}
	
	public boolean isTurningLeft()
	{
		int i = curBuf - packetDif;
		while(i < 0)
		{
			i += BUFFER_SIZE;
		}
		return (buffer[i] & LEFT) == LEFT;
	}
	
	public boolean isTurningRight()
	{
		int i = curBuf - packetDif;
		while(i < 0)
		{
			i += BUFFER_SIZE;
		}
		return (buffer[i] & RIGHT) == RIGHT;
	}
	
	public boolean isWarping()
	{
		int i = curBuf - packetDif;
		while(i < 0)
		{
			i += BUFFER_SIZE;
		}
		return (buffer[i] & HYPERSPACE) == HYPERSPACE;
	}
	
	void onApproval(int packetCompare)
	{		
		if(packetCount < packetCompare)
		{
			packetCompare -= 256;
		}
		lastPacketDif = packetDif;
		packetDif = packetCount - packetCompare;
		if(packetDif > 0)
		{
			CombatStatistics.PACKET_LOSS += packetDif;
		}
	}
	
	void onSend()
	{
		if(++curBuf == BUFFER_SIZE){
			curBuf = 0;
		}
		buffer[curBuf] = control;
		control = NONE;
		packetCount = (packetCount + 1) % 256;
	}
	
	void reset()
	{
		for(int i = 0; i < 10; i++)
		{
			buffer[i] = NONE;
		}
		packetCount = 0;
		packetDif = 0;
	}
	
	void setControl(byte control)
	{
		this.control = control;
	}
	
	public void start()
	{
		control  |= START;
	}
	
	public void turnLeft()
	{
		control |= LEFT;
	}
	
	public void turnRight()
	{
		control  |= RIGHT;
	}
	
	public void hyperspace()
	{
		control |= HYPERSPACE;
	}
	
	public boolean willAccelerate()
	{
		return (control & ACCELLERATE) == ACCELLERATE;
	}
	
	public boolean willFire()
	{
		return (control & FIRE) == FIRE;
	}
	
	public boolean willTurnLeft()
	{
		return (control & LEFT) == LEFT;
	}
	
	public boolean willTurnRight()
	{
		return (control & RIGHT) == RIGHT;
	}
}
